/* This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this software; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 59 Temple Place, Suite 330,
 * Boston, MA 02111-1307 USA
 */
/*
 * File: guile/mysql.c
 * Usage: implementation of MySQL routines
 *
 * Copyright (C) 1996-98 all source by Hal Roberts
 */

#include <time.h>
#include <string.h>

#include <guile/gh.h>

#ifdef GH_SQL_MYSQL
#include <mysql/mysql.h>
#elif GH_SQL_MSQL
#include <msql.h>
#endif

#include "sql_imp.h"

#ifdef GH_SQL_MSQL
int gh_sql_msql_affected_rows;
#endif

int gh_sql_imp_connect(void *db, char *host, char *name, char *pass) {

#ifdef GH_SQL_MYSQL
  return (int)mysql_connect((MYSQL *)db, host, name, pass);
#elif defined GH_SQL_MSQL
  *((int *)db) = msqlConnect(host);
  return (*((int *)db) >= 0);  
#endif
}

char *gh_sql_imp_error(void *db) {

#ifdef GH_SQL_MYSQL
  return mysql_error((MYSQL *)db);
#elif defined GH_SQL_MSQL
  return msqlErrMsg;
#endif
}

int gh_sql_imp_select_db(void *db, char *db_name) {

#ifdef GH_SQL_MYSQL
  return !mysql_select_db((MYSQL *)db, db_name);
#elif defined GH_SQL_MSQL
  return (msqlSelectDB(*((int *)db), db_name) >= 0);
#endif
}

void gh_sql_imp_close(void *db) {

#ifdef GH_SQL_MYSQL
  mysql_close((MYSQL *)db);
#elif defined GH_SQL_MSQL
  msqlClose(*((int *)db));
#endif
}

int gh_sql_imp_query(void *db, char *query) {

#ifdef GH_SQL_MYSQL
  return !mysql_query((MYSQL *)db, query);
#elif defined GH_SQL_MSQL
  gh_sql_msql_affected_rows = msqlQuery(*((int *)db), query);;

  return (gh_sql_msql_affected_rows >= 0);
#endif
}

void *gh_sql_imp_store_result(void *db) {

#ifdef GH_SQL_MYSQL
  return (void *)mysql_store_result((MYSQL *)db);
#elif defined GH_SQL_MSQL
  return (void *)msqlStoreResult();
#endif
}

int gh_sql_imp_num_fields(void *res) {

#ifdef GH_SQL_MYSQL
  return mysql_num_fields((MYSQL_RES *)res);
#elif defined GH_SQL_MSQL
  return msqlNumFields(((m_result *)res));
#endif
}

int gh_sql_imp_num_rows(void *res) {

#ifdef GH_SQL_MYSQL
  return mysql_num_rows((MYSQL_RES *)res);
#elif defined GH_SQL_MSQL
  return msqlNumRows(((m_result *)res));
#endif
}

char *gh_sql_imp_fetch_field_name(void *res) {

#ifdef GH_SQL_MYSQL
  return (mysql_fetch_field((MYSQL_RES *)res))->name;
#elif defined GH_SQL_MSQL
  return (msqlFetchField((m_result *)res))->name;
#endif
}

char **gh_sql_imp_fetch_row(void *res) {
  
#ifdef GH_SQL_MYSQL
  return (char **)mysql_fetch_row((MYSQL_RES *)res);
#elif defined GH_SQL_MSQL
  return (char **)msqlFetchRow((m_result *)res);
#endif
}

void gh_sql_imp_free_result(void *res) {

#ifdef GH_SQL_MYSQL
  mysql_free_result((MYSQL_RES *)res);
#elif defined GH_SQL_MSQL
  msqlFreeResult((m_result *)res);
#endif
}

int gh_sql_imp_affected_rows(void *db) {

#ifdef GH_SQL_MYSQL
  return mysql_affected_rows((MYSQL *)db);
#elif defined GH_SQL_MSQL
  return gh_sql_msql_affected_rows;
#endif
}

/* return a scm list representing a date / time */ 
SCM gh_sql_time_list(int year, int month, int day, 
			   int hour, int min, int sec) {
  
  SCM gh_time;
  
  gh_time = gh_cons(gh_int2scm(sec), SCM_EOL);
  gh_time = gh_cons(gh_int2scm(min), gh_time);	
  gh_time = gh_cons(gh_int2scm(hour), gh_time);
  gh_time = gh_cons(gh_int2scm(day), gh_time);
  gh_time = gh_cons(gh_int2scm(month), gh_time);
  gh_time = gh_cons(gh_int2scm(year), gh_time);
  
  return gh_time;
}

#ifdef GH_SQL_MYSQL

/* return either a scm string, a scm number, or a scm null depending
 * on the field type of the data at the given position in the given
 * row */

SCM gh_sql_mysql_sql2scm(MYSQL_RES *res, char **row, int pos) {

  struct tm time;
  
  char *s;
  char buf[32];
  
  /* for some reason, the time #include aboveis not declaring
     this function */
  extern char *strptime();
  
  /* if it's a null value, return a scm null */
  if (!row[pos])
    return SCM_EOL;

  /* set the cursor to the correct field */
  mysql_field_seek((MYSQL_RES *)res, pos);
  
  /* else, figure out what scm type it should be */
  switch ((mysql_fetch_field((MYSQL_RES *)res))->type) {
  case (FIELD_TYPE_LONGLONG):
  case (FIELD_TYPE_LONG):
  case (FIELD_TYPE_DOUBLE):
  case (FIELD_TYPE_FLOAT):  
    return gh_long2scm(atol(row[pos]));
    break;
  case (FIELD_TYPE_DECIMAL): 
  case (FIELD_TYPE_SHORT):  
  case (FIELD_TYPE_INT24):
    return gh_int2scm(atol(row[pos]));
    break;
  case (FIELD_TYPE_TINY_BLOB):
  case (FIELD_TYPE_MEDIUM_BLOB):
  case (FIELD_TYPE_LONG_BLOB):
  case (FIELD_TYPE_BLOB):
  case (FIELD_TYPE_VAR_STRING):
  case (FIELD_TYPE_STRING):
  case (FIELD_TYPE_CHAR):
    return gh_str02scm(row[pos]);
    break;
  case (FIELD_TYPE_NULL):   
    return SCM_EOL;
    break;
  case (FIELD_TYPE_DATE): /* the following time fields use strptime to
			   * read the ascii time into a struct tm
			   * structure and then make list
			   * representations of the times in scm */
    strptime(row[pos], "%Y-%m-%d", &time);
    return gh_sql_time_list(time.tm_year, time.tm_mon + 1, time.tm_mday, 
			      -1, -1, -1);
    break;
  case (FIELD_TYPE_TIME):
    strptime(row[pos], "%T", &time);
    return gh_sql_time_list(-1, -1, -1, time.tm_hour, time.tm_min, 
			      time.tm_sec);
    break;
  case (FIELD_TYPE_DATETIME):
    strptime(row[pos], "%Y-%m-%d %T", &time);
    return gh_sql_time_list(time.tm_year, time.tm_mon + 1, time.tm_mday,
			      time.tm_hour, time.tm_min, time.tm_sec);
    break;
  case (FIELD_TYPE_TIMESTAMP):
    /* strptime doesn't like the date string with no breaks, so let's
     * make it look like the datetime string */
    s = row[pos];
    snprintf(buf, 32, "%.4s-%.2s-%.2s %.2s:%.2s:%.2s", s, s + 4, s + 6,
	     s + 8, s + 10, s + 12);
    strptime(buf, "%Y-%m-%d %T", &time);
    return gh_sql_time_list(time.tm_year, time.tm_mon + 1, time.tm_mday,
			      time.tm_hour, time.tm_min, time.tm_sec);
    break;
  default:
    gh_sql_error("gh_sql2scm: Unknown field type");
    return SCM_BOOL_F;
    break;
  }
}

#elif defined GH_SQL_MSQL

/* return either a scm string, a scm number, or a scm null depending
 * on the field type of the data at the given position in the given
 * row */
SCM gh_sql_msql_sql2scm(m_result *res, char **row, int pos) {

  struct tm time;
  
  /* for some reason, the time #include aboveis not declaring
     this function */
  extern char *strptime();

  /* if it's a null value, return a scm null */
  if (!row[pos])
    return SCM_EOL;

  /* set the cursor to the correct field */
  msqlFieldSeek((m_result *)res, pos);
  
  /* else, figure out what scm type it should be */
  switch ((msqlFetchField((m_result *)res))->type) {
  case(UINT_TYPE):
  case(INT_TYPE):
    return gh_int2scm(atoi(row[pos]));
    break;
  case(IDENT_TYPE):	/* FIXME - What's an ident type ? */
  case(TEXT_TYPE):
  case(CHAR_TYPE):
    return gh_str02scm(row[pos]);
    break;
  case (DATE_TYPE):	  /* the following time fields use strptime to
			   * read the ascii time into a struct tm
			   * structure and then make list
			   * representations of the times in scm */
    if (strlen(row[pos])) {
      strptime(row[pos], "%d-%b-%Y", &time);
      return gh_sql_time_list(time.tm_year, time.tm_mon + 1, time.tm_mday, 
			      -1, -1, -1);
    } else
      return gh_sql_time_list(-1, -1, -1, -1, -1, -1);      
    break;
  case (TIME_TYPE):
    if (strlen(row[pos])) {
      strptime(row[pos], "%T", &time);
      return gh_sql_time_list(-1, -1, -1, time.tm_hour, time.tm_min, 
			      time.tm_sec);
    } else
      return gh_sql_time_list(-1, -1, -1, -1, -1, -1);      
    break;
  case(MONEY_TYPE):
  case(REAL_TYPE):
    return gh_long2scm(atol(row[pos]));
    break;
  case(NULL_TYPE):
    return SCM_EOL;
    break;
  default:
    gh_sql_error("gh_sql2scm: Unknown field type");
    return SCM_BOOL_F;
    break;
  }
}
#endif 

SCM gh_sql_imp_sql2scm(void *res, char **data, int pos) {

#ifdef GH_SQL_MYSQL
  return gh_sql_mysql_sql2scm((MYSQL_RES *)res, data, pos);
#elif defined GH_SQL_MSQL
  return gh_sql_msql_sql2scm((m_result *)res, data, pos);
#endif
}

void *gh_sql_alloc_db(void) {

#ifdef GH_SQL_MYSQL
  return malloc(sizeof (MYSQL));
#elif defined GH_SQL_MSQL
  return malloc(sizeof (int));
#endif
}

			     
