indexing

	description: "Provides an easy to access interface to the MySQL%
	             %database"
	date: "1999/06/00"
	copyright: "Copyright 1999 Michael Ravits"
	license: "Eiffel Forum Freeware License, version 1"
	
class SQL_DATABASE

inherit
	MEMORY
		export
			{NONE} all
		redefine
			dispose
		end
	EXCEPTIONS
	STRING_HELPER

creation
	make

feature  -- Initialization

	make is
			-- Initialize
		do
			mysql_pointer := mb_get_new_mysql_pointer
			mysql_result_pointer := mb_get_new_mysql_result_pointer
			mb_init( $mysql_pointer, $mysql_result_pointer )
		end

feature  -- Access

	row_count : INTEGER is
			-- Number of rows retrived
		require
			connected: is_connected
		do
			Result := mb_row_count( $mysql_result_pointer )
		end
	
	field_count : INTEGER is
			-- Number of fields retrived
		require
			connected: is_connected
		do
			Result := mb_field_count( $mysql_result_pointer )
		end
	
	field_name( n : INTEGER ) : STRING is
			-- Name of field number `n'
		require
			connected: is_connected
			in_range: n > 0 and n < (field_count + 1)
		do
			Result := mb_field_name( $mysql_result_pointer, n - 1 )
		end

	field_size( n : INTEGER ) : INTEGER is
			-- Size of field number `n'
		require
			connected: is_connected
			in_range: n > 0 and n < (field_count + 1)
		do
			Result := mb_field_size( $mysql_result_pointer, n - 1 )
		end

	get_field : STRING is
			-- Get selected field
		require
			connected: is_connected
		do
			if( current_row_number /= previous_row_number or current_row = default_pointer ) then
				mb_select_row( $mysql_result_pointer, current_row_number )
				current_row := mb_get_row( $mysql_result_pointer )
			end

			Result := mb_get_field( current_row, current_field_number )
				
			previous_row_number := current_row_number
		end
	
	error : STRING is
			-- Last SQL server error message
		do
			Result := mb_get_mysql_error( $mysql_pointer )
		end

	last_exception : STRING 
			-- Last raised eception name

feature  -- Status report

	is_connected : BOOLEAN
			-- Is connected to database?

feature  -- Element change

	set_query( q : STRING ) is
			-- Set query to be executed
		do
			query := q
		end

	set_host( h : STRING ) is
			-- Set host of database server
		do
			host := h
		end
	
	set_username( u : STRING ) is
			-- Set login name for database server
		do
			username := u
		end

	set_password( p : STRING ) is
			-- Set password for database server
		do
			password := p
		end
	
	set_database_name( n : STRING ) is
			-- Set database name
		do
			database_name := n
		end
	
	set_host_username_and_password( h, u, pas : STRING ) is
			-- Set host, login name and password
			-- of database server
		do
			host := h
			username := u
			password := pas
		end

	select_row( i : INTEGER ) is
			-- Select row number `i'
		require
			in_range: i > 0 and i < (row_count + 1)
		do
			current_row_number := i - 1
		end

	select_field( i : INTEGER ) is
			-- Select field number `i'
		require
			in_range: i > 0 and i < (field_count + 1)
		do
			current_field_number := i - 1
		end

	select_row_and_field( r, f : INTEGER ) is
			-- Select row number `r' and field number `f'
		do
			select_row( r )
			select_field( f )
		end
	

feature  -- Basic operations

	connect is
			-- Connect to database
		require
			not_connected: not is_connected
		local
			h : POINTER
			u : POINTER
			p : POINTER
		do
			if host /= Void and host /= "" then
				h := string_to_pointer( host )
			end
			
			if username /= Void and username /= "" then
				u := string_to_pointer( username )
			end
			
			if password /= Void and password /= "" then
				p := string_to_pointer( password )
			end
			
			if mb_connect( $mysql_pointer, h, u, p ) = true then
				is_connected := true
			else
				set_last_exception( "Could not connect to SQL server:" )
				raise( last_exception )
			end

		ensure
			connected:  is_connected
		end


	disconnect is
			-- Disconnect from database
		require
			connected: is_connected			
		do
			mb_disconnect( $mysql_pointer )
			is_connected := false
			
		ensure
			not_connected: not is_connected
		end
	
	execute is
			-- Execute SQL query
		require
			connected: is_connected
		do
			if mb_execute_query( $mysql_pointer,
					     $mysql_result_pointer,
					     string_to_pointer(query) ) = false then
					      
				set_last_exception( "Could not execute SQL query:" )
				raise( last_exception )
			end
		end

	select_database is
			-- Select the database we want to work with
		require
			connected: is_connected
		do
			if mb_select_database( $mysql_pointer,
						string_to_pointer(database_name) ) = false then
						
				set_last_exception( "Could not select database:" )
				raise( last_exception )
			end
		end

feature{NONE}  -- Implementation

	dispose is
			-- Close connections and free externaly allocated memory
			-- before the GC erases this object
		do
			if is_connected then
				disconnect
			end

			mb_free_result_data( $mysql_result_pointer )
			mb_free_mysql_pointer( $mysql_pointer )
		end

	set_last_exception( s : STRING ) is
			-- Set last raised exception name
		do
			last_exception := s
		end


feature{NONE}  -- Implementation, external routines


	mb_get_new_mysql_pointer : POINTER is
			external "C"
		end

	mb_get_new_mysql_result_pointer : POINTER is
			external "C"
		end

	mb_init( mysql, mysql_result : POINTER ) is
			external "C"
		end

	mb_connect( mysql, h, u, p : POINTER ) : BOOLEAN is
			external "C"
		end
	
	mb_disconnect( mysql : POINTER ) is
			external "C"
		end

	mb_select_database( mysql, d : POINTER ) : BOOLEAN is
			external "C"
		end

	mb_free_result_data( mysql_result : POINTER ) is
			external "C"
		end

	mb_free_mysql_pointer( mysql : POINTER ) is
			external "C" 
		end
	
	mb_execute_query( mysql, mysql_result, q : POINTER ) : BOOLEAN is
			external "C"
		end

	mb_row_count( mysql_result : POINTER ) : INTEGER is
			external "C"
		end

	mb_field_count( mysql_result : POINTER ) : INTEGER is
			external "C"
		end

	mb_field_name( mysql_result : POINTER; n : INTEGER ) : STRING is
			external "C"
		end

	mb_field_size( mysql_result : POINTER; n : INTEGER ) : INTEGER is
			external "C"
		end
		
	mb_select_row( mysql_result : POINTER; n : INTEGER ) is
			external "C"
		end

	mb_get_row( mysql_result : POINTER ) : POINTER is
			external "C"
		end

	mb_get_field( row : POINTER; n : INTEGER ) : STRING is
			external "C"
		end
	
	mb_get_mysql_error( mysql : POINTER ) : STRING is
			external "C"
		end


feature{NONE}  -- Implementation, private objects

	host          : STRING
	username      : STRING
	password      : STRING
	query         : STRING
	database_name : STRING
	
	mysql_pointer        : POINTER
	mysql_result_pointer : POINTER
	
	current_row          : POINTER
	current_row_number   : INTEGER
	previous_row_number  : INTEGER
	current_field_number : INTEGER

end -- class SQL_DATABASE
