/******************************************************************************
	  CCCC	    A	  BBBBB	  L	EEEEE  N     N	EEEEE	TTTTTTT
	 C    C    A A	  B    B  L	E      NN    N	E	   T
	C	  A   A	  B    B  L	E      N N   N	E	   T
	C	 AAAAAAA  BBBBB	  L	EEEEE  N  N  N	EEEEE	   T
	C        A     A  B    B  L	E      N   N N	E	   T
	 C    C  A     A  B    B  L	E      N    NN  E 	   T
	  CCCC	 A     A  BBBBB	  LLLL	EEEEE  N     N	EEEEE	   T
*******************************************************************************

CableNet Source Module:
	Copyright 1994-1995 (C) CableNet Limited. All Rights Reserved.

    Module Name:		$RCSfile: DfltsPaths.c,v $
    Module Description:	Defaults Handling library routines

Description:

    Defaults are stored in sets (directories) and groups (files).
    A default is stored in a group (of defaults) file in the format
    <default identifier>=<value>

    To retreive a default value you call a function passing ;
        - the set name   (or use the default/current set)
	- a group name   (or use the default/current group)
	- an identifier   which identifies the default value
	- a default return value    - returned if nothing is found

    The value returned is either a string or int depending on the
    function call used to access the default

    There are three levels of defaults in ascending order of precedence;
        - global (i.e. system defaults)
	- local  (i.e. local to the running program)
	- user   (specific to the current user)

    Therefore user defaults override all others and are ordinarily
    found in a tree below $(HOME)/defaults

    Local defaults are normally found in  /usr/local/defaults

    global defaults are normally found in /var/spool/defaults

Edit History:

	$Log: DfltsPaths.c,v $
 * Revision 1.3  1997/09/08  17:58:29  damian
 * return original values in get baseset
 *
 * Revision 1.2  1997/08/20  11:50:26  damian
 * last minor mods
 *
 * Revision 1.1  1997/08/19  15:27:32  damian
 * Initial revision
 *
 * Revision 1.7  1997/07/04  16:26:59  damian
 * hold the defaults in memory after reading the whole set
 *
 * Revision 1.6  1996/11/29  11:35:34  damian
 * remove ProgramName()
 *
 * Revision 1.5  1995/12/21  18:04:51  V
 * latest revision
 *
 * Revision 1.4  1995/08/29  09:58:52  V
 * many changes
 *
 * Revision 1.3  1995/02/27  17:40:04  V
 * cleaned up unused warnings
 *
 * Revision 1.2  1995/02/09  19:01:56  V
 * don't make the sets when setting the current set and group
 *
 * Revision 1.1  1995/02/03  18:27:53  V
 * Initial revision
 *


*/

/* RCS identification string (for "what" program) */
static char moduleRCSid[] = "@(#) $Id: DfltsPaths.c,v 1.3 1997/09/08 17:58:29 damian Exp $";

/* must come first header files */
#include "V.h"			/* virtual header file */
#include "Vport.h"		/* port header file */


/*
 *   system header files
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <dirent.h>
#include <sys/param.h>
#include <sys/stat.h>
#include <memory.h>
#include <unistd.h>
#include <syslog.h>

#ifdef Vsunos40
#include "Vansi.h"
#endif

/*
 * third party headers ie, X, informix, ctree
 */

/*
 * project header files
 */
#include "Vlib.h"

#include "dflts.h"

/*
 *  local application/library header files
 */

#include "dfltsP.h"

/* the global default base for all sets */
static char *defaults_base = "/etc/defaults";
static char *local_defaults_base = "/usr/local/defaults";

/*
 *  global or normal set/group paths
 global_base_set is an absolute path ie '/var/spool/defaults'
 global_set is an absolute path  ie '${base_set}/smallads/recover'
 global_group is a path relative to current set ie '${curr_set}/config'
 global_path is an absolute value to the current group
 ie /var/spool/defaults/smallads/recover/config
 */
static char *global_set = NULL;
static char *global_group = NULL;
static char *global_path = NULL;
static char *global_base_set = NULL;

/*
 * local set/group paths
 these can be used to provide an alternative set which can
 overide the values in the global defaults database
 typically you would first of all look for a value in the local
 set, and if you couldn't find it then look in the global set
 */
static char *local_set = NULL;
static char *local_group = NULL;
static char *local_base_set = NULL;

/*
 *   user set/group paths
 */
static char *user_set = NULL;
static char *user_group = NULL;
static char *user_path = NULL;
static char *user_base_set = NULL;

static char *current_path = NULL;


/*++ ***************  function  DfltMakeSet *************** */
/*
   Purpose:        ensure that a set (directory) exists, if it
   doesn't then make it (and recursively make parents)

   Globals changed: none

   Parameters:
   char     *path      == path to directory to ensure exists

   Return Values: int
   0    success
   < 0  error

   Dependencies:
   EndOfString(), CopyString()

 */
/************************************************************************ ++**/

#ifdef __STDC__
int
DfltMakeSet(char *setpath)
#else
int
DfltMakeSet(setpath)
char *setpath;

#endif
{
    char *set;
    char sysbuf[MAXPATHLEN + 128];
    char *t;
    int sysval = 0;
    struct stat st;

/*-- verify arguments */
    if (!setpath)
	return -1;

/*-- make a working copy */
    set = CopyString(setpath);

/*-- find the end of the path and remove any trailing '/' */
    t = EndOfString(set);

    /* remove trailing slashes */
    while (*t == '/')
	*t-- = '\0';

/*-- try to open this set */
    if (stat(set, &st) < 0) {
/*-- no joy */
	if ((t = strrchr(set, '/')) != NULL) {

	    *t = '\0';

/*-- recursivley make previous sets */
	    if (DfltMakeSet(set) < 0) {
		syslog(LOG_ERR, "couldn't make set %s", set);
		Free(set);
		return -1;
	    }
	    *t = '/';
/*-- now make this set */
	    sprintf(sysbuf, "mkdir %s", set);
	    sysval = system(sysbuf);
	    if ((sysval & 0xff) != 0) {
		syslog(LOG_ERR, "system error with %s\n",
		    sysbuf);
		Free(set);
		return -1;
	    }
	}
    } else
/*-- we opened this set ok, now check that it is a directory */
    if ((st.st_mode & S_IFMT) != S_IFDIR) {
/*-- not a directory, return error (-1) */
	Free(set);
	return -1;
    }
/*-- everything ok, return success (0) */
    Free(set);

    return 0;
}


/*++ **************** function DfltsInit() **********************

   Purpose:     Initialise the defaults system

   Globals changed:    base_set, local_base_set, user_base_set

   **** ++ */
#ifdef __STDC__
int
DfltsInit(char *gbase, char *lbase, char *ubase)
#else
int
DfltsInit(gbase,lbase,ubase)
char *gbase, *lbase, *ubase;
#endif
{
    char buf[MAXPATHLEN];
    char *home;

/*-- get the standard defaults base */
    if (gbase == NULL)
	global_base_set = CopyString(defaults_base);
    else
	global_base_set = CopyString(gbase);

/*-- get the local defaults base */
    if (lbase == NULL)
	local_base_set = CopyString(local_defaults_base);
    else
	local_base_set = CopyString(lbase);

/*-- get the standard defaults base */
    if (ubase == NULL) {
	home = getenv("HOME");

	if (home == NULL)
	    return -1;

	if (strcmp(home,"/") == 0)
	    home = "/root";

	sprintf(buf, "%s/defaults", home);

	user_base_set = CopyString(buf);
    } else
	user_base_set = CopyString(ubase);

    return 0;
}


/*++ ****************  function DfltGetBaseSet ******************** */
/*
   Purpose:        determine the current base_set

   Parameters:
   int      dflt_set   == which set to use

   Return Values:
   char        *base       == current base set
 */
/*********************************************************************** ++*/

#ifdef __STDC__
char *
DfltGetBaseSet(int dflt_set)
#else
char *
DfltGetBaseSet(dflt_set)
int dflt_set;

#endif
{

    switch (dflt_set) {
    case GLOBAL_DFLT_SET:
	return defaults_base;
	break;
    case LOCAL_DFLT_SET:
	return local_defaults_base;
	break;
    case USER_DFLT_SET:
	return user_base_set;
	break;
    case ANY_DFLT_SET:
    default:
	return NULL;
	break;
    }

    return NULL;
}

/*++ ****************  function DfltGetCurrentPath ******************** */
/*
   Purpose:     determine the current absolute set/group path

   Parameters:
   int      dflt_set    == which set to use

   Return Values:
   char        *path       ==  current base path
 */
/*********************************************************************** ++*/

#ifdef __STDC__
char *
DfltGetCurrentPath(DfltsInfo * di)
#else
char *
DfltGetCurrentPath(di)
DfltsInfo *di;

#endif
{
    static char buf[MAXPATHLEN];
    char *base;

    if (current_path)
	return current_path;

    switch (di->base) {
    case GLOBAL_DFLT_SET:
    case LOCAL_DFLT_SET:
    case USER_DFLT_SET:
	if ((base = DfltGetBaseSet(di->base)) == NULL) {
	    DfltsInit(NULL,NULL,NULL);
	    base = DfltGetBaseSet(di->base);
	}
	if (di->set && di->group)
	    sprintf(buf, "%s/%s/%s", base, di->set, di->group);
	else if (di->set)
	    sprintf(buf, "%s/%s", base, di->set);
	else if (di->group)
	    sprintf(buf, "%s/%s", base, di->group);
	else
	    return NULL;

	break;

    case ANY_DFLT_SET:
    default:
	return NULL;
	break;
    }

    return buf;

}


/*++ ***************  function DfltSetCurrentPath *************** */
/*
   Purpose:        set the current path to an absolute value

   Globals changed: current_path

   Parameters:
   char     *path      ==   absolute path to set

   Return Values: void

   Dependencies:
   CopyString()

 */
/************************************************************************ ++**/
#ifdef __STDC__
void
DfltSetCurrentPath(DfltsInfo * di)
#else
void
DfltSetCurrentPath(di)
DfltsInfo *di;

#endif
{
    /* set an absolute path */
    char buf[MAXPATHLEN];

/*-- verify arguments */
    if (!di)
	return;

    if (DfltGetBaseSet(di->base) == NULL) {
	DfltsInit(NULL,NULL,NULL);
    }
/*-- make a temporary copy */
    if (di->set && di->group)
	sprintf(buf, "%s/%s/%s", DfltGetBaseSet(di->base),
	    di->set, di->group);
    else if (di->set)
	sprintf(buf, "%s/%s", DfltGetBaseSet(di->base),
	    di->set);
    else if (di->group)
	sprintf(buf, "%s/%s", DfltGetBaseSet(di->base),
	    di->group);
    else
	return;

    if (current_path)
	Free(current_path);

    current_path = CopyString(buf);

}

#ifdef __STDC__
void
DfltResetPaths(DfltsInfo * di)
#else
void
DfltResetPaths(di)
DfltsInfo *di;

#endif
{
    char buf[MAXPATHLEN];

    if (di->base != GLOBAL_DFLT_SET &&
	di->base != LOCAL_DFLT_SET &&
	di->base != USER_DFLT_SET &&
	di->base != ANY_DFLT_SET)
	return;

    sprintf(buf, "%s/%s", DfltGetCurrentSet(di->base),
	DfltGetCurrentGroup(di->base));

    switch (di->base) {
    case GLOBAL_DFLT_SET:
/*-- free any previous value */
	if (global_path)
	    Free(global_path);

/*-- set the new value */
	global_path = CopyString(buf);

	break;
    case LOCAL_DFLT_SET:
/*-- free any previous value */
	if (global_path)
	    Free(global_path);

/*-- set the new value */
	global_path = CopyString(buf);
	break;
    case USER_DFLT_SET:
/*-- free any previous value */
	if (user_path)
	    Free(user_path);

/*-- set the new value */
	user_path = CopyString(buf);
	break;
    case ANY_DFLT_SET:
	di->base = GLOBAL_DFLT_SET;
	DfltResetPaths(di);
	di->base = LOCAL_DFLT_SET;
	DfltResetPaths(di);
	di->base = USER_DFLT_SET;
	DfltResetPaths(di);
	di->base = ANY_DFLT_SET;
	break;
    default:
	return;
    }

}

/*++ ****************  function  DfltGetCurrentSet ******************** */
/*
   Purpose:    determine the current set

   Parameters: void

   Return Values:  char *
   group       == current set
 */
/*********************************************************************** ++*/

char *
DfltGetCurrentSet(int dflt_set)
{
    switch (dflt_set) {
    case GLOBAL_DFLT_SET:
	return global_set;
	break;
    case LOCAL_DFLT_SET:
	return local_set;
	break;
    case USER_DFLT_SET:
	return user_set;
	break;
    case ANY_DFLT_SET:
    default:
	return NULL;
	break;
    }

    return NULL;
}

/*++ ***************  function DfltSetCurrentSet *************** */
/*
   Purpose:   set the current set to an absolute value

   Globals changed:  curr_set

   Parameters:
   char     *path      == new current set from base set

   Return Values: int
   0       success
   < 0     error

   Dependencies:
   CopyString(), DfltGetBaseSet()

 */
/************************************************************************ ++**/

#ifdef __STDC__
int
DfltSetCurrentSet(DfltsInfo * di)
#else
int
DfltSetCurrentSet(di)
DfltsInfo *di;

#endif
{

  /*-- no need to verify arguments as calling this with NULL resets the
      current set to it's default value */
    if (!di && !di->set)
	return -1;

    switch (di->base) {
    case GLOBAL_DFLT_SET:
/*-- free any previous value */
	if (global_set)
	    Free(global_set);

	global_set = CopyString(di->set);
	break;
    case LOCAL_DFLT_SET:
	if (local_set)
	    Free(local_set);

	local_set = CopyString(di->set);
	break;
    case USER_DFLT_SET:
	if (user_set)
	    Free(user_set);

	user_set = CopyString(di->set);
	break;
    case ANY_DFLT_SET:
	if (di->set) {
	    di->base = GLOBAL_DFLT_SET;
	    DfltSetCurrentSet(di);
	    di->base = LOCAL_DFLT_SET;
	    DfltSetCurrentSet(di);
	    di->base = USER_DFLT_SET;
	    DfltSetCurrentSet(di);
	    di->base = ANY_DFLT_SET;
	} else {
	    if (global_set)
		Free(global_set);

	    if (local_set)
		Free(local_set);

	    if (user_set)
		Free(user_set);

	    DfltsInit(NULL,NULL,NULL);
	}
	break;
    default:
	return -1;
	break;
    }


    return 0;
}

/*++ ***************  function DfltSetCurrentGroup *************** */
/*
   Purpose:        set the current group to some value, also
   update the current path to reflect the change

   Globals changed: curr_group

   Parameters:
   char     *group      == new group
   int      set        == which set to use

   Return Values:int
   0       success
   < 0     error

   Dependencies:
   DfltSetCurrentPath(), DfltGetBaseSet(), CopyString()

 */
/************************************************************************ ++**/

#ifdef __STDC__
int
DfltSetCurrentGroup(DfltsInfo * di)
#else
int
DfltSetCurrentGroup(di)
DfltsInfo *di;

#endif
{

/*-- verify arguments */
    if (!di || !di->group)
	return -1;

    switch (di->base) {
    case GLOBAL_DFLT_SET:
/*-- free previous value */
	if (global_group)
	    Free(global_group);

/*-- copy the new group */
	global_group = CopyString(di->group);

	break;
    case LOCAL_DFLT_SET:
/*-- free previous value */
	if (local_group)
	    Free(local_group);

/*-- copy the new group */
	local_group = CopyString(di->group);

	break;
    case USER_DFLT_SET:
/*-- free previous value */
	if (user_group)
	    Free(user_group);

/*-- copy the new group */
	user_group = CopyString(di->group);

	break;
    case ANY_DFLT_SET:
	di->base = GLOBAL_DFLT_SET;
	DfltSetCurrentGroup(di);
	di->base = LOCAL_DFLT_SET;
	DfltSetCurrentGroup(di);
	di->base = USER_DFLT_SET;
	DfltSetCurrentGroup(di);
	di->base = ANY_DFLT_SET;
	break;
    default:
	return -1;
	break;
    }


    return 0;
}

/*++ ****************  function  DfltGetCurrentGroup ******************** */
/*
   Purpose:    determine the current group

   Parameters: int set

   Return Values:  char *
   group       == current group
 */
/*********************************************************************** ++*/

char *
DfltGetCurrentGroup(int dflt_set)
{
    switch (dflt_set) {
    case GLOBAL_DFLT_SET:
	return global_group;
	break;
    case LOCAL_DFLT_SET:
	return local_group;
	break;
    case USER_DFLT_SET:
	return user_group;
	break;
    case ANY_DFLT_SET:
    default:
	return NULL;
	break;
    }

    return NULL;
}



/*++ ****************  function DfltDeleteGroup ******************** */
/*
   Purpose:         delete an entire group by renaming the group file
   to ${group}.bak

   Parameters:
   char     *group      == group to delete

   Return Values: int
   0        success
   < 0      error

 */
/*********************************************************************** ++*/

#ifdef __STDC__
int
DfltDeleteGroup(char *path)
#else
int
DfltDeleteGroup(path)
char *group;

#endif
{
    char backup[MAXPATHLEN];

/*-- generate group backup name */
    sprintf(backup, "%s.bak", path);

/*-- use system rename function to rename the file */
    rename(path, backup);

    return 0;
}



#ifdef __STDC__
int
DestroySet(DfltsInfo * di)
#else
int
DestroySet(di)
DfltsInfo *di;

#endif
{
    char *setstr;
    char newset[MAXPATHLEN];
    int retval = 0;
    struct stat stbuf;
    char buf[MAXPATHLEN];

    sprintf(buf, "%s/%s", DfltGetBaseSet(di->base),
	di->set);

    if (stat(buf, &stbuf) < 0)
	return 0;

    DfltSetCurrentSet(di);

    setstr = DfltGetSet(di);

    if (setstr) {
	char *t, *next;

	t = setstr;

	if (t)
	    do {
		next = TerminateAtNextChar(t, ",");
		DfltSetCurrentSet(di);

		sprintf(buf, "%s/%s/%s", DfltGetBaseSet(di->base),
		    di->set, t);

		if (stat(buf, &stbuf) < 0) {
		    fprintf(stderr, "can't access set %s\n", buf);
		    return -1;
		}
		if ((stbuf.st_mode & S_IFMT) == S_IFREG) {
		    unlink(buf);
		} else {
		    sprintf(newset, "%s/%s", di->set, t);

		    di->set = newset;
		    if (DestroySet(di) < 0) {
			syslog(LOG_ERR, "error deleting set %s\n",
			    t);
			retval = -1;
		    }
		}

	    }
	    while ((t = next) != NULL);

	Free(setstr);
    } else
	return -1;

    return retval;

}

#ifdef __STDC__
int
DfltDeleteSet(DfltsInfo * di)
#else
int
DfltDeleteSet(di)
DfltsInfo *di;

#endif
{
    char backup[MAXPATHLEN];
    char newbak[MAXPATHLEN];

    sprintf(backup, "%s.bak", DfltGetCurrentSet(di->base));
    sprintf(newbak, "%s/%s", DfltGetBaseSet(di->base),
	DfltGetCurrentSet(di->base));

    /* ensure the backup set does not exist */
    di->set = backup;
    DestroySet(di);

    sprintf(backup, "%s/%s.bak", DfltGetBaseSet(di->base),
	DfltGetCurrentSet(di->base));

    rename(newbak, backup);

    return 0;
}
