/*******************************************************************************
 * Copyright (c) 2005, 2013 Oracle and/or its affiliates. All rights reserved.
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v1.0 and Eclipse Distribution License v. 1.0
 * which accompanies this distribution.
 * The Eclipse Public License is available at http://www.eclipse.org/legal/epl-v10.html
 * and the Eclipse Distribution License is available at
 * http://www.eclipse.org/org/documents/edl-v10.php.
 *
 * Contributors:
 *     Oracle - initial API and implementation
 *
 ******************************************************************************/
package org.eclipse.persistence.tools.utility.iterator;

import java.util.Iterator;
import java.util.NoSuchElementException;
import org.eclipse.persistence.tools.utility.ObjectTools;

/**
 * A <code>SingleElementIterator</code> holds a single element
 * and returns it with the first call to {@link #next()}, at
 * which point it will return <code>false</code> to any subsequent
 * call to {@link #hasNext()}.
 * <p>
 * A <code>SingleElementIterator</code> is equivalent to the
 * {@link Iterator} returned by:
 * 	{@link java.util.Collections#singleton(Object element)}<code>.iterator()</code>
 *
 * @param <E> the type of elements returned by the iterator
 *
 * @see org.eclipse.jpt.common.utility.internal.iterable.SingleElementIterable
 */
public class SingleElementIterator<E>
	implements Iterator<E>
{
	private final E element;
	private boolean done;


	/**
	 * Construct an iterator that returns only the specified element.
	 */
	public SingleElementIterator(E element) {
		super();
		this.element = element;
		this.done = false;
	}

	@Override
	public boolean hasNext() {
		return ! this.done;
	}

	@Override
	public E next() {
		if (this.done) {
			throw new NoSuchElementException();
		}
		this.done = true;
		return this.element;
	}

	@Override
	public void remove() {
		throw new UnsupportedOperationException();
	}

	@Override
	public String toString() {
		return ObjectTools.toString(this, this.element);
	}
}