/*******************************************************************************
 * Copyright (c) 2006, 2012 Oracle and/or its affiliates. All rights reserved.
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v1.0 and Eclipse Distribution License v. 1.0
 * which accompanies this distribution.
 * The Eclipse Public License is available at http://www.eclipse.org/legal/epl-v10.html
 * and the Eclipse Distribution License is available at
 * http://www.eclipse.org/org/documents/edl-v10.php.
 *
 * Contributors:
 *     Oracle - initial API and implementation
 *
 ******************************************************************************/
package org.eclipse.persistence.tools.mapping.orm.dom;

import java.util.ArrayList;
import java.util.List;
import javax.persistence.AccessType;
import org.eclipse.persistence.tools.mapping.orm.ExternalEntityListener;
import org.eclipse.persistence.tools.mapping.orm.ExternalPersistenceUnit;
import org.eclipse.persistence.tools.mapping.orm.ExternalTenantDiscriminatorColumn;
import org.eclipse.persistence.tools.utility.iterable.ListIterable;
import org.eclipse.persistence.tools.utility.iterable.ListListIterable;
import org.w3c.dom.Element;

/**
 * The external form of a persistence unit.
 *
 * @see ORMConfiguration
 *
 * @version 2.5
 * @author Pascal Filion
 */
@SuppressWarnings("nls")
final class PersistenceUnit extends AbstractExternalForm
                            implements ExternalPersistenceUnit {

	/**
	 * The attribute name used to store and retrieve the access property.
	 */
	static final String ACCESS = "access";

	/**
	 * The attribute name used to store and retrieve the cascade-persist property.
	 */
	static final String CASCADE_PERSIST = "cascade-persist";

	/**
	 * The attribute name used to store and retrieve the catalog property.
	 */
	static final String CATALOG = "catalog";

	/**
	 * The element name used to store and retrieve the delimited identifiers flag.
	 */
	static final String DELIMITED_IDENTIFIERS = "delimited-identifiers";

	/**
	 * The element name used to store and retrieve the description.
	 */
	static final String DESCRIPTION = "description";

	/**
	 * The attribute name used to store and retrieve the exclude-default-mappings property.
	 */
	static final String EXCLUDE_DEFAULT_MAPPINGS = "exclude-default-mappings";

	/**
	 * The attribute name used to store and retrieve the persistence-unit-defaults property.
	 */
	static final String PERSISTENCE_UNIT_DEFAULTS = "persistence-unit-defaults";

	/**
	 * The node name used to store and retrieve the {@link Element} encapsulated by this external form.
	 */
	static final String PERSISTENCE_UNIT_METADATA = "persistence-unit-metadata";

	/**
	 * The attribute name used to store and retrieve the schema property.
	 */
	static final String SCHEMA = "schema";

	/**
	 * The attribute name used to store and retrieve the direction property.
	 */
	static final String XML_MAPPING_METADATA_COMPLETE = "xml-mapping-metadata-complete";

	/**
	 * Creates a new <code>PersistenceUnit</code>.
	 *
	 * @param parent The parent of this external form
	 */
	PersistenceUnit(ORMConfiguration parent) {
		super(parent);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ExternalTenantDiscriminatorColumn addDiscriminatorColumn(int index) {
		TenantDiscriminatorColumn column = buildDiscriminatorColumn(index);
		column.addSelf();
		return column;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void addEntityListener(int index, String className) {
		EntityListener entityListener = buildEntityListener(index);
		entityListener.addSelf();
		entityListener.setClassName(className);
	}

	private TenantDiscriminatorColumn buildDiscriminatorColumn(int index) {
		return new TenantDiscriminatorColumn(this, index);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	protected List<String> buildElementNamesOrder() {

		List<String> names = new ArrayList<String>();
		names.add(DESCRIPTION);
		names.add(XML_MAPPING_METADATA_COMPLETE);
		names.add(EXCLUDE_DEFAULT_MAPPINGS);
		names.add(PERSISTENCE_UNIT_DEFAULTS);
		return names;
	}

	private EntityListener buildEntityListener(int index) {
		return new EntityListener(this, index);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ListIterable<ExternalTenantDiscriminatorColumn> discriminatorColumns() {
		int count = discriminatorColumnSize();
		List<ExternalTenantDiscriminatorColumn> columns = new ArrayList<ExternalTenantDiscriminatorColumn>(count);

		for (int index = count; --index >= 0;) {
			ExternalTenantDiscriminatorColumn column = buildDiscriminatorColumn(index);
			columns.add(0, column);
		}

		return new ListListIterable<ExternalTenantDiscriminatorColumn>(columns);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public int discriminatorColumnSize() {

		Element element = getChild(PERSISTENCE_UNIT_DEFAULTS);

		if (element == null) {
			return 0;
		}

		return getChildrenSize(element, TenantDiscriminatorColumn.TENANT_DISCRIMINATOR_COLUMN);
	}

	@Override
	public ExternalTenantDiscriminatorColumn getDiscriminatorColumn(int index) {
		Element puElem = getChild(PERSISTENCE_UNIT_DEFAULTS);

		if (puElem == null) {
			return null;
		}

		Element element = getChild(puElem, TenantDiscriminatorColumn.TENANT_DISCRIMINATOR_COLUMN, index);

		if (element == null) {
			return null;
		}

		return buildDiscriminatorColumn(index);
	}


	/**
	 * {@inheritDoc}
	 */
	@Override
	public ListIterable<ExternalEntityListener> entityListeners() {

		int count = entityListenersSize();
		List<ExternalEntityListener> entityListeners = new ArrayList<ExternalEntityListener>(count);

		for (int index = count; --index >= 0;) {
			ExternalEntityListener entityListener = buildEntityListener(index);
			entityListeners.add(0, entityListener);
		}

		return new ListListIterable<ExternalEntityListener>(entityListeners);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public int entityListenersSize() {

		Element element = getChild(PERSISTENCE_UNIT_DEFAULTS);

		if (element == null) {
			return 0;
		}

		element = getChild(element, EntityListener.ENTITY_LISTENERS);

		if (element == null) {
			return 0;
		}

		return getChildrenSize(element, EntityListener.ENTITY_LISTENER);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public AccessType getDefaultAccessType() {

		Element element = getChild(PERSISTENCE_UNIT_DEFAULTS);

		if (element != null) {
			return getChildEnumNode(element, ACCESS, AccessType.class);
		}

		return null;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getDefaultCatalogName() {

		Element element = getChild(PERSISTENCE_UNIT_DEFAULTS);

		if (element != null) {
			return getChildTextNode(element, CATALOG);
		}

		return null;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getDefaultDescription() {

		Element element = getChild(PERSISTENCE_UNIT_DEFAULTS);

		if (element != null) {
			return getChildTextNode(element, DESCRIPTION);
		}

		return null;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getDefaultSchemaName() {

		Element element = getChild(PERSISTENCE_UNIT_DEFAULTS);

		if (element != null) {
			return getChildTextNode(element, SCHEMA);
		}

		return null;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getDescription() {
		return getChildTextNode(DESCRIPTION);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	protected String getElementName() {
		return PERSISTENCE_UNIT_METADATA;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ExternalEntityListener getEntityListener(int index) {

		Element element = getChild(PERSISTENCE_UNIT_DEFAULTS);

		if (element == null) {
			return null;
		}

		element = getChild(element, EntityListener.ENTITY_LISTENERS);

		if (element == null) {
			return null;
		}

		element = getChild(element, EntityListener.ENTITY_LISTENER, index);

		if (element == null) {
			return null;
		}

		return buildEntityListener(index);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public boolean isDefault() {
		return getAllChildrenSize() == 0;
	}

	@Override
	public boolean isDelimitedIdentifers() {

		Element element = getChild(PERSISTENCE_UNIT_DEFAULTS);

		if (element != null) {
			return hasChild(element, DELIMITED_IDENTIFIERS);
		}
		else {
			return false;
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public boolean isExcludeDefaultMappings() {
		return hasChild(EXCLUDE_DEFAULT_MAPPINGS);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public boolean isMappingMetaDataComplete() {
		return hasChild(XML_MAPPING_METADATA_COMPLETE);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void removeDiscriminatorColumn(int index) {

		TenantDiscriminatorColumn entityListener = buildDiscriminatorColumn(index);
		entityListener.removeSelf();

		Element parentElement = getChild(PERSISTENCE_UNIT_DEFAULTS);

		if ((parentElement != null) && !hasAnyChildren(parentElement)) {
			remove(parentElement);

			if (!hasAnyChildren()) {
				removeSelf();
			}
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void removeEntityListener(int index) {

		EntityListener entityListener = buildEntityListener(index);
		entityListener.removeSelf();

		Element parentElement = getChild(PERSISTENCE_UNIT_DEFAULTS);
		Element element = getChild(parentElement, EntityListener.ENTITY_LISTENERS);

		if ((parentElement != null) && (element != null) && !hasAnyChildren(element)) {
			remove(parentElement, element);
		}

		if ((parentElement != null) && !hasAnyChildren(parentElement)) {
			remove(parentElement);

			if (!hasAnyChildren()) {
				removeSelf();
			}
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setDefaultAccessType(AccessType type) {

		Element element = getChild(PERSISTENCE_UNIT_DEFAULTS);

		if ((element == null) && (type == null)) {
			return;
		}

		if (element == null) {
			element = addChild(PERSISTENCE_UNIT_DEFAULTS);
		}

		updateTextNode(element, ACCESS, type);

		if (!hasAnyChildren(element)) {
			remove(element);

			if (!hasAnyChildren()) {
				removeSelf();
			}
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setDefaultCatalogName(String catalogName) {

		Element element = getChild(PERSISTENCE_UNIT_DEFAULTS);

		if ((element == null) && (catalogName == null)) {
			return;
		}

		if (element == null) {
			element = addChild(PERSISTENCE_UNIT_DEFAULTS);
		}

		updateTextNode(element, CATALOG, catalogName);

		if (!hasAnyChildren(element)) {
			remove(element);

			if (!hasAnyChildren()) {
				removeSelf();
			}
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setDefaultDescription(String description) {

		Element element = getChild(PERSISTENCE_UNIT_DEFAULTS);

		if ((element == null) && (description == null)) {
			return;
		}

		if (element == null) {
			element = addChild(PERSISTENCE_UNIT_DEFAULTS);
		}

		updateTextNode(element, DESCRIPTION, description);

		if (!hasAnyChildren(element)) {
			remove(element);

			if (!hasAnyChildren()) {
				removeSelf();
			}
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setDefaultSchemaName(String schemaName) {

		Element element = getChild(PERSISTENCE_UNIT_DEFAULTS);

		if ((element == null) && (schemaName == null)) {
			return;
		}

		if (element == null) {
			element = addChild(PERSISTENCE_UNIT_DEFAULTS);
		}

		updateTextNode(element, SCHEMA, schemaName);

		if (!hasAnyChildren(element)) {
			remove(element);

			if (!hasAnyChildren()) {
				removeSelf();
			}
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setDelimitedIdentifers(boolean delimitIdentifiers) {

		Element element = getChild(PERSISTENCE_UNIT_DEFAULTS);

		if (delimitIdentifiers) {
			if (element == null) {
				element = addChild(PERSISTENCE_UNIT_DEFAULTS);
			}
			addChild(element, DELIMITED_IDENTIFIERS);
		}
		else {
			if (element != null) {
				removeChild(element, DELIMITED_IDENTIFIERS);

				if (!hasAnyChildren(element)) {
					remove(element);

					if (!hasAnyChildren()) {
						removeSelf();
					}
				}
			}
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setDescription(String description) {

		if (description == null) {
			removeChild(DESCRIPTION);

			if (!hasAnyChildren()) {
				removeSelf();
			}
		}
		else {
			Element element = getChild(DESCRIPTION);

			if (element == null) {
				addChild(DESCRIPTION);
			}

			updateChildTextNode(DESCRIPTION, description);
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setExcludeDefaultMappings(boolean excludeDefaultMappings) {

		if (excludeDefaultMappings) {
			addChild(EXCLUDE_DEFAULT_MAPPINGS);
		}
		else {
			removeChild(EXCLUDE_DEFAULT_MAPPINGS);

			if (!hasAnyChildren()) {
				removeSelf();

				if (!hasAnyChildren()) {
					removeSelf();
				}
			}
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setMappingMetaDataComplete(boolean metaDataComplete) {

		if (metaDataComplete) {
			addChild(XML_MAPPING_METADATA_COMPLETE);
		}
		else {
			removeChild(XML_MAPPING_METADATA_COMPLETE);

			if (!hasAnyChildren()) {
				removeSelf();

				if (!hasAnyChildren()) {
					removeSelf();
				}
			}
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setUsesCascadePersistByDefault(boolean cascadePersistByDefault) {

		Element element = getChild(PERSISTENCE_UNIT_DEFAULTS);

		if ((element == null) && !cascadePersistByDefault) {
			return;
		}

		if (element == null) {
			element = addChild(PERSISTENCE_UNIT_DEFAULTS);
		}

		if (cascadePersistByDefault) {
			addChild(element, CASCADE_PERSIST);
		}
		else {
			removeChild(element, CASCADE_PERSIST);

			if (!hasAnyChildren(element)) {
				remove(element);

				if (!hasAnyChildren()) {
					removeSelf();
				}
			}
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public boolean usesCascadePersistByDefault() {

		Element element = getChild(PERSISTENCE_UNIT_DEFAULTS);

		if (element != null) {
			return hasChild(element, CASCADE_PERSIST);
		}

		return false;
	}
}