/*******************************************************************************
 * Copyright (c) 2011, 2013 Oracle and/or its affiliates. All rights reserved.
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v1.0 and Eclipse Distribution License v. 1.0
 * which accompanies this distribution.
 * The Eclipse Public License is available at http://www.eclipse.org/legal/epl-v10.html
 * and the Eclipse Distribution License is available at
 * http://www.eclipse.org/org/documents/edl-v10.php.
 *
 * Contributors:
 *     Oracle - initial API and implementation
 *
 ******************************************************************************/
package org.eclipse.persistence.tools.utility.collection;

import java.io.Serializable;
import java.util.Collection;
import java.util.Iterator;
import java.util.List;
import java.util.ListIterator;
import org.eclipse.persistence.tools.utility.ArrayTools;
import org.eclipse.persistence.tools.utility.ObjectTools;

/**
 * A <em>read-only</em> repeating element list (i.e. the list contains only
 * one or more references to the same element).
 */
@SuppressWarnings("nls")
public abstract class AbstractRepeatingElementList<E>
	implements List<E>, Serializable
{
	// never negative
	private final int size;

	/**
	 * Construct a <em>read-only</em> list with the specified number of
	 * objects in it.
	 */
	protected AbstractRepeatingElementList(int size) {
		super();
		if (size < 0) {
			throw new IllegalArgumentException("Invalid size: " + size);
		}
		this.size = size;
	}

	@Override
	public boolean add(E o) {
		throw new UnsupportedOperationException();
	}

	@Override
	public void add(int index, E element) {
		throw new UnsupportedOperationException();
	}

	/**
	 * Replicate behavior of
	 * {@link java.util.AbstractCollection#addAll(Collection)}.
	 */
	@Override
	public boolean addAll(Collection<? extends E> c) {
		if (c.isEmpty()) {
			return false;
		}
		throw new UnsupportedOperationException();
	}

	/**
	 * Replicate behavior of
	 * {@link java.util.AbstractList#addAll(int, Collection)}.
	 */
	@Override
	public boolean addAll(int index, Collection<? extends E> c) {
		return this.addAll(c);  // ignore the index
	}

	/**
	 * Replicate behavior of
	 * {@link java.util.AbstractCollection#clear()}.
	 */
	@Override
	public void clear() {
		if (this.size != 0) {
			throw new UnsupportedOperationException();
		}
	}

	@Override
	public boolean contains(Object o) {
		return (this.elementIs(o) && (this.size > 0));
	}

	/**
	 * Replicate behavior of
	 * {@link java.util.AbstractCollection#containsAll(Collection)}.
	 */
	@Override
	public boolean containsAll(Collection<?> c) {
		if (c.isEmpty()) {
			return true;
		}
		if (this.size == 0) {
			return false;
		}
		for (Iterator<?> iterator = c.iterator(); iterator.hasNext(); ) {
			if (this.elementIsNot(iterator.next())) {
				return false;
			}
		}
		return true;
	}

	@Override
	public E get(int index) {
		this.checkIndex(index);
		return this.getElement();
	}

	private void checkIndex(int index) {
		if ((index < 0) || (index >= this.size)) {
			throw new IndexOutOfBoundsException("Index: " + index + ", Size: " + this.size);
		}
	}

	@Override
	public int indexOf(Object o) {
		return (this.elementIs(o) && (this.size > 0)) ? 0 : -1;
	}

	@Override
	public boolean isEmpty() {
		return (this.size == 0);
	}

	@Override
	public Iterator<E> iterator() {
		return this.iterator(this.size);
	}

	protected abstract Iterator<E> iterator(int iteratorSize);

	@Override
	public int lastIndexOf(Object o) {
		return (this.elementIs(o) && (this.size > 0)) ? (this.size - 1) : -1;
	}

	@Override
	public ListIterator<E> listIterator() {
		return this.listIterator_(this.size);
	}

	@Override
	public ListIterator<E> listIterator(int index) {
		return this.listIterator_(this.size - index);
	}

	protected abstract ListIterator<E> listIterator_(int iteratorSize);

	/**
	 * Replicate behavior of
	 * {@link java.util.AbstractCollection#remove(Object)}.
	 */
	@Override
	public boolean remove(Object o) {
		if ((this.size > 0) && this.elementIs(o)) {
			throw new UnsupportedOperationException();
		}
		return false;
	}

	@Override
	public E remove(int index) {
		throw new UnsupportedOperationException();
	}

	/**
	 * Replicate behavior of
	 * {@link java.util.AbstractCollection#removeAll(Collection)}.
	 */
	@Override
	public boolean removeAll(Collection<?> c) {
		if ((this.size > 0) && c.contains(this.getElement())) {
			throw new UnsupportedOperationException();
		}
		return false;
	}

	/**
	 * Replicate behavior of
	 * {@link java.util.AbstractCollection#retainAll(Collection)}.
	 */
	@Override
	public boolean retainAll(Collection<?> c) {
		if ((this.size > 0) && ! c.contains(this.getElement())) {
			throw new UnsupportedOperationException();
		}
		return false;
	}

	@Override
	public E set(int index, E element) {
		throw new UnsupportedOperationException();
	}

	@Override
	public int size() {
		return this.size;
	}

	@Override
	public List<E> subList(int fromIndex, int toIndex) {
		return this.subList(toIndex - fromIndex);
	}

	protected abstract List<E> subList(int subListSize);

	@Override
	public Object[] toArray() {
		return ArrayTools.fill(new Object[this.size], this.getElement());
	}

	@Override
	@SuppressWarnings("unchecked")
	public <T> T[] toArray(T[] a) {
		int sz = this.size;
		return (T[]) ArrayTools.fill(((a.length >= sz) ? a : ArrayTools.newInstance(a, sz)), 0, sz, this.getElement());
	}

	private boolean elementIsNot(Object o) {
		return ! this.elementIs(o);
	}

	private boolean elementIs(Object o) {
		return ObjectTools.equals(o, this.getElement());
	}

	/**
	 * Return the list's repeating element.
	 */
	protected abstract E getElement();

	@Override
	public String toString() {
		return ObjectTools.toString(this, this.size);
	}

	private static final long serialVersionUID = 1L;
}