/*******************************************************************************
 * Copyright (c) 2007, 2013 Oracle and/or its affiliates. All rights reserved.
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v1.0 and Eclipse Distribution License v. 1.0
 * which accompanies this distribution.
 * The Eclipse Public License is available at http://www.eclipse.org/legal/epl-v10.html
 * and the Eclipse Distribution License is available at
 * http://www.eclipse.org/org/documents/edl-v10.php.
 *
 * Contributors:
 *     Oracle - initial API and implementation
 *
 ******************************************************************************/
package org.eclipse.persistence.tools.utility.tests.model.value;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Comparator;
import java.util.List;
import java.util.TreeSet;
import junit.framework.TestCase;
import org.eclipse.persistence.tools.utility.ReverseComparator;
import org.eclipse.persistence.tools.utility.collection.Bag;
import org.eclipse.persistence.tools.utility.collection.CollectionTools;
import org.eclipse.persistence.tools.utility.collection.HashBag;
import org.eclipse.persistence.tools.utility.collection.ListTools;
import org.eclipse.persistence.tools.utility.model.AbstractModel;
import org.eclipse.persistence.tools.utility.model.event.ListAddEvent;
import org.eclipse.persistence.tools.utility.model.event.ListChangeEvent;
import org.eclipse.persistence.tools.utility.model.event.ListClearEvent;
import org.eclipse.persistence.tools.utility.model.event.ListMoveEvent;
import org.eclipse.persistence.tools.utility.model.event.ListRemoveEvent;
import org.eclipse.persistence.tools.utility.model.event.ListReplaceEvent;
import org.eclipse.persistence.tools.utility.model.listener.ChangeAdapter;
import org.eclipse.persistence.tools.utility.model.listener.ChangeListener;
import org.eclipse.persistence.tools.utility.model.listener.ListChangeListener;
import org.eclipse.persistence.tools.utility.model.value.ListValueModel;
import org.eclipse.persistence.tools.utility.model.value.SimpleCollectionValueModel;
import org.eclipse.persistence.tools.utility.model.value.SortedListValueModelAdapter;
import org.eclipse.persistence.tools.utility.tests.TestTools;

@SuppressWarnings("nls")
public class SortedListValueModelAdapterTests extends TestCase {
	private SortedListValueModelAdapter<String> adapter;
	private SimpleCollectionValueModel<String> wrappedCollectionHolder;
	private Collection<String> wrappedCollection;


	public SortedListValueModelAdapterTests(String name) {
		super(name);
	}

	@Override
	protected void setUp() throws Exception {
		super.setUp();
		this.wrappedCollection = new HashBag<String>();
		this.wrappedCollectionHolder = new SimpleCollectionValueModel<String>(this.wrappedCollection);
		this.adapter = new SortedListValueModelAdapter<String>(this.wrappedCollectionHolder);
	}

	@Override
	protected void tearDown() throws Exception {
		TestTools.clear(this);
		super.tearDown();
	}

	private void verifyList(Collection<String> expected, ListValueModel<String> actual) {
		this.verifyList(expected, actual, null);
	}

	private void verifyList(Collection<String> expected, ListValueModel<String> actual, Comparator<String> comparator) {
		Collection<String> sortedSet = new TreeSet<String>(comparator);
		sortedSet.addAll(expected);
		List<String> expectedList = new ArrayList<String>(sortedSet);
		List<String> actualList = ListTools.list(actual.iterator());
		assertEquals(expectedList, actualList);
	}

	public void testAdd() {
		this.adapter.addListChangeListener(ListValueModel.LIST_VALUES, new TestListChangeListener() {
			@Override
			public void itemsAdded(ListAddEvent e) {/* OK */}
			@Override
			public void itemsReplaced(ListReplaceEvent e) {/* OK */}
		});
		this.wrappedCollectionHolder.add("foo");
		this.wrappedCollectionHolder.add("bar");
		this.wrappedCollectionHolder.add("baz");
		assertEquals(3, this.adapter.size());
		this.verifyList(this.wrappedCollection, this.adapter);
	}

	public void testAddItem() {
		List<String> synchList = new CoordinatedList<String>(this.adapter);
		Bag<String> synchCollection = new CoordinatedBag<String>(this.wrappedCollectionHolder);
		this.wrappedCollectionHolder.add("foo");
		assertTrue(this.wrappedCollection.contains("foo"));
		this.wrappedCollectionHolder.add("bar");
		this.wrappedCollectionHolder.add("baz");
		this.wrappedCollectionHolder.add("joo");
		this.wrappedCollectionHolder.add("jar");
		this.wrappedCollectionHolder.add("jaz");
		assertEquals(6, this.wrappedCollection.size());

		this.verifyList(this.wrappedCollection, this.adapter);
		assertEquals(this.wrappedCollection, CollectionTools.collection(synchList.iterator()));
		assertEquals(this.wrappedCollection, synchCollection);
	}

	public void testRemoveItem() {
		List<String> synchList = new CoordinatedList<String>(this.adapter);
		Bag<String> synchCollection = new CoordinatedBag<String>(this.wrappedCollectionHolder);
		this.wrappedCollectionHolder.add("foo");
		this.wrappedCollectionHolder.add("bar");
		this.wrappedCollectionHolder.add("baz");
		this.wrappedCollectionHolder.add("joo");
		this.wrappedCollectionHolder.add("jar");
		this.wrappedCollectionHolder.add("jaz");
		this.wrappedCollectionHolder.remove("jaz");
		assertFalse(this.wrappedCollection.contains("jaz"));
		this.wrappedCollectionHolder.remove("foo");
		assertFalse(this.wrappedCollection.contains("foo"));
		assertEquals(4, this.wrappedCollection.size());

		this.verifyList(this.wrappedCollection, this.adapter);
		assertEquals(this.wrappedCollection, CollectionTools.collection(synchList.iterator()));
		assertEquals(this.wrappedCollection, synchCollection);
	}

	public void testListSynch() {
		this.adapter.addListChangeListener(ListValueModel.LIST_VALUES, new TestListChangeListener() {
			@Override
			public void itemsAdded(ListAddEvent e) {/* OK */}
			@Override
			public void itemsRemoved(ListRemoveEvent e) {/* OK */}
			@Override
			public void itemsReplaced(ListReplaceEvent e) {/* OK */}
		});
		this.wrappedCollectionHolder.add("foo");
		this.wrappedCollectionHolder.add("bar");
		this.wrappedCollectionHolder.add("baz");
		this.wrappedCollectionHolder.add("joo");
		this.wrappedCollectionHolder.add("jar");
		this.wrappedCollectionHolder.add("jaz");
		this.wrappedCollectionHolder.remove("jaz");
		assertFalse(this.wrappedCollection.contains("jaz"));
		this.wrappedCollectionHolder.remove("foo");
		assertFalse(this.wrappedCollection.contains("foo"));
		assertEquals(4, this.wrappedCollection.size());

		this.verifyList(this.wrappedCollection, this.adapter);
	}

	public void testSetComparator() {
		List<String> synchList = new CoordinatedList<String>(this.adapter);
		Bag<String> synchCollection = new CoordinatedBag<String>(this.wrappedCollectionHolder);
		this.wrappedCollectionHolder.add("foo");
		assertTrue(this.wrappedCollection.contains("foo"));
		this.wrappedCollectionHolder.add("bar");
		this.wrappedCollectionHolder.add("baz");
		this.wrappedCollectionHolder.add("joo");
		this.wrappedCollectionHolder.add("jar");
		this.wrappedCollectionHolder.add("jaz");
		assertEquals(6, this.wrappedCollection.size());

		this.verifyList(this.wrappedCollection, this.adapter);
		assertEquals(this.wrappedCollection, CollectionTools.collection(synchList.iterator()));
		assertEquals(this.wrappedCollection, synchCollection);

		this.adapter.setComparator(new ReverseComparator<String>());
		this.verifyList(this.wrappedCollection, this.adapter, new ReverseComparator<String>());
		assertEquals(this.wrappedCollection, CollectionTools.collection(synchList.iterator()));
		assertEquals(this.wrappedCollection, synchCollection);
	}

	public void testHasListeners() {
		assertFalse(((AbstractModel) this.adapter).hasAnyListChangeListeners(ListValueModel.LIST_VALUES));
		CoordinatedList<String> synchList = new CoordinatedList<String>(this.adapter);
		assertTrue(((AbstractModel) this.adapter).hasAnyListChangeListeners(ListValueModel.LIST_VALUES));
		this.adapter.removeListChangeListener(ListValueModel.LIST_VALUES, synchList);
		assertFalse(((AbstractModel) this.adapter).hasAnyListChangeListeners(ListValueModel.LIST_VALUES));

		ChangeListener cl = new ChangeAdapter();
		this.adapter.addChangeListener(cl);
		assertTrue(((AbstractModel) this.adapter).hasAnyListChangeListeners(ListValueModel.LIST_VALUES));
		this.adapter.removeChangeListener(cl);
		assertFalse(((AbstractModel) this.adapter).hasAnyListChangeListeners(ListValueModel.LIST_VALUES));
	}

	public void testCollectionChange() {
		this.wrappedCollectionHolder.add("fred");
		this.adapter.addListChangeListener(ListValueModel.LIST_VALUES, new TestListChangeListener() {
			@Override
			public void itemsAdded(ListAddEvent e) {/* OK */}
			@Override
			public void itemsReplaced(ListReplaceEvent e) {/* OK */}
		});
		this.wrappedCollectionHolder.setValues(Arrays.asList(new String[] {"foo", "bar", "baz"}));
		assertEquals(3, this.adapter.size());
		this.verifyList(this.wrappedCollection, this.adapter);
	}

	class TestListChangeListener implements ListChangeListener {
		@Override
		public void itemsAdded(ListAddEvent e) {
			fail("unexpected event");
		}
		@Override
		public void itemsRemoved(ListRemoveEvent e) {
			fail("unexpected event");
		}
		@Override
		public void itemsReplaced(ListReplaceEvent e) {
			fail("unexpected event");
		}
		@Override
		public void itemsMoved(ListMoveEvent e) {
			fail("unexpected event");
		}
		@Override
		public void listCleared(ListClearEvent e) {
			fail("unexpected event");
		}
		@Override
		public void listChanged(ListChangeEvent e) {
			fail("unexpected event");
		}
	}
}