/*******************************************************************************
 * Copyright (c) 2006, 2012 Oracle and/or its affiliates. All rights reserved.
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v1.0 and Eclipse Distribution License v. 1.0
 * which accompanies this distribution.
 * The Eclipse Public License is available at http://www.eclipse.org/legal/epl-v10.html
 * and the Eclipse Distribution License is available at
 * http://www.eclipse.org/org/documents/edl-v10.php.
 *
 * Contributors:
 *     Oracle - initial API and implementation
 *
 ******************************************************************************/
package org.eclipse.persistence.tools.mapping.orm.dom;

import java.util.List;
import org.eclipse.persistence.tools.mapping.orm.ExternalFetchGroup;
import org.w3c.dom.Element;

/**
 * The external form of a fetch group, which is a child of an entity.
 *
 * @see EmbeddableEntity
 *
 * @version 2.5
 * @author Les Davis
 */
@SuppressWarnings("nls")
final class FetchGroup extends AbstractExternalForm
                       implements ExternalFetchGroup {

	/**
	 * The position of the element within the list of children with the same type owned by the parent.
	 */
	private int index;

	/**
	 * The node name used to store and retrieve the {@link Element} encapsulated by this external form.
	 */
	static final String FETCH_GROUP = "fetch-group";

	/**
	 * The attribute name used to store and retrieve the load property.
	 */
	static final String LOAD = "load";

	/**
	 * The attribute name used to store and retrieve the name property.
	 */
	static final String NAME = "name";

	/**
	 * Creates a new <code>FetchGroup</code>.
	 *
	 * @param parent The parent of this external form
	 * @param index The position of the external form within the list of the same type
	 */
	FetchGroup(AbstractExternalForm parent, int index) {
		super(parent);
		this.index = index;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	protected Element addSelf(String elementName, List<String> elementNamesOrder) {

		if (index == -1) {
			return super.addSelf(elementName, elementNamesOrder);
		}

		return addChild(getParent(), elementName, index, elementNamesOrder);
	}
	
	/**
	 * {@inheritDoc}
	 */
	@Override
	protected List<String> buildAttributeNamesOrder() {
		List<String> names = super.buildAttributeNamesOrder();
		names.add(NAME);
		names.add(LOAD);
		return names;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public Boolean doesLoad() {
		return getBooleanAttribute(LOAD);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	protected String getElementName() {
		return FETCH_GROUP;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public int getIndex() {
		return index;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getName() {
		return getAttribute(NAME);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final void removeSelf() {
		if (index == -1) {
			super.removeSelf();
		}
		else {
			removeChild(getParent(), getElementName(), index);
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setName(String name) {
		setAttribute(NAME, name);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setShouldLoad(Boolean load) {
		setAttribute(LOAD, load);
	}
}