/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * license agreements; and to You under the Apache License, version 2.0:
 *
 *   https://www.apache.org/licenses/LICENSE-2.0
 *
 * This file is part of the Apache Pekko project, which was derived from Akka.
 */

/*
 * Copyright (C) 2009-2022 Lightbend Inc. <https://www.lightbend.com>
 */

package org.apache.pekko.actor.testkit.typed.javadsl

import java.util.concurrent.ThreadLocalRandom

import scala.collection.immutable

import org.apache.pekko
import pekko.actor.testkit.typed.internal.TestInboxImpl
import pekko.actor.typed.ActorRef
import pekko.annotation.{ ApiMayChange, DoNotInherit }
import pekko.pattern.StatusReply
import pekko.util.ccompat.JavaConverters._

object TestInbox {
  import pekko.actor.testkit.typed.scaladsl.TestInbox.address

  def create[T](name: String): TestInbox[T] = {
    val uid = ThreadLocalRandom.current().nextInt()
    new TestInboxImpl((address / name).withUid(uid))
  }
  def create[T](): TestInbox[T] = {
    val uid = ThreadLocalRandom.current().nextInt()
    new TestInboxImpl((address / "inbox").withUid(uid))
  }
}

/**
 * Utility for use as an [[ActorRef]] when *synchronously* testing [[pekko.actor.typed.Behavior]]
 * with [[pekko.actor.testkit.typed.javadsl.BehaviorTestKit]].
 *
 * If you plan to use a real [[pekko.actor.typed.ActorSystem]] then use [[pekko.actor.testkit.typed.javadsl.TestProbe]]
 * for asynchronous testing.
 *
 * Use `TestInbox.create` factory methods to create instances
 *
 * Not for user extension
 */
@DoNotInherit
abstract class TestInbox[T] {

  /**
   * The actor ref of the inbox
   */
  def getRef(): ActorRef[T]

  /**
   * Get and remove the oldest message
   */
  def receiveMessage(): T

  /**
   * Assert and remove the the oldest message.
   */
  def expectMessage(expectedMessage: T): TestInbox[T]

  /**
   * Collect all messages in the inbox and clear it out
   */
  def getAllReceived(): java.util.List[T] = internalReceiveAll().asJava

  protected def internalReceiveAll(): immutable.Seq[T]

  def hasMessages: Boolean

  // TODO expectNoMsg etc
}

/**
 * Similar to an [[akka.actor.testkit.typed.javadsl.TestInbox]], but can only ever give access to a single message (a reply).
 *
 * Not intended for user creation: the [[akka.actor.testkit.typed.javadsl.BehaviorTestKit]] will provide these to
 * denote that at most a single reply is expected.
 *
 * @since 1.3.0
 */
@DoNotInherit
@ApiMayChange
trait ReplyInbox[T] {

  /**
   * Get and remove the reply.  Subsequent calls to `receiveReply`, `expectReply`, and `expectNoReply` will fail and `hasReplies`
   * will be false after calling this method
   */
  def receiveReply(): T

  /**
   * Assert and remove the message.  Subsequent calls to `receiveReply`, `expectReply`, and `expectNoReply` will fail and `hasReplies`
   * will be false after calling this method
   */
  def expectReply(expectedReply: T): Unit

  def expectNoReply(): ReplyInbox[T]
  def hasReply: Boolean
}

/**
 * A [[akka.actor.testkit.typed.javadsl.ReplyInbox]] which specially handles [[akka.pattern.StatusReply]].
 *
 * Note that there is no provided ability to expect a specific `Throwable`, as it's recommended to prefer
 * a string error message or to enumerate failures with specific types.
 *
 * Not intended for user creation: the [[akka.actor.testkit.typed.javadsl.BehaviorTestKit]] will provide these to
 * denote that at most a single reply is expected.
 */
@DoNotInherit
@ApiMayChange
trait StatusReplyInbox[T] {

  /**
   * Get and remove the status reply.  Subsequent calls to any `receive` or `expect` method will fail and `hasReply`
   * will be false after calling this method.
   */
  def receiveStatusReply(): StatusReply[T]

  /**
   * Get and remove the successful value of the status reply.  This will fail if the status reply is an error.
   * Subsequent calls to any `receive` or `expect` method will fail and `hasReply` will be false after calling this
   * method.
   */
  def receiveValue(): T

  /**
   * Get and remove the error value of the status reply.  This will fail if the status reply is a success.
   * Subsequent calls to any `receive` or `expect` method will fail and `hasReply` will be false after calling this
   * method.
   */
  def receiveError(): Throwable

  /**
   * Assert that the status reply is a success with this value and remove the status reply.  Subsequent calls to any
   * `receive` or `expect` method will fail and `hasReply` will be false after calling this method.
   */
  def expectValue(expectedValue: T): Unit

  /**
   * Assert that the status reply is a failure with this error message and remove the status reply.  Subsequent
   * calls to any `receive` or `expect` method will fail and `hasReply` will be false after calling this method.
   */
  def expectErrorMessage(errorMessage: String): Unit

  /**
   * Assert that this inbox has *never* received a reply.
   */
  def expectNoReply(): StatusReplyInbox[T]

  def hasReply: Boolean
}
