#include <stdio.h>
#include <unistd.h>
#include <string.h>
#include <ctype.h>
#include <sys/fcntl.h>
#include <sys/disklabel.h>
#include <sys/ioctl.h>
#include <sys/stat.h>
#include <sys/scsiio.h>
#include <sys/endian.h>
#include <dev/ata/atareg.h>
#include <dev/ic/wdcreg.h>
#include <sys/ataio.h>
#include "disklabel_rdb.h"

struct part_mem {
	struct part_mem *next;
	uint32_t dskblk;
	int flags;
	char data[512];
};

struct lseg_mem {
	struct lseg_mem *next;
	uint32_t dskblk;
	int flags;
	char data[512];
};

struct fsh_mem {
	struct fsh_mem *next;
	uint32_t dskblk;
	int flags;
	struct lseg_mem *lsegs;
	char data[512];
};

struct rdb_mem {
	uint32_t dskblk;
	int flags;
	struct part_mem *parts;
	struct fsh_mem *fshs;
	struct lseg_mem *drvini;
	char data[512];
};

#define DOSPARTOFF	446
#define BOOT_MAGIC	0xaa55
#define DOSSECT(s,c)	(((s) & 0x3f) | (((c) >> 2) & 0xc0))
#define DOSCYL(c)	((c) & 0xff)
/* Isolate the relevant bits to get sector and cylinder. */
#define	DPSECT(s)	((s) & 0x3f)
#define	DPCYL(c, s)	((c) + (((s) & 0xc0) << 2))

struct dos_partition {
	unsigned char	dp_flag;
	unsigned char	dp_shd;
	unsigned char	dp_ssect;
	unsigned char	dp_scyl;
	unsigned char	dp_typ;
	unsigned char	dp_ehd;
	unsigned char	dp_esect;
	unsigned char	dp_ecyl;
	unsigned long	dp_start;
	unsigned long	dp_size;
};

struct mboot {
	unsigned char padding[2];
	unsigned char bootinst[DOSPARTOFF];
	struct dos_partition parts[4];
	unsigned short int signature;
};

struct rdb_mem *rdb;
struct mboot *mboot;

struct disklabel label;

uint32_t dos_cylinders;
uint32_t dos_cylindersectors;
uint32_t dos_heads;
uint32_t dos_sectors;

int fd;

int msdos_part;
int verbose;
int write_it;

unsigned char bootcode[] = {
0x33, 0xc0, 0xfa, 0x8e, 0xd0, 0xbc, 0x00, 0x7c, 0x8e, 0xc0, 0x8e, 0xd8, 0xfb, 0x8b, 0xf4, 0xbf,
0x00, 0x06, 0xb9, 0x00, 0x02, 0xfc, 0xf3, 0xa4, 0xea, 0x1d, 0x06, 0x00, 0x00, 0xb0, 0x04, 0xbe,
0xbe, 0x07, 0x80, 0x3c, 0x80, 0x74, 0x0c, 0x83, 0xc6, 0x10, 0xfe, 0xc8, 0x75, 0xf4, 0xbe, 0xbd,
0x06, 0xeb, 0x43, 0x8b, 0xfe, 0x8b, 0x14, 0x8b, 0x4c, 0x02, 0x83, 0xc6, 0x10, 0xfe, 0xc8, 0x74,
0x0a, 0x80, 0x3c, 0x80, 0x75, 0xf4, 0xbe, 0xbd, 0x06, 0xeb, 0x2b, 0xbd, 0x05, 0x00, 0xbb, 0x00,
0x7c, 0xb8, 0x01, 0x02, 0xcd, 0x13, 0x73, 0x0c, 0x33, 0xc0, 0xcd, 0x13, 0x4d, 0x75, 0xef, 0xbe,
0x9e, 0x06, 0xeb, 0x12, 0x81, 0x3e, 0xfe, 0x7d, 0x55, 0xaa, 0x75, 0x07, 0x8b, 0xf7, 0xea, 0x00,
0x7c, 0x00, 0x00, 0xbe, 0x85, 0x06, 0x2e, 0xac, 0x0a, 0xc0, 0x74, 0x06, 0xb4, 0x0e, 0xcd, 0x10,
0xeb, 0xf4, 0xfb, 0xeb, 0xfe,
'M', 'i', 's', 's', 'i', 'n', 'g', ' ',
	'o', 'p', 'e', 'r', 'a', 't', 'i', 'n', 'g', ' ', 's', 'y', 's', 't', 'e', 'm', 0,
'E', 'r', 'r', 'o', 'r', ' ', 'l', 'o', 'a', 'd', 'i', 'n', 'g', ' ',
	'o', 'p', 'e', 'r', 'a', 't', 'i', 'n', 'g', ' ', 's', 'y', 's', 't', 'e', 'm', 0,
'I', 'n', 'v', 'a', 'l', 'i', 'd', ' ',
	'p', 'a', 'r', 't', 'i', 't', 'i', 'o', 'n', ' ', 't', 'a', 'b', 'l', 'e', 0,
'A', 'u', 't', 'h', 'o', 'r', ' ', '-', ' ',
	'S', 'i', 'e', 'g', 'm', 'a', 'r', ' ', 'S', 'c', 'h', 'm', 'i', 'd', 't', 0,0,0,

  0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
  0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
  0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
  0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
  0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
  0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
  0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
  0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
  0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
  0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
  0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
  0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
  0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0
};

void load_rdb(void);
void load_part(uint32_t);
void load_fs(uint32_t);
void load_drvini(uint32_t);
void display_rdb(void);
void display_part();
void display_fsh();
void display_drvini();
void add_part(struct partblock *);
void getlabel(int);
void setlabel(int, int);
void showpartitions(FILE *, struct disklabel *, int);
void get_params();
int read_block(void *, uint32_t, char*);
void intuit_geometry();
int get_mapping(int, int *, int *, int *, long *);
void dos(int, unsigned char *, unsigned char *, unsigned char *);
void display_msdos();
void display_dospart(int);
int xsum(uint32_t *);
void giveup(char *);

main (argc, argv)
int argc;
char *argv[];
{
	char *devname;
	char namebuf[128];
	char *np;

	if (argc > 1 && argv[1][0] == '-') {
		++argv[1];
		while (argv[1][0]) {
			switch(argv[1][0]) {
			case 'm':
				msdos_part = 1;
				break;
			case 'v':
				verbose = 1;
				break;
			case 'w':
				write_it = 1;
				break;
			default:
				;
			}
			++argv[1];
		}
		--argc;
		++argv;
	}
	if (argc > 1)
		devname = argv[1];
	else
		devname = "/dev/rsd0c";
	np = strchr (devname, ':');
	if (*devname != '/') {
		sprintf (namebuf, "/dev/r%sc", devname);
		devname = namebuf;
	}
	fd = open (devname, write_it ? O_RDWR : O_RDONLY);
	if (fd < 0)
		perror("open");
	else {
		getlabel(fd);
		load_rdb();		/* load existing RDB info */
		if (rdb) {
			display_rdb();
		}
		if (verbose) {
			printf("\n");
			showpartitions(stdout, &label, 0);
			printf("\n");
		}
		if (write_it) {
			/* check changed */
			setlabel(fd, 0);
		}
	}
	close (fd);
}

void load_rdb()
{
	int blk;
	struct rdblock *rdbp;
	struct mboot *mb;

	rdb = (struct rdb_mem *)malloc(sizeof(struct rdb_mem));
	if (rdb == NULL)
		giveup("RDB malloc");
	rdb->parts = NULL;
	rdb->fshs = NULL;
	rdb->drvini = NULL;
	rdbp = (struct rdblock *)rdb->data;
	mb = (struct mboot *)((char *)rdbp - 2);

	for (blk = 0; blk < RDB_MAXBLOCKS; ++blk) {
		if (read_block(rdbp, blk, "RDB"))
			giveup("RDB read");
		if (be32toh(rdbp->id) == RDBLOCK_ID)
			break;
		if (blk == 0 && le16toh(mb->signature) == BOOT_MAGIC) {
			mboot = (struct mboot *)malloc(sizeof (struct mboot));
			memcpy(mboot->bootinst, rdbp, 512);
			intuit_geometry();
			msdos_part = 1;		/* ? */
			printf("+++ Found MSDOS partition table:");
			printf(" %d cyl %d hd %d sect +++\n", dos_cylinders,
			    dos_heads, dos_sectors);
		}
	}
	if (blk >= RDB_MAXBLOCKS) {
		fprintf(stderr, "RDB not found\n");
		free(rdb);
		rdb = NULL;
		return;
	}
	if (xsum((uint32_t *)rdbp))
		fprintf(stderr, "RDB Checksum error\n");

	rdb->dskblk = blk;
	rdb->flags = 0;
	if (be32toh(rdbp->badbhead) != RDBNULL)
		fprintf(stderr, "*** bad blocks present ***\n");
	if (be32toh(rdbp->partbhead) != RDBNULL)
		load_part(be32toh(rdbp->partbhead));
	if (be32toh(rdbp->fsbhead) != RDBNULL)
		load_fs(be32toh(rdbp->fsbhead));
	if (be32toh(rdbp->driveinit) != RDBNULL)
		load_drvini(be32toh(rdbp->driveinit));
}

void load_part(blk)
uint32_t blk;
{
	struct partblock *pb;
	struct part_mem *pbp = (struct part_mem *)&rdb->parts;

	while (blk != RDBNULL) {
		pbp->next = (struct part_mem *)malloc(sizeof (struct part_mem));
		if (pbp->next == NULL)
			giveup("PART malloc");
		pbp = pbp->next;
		pbp->next = NULL;
		pbp->dskblk = blk;
		pbp->flags = 0;
		pb = (struct partblock *)pbp->data;
		if (read_block(pb, blk, "PART"))
			giveup("PART read");
		if (be32toh(pb->id) != PARTBLOCK_ID)
			giveup("PARTBLOCK invalid ID");
		if (xsum((uint32_t *)pb))
			fprintf(stderr, "PART Checksum Error\n");
		add_part(pb);
		blk = be32toh(pb->next);
	}
}

void load_fs(blk)
uint32_t blk;
{
	struct fsblock *fs;
	struct fsh_mem *fsp = (struct fsh_mem *)&rdb->fshs;
	struct lsegblock *ls;
	struct lseg_mem *lsp;

	while (blk != RDBNULL) {
		fsp->next = (struct fsh_mem *)malloc(sizeof (struct fsh_mem));
		if (fsp->next == NULL)
			giveup("FSH malloc");
		fsp = fsp->next;
		fsp->next = NULL;
		fsp->lsegs = NULL;
		fsp->dskblk = blk;
		fsp->flags = 0;
		fs = (struct fsblock *)fsp->data;
		if (read_block(fs, blk, "FSH"))
			giveup("FSH read");
		if (be32toh(fs->id) != FSBLOCK_ID)
			giveup("FSBLOCK invalid ID");
		if (xsum((uint32_t *)fs))
			fprintf(stderr, "FSH Checksum Error\n");
		blk = be32toh(fs->lsegblocks);
		lsp = (struct lseg_mem *)&fsp->lsegs;
		while (blk != RDBNULL) {
			lsp->next = (struct lseg_mem *)malloc(sizeof (struct lseg_mem));
			if (lsp->next == NULL)
				giveup("FS LSEG malloc");
			lsp = lsp->next;
			lsp->next = NULL;
			lsp->dskblk = blk;
			lsp->flags = 0;
			ls = (struct lsegblock *)lsp->data;
			if (read_block(ls, blk, "FS LSEG"))
				giveup("FS LSEG read");
			if (be32toh(ls->id) != LSEGBLOCK_ID)
				giveup("FS LSEGBLOCK invalid ID");
			if (xsum((uint32_t *)ls))
				fprintf(stderr, "FS LSEG Checksum Error\n");
			blk = be32toh(ls->next);
		}
		blk = be32toh(fs->next);
	}
}

void load_drvini(blk)
uint32_t blk;
{
	struct lseg_mem *lsp = (struct lseg_mem *)&rdb->drvini;
	struct lsegblock *ls;

	while (blk != RDBNULL) {
		lsp->next = (struct lseg_mem *)malloc(sizeof (struct lseg_mem));
		if (lsp->next == NULL)
			giveup("DRVINI malloc");
		lsp = lsp->next;
		lsp->next = NULL;
		lsp->dskblk = blk;
		lsp->flags = 0;
		ls = (struct lsegblock *)lsp->data;
		if (read_block(ls, blk, "DRVINI"))
			giveup("DRVINI read");
		if (be32toh(ls->id) != LSEGBLOCK_ID)
			giveup("DRVINI LSEGBLOCK invalid ID");
		if (xsum((uint32_t *)ls))
			fprintf(stderr, "DRVINI Checksum Error\n");
		blk = be32toh(ls->next);
	}
}

void display_rdb()
{
	struct rdblock *rdbp;
	char str[20];

	if (msdos_part && mboot)
		display_msdos();
	printf("RDSK block at %d Flags:", rdb->dskblk);
	rdbp = (struct rdblock *) rdb->data;
	str[0] = ' ';
	if (be32toh(rdbp->flags) & RDBF_LAST) {
		printf(" LAST");
		str[0] = ',';
	}
	if (be32toh(rdbp->flags) & RDBF_LASTLUN) {
		printf("%cLASTLUN", str[0]);
		str[0] = ',';
	}
	if (be32toh(rdbp->flags) & RDBF_LASTUNIT) {
		printf("%cLASTID", str[0]);
		str[0] = ',';
	}
	if (be32toh(rdbp->flags) & RDBF_NORESELECT) {
		printf("%cNORESELECT", str[0]);
		str[0] = ',';
	}
	if (be32toh(rdbp->flags) & RDBF_DISKID) {
		printf("%cDISKID", str[0]);
		str[0] = ',';
	}
	if (be32toh(rdbp->flags) & RDBF_CTRLID) {
		printf("%cCTRLID", str[0]);
		str[0] = ',';
	}
	if (be32toh(rdbp->flags) & RDBF_SYNC) {
		printf("%cSYNC", str[0]);
		str[0] = ',';
	}
	printf("\n");
	if (be32toh(rdbp->flags) & RDBF_DISKID) {
		strncpy(str, rdbp->diskvendor, 8);
		str[8] = 0;
		printf("Disk Vendor: '%s'", str);
		strncpy(str, rdbp->diskproduct, 16);
		str[16] = 0;
		printf(" Product: '%s'", str);
		strncpy(str, rdbp->diskrevision, 4);
		str[4] = 0;
		printf(" Revision: '%s'\n", str);
	}
	if (be32toh(rdbp->flags) & RDBF_CTRLID) {
		strncpy(str, rdbp->contvendor, 8);
		str[8] = 0;
		printf("Controller Vendor: '%s'", str);
		strncpy(str, rdbp->contproduct, 16);
		str[16] = 0;
		printf(" Product: '%s'", str);
		strncpy(str, rdbp->contrevision, 4);
		str[4] = 0;
		printf(" Revision: '%s'\n", str);
	}
	printf("RDB: Cylinders %d Head%s %d Sectors %d", be32toh(rdbp->ncylinders),
	    be32toh(rdbp->nheads) == 1 ? "" : "s", be32toh(rdbp->nheads), be32toh(rdbp->nsectors));
	printf(" LowCyl %d HighCyl %d Sect/Cyl %d\n",
	    be32toh(rdbp->lowcyl), be32toh(rdbp->highcyl), be32toh(rdbp->secpercyl));
	display_part();
	display_fsh();
	display_drvini();
}

void display_part()
{
	struct part_mem *pbp = rdb->parts;
	struct partblock *pb;
	char partname[32];
	char dostype[5];

	while (pbp) {
		pb = (struct partblock *)pbp->data;
		if (be32toh(pb->id) != PARTBLOCK_ID)
			break;
		printf("PART @ %2d %5d-%5d %2d %3d %08x", pbp->dskblk,
		    be32toh(pb->e.lowcyl), be32toh(pb->e.highcyl),
		    be32toh(pb->e.numheads), be32toh(pb->e.secpertrk),
		    be32toh(pb->e.mask));
/** FIXME **/
		memcpy(dostype, (char *)&pb->e.dostype, 4);
		dostype[4] = 0;
		if (dostype[3] < 10)
			dostype[3] += '0';
		printf(" %-4s", dostype);
		strncpy(partname, pb->partname + 1, pb->partname[0]);
		partname[pb->partname[0]] = 0;
		printf(" %s", partname);
		if (be32toh(pb->flags)) {
			printf(" (");
			if (be32toh(pb->flags) & PBF_BOOTABLE) {
				printf("Bootable:%d", be32toh(pb->e.bootpri));
				if (be32toh(pb->e.tabsize) >= 19 && be32toh(pb->e.bootblocks))
					printf("[%d]", be32toh(pb->e.bootblocks));
				if (be32toh(pb->flags) & ~PBF_BOOTABLE)
					printf(",");
			}
			if (be32toh(pb->flags) & PBF_NOMOUNT)
				printf("NoMount");
			printf(")");
		}
		printf("\n");
		pbp = pbp->next;
	}
}

void display_fsh()
{
	char dostype[5];
	struct fsh_mem *fsp = rdb->fshs;
	struct fsblock *fs;
	struct lsegblock *ls;
	struct lseg_mem *lsp;
	int nblks, nbytes;

	while (fsp) {
		fs = (struct fsblock *)fsp->data;
		if (be32toh(fs->id) != FSBLOCK_ID) {
			fprintf(stderr, "FileSystem Header bad\n");
			break;
		}
		printf("FSH @ %2d->%2d", fsp->dskblk, be32toh(fs->lsegblocks));
/** FIXME **/
		memcpy(dostype, (char *)&fs->dostype, 4);
		dostype[4] = 0;
		if (dostype[3] < 10)
			dostype[3] += '0';
		printf(" %-4s v %d", dostype, be32toh(fs->version));
		nblks = nbytes = 0;
		lsp = fsp->lsegs;
		while (lsp) {
			ls = (struct lsegblock *)lsp->data;
			if (be32toh(ls->id) != LSEGBLOCK_ID)
				break;
			nblks++;
			nbytes += (be32toh(ls->nsumlong) - 5) * 4;
			lsp = lsp->next;
		}
		printf (" %d blocks %d[%x] bytes\n", nblks, nbytes, nbytes);
		fsp = fsp->next;
	}
}

void display_drvini()
{
	struct lseg_mem *drvp = rdb->drvini;
	struct lsegblock *ls;
	int nblks, nbytes;

	if (drvp == NULL)
		return;
	printf("DriveInit @ %d:  ", drvp->dskblk);
	nblks = nbytes = 0;
	while (drvp) {
		ls = (struct lsegblock *)drvp->data;
		if (be32toh(ls->id) != LSEGBLOCK_ID)
			break;
		++nblks;
		nbytes += ((ls->nsumlong) - 5) * 4;
		drvp = drvp->next;
	}
	printf("%d blocks %d[%x] bytes\n", nblks, nbytes, nbytes);
}

void
add_part(struct partblock *pb)
{
	struct partition	npe;
	int			i, j, unused;
	off_t			loff;

	memset((void *)&npe, 0, sizeof(npe));
	npe.p_size = be32toh(pb->e.numheads) * be32toh(pb->e.secpertrk) *
	    (be32toh(pb->e.highcyl) - be32toh(pb->e.lowcyl) + 1);
	npe.p_offset = be32toh(pb->e.lowcyl) *
	    be32toh(pb->e.numheads) * be32toh(pb->e.secpertrk);
	npe.p_fstype = be32toh(pb->e.dostype) & 255;
	if (npe.p_fstype != FS_BSDFFS)
		return;

	unused = -1;
	for (j = 0; j < label.d_npartitions; ++j) {
		struct partition *lpe;

		if (j == RAW_PART)
			continue;
		lpe = &label.d_partitions[j];
		if (lpe->p_size == npe.p_size &&
		    lpe->p_offset == npe.p_offset) {
		    	unused = -2;
		    	break;
		}
		if (unused == -1 && lpe->p_size == 0 &&
		    lpe->p_fstype == FS_UNUSED)
		    	unused = j;
	}
	if (unused == -2)
		return;
	if (unused == -1) {
		if (label.d_npartitions < MAXPARTITIONS) {
			unused = label.d_npartitions;
			label.d_npartitions++;
		} else {
			return;
		}
	}
	switch (npe.p_fstype) {
	case FS_BSDFFS:
		npe.p_fsize = 1024;
		npe.p_frag = 8;
		npe.p_cpg = 16;
		break;
	}
	label.d_partitions[unused] = npe;
}

void
getlabel(int sd)
{

	if (ioctl(sd, DIOCGDINFO, &label) < 0) {
		perror("get label");
		exit(1);
	}
	/*
	 * Some ports seem to not set the number of partitions
	 * correctly, albeit they seem to set the raw partition ok!
	 */
	if (label.d_npartitions <= getrawpartition())
		label.d_npartitions = getrawpartition() + 1;
}

void
setlabel(int sd, int doraw)
{
	int one = 1;
	
	label.d_checksum = 0;
	label.d_checksum = dkcksum(&label);
	if (ioctl(sd, doraw ? DIOCWDINFO : DIOCSDINFO, &label) < 0) {
		perror("set label");
		exit(1);
	}
	if (!doraw)
		/* If we haven't written to the disk, don't discard on close */
		ioctl(sd, DIOCKLABEL, &one);
}

void get_params()
{
	int i;
	struct rdblock *rdbp = (struct rdblock *)rdb->data;
	struct ataparams *atabuf;
	struct disklabel disklabel;
	struct scsireq scsireq;
	struct atareq req;
	char inqbuf[512];
#if BYTE_ORDER == LITTLE_ENDIAN
#endif
	
	if (ioctl(fd, DIOCGDINFO, &disklabel) == -1) {
		giveup("DIOCGDINFO");
		return;
	}

	rdbp->ncylinders = disklabel.d_ncylinders;
	rdbp->nheads = disklabel.d_ntracks;
	rdbp->nsectors = disklabel.d_nsectors;
	rdbp->secpercyl = rdbp->nheads * rdbp->nsectors;
	/* Adjust ncylinders according to disklabel.d_secperunit? */
	if (rdbp->ncylinders < disklabel.d_secperunit / rdbp->secpercyl)
		rdbp->ncylinders = disklabel.d_secperunit / rdbp->secpercyl;
	rdbp->lowcyl = 1;
	rdbp->highcyl = rdbp->ncylinders - 1;
	rdbp->rdblowb = 0;
	rdbp->rdbhighb = rdbp->nsectors;

	memset(&scsireq, 0, sizeof(struct scsireq));
	scsireq.timeout = 10;
	scsireq.cmd[0] = 0x12;
	scsireq.cmd[4] = 255;
	scsireq.cmdlen = 6;
	scsireq.senselen = sizeof(scsireq.sense);
	scsireq.databuf = inqbuf;
	scsireq.datalen = 255;
	i = ioctl(fd, SCIOCCOMMAND, &scsireq);
	if (i != -1) {
		strncpy(rdbp->diskvendor, inqbuf + 8, 8);
		strncpy(rdbp->diskproduct, inqbuf + 16, 16);
		strncpy(rdbp->diskrevision, inqbuf + 32, 4);
	} else {
		perror("SCIOCCOMMAND");
		atabuf = (struct ataparams *) inqbuf;
		req.flags = ATACMD_READ;
		req.command = WDCC_IDENTIFY;
		req.databuf =  inqbuf;
		req.datalen = sizeof(inqbuf);
		req.timeout = 1000;
		i = ioctl(fd, ATAIOCCOMMAND, &req);
		if (i != -1) {
			char *p;
#if BYTE_ORDER == LITTLE_ENDIAN
/* shuffle string byte order */
#endif
			/* strip blanks off the info strings */
			/* get vendor, product, revision */
			p = strchr(atabuf->atap_model, ' ');
			if (p) {
				*p++ = 0;
				strncpy(rdbp->diskvendor, atabuf->atap_model, 8);
			} else {
				strcpy(rdbp->diskvendor, "UNKNOWN");
				p = atabuf->atap_model;
			}
			strncpy(rdbp->diskproduct, p, 16);
			strncpy(rdbp->diskrevision, atabuf->atap_revision, 4);
		} else
			perror("ATAIOCCOMMAND");
	}
	rdbp->flags |= RDBF_DISKID;
}

void
intuit_geometry()
{
	int cylinders = -1, heads = -1, sectors = -1, i, j;
	int c1, h1, s1, c2, h2, s2;
	long a1, a2;
	quad_t num, denom;

	/* Try to deduce the number of heads from two different mappings. */
	for (i = 0; i < 4 * 2; i++) {
		if (get_mapping(i, &c1, &h1, &s1, &a1) < 0)
			continue;
		for (j = 0; j < 8; j++) {
			if (get_mapping(j, &c2, &h2, &s2, &a2) < 0)
				continue;
			num = (quad_t)h1*(a2-s2) - h2*(a1-s1);
			denom = (quad_t)c2*(a1-s1) - c1*(a2-s2);
			if (denom != 0 && num % denom == 0) {
				heads = num / denom;
				break;
			}
		}
		if (heads != -1)	
			break;
	}

	if (heads == -1)
		return;

	/* Now figure out the number of sectors from a single mapping. */
	for (i = 0; i < 4 * 2; i++) {
		if (get_mapping(i, &c1, &h1, &s1, &a1) < 0)
			continue;
		num = a1 - s1;
		denom = c1 * heads + h1;
		if (denom != 0 && num % denom == 0) {
			sectors = num / denom;
			break;
		}
	}

	if (sectors == -1)
		return;

	/* Estimate the number of cylinders. */
	cylinders = dos_cylinders * dos_cylindersectors / heads / sectors;

	/* Now verify consistency with each of the partition table entries.
	 * Be willing to shove cylinders up a little bit to make things work,
	 * but translation mismatches are fatal. */
	for (i = 0; i < 4 * 2; i++) {
		if (get_mapping(i, &c1, &h1, &s1, &a1) < 0)
			continue;
		if (sectors * (c1 * heads + h1) + s1 != a1)
			return;
		if (c1 >= cylinders)
			cylinders = c1 + 1;
	}

	/* Everything checks out.  Reset the geometry to use for further
	 * calculations. */
	dos_cylinders = cylinders;
	dos_heads = heads;
	dos_sectors = sectors;
	dos_cylindersectors = heads * sectors;
}

/* For the purposes of intuit_translated_geometry(), treat the partition
 * table as a list of eight mapping between (cylinder, head, sector)
 * triplets and absolute sectors.  Get the relevant geometry triplet and
 * absolute sectors for a given entry, or return -1 if it isn't present.
 * Note: for simplicity, the returned sector is 0-based. */
int
get_mapping(i, cylinder, head, sector, absolute)
	int i, *cylinder, *head, *sector;
	long *absolute;
{
	struct dos_partition *part = &mboot->parts[i / 2];

	if (part->dp_typ == 0)
		return -1;
	if (i % 2 == 0) {
		*cylinder = DPCYL(part->dp_scyl, part->dp_ssect);
		*head = part->dp_shd;
		*sector = DPSECT(part->dp_ssect) - 1;
		*absolute = le32toh(part->dp_start);
	} else {
		*cylinder = DPCYL(part->dp_ecyl, part->dp_esect);
		*head = part->dp_ehd;
		*sector = DPSECT(part->dp_esect) - 1;
		*absolute = le32toh(part->dp_start) + le32toh(part->dp_size) - 1;
	}
	return 0;
}

void
dos(sector, cylinderp, headp, sectorp)
	int sector;
	unsigned char *cylinderp, *headp, *sectorp;
{
	int cylinder, head;

	cylinder = sector / dos_cylindersectors;
	sector -= cylinder * dos_cylindersectors;

	head = sector / dos_sectors;
	sector -= head * dos_sectors;

	*cylinderp = cylinder & 0xff;
	*headp = head;
	*sectorp = ((sector + 1) & 0x3f) + ((cylinder >> 2) & 0xc0);
}

void display_msdos()
{
	int i;

	printf("MSDOS partition tables:\n");
	for (i = 0; i < 4; ++i) {
		display_dospart(i);
	}
}

void display_dospart(i)
{
	struct dos_partition *mpartp;

	mpartp = &mboot->parts[i];
	if (mpartp->dp_typ == 0) {
		printf("%d: <UNUSED>\n", i);
		return;
	}
	printf("%d: %2x %2x %8d %8d (%d,%d,%d) (%d,%d,%d)\n",
	    i, mpartp->dp_typ, mpartp->dp_flag,
	    le32toh(mpartp->dp_start), le32toh(mpartp->dp_size),
	    mpartp->dp_scyl + ((mpartp->dp_ssect & 0xc0) << 2),
	    mpartp->dp_shd, (mpartp->dp_ssect - 1) & 0x3f, 
	    mpartp->dp_ecyl + ((mpartp->dp_esect & 0xc0) << 2),
	    mpartp->dp_ehd, (mpartp->dp_esect - 1) & 0x3f);
}

int read_block(buf, blk, what)
void *buf;
uint32_t blk;
char *what;
{
	if (lseek(fd, (off_t)(blk * 512), SEEK_SET) < 0) {
		perror("write_block lseek");
		giveup(what);
	}
	if (read(fd, buf, (off_t)512) != 512) {
		perror("write_block write");
		giveup(what);
	}
	return (0);
}
int xsum(buf)
uint32_t *buf;
{
	int i, n, sum;

	n = be32toh(buf[1]);
	sum = 0;
	for (i = 0; i < n; ++i)
		sum += be32toh(buf[i]);
	return(sum);
}

void giveup(what)
char *what;
{
	perror(what);
	close(fd);
	exit(1);
}
