/****************************************************************
Copyright (C) Lucent Technologies 1997
All Rights Reserved

Permission to use, copy, modify, and distribute this software and
its documentation for any purpose and without fee is hereby
granted, provided that the above copyright notice appear in all
copies and that both that the copyright notice and this
permission notice and warranty disclaimer appear in supporting
documentation, and that the name Lucent Technologies or any of
its entities not be used in advertising or publicity pertaining
to distribution of the software without specific, written prior
permission.

LUCENT DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS.
IN NO EVENT SHALL LUCENT OR ANY OF ITS ENTITIES BE LIABLE FOR ANY
SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER
IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,
ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF
THIS SOFTWARE.
****************************************************************/

#define DEBUG
#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <errno.h>
#include <stdlib.h>
#include <stdarg.h>

#include "awklib.h"
#include "awkgram.h"
#include "proto.h"

static	void	makefields(awk_t *, int, int);

static awkcell_t dollar0 = { OCELL, CFLD, NULL, "", 0.0, REC|STR|DONTFREE, NULL };
static awkcell_t dollar1 = { OCELL, CFLD, NULL, "", 0.0, FLD|STR|DONTFREE, NULL };

/* get ARGV[n] */
static char *
getargv(awk_t *awkp, int n)
{
	awkcell_t *x;
	char *s, temp[50];

	snprintf(temp, sizeof(temp), "%d", n);
	x = setawkvar(awkp, temp, "", 0.0, STR, awkp->ARGVtab);
	s = awklib_getsval(awkp, x);
	DPRINTF(awkp, ("getargv(%d) returns |%s|\n", n, s) );
	return s;
}

static void
initgetrec(awk_t *awkp)
{
	int i;
	char *p;

	for (i = 1; i < *awkp->ARGC; i++) {
		if (!awklib_isclvar(p = getargv(awkp, i))) {	/* find 1st real filename */
			awklib_setsval(awkp, awklib_lookup("FILENAME", awkp->symtab), getargv(awkp, i));
			return;
		}
		awklib_setclvar(awkp, p);	/* a commandline assignment before filename */
		awkp->argno++;
	}
	awkp->infile = stdin;		/* no filenames, so use stdin */
}

/* clean out fields n1 .. n2 inclusive */
/* nvals remain intact */
static void
cleanfld(awk_t *awkp, int n1, int n2)
{
	awkcell_t *p;
	int i;

	for (i = n1; i <= n2; i++) {
		p = awkp->fldtab[i];
		if (FREEABLE(p))
			XFREE(p->sval);
		p->sval = "";
		p->tval = FLD | STR | DONTFREE;
	}
}

/* make new fields up to at least $n */
static void
growfldtab(awk_t *awkp, unsigned n)
{
	unsigned nf = 2 * awkp->nfields;
	size_t s;

	if (n > nf)
		nf = n;
	s = (nf+1) * (sizeof(awkcell_t *));  /* freebsd: how much do we need? */
	if (s / sizeof(awkcell_t *) - 1 == nf) /* didn't overflow */
		awkp->fldtab = realloc(awkp->fldtab, s);
	else					/* overflow sizeof int */
		XFREE(awkp->fldtab);	/* make it null */
	if (awkp->fldtab == NULL)
		FATAL(awkp, "out of space creating %d fields", nf);
	makefields(awkp, awkp->nfields+1, nf);
	awkp->nfields = nf;
}

/* build fields from reg expr in FS */
static int
refldbld(awk_t *awkp, const char *rec, const char *fs)
{
	/* this relies on having fields[] the same length as $0 */
	/* the fields are all stored in this one array with \0's */
	char *fr;
	int i, tempstat, n;
	fa_t *pfa;

	n = strlen(rec);
	if (n > awkp->fieldssize) {
		XFREE(awkp->fields);
		if ((awkp->fields = malloc(n+1)) == NULL)
			FATAL(awkp, "out of space for fields in refldbld %d", n);
		awkp->fieldssize = n;
	}
	fr = awkp->fields;
	*fr = '\0';
	if (*rec == '\0')
		return 0;
	pfa = awklib_makedfa(awkp, fs, 1);
	DPRINTF(awkp, ("into refldbld, rec = <%s>, pat = <%s>\n", rec, fs) );
	tempstat = pfa->initstat;
	for (i = 1; ; i++) {
		if (i > awkp->nfields)
			growfldtab(awkp, i);
		if (FREEABLE(awkp->fldtab[i]))
			XFREE(awkp->fldtab[i]->sval);
		awkp->fldtab[i]->tval = FLD | STR | DONTFREE;
		awkp->fldtab[i]->sval = fr;
		DPRINTF(awkp, ("refldbld: i=%d\n", i) );
		if (awklib_nematch(awkp, pfa, rec)) {
			pfa->initstat = 2;	/* horrible coupling to b.c */
			DPRINTF(awkp, ("match %s (%d chars)\n", awkp->patbeg, awkp->patlen) );
			strncpy(fr, rec, ((char*)awkp->patbeg)-rec);
			fr += ((char*)awkp->patbeg) - rec + 1;
			*(fr-1) = '\0';
			rec = (char *)awkp->patbeg + awkp->patlen;
		} else {
			DPRINTF(awkp, ("no match %s\n", rec) );
			strcpy(fr, rec);
			pfa->initstat = tempstat;
			break;
		}
	}
	return i;		
}

static void
bclass(awk_t *awkp, int c)
{
	switch (c) {
	case '{': awkp->bracecnt++; break;
	case '}': awkp->bracecnt--; break;
	case '[': awkp->brackcnt++; break;
	case ']': awkp->brackcnt--; break;
	case '(': awkp->parencnt++; break;
	case ')': awkp->parencnt--; break;
	}
}

/* try to print context around error */
static void
eprint(awk_t *awkp)
{
	char *p, *q;

	if (awkp->stage == AWKLIB_ERROR_PRINTING || awkp->stage == AWKLIB_RUNNING || awkp->eprintc++ > 0)
		return;
	p = awkp->ep - 1;
	if (p > awkp->ebuf && *p == '\n')
		p--;
	for ( ; p > awkp->ebuf && *p != '\n' && *p != '\0'; p--)
		;
	while (*p == '\n')
		p++;
	(void) fprintf(stderr, " context is\n\t");
	for (q=awkp->ep-1; q>=p && *q!=' ' && *q!='\t' && *q!='\n'; q--)
		;
	for ( ; p < q; p++)
		if (*p)
			putc(*p, stderr);
	(void) fprintf(stderr, " >>> ");
	for ( ; p < awkp->ep; p++)
		if (*p)
			putc(*p, stderr);
	(void) fprintf(stderr, " <<< ");
#if 0
	/*
	 * The following code was used to print the rest of the line of
	 * error context. It naively counts brackets, parens and braces in
	 * order to minimize the parsing effect of dropping the rest of the
	 * line but it does not work in all the cases. It is too much work
	 * to save the current program input point and restore it in all the
	 * cases just for the benefit of error printing so for now this
	 * code is disabled. In particular this code is confused if the
	 * [ { ( ) } ] is inside a quoted string or a pattern.
	 */
	if (*awkp->ep) {
		int c;
		while ((c = input()) != '\n' && c != '\0' && c != EOF) {
			putc(c, stderr);
			bclass(awkp, c);
		}
	}
#endif
	putc('\n', stderr);
	awkp->ep = awkp->ebuf;
}

/* current source file name */
static char *
cursource(awk_t *awkp)
{
	return (awkp->scriptc > 0) ? awkp->scripts[awkp->curscript] : NULL;
}

static void
bcheck2(int n, int c1, int c2)
{
	if (n == 1)
		(void) fprintf(stderr, "\tmissing %c\n", c2);
	else if (n > 1)
		(void) fprintf(stderr, "\t%d missing %c's\n", n, c2);
	else if (n == -1)
		(void) fprintf(stderr, "\textra %c\n", c2);
	else if (n < -1)
		(void) fprintf(stderr, "\t%d extra %c's\n", -n, c2);
}

static void
error(awk_t *awkp)
{
	(void) fprintf(stderr, "\n");
	if (awkp->stage != AWKLIB_ERROR_PRINTING && awkp->NR && *awkp->NR > 0) {
		(void) fprintf(stderr, " input record number %d", (int) (*awkp->FNR));
		if (strcmp(*awkp->FILENAME, "-") != 0)
			(void) fprintf(stderr, ", file %s", *awkp->FILENAME);
		(void) fprintf(stderr, "\n");
	}
	if (awkp->stage != AWKLIB_ERROR_PRINTING && awkp->curnode)
		(void) fprintf(stderr, " source line number %d", awkp->curnode->lineno);
	else if (awkp->stage != AWKLIB_ERROR_PRINTING && awkp->lineno)
		(void) fprintf(stderr, " source line number %d", awkp->lineno);
	if (awkp->stage == AWKLIB_COMPILING && cursource(awkp) != NULL)
		(void) fprintf(stderr, " source file %s", cursource(awkp));
	(void) fprintf(stderr, "\n");
	eprint(awkp);
}


void
awklib_recinit(awk_t *awkp, unsigned int n)
{
	if ((awkp->record = malloc(n)) == NULL ||
	    (awkp->fields = malloc(n)) == NULL ||
	    (awkp->fldtab = malloc((awkp->nfields+1) * sizeof(awkcell_t *))) == NULL ||
	    (awkp->fldtab[0] = malloc(sizeof(awkcell_t))) == NULL)
		FATAL(awkp, "out of space for $0 and fields");
	*awkp->fldtab[0] = dollar0;
	awkp->fldtab[0]->sval = (char *)awkp->record;
	awkp->fldtab[0]->nval = awklib_tostring(awkp, "0");
	makefields(awkp, 1, awkp->nfields);
}

/* create $n1..$n2 inclusive */
static void
makefields(awk_t *awkp, int n1, int n2)
{
	char temp[50];
	int i;

	for (i = n1; i <= n2; i++) {
		awkp->fldtab[i] = malloc(sizeof (awkcell_t));
		if (awkp->fldtab[i] == NULL)
			FATAL(awkp, "out of space in makefields %d", i);
		*awkp->fldtab[i] = dollar1;
		snprintf(temp, sizeof(temp), "%d", i);
		awkp->fldtab[i]->nval = awklib_tostring(awkp, temp);
	}
}

/* get next input record */
int
awklib_getrec(awk_t *awkp, uint8_t **pbuf, int *pbufsize, int isrecord)
{			/* note: cares whether buf == record */
	int c;
	uint8_t *buf = *pbuf;
	uint8_t saveb0;
	int bufsize = *pbufsize, savebufsize = bufsize;

	if (!awkp->init_getrec) {
		awkp->init_getrec = 1;
		initgetrec(awkp);
	}
	DPRINTF(awkp, ("RS=<%s>, FS=<%s>, ARGC=%g, FILENAME=%s\n",
		*awkp->RS, *awkp->FS, *awkp->ARGC, *awkp->FILENAME) );
	if (isrecord) {
		awkp->donefld = 0;
		awkp->donerec = 1;
	}
	saveb0 = buf[0];
	buf[0] = 0;
	while (awkp->argno < *awkp->ARGC || awkp->infile == stdin) {
		DPRINTF(awkp, ("argno=%d, file=|%s|\n", awkp->argno, awkp->file) );
		if (awkp->infile == NULL) {	/* have to open a new file */
			awkp->file = getargv(awkp, awkp->argno);
			if (*awkp->file == '\0') {	/* it's been zapped */
				awkp->argno++;
				continue;
			}
			if (awklib_isclvar(awkp->file)) {	/* a var=value arg */
				awklib_setclvar(awkp, awkp->file);
				awkp->argno++;
				continue;
			}
			*awkp->FILENAME = awkp->file;
			DPRINTF(awkp, ("opening file %s\n", awkp->file) );
			if (*awkp->file == '-' && *(awkp->file+1) == '\0')
				awkp->infile = stdin;
			else if ((awkp->infile = fopen(awkp->file, "r")) == NULL)
				FATAL(awkp, "can't open file %s", awkp->file);
			awklib_setfval(awkp, awkp->fnrloc, 0.0);
		}
		c = awklib_readrec(awkp, &buf, &bufsize, awkp->infile);
		if (c != 0 || buf[0] != '\0') {	/* normal record */
			if (isrecord) {
				if (FREEABLE(awkp->fldtab[0]))
					XFREE(awkp->fldtab[0]->sval);
				awkp->fldtab[0]->sval = (char *)buf;	/* buf == record */
				awkp->fldtab[0]->tval = REC | STR | DONTFREE;
				if (awklib_is_number(awkp->fldtab[0]->sval)) {
					awkp->fldtab[0]->fval = atof(awkp->fldtab[0]->sval);
					awkp->fldtab[0]->tval |= NUM;
				}
			}
			awklib_setfval(awkp, awkp->nrloc, awkp->nrloc->fval+1);
			awklib_setfval(awkp, awkp->fnrloc, awkp->fnrloc->fval+1);
			*pbuf = buf;
			*pbufsize = bufsize;
			return 1;
		}
		/* EOF arrived on this file; set up next */
		if (awkp->infile != stdin)
			fclose(awkp->infile);
		awkp->infile = NULL;
		awkp->argno++;
	}
	buf[0] = saveb0;
	*pbuf = buf;
	*pbufsize = savebufsize;
	return 0;	/* true end of file */
}

void
awklib_nextfile(awk_t *awkp)
{
	if (awkp->infile != NULL && awkp->infile != stdin)
		fclose(awkp->infile);
	awkp->infile = NULL;
	awkp->argno++;
}

/* read one record into buf */
int
awklib_readrec(awk_t *awkp, uint8_t **pbuf, int *pbufsize, FILE *inf)
{
	int sep, c;
	uint8_t *rr, *buf = *pbuf;
	int bufsize = *pbufsize;
	size_t len;

	if ((len = strlen(*awkp->FS)) <= awkp->len_inputFS) {
		strcpy(awkp->inputFS, *awkp->FS);	/* for subsequent field splitting */
	} else {
		if (awkp->inputFS != awkp->static_inputFS)
			free(awkp->inputFS);
		awkp->inputFS = malloc(len + 1);
		if (awkp->inputFS == NULL)
			FATAL(awkp, "field separator %.10s... is too long", *awkp->FS);
		awkp->len_inputFS = len;
		memcpy(awkp->inputFS, *awkp->FS, len + 1);
	}
	if ((sep = **awkp->RS) == 0) {
		sep = '\n';
		while ((c=getc(inf)) == '\n' && c != EOF)	/* skip leading \n's */
			;
		if (c != EOF)
			ungetc(c, inf);
	}
	for (rr = buf; ; ) {
		for (; (c=getc(inf)) != sep && c != EOF; ) {
			if (rr-buf+1 > bufsize)
				if (!awklib_adjbuf(awkp, &buf, &bufsize, 1+rr-buf, awkp->recsize, &rr, "readrec 1"))
					FATAL(awkp, "input record `%.30s...' too long", buf);
			*rr++ = c;
		}
		if (**awkp->RS == sep || c == EOF)
			break;
		if ((c = getc(inf)) == '\n' || c == EOF) /* 2 in a row */
			break;
		if (!awklib_adjbuf(awkp, &buf, &bufsize, 2+rr-buf, awkp->recsize, &rr, "readrec 2"))
			FATAL(awkp, "input record `%.30s...' too long", buf);
		*rr++ = '\n';
		*rr++ = c;
	}
	if (!awklib_adjbuf(awkp, &buf, &bufsize, 1+rr-buf, awkp->recsize, &rr, "readrec 3"))
		FATAL(awkp, "input record `%.30s...' too long", buf);
	*rr = 0;
	DPRINTF(awkp, ("readrec saw <%s>, returns %d\n", buf, c == EOF && rr == buf ? 0 : 1) );
	*pbuf = buf;
	*pbufsize = bufsize;
	return c == EOF && rr == buf ? 0 : 1;
}

/* set var=value from s */
void
awklib_setclvar(awk_t *awkp, char *s)
{
	char *p;
	awkcell_t *q;

	for (p=s; *p != '='; p++)
		;
	*p++ = 0;
	p = awklib_qstring(awkp, p, '\0');
	q = setawkvar(awkp, s, p, 0.0, STR, awkp->symtab);
	awklib_setsval(awkp, q, p);
	if (awklib_is_number(q->sval)) {
		q->fval = atof(q->sval);
		q->tval |= NUM;
	}
	DPRINTF(awkp, ("command line set %s to |%s|\n", s, p) );
}


/* create fields from current record */
void
awklib_fldbld(awk_t *awkp)
{
	/* this relies on having fields[] the same length as $0 */
	/* the fields are all stored in this one array with \0's */
	char *r, *fr, sep;
	awkcell_t *p;
	int i, j, n;

	if (awkp->donefld)
		return;
	if (!ISSTR(awkp->fldtab[0]))
		awklib_getsval(awkp, awkp->fldtab[0]);
	r = awkp->fldtab[0]->sval;
	n = strlen(r);
	if (n > awkp->fieldssize) {
		XFREE(awkp->fields);
		if ((awkp->fields = malloc(n+1)) == NULL)
			FATAL(awkp, "out of space for fields in awklib_fldbld %d", n);
		awkp->fieldssize = n;
	}
	fr = awkp->fields;
	i = 0;	/* number of fields accumulated here */
	if (awkp->inputFS[0] && awkp->inputFS[1]) {	/* it's a regular expression */
		i = refldbld(awkp, r, awkp->inputFS);
	} else if ((sep = *awkp->inputFS) == ' ') {	/* default whitespace */
		for (i = 0; ; ) {
			while (*r == ' ' || *r == '\t' || *r == '\n')
				r++;
			if (*r == 0)
				break;
			i++;
			if (i > awkp->nfields)
				growfldtab(awkp, i);
			if (FREEABLE(awkp->fldtab[i]))
				XFREE(awkp->fldtab[i]->sval);
			awkp->fldtab[i]->sval = fr;
			awkp->fldtab[i]->tval = FLD | STR | DONTFREE;
			do
				*fr++ = *r++;
			while (*r != ' ' && *r != '\t' && *r != '\n' && *r != '\0');
			*fr++ = 0;
		}
		*fr = 0;
	} else if ((sep = *awkp->inputFS) == 0) {		/* new: FS="" => 1 char/field */
		for (i = 0; *r != 0; r++) {
			char buf[2];
			i++;
			if (i > awkp->nfields)
				growfldtab(awkp, i);
			if (FREEABLE(awkp->fldtab[i]))
				XFREE(awkp->fldtab[i]->sval);
			buf[0] = *r;
			buf[1] = 0;
			awkp->fldtab[i]->sval = awklib_tostring(awkp, buf);
			awkp->fldtab[i]->tval = FLD | STR;
		}
		*fr = 0;
	} else if (*r != 0) {	/* if 0, it's a null field */
		/* subtlecase : if length(FS) == 1 && length(RS > 0)
		 * \n is NOT a field separator (cf awk book 61,84).
		 * this variable is tested in the inner while loop.
		 */
		int rtest = '\n';  /* normal case */
		if (strlen(*awkp->RS) > 0)
			rtest = '\0';
		for (;;) {
			i++;
			if (i > awkp->nfields)
				growfldtab(awkp, i);
			if (FREEABLE(awkp->fldtab[i]))
				XFREE(awkp->fldtab[i]->sval);
			awkp->fldtab[i]->sval = fr;
			awkp->fldtab[i]->tval = FLD | STR | DONTFREE;
			while (*r != sep && *r != rtest && *r != '\0')	/* \n is always a separator */
				*fr++ = *r++;
			*fr++ = 0;
			if (*r++ == 0)
				break;
		}
		*fr = 0;
	}
	if (i > awkp->nfields)
		FATAL(awkp, "record `%.30s...' has too many fields; can't happen", r);
	cleanfld(awkp, i+1, awkp->lastfld);	/* clean out junk from previous record */
	awkp->lastfld = i;
	awkp->donefld = 1;
	for (j = 1; j <= awkp->lastfld; j++) {
		p = awkp->fldtab[j];
		if(awklib_is_number(p->sval)) {
			p->fval = atof(p->sval);
			p->tval |= NUM;
		}
	}
	awklib_setfval(awkp, awkp->nfloc, (awkfloat_t) awkp->lastfld);
	if (awkp->dbg) {
		for (j = 0; j <= awkp->lastfld; j++) {
			p = awkp->fldtab[j];
			printf("field %d (%s): |%s|\n", j, p->nval, p->sval);
		}
	}
}

/* add field n after end of existing lastfld */
void
awklib_newfld(awk_t *awkp, int n)
{
	if (n > awkp->nfields)
		growfldtab(awkp, n);
	cleanfld(awkp, awkp->lastfld+1, n);
	awkp->lastfld = n;
	awklib_setfval(awkp, awkp->nfloc, (awkfloat_t) n);
}

/* get nth field */
awkcell_t *
awklib_fieldadr(awk_t *awkp, int n)
{
	if (n < 0)
		FATAL(awkp, "trying to access out of range field %d", n);
	if (n > awkp->nfields)	/* fields after NF are empty */
		growfldtab(awkp, n);	/* but does not increase NF */
	return(awkp->fldtab[n]);
}

/* create $0 from $1..$NF if necessary */
void
awklib_recbld(awk_t *awkp)
{
	int i;
	uint8_t *r;
	char *p;

	if (awkp->donerec == 1)
		return;
	r = awkp->record;
	for (i = 1; i <= *awkp->NF; i++) {
		p = awklib_getsval(awkp, awkp->fldtab[i]);
		if (!awklib_adjbuf(awkp, &awkp->record, &awkp->recsize, 1+strlen(p)+(int)(r - awkp->record), awkp->recsize, &r, "recbld 1"))
			FATAL(awkp, "created $0 `%.30s...' too long", awkp->record);
		while ((*r = *p++) != 0)
			r++;
		if (i < *awkp->NF) {
			if (!awklib_adjbuf(awkp, &awkp->record, &awkp->recsize, 2+strlen(*awkp->OFS)+(int)(r - awkp->record), awkp->recsize, &r, "recbld 2"))
				FATAL(awkp, "created $0 `%.30s...' too long", awkp->record);
			for (p = *awkp->OFS; (*r = *p++) != 0; )
				r++;
		}
	}
	if (!awklib_adjbuf(awkp, &awkp->record, &awkp->recsize, 2+(int)(r - awkp->record), awkp->recsize, &r, "recbld 3"))
		FATAL(awkp, "built giant record `%.30s...'", awkp->record);
	*r = '\0';
	DPRINTF(awkp, ("in recbld inputFS=%s, fldtab[0]=%p\n", awkp->inputFS, awkp->fldtab[0]) );

	if (FREEABLE(awkp->fldtab[0]))
		XFREE(awkp->fldtab[0]->sval);
	awkp->fldtab[0]->tval = REC | STR | DONTFREE;
	awkp->fldtab[0]->sval = (char *)awkp->record;

	DPRINTF(awkp, ("in recbld inputFS=%s, fldtab[0]=%p\n", awkp->inputFS, awkp->fldtab[0]) );
	DPRINTF(awkp, ("recbld = |%s|\n", awkp->record) );
	awkp->donerec = 1;
}

void
awklib_error(const char *s)
{
	(void) fprintf(stderr, "%s", s);
}

void
SYNTAX(awk_t *awkp, const char *fmt, ...)
{
	va_list varg;

	if (awkp->syntaxc++ > 2)
		return;
	(void) fprintf(stderr, "%s: ", awkp->cmdname);
	va_start(varg, fmt);
	vfprintf(stderr, fmt, varg);
	va_end(varg);
	(void) fprintf(stderr, " at source line %d", awkp->lineno);
	if (awkp->curfname != NULL)
		(void) fprintf(stderr, " in function %s", awkp->curfname);
	if (awkp->stage == AWKLIB_COMPILING && cursource(awkp) != NULL)
		(void) fprintf(stderr, " source file %s", cursource(awkp));
	(void) fprintf(stderr, "\n");
	awkp->errorflag = 2;
	eprint(awkp);
}

void
awklib_fpecatch(int n)
{
	(void) fprintf(stderr, "floating point exception %d", n);
}

void
awklib_bracecheck(awk_t *awkp)
{
	int c;

	if (awkp->bracechecked++)
		return;
	while ((c = input(awkp)) != EOF && c != '\0')
		bclass(awkp, c);
	bcheck2(awkp->bracecnt, '{', '}');
	bcheck2(awkp->brackcnt, '[', ']');
	bcheck2(awkp->parencnt, '(', ')');
}

void
FATAL(awk_t *awkp, const char *fmt, ...)
{
	va_list varg;

	fflush(stdout);
	(void) fprintf(stderr, "%s: ", awkp->cmdname);
	va_start(varg, fmt);
	vfprintf(stderr, fmt, varg);
	va_end(varg);
	error(awkp);
	if (awkp->dbg > 1)		/* core dump if serious debugging on */
		abort();
	exit(2);
}

void
WARNING(awk_t *awkp, const char *fmt, ...)
{
	va_list varg;

	fflush(stdout);
	(void) fprintf(stderr, "%s: ", awkp->cmdname);
	va_start(varg, fmt);
	vfprintf(stderr, fmt, varg);
	va_end(varg);
	error(awkp);
}

double
errcheck(awk_t *awkp, double x, const char *s)
{

	if (errno == EDOM) {
		errno = 0;
		WARNING(awkp, "%s argument out of domain", s);
		x = 1;
	} else if (errno == ERANGE) {
		errno = 0;
		WARNING(awkp, "%s result out of range", s);
		x = 1;
	}
	return x;
}

/* is s of form var=something ? */
int
awklib_isclvar(const char *s)
{
	const char *os = s;

	if (!(isalpha((uint8_t) *s) || *s == '_')) {
		return 0;
	}
	for ( ; *s; s++) {
		if (!(isalnum((uint8_t) *s) || *s == '_')) {
			break;
		}
	}
	return *s == '=' && s > os && *(s+1) != '=';
}

/* strtod is supposed to be a proper test of what's a valid number */
/* appears to be broken in gcc on linux: thinks 0x123 is a valid FP number */
/* wrong: violates 4.10.1.4 of ansi C standard */

#include <math.h>
int awklib_is_number(const char *s)
{
	double r;
	char *ep;

	errno = 0;
	r = strtod(s, &ep);
	if (ep == s || r == HUGE_VAL || errno == ERANGE)
		return 0;
	while (*ep == ' ' || *ep == '\t' || *ep == '\n')
		ep++;
	if (*ep == '\0')
		return 1;
	else
		return 0;
}
