
/*
 * util.c
 *
 *   July 10, 1994 -- Lawrence Kesteloot
 *
 *   Random utility functions.
 */

#include <stdio.h>
#include <unistd.h>
#include <stdlib.h>
#include <fcntl.h>
#include <ctype.h>
#include <string.h>

#include "util.h"
#include "hfs.h"

#define	MAXCACHE	16

/*
 * getbytes()
 *
 *   Reads a chunk of bytes from a file.
 */

void getbytes (int f, int loc, uchar *buf, int len)
{
	if (lseek (f, loc, SEEK_SET) == -1) {
		perror ("getbytes(): lseek");
		exit (1);
	}
	if (read (f, buf, len) == -1) {
		perror ("getbytes(): read");
		exit (1);
	}
}

/*
 * printbytes()
 *
 *   Prints an array in hex and ASCII.
 */

void printbytes (uchar *buf, int len)
{
	int	i, j;

	for (i = 0; i < len; i += 16) {
		printf ("%04x: ", i);
		for (j = 0; j < 16; j++) {
			if (j == 8) {
				printf (" ");
			}
			printf ("%02x ", (unsigned int)buf[i + j]);
		}
		printf ("  ");
		for (j = 0; j < 16; j++) {
			if (j == 8) {
				printf (" ");
			}
			printf ("%c", isprint (buf[i + j]) ? buf[i + j] : '.');
		}
		printf ("\n");
	}
}

/*
 * searchbytes()
 *
 *   Searches through an array for a substring.  Returns the index
 *   into the array of the first match, or -1 if it wasn't found.
 */

int searchbytes (uchar *in, int insize, uchar *what, int whatsize)
{
	int index;

	for (index = 0; index < insize - whatsize; index++) {
		if (memcmp (in + index, what, whatsize) == 0) {
			return index;
		}
	}

	return -1;
}

/*
 * readshortblock()
 *
 *   Reads a block of BLKSIZ bytes from the disk, using caching.  This
 *   is used by the routines that access the btree and such.  It should
 *   not be used to cache actual file data.
 *
 *   With MAXCACHE == 16, this routine gets about 98% cache hit in
 *   a 500k copy and cuts copy time by about a third.
 *
 *   XXX: Maybe should check to see if it's the same file.
 */

void readshortblock (int f, ulong loc, uchar *buf)
{
	static int	cnum;
	static ulong	cloc[MAXCACHE];
	static uchar	cbuf[MAXCACHE][BLKSIZ];
	int		i;

	for (i = 0; i < cnum; i++) {
		if (cloc[i] == loc) {
			memcpy (buf, cbuf[i], BLKSIZ);
			return;
		}
	}

	if (lseek (f, loc, SEEK_SET) == -1) {
		perror ("readshortblock(): lseek");
		exit (1);
	}
	if (read (f, buf, BLKSIZ) == -1) {
		perror ("readshortblock(): read");
		exit (1);
	}

	if (cnum < MAXCACHE) {
		cloc[cnum] = loc;
		memcpy (cbuf[cnum], buf, BLKSIZ);
		cnum++;
	} else {
		/* This works just fine: */
		i = rand () % MAXCACHE;
		cloc[i] = loc;
		memcpy (cbuf[i], buf, BLKSIZ);
	}
}

/*
 * evalexpr()
 *
 *   Evaluates an expression and returns its value.  This is the
 *   standard recursive breakout algorithm.  The argument must not
 *   have any spaces.  This routine is not error tolerant.
 */

static int evalexpr (char *s, int l, int r)
{
	int	i, paren;

	paren = 0;
	for (i = r - 1; i > l; i--) {
		if (s[i] == ')') {
			paren++;
		}
		if (s[i] == '(') {
			paren--;
		}
		if (s[i] == '+' && paren == 0) {
			return evalexpr (s, l, i - 1) + evalexpr (s, i + 1, r);
		}
		if (s[i] == '-' && paren == 0) {
			return evalexpr (s, l, i - 1) - evalexpr (s, i + 1, r);
		}
	}

	paren = 0;
	for (i = r - 1; i > l; i--) {
		if (s[i] == ')') {
			paren++;
		}
		if (s[i] == '(') {
			paren--;
		}
		if (s[i] == '*' && paren == 0) {
			return evalexpr (s, l, i - 1) * evalexpr (s, i + 1, r);
		}
		if (s[i] == '/' && paren == 0) {
			return evalexpr (s, l, i - 1) / evalexpr (s, i + 1, r);
		}
	}

	if (s[l] == '(' && s[r] == ')') {
		return evalexpr (s, l + 1, r - 1);
	}

	return strtol (s + l, NULL, 0);
}

/*
 * expr()
 *
 *   Front-end to evalexpr().
 */

int expr (char *s)
{
	return evalexpr (s, 0, strlen (s) - 1);
}
