/*--------------------------------------------------------------------------+
$Id: SimulinkModelTest.java 26285 2010-02-18 11:22:54Z juergens $
|                                                                          |
| Copyright 2005-2010 Technische Universitaet Muenchen                     |
|                                                                          |
| Licensed under the Apache License, Version 2.0 (the "License");          |
| you may not use this file except in compliance with the License.         |
| You may obtain a copy of the License at                                  |
|                                                                          |
|    http://www.apache.org/licenses/LICENSE-2.0                            |
|                                                                          |
| Unless required by applicable law or agreed to in writing, software      |
| distributed under the License is distributed on an "AS IS" BASIS,        |
| WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. |
| See the License for the specific language governing permissions and      |
| limitations under the License.                                           |
+--------------------------------------------------------------------------*/
package edu.tum.cs.simulink.model;

import java.io.FileNotFoundException;
import java.util.ArrayList;

import edu.tum.cs.commons.clone.DeepCloneException;
import edu.tum.cs.commons.collections.IdentityHashSet;
import edu.tum.cs.commons.logging.SimpleLogger;
import edu.tum.cs.commons.test.CCSMTestCaseBase;
import edu.tum.cs.commons.test.DeepCloneTestUtils;
import edu.tum.cs.simulink.builder.SimulinkModelBuilder;
import edu.tum.cs.simulink.builder.SimulinkModelBuildingException;

/**
 * Test for the Stateflow part of the model.
 * 
 * @author deissenb
 * @author $Author: juergens $
 * @version $Rev: 26285 $
 * @levd.rating GREEN Hash: 7B81E960677A2270C51A6CA89CC05E0C
 */
public class SimulinkModelTest extends CCSMTestCaseBase {

	/** Model under test. */
	private SimulinkModel model;

	/** Load model and set block. */
	@Override
	public void setUp() throws FileNotFoundException,
			SimulinkModelBuildingException {
		SimulinkModelBuilder builder = new SimulinkModelBuilder(
				useTestFile("remove.mdl"), new SimpleLogger());
		model = builder.buildModel();
	}

	/** Test if annotation removal works. */
	public void testRemoveAnnotation() {
		IdentityHashSet<SimulinkAnnotation> annotations = DeepCloneTestUtils
				.getAllReferencedObjects(model, SimulinkAnnotation.class,
						getClass().getPackage().getName());

		for (SimulinkAnnotation annotation : annotations) {
			SimulinkBlock parent = annotation.getParent();
			annotation.remove();
			assertFalse(parent.getAnnotations().contains(annotation));
			assertNull(annotation.getParent());
		}
	}

	/** Test if block removal works. */
	public void testRemoveBlock() {
		// this is not done in a recursive manner as we would need to care about
		// removal order
		for (SimulinkBlock block : new ArrayList<SimulinkBlock>(model
				.getSubBlocks())) {
			SimulinkBlock parent = block.getParent();
			block.remove();
			assertNull(block.getParent());
			assertFalse(parent.getSubBlocks().contains(block));
			assertTrue(block.getSubBlocks().isEmpty());
			assertTrue(block.getOutPorts().isEmpty());
			assertTrue(block.getInPorts().isEmpty());
		}
	}

	/** Test if inport removal works. */
	public void testRemoveInPort() {
		IdentityHashSet<SimulinkInPort> inPorts = DeepCloneTestUtils
				.getAllReferencedObjects(model, SimulinkInPort.class,
						getClass().getPackage().getName());

		for (SimulinkInPort inPort : inPorts) {
			SimulinkBlock block = inPort.getBlock();
			inPort.remove();
			assertFalse(block.getOutPorts().contains(inPort));
			assertNull(inPort.getLine());
			assertNull(inPort.getBlock());
		}
	}

	/** Test if line removal works. */
	public void testRemoveLine() {
		IdentityHashSet<SimulinkLine> lines = DeepCloneTestUtils
				.getAllReferencedObjects(model, SimulinkLine.class, getClass()
						.getPackage().getName());

		for (SimulinkLine line : lines) {
			SimulinkOutPort src = line.getSrcPort();
			SimulinkInPort dst = line.getDstPort();
			line.remove();
			assertFalse(src.getLines().contains(line));
			assertNull(dst.getLine());
			assertNull(line.getSrcPort());
			assertNull(line.getDstPort());
		}
	}

	/** Test if out port removal works. */
	public void testRemoveOutPort() {
		IdentityHashSet<SimulinkOutPort> outPorts = DeepCloneTestUtils
				.getAllReferencedObjects(model, SimulinkOutPort.class,
						getClass().getPackage().getName());

		for (SimulinkOutPort outPort : outPorts) {
			SimulinkBlock block = outPort.getBlock();
			outPort.remove();
			assertFalse(block.getOutPorts().contains(outPort));
			assertTrue(outPort.getLines().isEmpty());
			assertNull(outPort.getBlock());
		}
	}

	/**
	 * Test if deep cloning works for flattened models, ie. models with
	 * cross-subsystem lines.
	 */
	public void testDeepCloneForFlattenedmodels() throws FileNotFoundException,
			SimulinkModelBuildingException, DeepCloneException {
		SimulinkModelBuilder builder = new SimulinkModelBuilder(
				useTestFile("cross_subsystem_line.mdl"), new SimpleLogger());
		model = builder.buildModel();
		SimulinkBlock in = model
				.getBlock("cross_subsystem_line/Subsystem1/In1");
		SimulinkBlock out = model
				.getBlock("cross_subsystem_line/Subsystem2/Out1");

		SimulinkOutPort outPort = new SimulinkOutPort(in, "2");
		SimulinkInPort inPort = new SimulinkInPort(out, "2");

		new SimulinkLine(outPort, inPort);
		model.deepClone();
	}

}