/*--------------------------------------------------------------------------+
$Id: XMLReaderTest.java 26268 2010-02-18 10:44:30Z juergens $
|                                                                          |
| Copyright 2005-2010 Technische Universitaet Muenchen                     |
|                                                                          |
| Licensed under the Apache License, Version 2.0 (the "License");          |
| you may not use this file except in compliance with the License.         |
| You may obtain a copy of the License at                                  |
|                                                                          |
|    http://www.apache.org/licenses/LICENSE-2.0                            |
|                                                                          |
| Unless required by applicable law or agreed to in writing, software      |
| distributed under the License is distributed on an "AS IS" BASIS,        |
| WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. |
| See the License for the specific language governing permissions and      |
| limitations under the License.                                           |
+--------------------------------------------------------------------------*/
package edu.tum.cs.commons.xml;

import java.io.File;
import java.io.IOException;

import org.xml.sax.SAXException;

import edu.tum.cs.commons.test.CCSMTestCaseBase;
import edu.tum.cs.commons.xml.example.EXMLAttribute;
import edu.tum.cs.commons.xml.example.EXMLElement;
import edu.tum.cs.commons.xml.example.Model;
import edu.tum.cs.commons.xml.example.ModelPersistenceException;
import edu.tum.cs.commons.xml.example.ModelReader;
import edu.tum.cs.commons.xml.example.ModelResolver;
import edu.tum.cs.commons.xml.example.ModelUtils;

/**
 * Test class for {@link XMLReader}.
 * 
 * @author Florian Deissenboeck
 * @author $Author: juergens $
 * @version $Rev: 26268 $
 * @levd.rating GREEN Hash: 5A4CCB1006F7E0B2811E44DFF0DCF13F
 */
public class XMLReaderTest extends CCSMTestCaseBase {

	/**
	 * Create an example model, write it to a file, re-read and compare it to
	 * the original model.
	 */
	public void test() throws IOException, ModelPersistenceException {

		Model originalModel = ModelUtils.createExampleModel();

		File file = new File(getTmpDirectory(), "test.xml");

		ModelUtils.writeModel(originalModel, file);

		Model newModel = new Model();

		assertFalse(originalModel.equals(newModel));

		ModelReader reader = new ModelReader(file, newModel);

		reader.readModel();

		assertTrue(originalModel.equals(newModel));
	}

	/** Test if an illegal model raises the expected exception. */
	public void testIllegalModel() throws IOException {
		Model originalModel = ModelUtils.createIllegalExampleModel();

		File file = new File(getTmpDirectory(), "test.xml");

		ModelUtils.writeModel(originalModel, file);

		Model newModel = new Model();

		ModelReader reader = new ModelReader(file, newModel);

		try {
			reader.readModel();
			fail("expected exception");
		} catch (ModelPersistenceException e) {
			// expected
		}
	}

	/**
	 * Tests if Processors can access XML elements arbitrarily deep in the XML
	 * document
	 */
	public void testProcessDecendants() throws Exception {
		File testFile = useTestFile("test04.xml");

		OrderCounter orderCounter = new OrderCounter(testFile);
		int orderCount = orderCounter.countOrders();

		assertEquals(6, orderCount);
	}

	/** Counts the number of Order elements, on */
	private class OrderCounter extends
			XMLReader<EXMLElement, EXMLAttribute, ModelPersistenceException> {

		/** Constructor */
		public OrderCounter(File file) {
			super(file, new ModelResolver());
		}

		/** Count number of order processed tags */
		public int countOrders() throws ModelPersistenceException {
			try {
				parseFile();
			} catch (SAXException e) {
				throw new ModelPersistenceException("Parsing Error: "
						+ e.getMessage());
			} catch (IOException e) {
				throw new ModelPersistenceException("IO Error");
			}

			OrderProcessor orderProcessor = new OrderProcessor();
			processDecendantElements(orderProcessor);

			return orderProcessor.getOrderCount();
		}

		/** Processes Order elements */
		private class OrderProcessor implements
				IXMLElementProcessor<EXMLElement, ModelPersistenceException> {

			/** Counts number of processed order elements */
			private int orderCount = 0;

			/** This method defines the target element of this processor. */
			public EXMLElement getTargetElement() {
				return EXMLElement.ORDER;
			}

			/** Process {@link EXMLElement#ORDER} element. */
			public void process() {
				orderCount++;
			}

			/** Gets the order count */
			public int getOrderCount() {
				return orderCount;
			}
		}

	}

}