/*--------------------------------------------------------------------------+
$Id: ReflectionUtilsTest.java 26268 2010-02-18 10:44:30Z juergens $
|                                                                          |
| Copyright 2005-2010 Technische Universitaet Muenchen                     |
|                                                                          |
| Licensed under the Apache License, Version 2.0 (the "License");          |
| you may not use this file except in compliance with the License.         |
| You may obtain a copy of the License at                                  |
|                                                                          |
|    http://www.apache.org/licenses/LICENSE-2.0                            |
|                                                                          |
| Unless required by applicable law or agreed to in writing, software      |
| distributed under the License is distributed on an "AS IS" BASIS,        |
| WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. |
| See the License for the specific language governing permissions and      |
| limitations under the License.                                           |
+--------------------------------------------------------------------------*/
package edu.tum.cs.commons.reflect;

import java.io.File;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.util.AbstractCollection;
import java.util.AbstractList;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;

import junit.framework.TestCase;
import edu.tum.cs.commons.string.StringUtils;

/**
 * Tests for <code>ReflectionUtils</code>
 * 
 * 
 * @author Florian Deissenboeck
 * @author $Author: juergens $
 * @version $Rev: 26268 $ <<<<<<< .mine
 * @levd.rating GREEN Hash: 128B02EBEAEF446CF12D7D2CF2E315B2 =======
 * @levd.rating YELLOW Rev: 16049 >>>>>>> .r16049
 */
public class ReflectionUtilsTest extends TestCase {

	/** For testing purposes. */
	private enum TestEnum {
		/** value */
		ALPHA
	}

	/**
	 * Test for {@link ReflectionUtils#getFormalParameters(Method)}.
	 */
	public void testGetFormalParameters() throws SecurityException,
			NoSuchMethodException {

		Class<?>[] parameterTypes0 = { String.class };
		check(String.class, "compareTo", parameterTypes0);

		Class<?>[] parameterTypes1 = { Locale.class, String.class,
				Object[].class };
		check(String.class, "format", parameterTypes1);

		Class<?>[] parameterTypes2 = {};
		check(String.class, "trim", parameterTypes2);

		Class<?>[] parameterTypes3 = { char[].class, int.class, int.class };
		check(String.class, "copyValueOf", parameterTypes3);
	}

	/**
	 * Check if for a given method
	 * {@link ReflectionUtils#getFormalParameters(Method)} creates the correct
	 * formal parameters.
	 */
	private void check(Class<?> clazz, String methodName,
			Class<?>[] parameterTypes) throws SecurityException,
			NoSuchMethodException {
		Method method = clazz.getMethod(methodName, parameterTypes);
		FormalParameter[] parameters = ReflectionUtils
				.getFormalParameters(method);

		assertEquals(parameterTypes.length, parameters.length);

		for (int i = 0; i < parameterTypes.length; i++) {
			assertEquals(parameterTypes[i], parameters[i].getType());
		}
	}

	/**
	 * Test {@link ReflectionUtils#invoke(Method, Object, java.util.Map)} with a
	 * method with one parameter.
	 */
	public void testInvokeOneParameter() throws SecurityException,
			NoSuchMethodException, IllegalArgumentException,
			IllegalAccessException, InvocationTargetException {
		String testString = "test";

		Class<?>[] parameterTypes0 = { String.class };

		Method method = String.class.getMethod("matches", parameterTypes0);

		FormalParameter[] formalParameters = ReflectionUtils
				.getFormalParameters(method);

		HashMap<FormalParameter, Object> parameters = new HashMap<FormalParameter, Object>();

		parameters.put(formalParameters[0], "test");

		assertTrue((Boolean) ReflectionUtils.invoke(method, testString,
				parameters));

		parameters.put(formalParameters[0], "x");
		assertFalse((Boolean) ReflectionUtils.invoke(method, testString,
				parameters));
	}

	/**
	 * Test for {@link ReflectionUtils#resolveType(String)}
	 */
	public void testResolveType() throws ClassNotFoundException {
		assertEquals(String.class, ReflectionUtils
				.resolveType("java.lang.String"));
		assertEquals(Boolean.class, ReflectionUtils
				.resolveType("java.lang.Boolean"));

		assertEquals(byte.class, ReflectionUtils.resolveType("byte"));
		assertEquals(char.class, ReflectionUtils.resolveType("char"));
		assertEquals(double.class, ReflectionUtils.resolveType("double"));
		assertEquals(float.class, ReflectionUtils.resolveType("float"));
		assertEquals(int.class, ReflectionUtils.resolveType("int"));
		assertEquals(long.class, ReflectionUtils.resolveType("long"));
		assertEquals(short.class, ReflectionUtils.resolveType("short"));
		assertEquals(boolean.class, ReflectionUtils.resolveType("boolean"));
	}

	/**
	 * Test {@link ReflectionUtils#invoke(Method, Object, java.util.Map)} with a
	 * method with one primitive parameter.
	 */
	public void testInvokeOnePrimitiveParameter() throws SecurityException,
			NoSuchMethodException, IllegalArgumentException,
			IllegalAccessException, InvocationTargetException {
		String testString = "test";

		Class<?>[] parameterTypes0 = { int.class };

		Method method = String.class.getMethod("charAt", parameterTypes0);

		FormalParameter[] formalParameters = ReflectionUtils
				.getFormalParameters(method);

		HashMap<FormalParameter, Object> parameters = new HashMap<FormalParameter, Object>();

		parameters.put(formalParameters[0], 1);

		assertEquals('e', ReflectionUtils
				.invoke(method, testString, parameters));

	}

	/**
	 * Test {@link ReflectionUtils#invoke(Method, Object, java.util.Map)} with a
	 * method with two parameters.
	 */
	public void testInvokeTwoParameters() throws SecurityException,
			NoSuchMethodException, IllegalArgumentException,
			IllegalAccessException, InvocationTargetException {
		String testString = "test";

		Class<?>[] parameterTypes0 = { String.class, String.class };

		Method method = String.class.getMethod("replaceAll", parameterTypes0);

		FormalParameter[] formalParameters = ReflectionUtils
				.getFormalParameters(method);

		HashMap<FormalParameter, Object> parameters = new HashMap<FormalParameter, Object>();

		parameters.put(formalParameters[0], "t");
		parameters.put(formalParameters[1], "x");

		assertEquals("xesx", ReflectionUtils.invoke(method, testString,
				parameters));
	}

	/**
	 * Test {@link ReflectionUtils#invoke(Method, Object, java.util.Map)} raises
	 * an exception if the formal parameters don't belong the method.
	 */
	public void testInvokeTwoIllegal() throws SecurityException,
			NoSuchMethodException, IllegalArgumentException,
			IllegalAccessException, InvocationTargetException {

		String testString = "test";

		Class<?>[] parameterTypes0 = { int.class };

		Method method0 = String.class.getMethod("charAt", parameterTypes0);

		FormalParameter[] formalParameters0 = ReflectionUtils
				.getFormalParameters(method0);

		Class<?>[] parameterTypes1 = { String.class };

		Method method1 = String.class.getMethod("matches", parameterTypes1);

		FormalParameter[] formalParameters1 = ReflectionUtils
				.getFormalParameters(method1);

		HashMap<FormalParameter, Object> parameters = new HashMap<FormalParameter, Object>();

		parameters.put(formalParameters0[0], "t");
		parameters.put(formalParameters1[0], "x");

		try {
			ReflectionUtils.invoke(method0, testString, parameters);
			fail();
		} catch (IllegalArgumentException ex) {
			// expected
		}
	}

	/**
	 * Test {@link ReflectionUtils#isAssignable(Class, Class)}.
	 */
	public void testIsAssignable() {

		// check reference types
		assertTrue(ReflectionUtils.isAssignable(String.class, Object.class));
		assertTrue(ReflectionUtils.isAssignable(Object.class, Object.class));
		assertFalse(ReflectionUtils.isAssignable(Object.class, String.class));

		// check primitives
		assertTrue(ReflectionUtils.isAssignable(Character.class, char.class));
		assertTrue(ReflectionUtils.isAssignable(char.class, Character.class));
		assertTrue(ReflectionUtils.isAssignable(char.class, char.class));
		assertTrue(ReflectionUtils.isAssignable(Character.class,
				Character.class));

		assertTrue(ReflectionUtils.isAssignable(Byte.class, byte.class));
		assertTrue(ReflectionUtils.isAssignable(byte.class, Byte.class));
		assertTrue(ReflectionUtils.isAssignable(byte.class, byte.class));
		assertTrue(ReflectionUtils.isAssignable(Byte.class, Byte.class));

		assertTrue(ReflectionUtils.isAssignable(Short.class, short.class));
		assertTrue(ReflectionUtils.isAssignable(short.class, Short.class));
		assertTrue(ReflectionUtils.isAssignable(short.class, short.class));
		assertTrue(ReflectionUtils.isAssignable(Short.class, Short.class));

		assertTrue(ReflectionUtils.isAssignable(Integer.class, int.class));
		assertTrue(ReflectionUtils.isAssignable(int.class, Integer.class));
		assertTrue(ReflectionUtils.isAssignable(int.class, int.class));
		assertTrue(ReflectionUtils.isAssignable(Integer.class, Integer.class));

		assertTrue(ReflectionUtils.isAssignable(Long.class, long.class));
		assertTrue(ReflectionUtils.isAssignable(long.class, Long.class));
		assertTrue(ReflectionUtils.isAssignable(long.class, long.class));
		assertTrue(ReflectionUtils.isAssignable(Long.class, Long.class));

		assertTrue(ReflectionUtils.isAssignable(Float.class, float.class));
		assertTrue(ReflectionUtils.isAssignable(float.class, Float.class));
		assertTrue(ReflectionUtils.isAssignable(float.class, float.class));
		assertTrue(ReflectionUtils.isAssignable(Float.class, Float.class));

		assertTrue(ReflectionUtils.isAssignable(Double.class, double.class));
		assertTrue(ReflectionUtils.isAssignable(double.class, Double.class));
		assertTrue(ReflectionUtils.isAssignable(double.class, double.class));
		assertTrue(ReflectionUtils.isAssignable(Double.class, Double.class));

		assertTrue(ReflectionUtils.isAssignable(Boolean.class, boolean.class));
		assertTrue(ReflectionUtils.isAssignable(boolean.class, Boolean.class));
		assertTrue(ReflectionUtils.isAssignable(boolean.class, boolean.class));
		assertTrue(ReflectionUtils.isAssignable(Boolean.class, Boolean.class));

	}

	/**
	 * Test if {@link ReflectionUtils#convertPrimitive(String, Class)} raises
	 * the expected exception.
	 */
	public void testConvertPrimitiveForNonPrimitive()
			throws TypeConversionException {
		try {
			ReflectionUtils.convertPrimitive("test", Object.class);
			fail();
		} catch (IllegalArgumentException e) {
			// expected
		}
	}

	/**
	 * Test if {@link ReflectionUtils#convertPrimitive(String, Class)}.
	 */
	public void testConvertPrimitive() throws TypeConversionException {
		assertTrue(ReflectionUtils.convertPrimitive("1", boolean.class));
		assertTrue(ReflectionUtils.convertPrimitive("yes", boolean.class));
		assertTrue(ReflectionUtils.convertPrimitive("on", boolean.class));
		assertTrue(ReflectionUtils.convertPrimitive("true", boolean.class));
		assertFalse(ReflectionUtils.convertPrimitive("x", boolean.class));

		assertEquals((Character) 't', ReflectionUtils.convertPrimitive("test",
				char.class));

		assertEquals((Byte) (byte) 123, ReflectionUtils.convertPrimitive("123",
				byte.class));
		assertEquals((Short) (short) 123, ReflectionUtils.convertPrimitive(
				"123", short.class));
		assertEquals((Integer) 123, ReflectionUtils.convertPrimitive("123",
				int.class));
		assertEquals((Long) 123l, ReflectionUtils.convertPrimitive("123",
				long.class));
		assertEquals(123f, ReflectionUtils.convertPrimitive("123", float.class));
		assertEquals(123d, ReflectionUtils
				.convertPrimitive("123", double.class));
	}

	/**
	 * Test if {@link ReflectionUtils#convertPrimitive(String, Class)} raises
	 * the expected exception.
	 */
	public void testNumberConversionErrors() {
		try {
			ReflectionUtils.convertPrimitive("abc", int.class);
			fail();
		} catch (TypeConversionException e) {
			assertTrue(e.getCause() instanceof NumberFormatException);
		}
	}

	/**
	 * Test {@link ReflectionUtils#convertString(String, Class)}.
	 */
	public void testConvertString() throws TypeConversionException {
		assertEquals(StringUtils.EMPTY_STRING, ReflectionUtils.convertString(
				null, String.class));

		assertEquals("hallo", ReflectionUtils.convertString("hallo",
				String.class));

		assertEquals((Integer) 123, ReflectionUtils.convertString("123",
				int.class));

		assertEquals(TestEnum.ALPHA, ReflectionUtils.convertString("alpha",
				TestEnum.class));

		assertEquals((Integer) 123, ReflectionUtils.convertString("123",
				Integer.class));

		assertEquals(new File("test"), ReflectionUtils.convertString("test",
				File.class));

	}

	/**
	 * Test {@link ReflectionUtils#convertString(String, Class)} for conversion
	 * to object. This test has been introduced to reproduce CR#1539.
	 */
	public void testConvertStringToObject() throws Exception {
		assertEquals("o", ReflectionUtils.convertString("o", Object.class));
	}

	/**
	 * Test if {@link ReflectionUtils#convertString(String, Class)} raises the
	 * expected exception.
	 */
	public void testConvertStringExceptions() {
		checkException(null, int.class);

		checkException("", int.class);

		checkException("abc", int.class);
		checkException("abc", Integer.class);

		checkException("abc", TestEnum.class);

		checkException("abc", Class.class);
	}

	/**
	 * Test for {@link ReflectionUtils#getSuperClasses(Class)}.
	 * 
	 */
	public void testGetSuperClasses() {
		List<Class<?>> superClasses = ReflectionUtils
				.getSuperClasses(Object.class);
		assertTrue(superClasses.isEmpty());

		superClasses = ReflectionUtils.getSuperClasses(String.class);
		assertEqualLists(superClasses, Object.class);

		superClasses = ReflectionUtils.getSuperClasses(ArrayList.class);
		assertEqualLists(superClasses, AbstractList.class,
				AbstractCollection.class, Object.class);

		superClasses = ReflectionUtils
				.getSuperClasses(new String[0].getClass());
		assertEqualLists(superClasses, Object.class);

		superClasses = ReflectionUtils.getSuperClasses(int.class);
		assertTrue(superClasses.isEmpty());

		superClasses = ReflectionUtils.getSuperClasses(Map.class);
		assertTrue(superClasses.isEmpty());

		superClasses = ReflectionUtils.getSuperClasses(TestEnum.class);
		assertEqualLists(superClasses, Enum.class, Object.class);
	}

	/**
	 * Test the {@link ReflectionUtils#performNearestClassLookup(Class, Map)}
	 * method.
	 */
	public void testPerformNearestClassLookup() {
		Map<Class<?>, Class<?>> m = new HashMap<Class<?>, Class<?>>();

		assertNull(ReflectionUtils.performNearestClassLookup(String.class, m));

		m.put(Object.class, Object.class);
		assertSame(Object.class, ReflectionUtils.performNearestClassLookup(
				String.class, m));

		m.put(String.class, String.class);
		assertSame(String.class, ReflectionUtils.performNearestClassLookup(
				String.class, m));

		assertSame(Object.class, ReflectionUtils.performNearestClassLookup(
				ArrayList.class, m));

		m.put(Iterable.class, Iterable.class);
		assertSame(Iterable.class, ReflectionUtils.performNearestClassLookup(
				ArrayList.class, m));

		m.put(List.class, List.class);
		assertSame(List.class, ReflectionUtils.performNearestClassLookup(
				ArrayList.class, m));

		m.put(AbstractList.class, AbstractList.class);
		assertSame(AbstractList.class, ReflectionUtils
				.performNearestClassLookup(ArrayList.class, m));

		m.put(ArrayList.class, ArrayList.class);
		assertSame(ArrayList.class, ReflectionUtils.performNearestClassLookup(
				ArrayList.class, m));
	}

	/**
	 * Check if the provided list equals the classes provided as varargs.
	 * 
	 * @param superClasses
	 * @param classes
	 */
	private void assertEqualLists(List<Class<?>> superClasses,
			Class<?>... classes) {
		assertEquals(classes.length, superClasses.size());
		for (int i = 0; i < classes.length; i++) {
			assertEquals(classes[i], superClasses.get(i));
		}
	}

	/**
	 * Test if an exception is raised.
	 */
	private void checkException(String value, Class<?> targetType) {
		try {
			ReflectionUtils.convertString(value, targetType);
			fail();
		} catch (TypeConversionException e) {
			// expected
		}
	}

	/**
	 * Test the {@link ReflectionUtils#listInstances(List, Class)} method.
	 */
	public void testListInstancesOf() {
		List<Object> input = new ArrayList<Object>();

		// test empty list
		assertTrue(ReflectionUtils.listInstances(input, Object.class).isEmpty());

		// test list with one object
		Object o1 = new Object();
		input.add(o1);
		List<Object> filtered1 = ReflectionUtils.listInstances(input,
				Object.class);
		assertEquals(1, filtered1.size());
		assertSame(o1, filtered1.get(0));

		// test with two objects of different type
		String s1 = "some string";
		input.add(s1);
		List<Object> objectList = ReflectionUtils.listInstances(input,
				Object.class);
		assertEquals(2, objectList.size());
		assertSame(o1, objectList.get(0));
		assertSame(s1, objectList.get(1));

		List<String> stringList = ReflectionUtils.listInstances(input,
				String.class);
		assertEquals(1, stringList.size());
		assertSame(s1, stringList.get(0));
	}
}