/* GeneralMatchingNumber.java
 * =========================================================================
 * This file is part of the GrInvIn project - http://www.grinvin.org
 * 
 * Copyright (C) 2005-2008 Universiteit Gent
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * A copy of the GNU General Public License can be found in the file
 * LICENSE.txt provided with the source distribution of this program (see
 * the META-INF directory in the source jar). This license can also be
 * found on the GNU website at http://www.gnu.org/licenses/gpl.html.
 * 
 * If you did not receive a copy of the GNU General Public License along
 * with this program, contact the lead developer, or write to the Free
 * Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 */

package org.grinvin.invariants.computers.standard.matching;

import java.util.ArrayList;
import java.util.Iterator;

/**
 * This abstract class contains some common methods used by
 * the other matching number classes.
 */
public abstract class GeneralMatchingNumber {
    /** Adjacency list. */
    protected int[][] adjlist;
    /** Array indicating the incident vertices of the vertices
     *  of the edges belonging the matching.
     *  If a vertex is unsaturated(does not belong to the current
     *  matching) then -1 is used as incident vertex to indicate
     *  that the specific vertex is unsaturated.
     */
    protected int[] matching;
    /** The current size of the matching. */
    protected int matchingSize;
    /** Integer to indicate a vertex is unsaturated. */
    protected static final int UNSATURATED = -1;
    // -----------------------------------
    // Constants for initial matching type
    // -----------------------------------
    /** Integer constant to start with empty initial matching. */
    protected static final int EMPTY = 0;
    /** Integer constant to use a simple greedy heuristic 
     *  to find a maximal initial matching. */
    protected static final int SIMPLE_GREEDY_HEUR = 1;
    /** Integer constant to use a degree greedy heuristic 
     *  to find a maximal initial matching. */
    protected static final int DEGREE_GREEDY_HEUR = 2;
    
    /** Creates a new instance of GeneralMatchingNumber */
    public GeneralMatchingNumber() {
        adjlist = null;
        matching = null;
        matchingSize = 0;
    }
    
    /**
     * Sets current matching to given matching.
     * @param matching The matching.
     */
    protected void setMatching(int[] matching) {
        this.matching = matching;
    }
    
    /**
     * This method returns the current matching.
     * @return The current matching.
     */
    protected int[] getMatching() {
        return this.matching;
    }
    
    /**
     * Method for finding a initial matching using heuristics.
     * @param heuristic = EMPTY Initial matching = empty
     * @param heuristic = SIMPLE_GREEDY_HEUR Initial matching = maximal
     * @param heuristic = DEGREE_GREEDY_HEUR Initial matching = maximal
     */
    protected void initializeMatching(int heuristic) {
        if(matching != null && matchingSize > 0) {
            return;
        }
        matching = new int[adjlist.length];
        for(int i = 0; i < matching.length; i++)
            matching[i] = UNSATURATED;
        matchingSize = 0;
        switch(heuristic) {
            case SIMPLE_GREEDY_HEUR:
                greedyHeuristic();
                break;
            case DEGREE_GREEDY_HEUR:
                degreeHeuristic();
                break;
            default:
        }
    }
    
    /**
     * Use a very simple, straight forward and cheap greedy heuristic
     * to find a maximal initial matching.
     */
    private void greedyHeuristic() {
        for(int i = 0; i < adjlist.length; i++) {
            for(int j = 0; matching[i] == UNSATURATED 
                           && j < adjlist[i].length; j++) {
                if(matching[adjlist[i][j]] == UNSATURATED) {
                    matching[i] = adjlist[i][j];
                    matching[adjlist[i][j]] = i;
                    matchingSize++;
                }
            }
        }
    }
    
    /**
     * Use a more advanced but also 'cheap' greedy heuristic to
     * find a maximal inital matching based on the vertices
     * degrees.
     */
    private void degreeHeuristic() {
        // Make array of buckets represented as ArrayLists
        VertexArrayList[] buckets = new VertexArrayList[adjlist.length-1];
        for(int i = 0; i < buckets.length; i++) {
            buckets[i] = new VertexArrayList();
        }
        for(int i = 0; i < adjlist.length; i++) {
            if(adjlist[i].length != 0) { // remove isolated vertices
                buckets[adjlist[i].length-1].add(new Vertex(i));
            }
        }
        // Find a maximal matching
        for(int i = 0; i < buckets.length; i++) {
            Iterator<Vertex> it = buckets[i].iterator();
            while(it.hasNext()) {
                int v = it.next().vertexId;
                if(matching[v] == UNSATURATED) {
                    int bestNeighbour = -1;
                    int lowestDegree = -1;
                    int curNeighbour = -1;
                    int curDegree = -1;
                    for(int j = 0; j < adjlist[v].length; j++) {
                        curNeighbour = adjlist[v][j];
                        curDegree = adjlist[curNeighbour].length;
                        if(matching[curNeighbour] == UNSATURATED) {
                            if(bestNeighbour == -1 || curDegree < lowestDegree) {
                                bestNeighbour = curNeighbour;
                                lowestDegree = curDegree;
                            }
                        }
                    }
                    if(bestNeighbour != -1) {
                        matching[v] = bestNeighbour;
                        matching[bestNeighbour] = v;
                        matchingSize++;
                    }
                }
            }
       }
    }
    
    /**
     * Wrapperclass for vertices in ArrayList
     */
    private class Vertex {
        private int vertexId;
        
        public Vertex(int id) {
            this.vertexId = id;
        }
    }
    
    /**
     * Class to get around "generic array problem" in Java 1.5
     */
    private class VertexArrayList extends ArrayList<Vertex> {}
}
