/* Girth.java
 * =========================================================================
 * This file is part of the GrInvIn project - http://www.grinvin.org
 * 
 * Copyright (C) 2005-2008 Universiteit Gent
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * A copy of the GNU General Public License can be found in the file
 * LICENSE.txt provided with the source distribution of this program (see
 * the META-INF directory in the source jar). This license can also be
 * found on the GNU website at http://www.gnu.org/licenses/gpl.html.
 * 
 * If you did not receive a copy of the GNU General Public License along
 * with this program, contact the lead developer, or write to the Free
 * Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 */

package org.grinvin.invariants.computers.standard;

import java.util.Arrays;

import org.grinvin.graphs.GraphBundleView;
import org.grinvin.invariants.computers.AbstractInvariantComputer;
import org.grinvin.invariants.values.IntegerValue;

/**
 * Compute the girth (length of the shortest cycle) of a graph.
 */
public class Girth extends AbstractInvariantComputer {
    
    //implements compute
    public IntegerValue compute(GraphBundleView bundle) {
        int[][] alist = bundle.adjacencyList();
        int n = alist.length;
        
        int girth = n + 1;
        for (int i=n-1; i >= 0; i--)
            girth = pseudoGirth(i, girth, alist, n); // TODO: remove vertices which have already been tried
        if (girth <= n)
            return new IntegerValue(girth, this);
        else
            return IntegerValue.undefinedValue(this); // tree
        
    }
    
    /**
     * Return the minimum of the given girth and the length of the shortest
     * circular path through the given root vertex.
     * @param alist adjacency list of the graph
     */
    private int pseudoGirth(int root, int girth, int[][] alist, int n) {
        
        int[] queue = new int[n]; // TODO: share between calls
        int head = 0;
        int tail = 0;
        
        int[] levels = new int[n]; // TODO: share between calls
        Arrays.fill(levels, -1);
        queue[head] = root;
        
        int lim = girth / 2;
        
        head ++;
        levels[root] = 0;
        while (head > tail) {
            int vertex = queue[tail];
            int d = levels[vertex];
            if (d >= lim)
                return girth; // can never improve current girth
            tail ++;
            for (int nb : alist[vertex]) {
                int e = levels[nb];
                if (e < 0) {
                    // not yet encountered
                    levels[nb] = d+1;
                    queue[head] = nb;
                    head ++;
                } else if (e == d) {
                    // odd cycle
                    return 2*d + 1;
                } else if (e > d) {
                    // even cycle
                    if (girth > 2*e) {
                        girth = 2*e; // == 2*d+2
                    }
                } else {
                    // returns towards root, ignore
                }
            }
        }
        return girth;
    }
    
    public String getInvariantId() {
        return "org.grinvin.invariants.Girth";
    }
    
}
