/*
 * Copyright (c) 1992/3 Theo de Raadt <deraadt@fsa.ca>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. The name of the author may not be used to endorse or promote
 *    products derived from this software without specific prior written
 *    permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS
 * OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#if defined(HAVE_CONFIG_H)
#include "config.h"
#endif

#include <sys/param.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <stdio.h>
#include <netdb.h>
#include <getopt.h>
#include <string.h>
#include <locale.h>
#include <libintl.h>
#include <rpc/rpc.h>
#include <rpc/xdr.h>
#include <arpa/inet.h>
#include <rpcsvc/yp_prot.h>
#include <rpcsvc/ypclnt.h>

#ifndef _
#define _(String) gettext (String)
#endif

void
usage (void)
{
  fprintf (stderr, _("Usage:\n"));
  fprintf (stderr, _("\typset [-h host] [-d domain] server\n"));
  exit (1);
}

int
bind_tohost (struct sockaddr_in *sin, char *dom, char *server, char *host)
{
  struct ypbind_setdom ypsd;
  struct timeval tv;
  struct hostent *hp;
  CLIENT *client;
  int sock, port;
  int r;
  unsigned long server_addr;

  if ((port = htons (getrpcport (server, YPPROG, YPPROC_NULL, IPPROTO_UDP)))
      == 0)
    {
      fprintf (stderr, _("%s not running ypserv.\n"), server);
      exit (1);
    }

  bzero (&ypsd, sizeof ypsd);

  if ((hp = gethostbyname (server)) != NULL)
    {
      /* is this the most compatible way?? */
      bcopy (hp->h_addr_list[0], &ypsd.ypsetdom_binding.ypbind_binding_addr,
	     sizeof (ypsd.ypsetdom_binding.ypbind_binding_addr));
    }
  else
    if ((long) (server_addr = inet_addr (server)) == -1)
      {
	fprintf (stderr, _("can't find address for %s\n"), server);
	exit (1);
      }
    else
      bcopy (&server_addr, &ypsd.ypsetdom_binding.ypbind_binding_addr,
	     sizeof (server_addr));

  ypsd.ypsetdom_domain = dom;
  ypsd.ypsetdom_binding.ypbind_binding_port = port;
  ypsd.ypsetdom_vers = YPVERS;

  tv.tv_sec = 15;
  tv.tv_usec = 0;
  sock = RPC_ANYSOCK;
  client = clntudp_create (sin, YPBINDPROG, YPBINDVERS, tv, &sock);
  if (client == NULL)
    {
      fprintf (stderr, _("can't yp_bind: Reason: %s\n"),
	       yperr_string (YPERR_YPBIND));
      return YPERR_YPBIND;
    }
  client->cl_auth = authunix_create_default ();

  r = clnt_call (client, YPBINDPROC_SETDOM,
		 (xdrproc_t) xdr_ypbind_setdom, (caddr_t) &ypsd,
		 (xdrproc_t) xdr_void, NULL, tv);
  if (r)
    {
      fprintf (stderr, _("Cannot ypset for domain %s on host %s.\n"),
	       dom, host);
      clnt_destroy (client);
      return YPERR_YPBIND;
    }
  clnt_destroy (client);
  return 0;
}

int
main (int argc, char **argv)
{
  struct sockaddr_in sin;
  struct hostent *hent;
  extern char *optarg;
  extern int optind;
  char *domainname;
  char *hostname = (char *) "localhost";
  int c;

  setlocale (LC_MESSAGES, "");
  bindtextdomain (PACKAGE, LOCALEDIR);
  textdomain (PACKAGE);

  yp_get_default_domain (&domainname);

  bzero (&sin, sizeof sin);
  sin.sin_family = AF_INET;
  sin.sin_addr.s_addr = htonl (0x7f000001);

  while ((c = getopt (argc, argv, "h:d:")) != -1)
    switch (c)
      {
      case 'd':
	domainname = optarg;
	break;
      case 'h':
	if ((sin.sin_addr.s_addr = inet_addr (optarg)) == (u_long)-1)
	  {
	    hent = gethostbyname (optarg);
	    if (hent == NULL)
	      {
		fprintf (stderr, _("ypset: host %s unknown\n"), optarg);
		exit (1);
	      }
	    bcopy (&hent->h_addr_list[0], &sin.sin_addr,
		   sizeof sin.sin_addr);
	  }
	hostname = optarg;
	break;
      default:
	usage ();
      }

  if (optind + 1 != argc)
    usage ();

  if (bind_tohost (&sin, domainname, argv[optind], hostname))
    exit (1);
  exit (0);
}
