/*
 * brass - Braille and speech server
 *
 * Copyright (C) 2001 by Roger Butenuth, All rights reserved.
 *
 * This is free software, placed under the terms of the
 * GNU General Public License, as published by the Free Software
 * Foundation.  Please see the file COPYING for details.
 *
 * $Id: viavoice.c,v 1.4 2003/02/24 20:57:04 butenuth Exp $
 */
#include <stdio.h>
#include <string.h>
#include <assert.h>
#include <dlfcn.h>
#include <eci.h>                /* viavoice include */

#include "synthesizer.h"

static int  s_close(synth_t *s);
static int  s_synth(synth_t *s, unsigned char *buffer);
static int  s_flush(synth_t *s);
static int  s_clear(synth_t *s);
#if 0
static int  s_index_set(struct synth_struct *s);
static int  s_index_wait(struct synth_struct *s, int id, int timeout);
#endif
static int  s_get_param(struct synth_struct *s, synth_par_t par, int *value);
static int  s_set_param(struct synth_struct *s, synth_par_t par, int value);

static void viavoice_error(const char *detail);

typedef struct synth_state {
    int  param[S_MAX];
    int  initialized;
} synth_state_t;

static synth_state_t private_state[2];

static synth_t state[] = {
    {
        &private_state[0],
        &languages[LANG_ENGLISH],
        "Viavoice/US English",
        NULL,
        s_close,
        s_synth,
        s_flush,
        s_clear,
        NULL,                   /* s_index_set, */
        NULL,                   /* s_index_wait, */
        s_get_param,
        s_set_param
    }, {
        &private_state[1],
        &languages[LANG_GERMAN],
        "Viavoice/German",
        NULL,
        s_close,
        s_synth,
        s_flush,
        s_clear,
        NULL,                   /* s_index_set, */
        NULL,                   /* s_index_wait, */
        s_get_param,
        s_set_param
    }
};


static int     current_language = -1;
static int     ref_count = 0;
static void    *thread_lib;     /* handle for pthreads library */
static void    *viavoice_lib;   /* handle for the viavoice library */
static ECIHand hECI;            /* handle for a viavoice instance */
static FILE    *debug_fp = NULL;

/*static int sync_mark_no = 0;*/	/* currently used number */
/*static struct timeval mark;*/	/* time the mark has been set */


/*
 * ----------------------------------------------------------------------
 * Called before library is loaded.
 * ----------------------------------------------------------------------
 */
void _init(void)
{
}


/*
 * ----------------------------------------------------------------------
 * Called before library is unloaded.
 * ----------------------------------------------------------------------
 */
void _fini(void)
{
}


/*
 * ----------------------------------------------------------------------
 * General open function for german and english synthesizer.
 * Second open increments refcount.
 * Return 0 on succes, 1 on error.
 * ----------------------------------------------------------------------
 */
synth_t *synth_open(void *context, lookup_string_t lookup)
{
    synth_t *s;
    char    *language = (*lookup)(context, "language");
    char    *thread_str = (*lookup)(context, "pthreads");
    char    *viavoice_str = (*lookup)(context, "viavoice");
    int     langi;

    debug_fp = stderr;
    if (language == NULL) {
        fprintf(debug_fp, "variable \"language\" not defined\n");
        return NULL;
    }
    if (ref_count == 0) {
        if (thread_str == NULL) {
            fprintf(debug_fp, "variable \"pthreads\" not defined\n");
            return NULL;
        }
        if (viavoice_str == NULL) {
            fprintf(debug_fp, "variable \"viavoice\" not defined\n");
            return NULL;
        }
        thread_lib = dlopen(thread_str, RTLD_GLOBAL);
        if (thread_lib == NULL) {
            fprintf(debug_fp, "can't open \"%s\"\n", thread_str);
            return NULL;
        }
        viavoice_lib = dlopen(viavoice_str, RTLD_GLOBAL);
        if (viavoice_lib == NULL) {
            fprintf(debug_fp, "can't open \"%s\"\n", viavoice_str);
            return NULL;
        }
        hECI = eciNew();
        if (NULL_ECI_HAND == hECI) {
            fprintf(debug_fp, "can't inialize Viavoice\n");
            return NULL;
        }
    }
    ref_count++;

    if (!strcasecmp(language, "english")) {
        langi = 0;
        s = &state[langi];
    } else if (!strcasecmp(language, "german")) {
        langi = 1;
        s = &state[langi];
    } else {
        langi = -1;
        s = NULL;
    }

    if (s != NULL && !s->state->initialized) {
        s->state->param[S_SPEED]  = 1000;
        s->state->param[S_PITCH]  = 1000;
        s->state->param[S_VOLUME] = 1000;
        s->state->initialized = 1;
    }

    return s;
}


/*
 * ----------------------------------------------------------------------
 * General close. Decrement refcount, do real close when count reaches
 * zero.
 * ----------------------------------------------------------------------
 */
static int s_close(synth_t *s)
{
    ref_count--;
    if (ref_count == 0) {
        printf(">>> eci delete\n");
        eciDelete(hECI);
#if 0
        /* closing the library causes a sigsegv */
        dlclose(viavoice_lib);
        dlclose(thread_lib);
#endif
        printf("<<< eci delete\n");
    }

    return 0;
}


/*
 * ----------------------------------------------------------------------
 * Verify that the synthesizer is set to the correct language.
 * Switch if necessary.
 * ----------------------------------------------------------------------
 */
static void verify_language(struct synth_struct *s)
{
    int value = -1;

    if (s->lang->lang == LANG_ENGLISH
        && current_language != LANG_ENGLISH) {
        value = eciGeneralAmericanEnglish;
    } else if (s->lang->lang == LANG_GERMAN &&
               current_language != LANG_GERMAN) {
        value = eciStandardGerman;
    }
    if (value != -1) {
        if (eciSetParam(hECI, eciLanguageDialect, value) < 0) {
            viavoice_error("verify_language, eciSetParam");
        }
    }
}


/*
 * ----------------------------------------------------------------------
 * Copy Text to server. Text is not spoken until a flush command
 * arrives.
 * ----------------------------------------------------------------------
 */
static int s_synth(struct synth_struct *s, unsigned char *buffer)
{
    assert(s->state->initialized);

    if (!eciAddText(hECI, (char*)buffer)) {
        viavoice_error("s_synth, add_text");
        return 1;
    }
    return 0;
}


/*
 * ----------------------------------------------------------------------
 * Flush synthesizer. This triggers the synthesizer and starts 
 * the synthesis.
 * ----------------------------------------------------------------------
 */
static int s_flush(synth_t *s)
{
    if (!eciSynthesize(hECI)) {
        viavoice_error("s_flush");
        return 1;
    }
    return 0;
}


/*
 * ----------------------------------------------------------------------
 * Remove anything in the synthesizier speech queue.
 * ----------------------------------------------------------------------
 */
static int s_clear(synth_t *s)
{
    if (!eciStop(hECI)) {
        viavoice_error("s_clear");
        return 1;
    }
    return 0;
}


#if 0
/*
 * ----------------------------------------------------------------------
 * ToDo
 * ----------------------------------------------------------------------
 */
static int s_index_set(struct synth_struct *s)
{
    return 0;
}

/*
 * ----------------------------------------------------------------------
 * ToDo
 * ----------------------------------------------------------------------
 */
static int s_index_wait(struct synth_struct *s, int id, int timeout)
{
    int res = 0;
    return res;
}
#endif


/*
 * ----------------------------------------------------------------------
 * Get a synthesizer parameter.
 * ----------------------------------------------------------------------
 */
static int s_get_param(struct synth_struct *s, synth_par_t par, int *value)
{
    if (par >= 0 && par < S_MAX) {
	*value = s->state->param[par];
	return 0;
    } else
	return 1;
}


/*
 * ----------------------------------------------------------------------
 * Set a parameter of the synthesizer.
 * ----------------------------------------------------------------------
 */
static int s_set_param(struct synth_struct *s, synth_par_t par, int value)
{
    int eci_voice = 0;              /* currently active voice */
    int eci_par;
    int eci_val;

    verify_language(s);

    switch (par) {
    case S_SPEED:               /* default: 50, range: 0-250 */
        eci_par = eciSpeed;
        eci_val = (50 * value) / 1000;
        if (eci_val < 0) eci_val = 0;
        if (eci_val > 250) eci_val = 250;
        break;
    case S_PITCH:               /* default: 65, range: 0-100 */
        eci_par = eciPitchBaseline;
        eci_val = (65 * value) / 1000;
        if (eci_val < 0) eci_val = 0;
        if (eci_val > 100) eci_val = 100;
        break;
    case S_VOLUME:              /* default: 92, range: 0-100 */
        eci_par = eciVolume;
        eci_val = (92 * value) / 1000;
        if (eci_val < 0) eci_val = 0;
        if (eci_val > 100) eci_val = 250;
        break;
    default:
        return 1;
    }
    if (eciSetVoiceParam(hECI, eci_voice, eci_par, eci_val) < 0) {
        viavoice_error("s_set_param");
        return 1;
    }
    s->state->param[par] = value;

    return 0;
}


static void viavoice_error(const char *detail)
{
    char error[100];

    eciErrorMessage(hECI, error);
    fprintf(debug_fp, "viavoice error (%s): %s\n", detail, error);
    eciClearErrors(hECI);
}

