/*
 * brass - Braille and speech server
 *
 * Copyright (C) 2001-2003 by Roger Butenuth, All rights reserved.
 *
 * This is free software, placed under the terms of the
 * GNU General Public License, as published by the Free Software
 * Foundation.  Please see the file COPYING for details.
 *
 * A pseudo-synthesizer for debugging: Trace function calls to stdout.
 *
 * $Id: stdout_synth.c,v 1.1 2003/02/16 17:14:41 butenuth Exp $
 */
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <assert.h>

#include "synthesizer.h"

static int  s_close(synth_t *s);
static int  s_synth(synth_t *s, unsigned char *buffer);
static int  s_flush(synth_t *s);
static int  s_clear(synth_t *s);
static int  s_index_set(struct synth_struct *s);
static int  s_index_wait(struct synth_struct *s, int id, int timeout);
static int  s_get_param(struct synth_struct *s, synth_par_t par, int *value);
static int  s_set_param(struct synth_struct *s, synth_par_t par, int value);


typedef struct synth_state {
    int  param[S_MAX];
    int  initialized;
} synth_state_t;


static synth_t dummy = {
    NULL,			/* state */
    &languages[LANG_DUMMY],
    "Stdout",
    NULL,			/* library handle */
    s_close,
    s_synth,
    s_flush,
    s_clear,
    s_index_set,
    s_index_wait,
    s_get_param,
    s_set_param,
};



/*
 * ----------------------------------------------------------------------
 * General open function.
 * Return 0 on succes, 1 on error.
 * ----------------------------------------------------------------------
 */
synth_t *synth_open(void *context, lookup_string_t lookup)
{
    synth_t       *s = NULL;
    synth_state_t *state = NULL;
    
    s = malloc(sizeof(synth_t));
    if (s == NULL)
	goto failure;
    *s = dummy;
    state = malloc(sizeof(synth_state_t));
    if (s == NULL)
	goto failure;
    memset(state, 0, sizeof(synth_state_t));
    state->initialized = 1;
    s->lib_handle = NULL;
    s->state = state;

    printf("synth_open()\n");

    return s;
 failure:
    if (s)
	free(s);
    if (state)
	free(state);
    return NULL;
}


/*
 * ----------------------------------------------------------------------
 * General close.
 * ----------------------------------------------------------------------
 */
static int s_close(synth_t *s)
{

    assert(s->state->initialized);
    s->state->initialized = 0;

    printf("s_close()\n");
   
    free(s->state);
    free(s);
    return 0;
}


/*
 * ----------------------------------------------------------------------
 * Synthesize the null terminated String.
 * ----------------------------------------------------------------------
 */
static int s_synth(struct synth_struct *s, unsigned char *buffer)
{
    assert(s->state->initialized);

    printf("s_synth(s, \"%s\")\n", buffer);

    return 0;
}


/*
 * ----------------------------------------------------------------------
 * Flush the queue to the synthesizer. All queued text is now spoken.
 * ----------------------------------------------------------------------
 */
static int s_flush(synth_t *s)
{
    assert(s->state->initialized);

    printf("s_flush()\n");

    return 0;
}


/*
 * ----------------------------------------------------------------------
 * Remove all text from the queue. This makes it possible to synthesize
 * new text without waiting for the old text being spoken.
 * ----------------------------------------------------------------------
 */
static int s_clear(synth_t *s)
{
    assert(s->state->initialized);

    printf("s_clear()\n");

    return 0;
}


static int s_index_set(struct synth_struct *s)
{
    return 0;
}

static int s_index_wait(struct synth_struct *s, int id, int timeout)
{
    return 1;			/* code for "index reached" */
}


/*
 * ----------------------------------------------------------------------
 * Get a synthesizer parameter.
 * ----------------------------------------------------------------------
 */
static int s_get_param(struct synth_struct *s, synth_par_t par, int *value)
{
    if (par >= 0 && par < S_MAX) {
	*value = s->state->param[par];
	return 0;
    } else
	return 1;
}


/*
 * ----------------------------------------------------------------------
 * Set a parameter of the synthesizer.
 * ----------------------------------------------------------------------
 */
static int s_set_param(struct synth_struct *s, synth_par_t par, int value)
{
    if (par >= 0 && par < S_MAX) {
	s->state->param[par] = value;
	return 0;
    } else
	return 1;
}
