/*
 * brass - Braille and speech server
 *
 * Copyright (C) 2001 by Roger Butenuth, All rights reserved.
 *
 * This is free software, placed under the terms of the
 * GNU General Public License, as published by the Free Software
 * Foundation.  Please see the file COPYING for details.
 *
 * $Id: dolphin.c,v 1.6 2003/02/24 20:57:04 butenuth Exp $
 */
#include <stdio.h>
#include <string.h>
#include <assert.h>
#include <unistd.h>
#include <fcntl.h>
#include <errno.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <sys/time.h>
#include <sys/wait.h>
#include <termios.h>

#include "synthesizer.h"

static int s_close(synth_t *s);
static int s_synth(synth_t *s, unsigned char *buffer);
static int s_flush(synth_t *s);
static int s_clear(synth_t *s);
#if 0
static int s_index_set(struct synth_struct *s);
static int s_index_wait(struct synth_struct *s, int id, int timeout);
#endif
static int s_get_param(struct synth_struct *s, synth_par_t par, int *value);
static int s_set_param(struct synth_struct *s, synth_par_t par, int value);

typedef struct synth_state {
    int  param[S_MAX];
    int  initialized;
    int  rom;
} synth_state_t;

static synth_state_t private_state[2];

static FILE   *sy_fp;
static struct termios old_tio;
static int    sy_fd;
static int    sy_ref = 0;
static int    current_language = -1;

static int sync_mark_no = 0;	/* number to expect */
/*static struct timeval mark;*/	/* time the mark has been set */

static synth_t state[] = {
    {
        &private_state[0],
        &languages[LANG_ENGLISH],
        "Dolphin/British English",
        NULL,			/* lib_handle */
        s_close,
        s_synth,
        s_flush,
        s_clear,
        NULL,                   /* s_index_set, */
        NULL,                   /* s_index_wait, */
        s_get_param,
        s_set_param
    }, {
        &private_state[1],
        &languages[LANG_GERMAN],
        "Dolphin/German",
        NULL,			/* lib_handle */
        s_close,
        s_synth,
        s_flush,
        s_clear,
        NULL,                   /* s_index_set, */
        NULL,                   /* s_index_wait, */
        s_get_param,
        s_set_param
    }
};


/*
 * charset conversion table from iso latin-1 == iso 8859-1 to cp437==ibmpc
 * for chars >=128. 
 */
static unsigned char latin2cp437[128] = {
    199, 252, 233, 226, 228, 224, 229, 231,
    234, 235, 232, 239, 238, 236, 196, 197,
    201, 181, 198, 244, 247, 242, 251, 249,
    223, 214, 220, 243, 183, 209, 158, 159,
    255, 173, 155, 156, 177, 157, 188,  21,
    191, 169, 166, 174, 170, 237, 189, 187,
    248, 241, 253, 179, 180, 230,  20, 250,
    184, 185, 167, 175, 172, 171, 190, 168,
    192, 193, 194, 195, 142, 143, 146, 128,
    200, 144, 202, 203, 204, 205, 206, 207,
    208, 165, 210, 211, 212, 213, 153, 215,
    216, 217, 218, 219, 154, 221, 222, 225,
    133, 160, 131, 227, 132, 134, 145, 135,
    138, 130, 136, 137, 141, 161, 140, 139,
    240, 164, 149, 162, 147, 245, 148, 246,
    176, 151, 163, 150, 129, 178, 254, 152
};

typedef struct {
    int d_val;                  /* Dolphin value */
    int b_val;                  /* Brass value */
} table_t;

/*
 * Table for speed value conversion.
 */
static table_t speed_table[] = {
    { 1,  826 },
    { 2,  910 },
    { 3, 1000 },                /* default */
    { 4, 1100 },
    { 5, 1210 },
    { 6, 1330 },
    { 7, 1464 },
    { 8, 1610 },
    { 9, 1771 },
    { -1, 0 }
};

/*
 * Table for pitch value conversion.
 */
static table_t pitch_table[] = {
    {  1,  513 },
    {  2,  564 },
    {  3,  620 },
    {  4,  683 },
    {  5,  750 },
    {  6,  826 },
    {  7,  910 },
    {  8, 1000 },               /* default */
    {  9, 1100 },
    { 10, 1210 },
    { 11, 1330 },
    { 12, 1464 },
    { 13, 1610 },
    { 14, 1771 },
    { 15, 1949 },
    { -1, 0 }
};

/*
 * Table for volume value conversion.
 */
static table_t volume_table[] = {
    {  1,  424 },
    {  2,  466 },
    {  3,  513 },
    {  4,  564 },
    {  5,  620 },
    {  6,  683 },
    {  7,  750 },
    {  8,  826 },
    {  9,  910 },
    { 10, 1000 },               /* default */
    { 11, 1100 },
    { 12, 1210 },
    { 13, 1330 },
    { 14, 1464 },
    { 15, 1610 },
    { -1, 0 }
};

/*
 * ----------------------------------------------------------------------
 * Called before library is loaded.
 * ----------------------------------------------------------------------
 */
void _init(void)
{
}


/*
 * ----------------------------------------------------------------------
 * Called before library is unloaded.
 * ----------------------------------------------------------------------
 */
void _fini(void)
{
}


/*
 * ----------------------------------------------------------------------
 * General open function for german and english synthesizer.
 * Second open increments refcount.
 * Return 0 on succes, 1 on error.
 * Be aware: The file pointer (sy_fp) is only used for writes, 
 * reads use sy_fd directly.
 * Return 0 on succes, or an error number.
 * ----------------------------------------------------------------------
 */
synth_t *synth_open(void *context, lookup_string_t lookup)
{
    synth_t *s = NULL;
    char    *language = (*lookup)(context, "language");
    char    *device = (*lookup)(context, "device");
    int     r = 0;
    struct  termios new_tio;
    int     langi;

    /*
     * Configuration parameter checking.
     */
    if (language == NULL) {
        fprintf(stderr, "variable \"language\" not defined\n");
        return NULL;
    }
    if (!strcasecmp(language, "english")) {
        langi = 0;
        s = &state[langi];
        s->state->rom = 2;
    } else if (!strcasecmp(language, "german")) {
        langi = 1;
        s = &state[langi];
        s->state->rom = 1;
    } else {
        fprintf(stderr, "\"language\" must be english or german!\n");
        return NULL;
    }
    if (device == NULL) {
        fprintf(stderr, "variable \"device\" not defined\n");
        return NULL;
    }
    /*
     * General initialization when refcount is zero.
     */
    if (sy_ref == 0) {
	sy_fd = open(device, O_RDWR);
	if (sy_fd < 0) {
	    fprintf(stderr, "can't open %s: %s\n",
                    device, strerror(errno));
	    return NULL;
	}
	tcgetattr(sy_fd, &old_tio); /* save current port settings */

        memset(&new_tio, 0, sizeof(new_tio));
        new_tio.c_cflag = B9600 | CRTSCTS | CS8 | CLOCAL | CREAD;
        new_tio.c_iflag = IGNPAR;
        new_tio.c_oflag = 0;
        new_tio.c_lflag = 0; /* set input mode (non-canonical, no echo,...) */
        new_tio.c_cc[VTIME] = 10; /* inter-character timer unused (t * 0.1s) */
        new_tio.c_cc[VMIN]  = 0; /* blocking read until 1 char received */

        tcflush(sy_fd, TCIFLUSH);
        tcflush(sy_fd, TCOFLUSH);
        tcsetattr(sy_fd, TCSANOW, &new_tio);

	sy_fp = fdopen(sy_fd, "w");
	if (sy_fp == NULL)
	    return NULL;
        /*
         * We could send the initialization string ("@YF3N8") at this
         * point, but it seems not all Dolphin versions support it.
         */
        fprintf(sy_fp, "@G0\r\n"); /* greeting off */
	current_language = -1;
    }
    if (r == 0)
	sy_ref++;

    if (s->state->initialized) {
        fprintf(stderr, "%s already open\n", s->name);
        return NULL;
    }
    s->state->param[S_SPEED]  = 1000;
    s->state->param[S_PITCH]  = 1000;
    s->state->param[S_VOLUME] = 1000;
    s->state->initialized = 1;
    
    return s;
}


/*
 * ----------------------------------------------------------------------
 * Decrement refcount, do real close when count reaches zero.
 * ----------------------------------------------------------------------
 */
static int s_close(synth_t *s)
{
    assert(s->state->initialized);
    s->state->initialized = 0;

    assert(sy_ref > 0);
    sy_ref--;
    if (sy_ref == 0) {
	tcsetattr(sy_fd, TCSANOW, &old_tio);
	fclose(sy_fp);
	sy_fp = NULL;
    }

    return 0;
}


/*
 * ----------------------------------------------------------------------
 * Verify that the synthesizer is set to the correct language.
 * Switch if necessary and reset parameters.
 * ----------------------------------------------------------------------
 */
static void verify_language(struct synth_struct *s)
{
    int p;

    if (s->lang->lang == LANG_GERMAN &&	current_language != LANG_GERMAN) {
	fprintf(sy_fp, "@=%d,\r\n", s->state->rom);
	current_language = LANG_GERMAN;
	for (p = 0; p < S_MAX; p++) {
	    s_set_param(s, p, s->state->param[p]);
	}
    }
    if (s->lang->lang == LANG_ENGLISH &&
	current_language != LANG_ENGLISH) {
	fprintf(sy_fp, "@=%d,\r\n", s->state->rom);
	current_language = LANG_ENGLISH;
	for (p = 0; p < S_MAX; p++) {
	    s_set_param(s, p, s->state->param[p]);
	}
    }
}


#if 0
/*
 * ----------------------------------------------------------------------
 * Milliseconds from t1 (start) to t2 (end).
 * ----------------------------------------------------------------------
 */
static int elapsed_msec(struct timeval *t1, struct timeval *t2)
{
    unsigned diff;

    diff = ((t2->tv_sec - t1->tv_sec) * 1000L +
            (t2->tv_usec - t1->tv_usec) / 1000L);

    return diff;
}
#endif


/*
 * ----------------------------------------------------------------------
 * 
 * ----------------------------------------------------------------------
 */
static int s_synth(struct synth_struct *s, unsigned char *buffer)
{
    int i, len;
    int c;

    assert(s->state->initialized);

    verify_language(s);
    /*
     * Write string to synthesizer device, convert character when necessary
     * and filter control characters.
     */
    len = strlen(buffer);
    for (i = 0; i < len; i++) {
	c = buffer[i];
	if (c < ' ' || c == '@')
	    ;			/* ignore */
	else if (c < 0x80)
	    fputc(c, sy_fp);	/* normal character, simply write */
	else /* 0x80 <= c <= 0xff */
	    fputc(latin2cp437[c - 0x80], sy_fp);
    }
    fflush(sy_fp);
    return 0;
}


/*
 * ----------------------------------------------------------------------
 * Flush date queued in the driver or synthesizer.
 * ----------------------------------------------------------------------
 */
static int s_flush(synth_t *s)
{
    assert(s->state->initialized);

    fprintf(sy_fp, "\r\n");
    fflush(sy_fp);

    return 0;
}


/*
 * ----------------------------------------------------------------------
 * Remove any speech data in the driver or synthesizer.
 * ----------------------------------------------------------------------
 */
static int s_clear(synth_t *s)
{
    assert(s->state->initialized);

    fputc(0x18, sy_fp);
    fflush(sy_fp);
    sync_mark_no = 0;

    return 0;
}


#if 0
/*
 * ----------------------------------------------------------------------
 *
 * ----------------------------------------------------------------------
 */
static int s_index_set(struct synth_struct *s)
{
    return 0;
}

/*
 * ----------------------------------------------------------------------
 * Indexing, return value is:
 *  0 sync point not reached
 *  1 sync point reached
 *  2 timeout
 * The Dolphin synthesizers work with an increment/decrement scheme:
 * Every time a mark is sent to the synthesizer, an internal counter is
 * inceremented. Whenever a mark is spoken, it is decremented. The
 * value of the counter can be retrieved from the synthesizer. This
 * causes problem with timeouts because in that case the mark is still
 * on the way and decrements the value later. This is the reason we
 * have to keep track of the value over sync commands. To be sure we
 * dont count totally wrong, store the correct value after every sync
 * that was not a timeout.
 * All this is not very well tested.
 * ----------------------------------------------------------------------
 */
static int s_index_wait(struct synth_struct *s, int id, int timeout)

{
    struct timeval cur;
    char           buf[100];
    int            res = 0;
    int            gone;
    int            value;

    assert(s->state->initialized);

    verify_language(s);

    (void)value; (void)gone; (void)buf; (void)cur;
#if 0
    switch (par) {
    case SYNC_POLL:		/* have we reached the mark? */
	/*
	 * Write the question string to the synth, the question is
	 * answered immediately, regardless whether the index has
	 * been reached or not.
	 */
	fprintf(sy_fp, "@I?"); fflush(sy_fp);
	do {
	    read(sy_fd, buf, 1);
	} while (buf[0] != 'I');
	read(sy_fd, buf, 3);
	value = buf[0] - '0' + ((buf[1] - '0') << 4);
	if (value < sync_mark_no) {
	    res = 1;
	    sync_mark_no = value;
	} else {
	    res = 0;
	}
	/*
	 * In case we did not get the right index, check for timeout.
	 */
	if (res == 0) {
	    gettimeofday(&cur, NULL);
	    if (elapsed_msec(&mark, &cur) > timeout)
		res = 2;
	}
	break;
    case SYNC_SET:		/* set mark */
    case SYNC_SET_WAIT:		/* set mark and wait (combined) */
	/*
	 * Increment the sync mark number and set a mark (which increments
	 * the number in the synthesizer).
	 */
	sync_mark_no++;
	fprintf(sy_fp, "@I+\r\n"); fflush(sy_fp);
	gettimeofday(&mark, NULL);
	res = 0;
	/*
	 * "Set" and "set wait" are similar. In case of "set wait" fall
	 * through in the wait case.
	 */
	if (par == SYNC_SET)
	    break;
    case SYNC_WAIT:		/* wait for mark */
	do {
	    gettimeofday(&cur, NULL);
	    gone = elapsed_msec(&mark, &cur);
	    if (gone < timeout) {
		/*
		 * Write the question string to the synth, the question is
		 * answered immediately, regardless whether the index has
		 * been reached or not.
		 */
		fprintf(sy_fp, "@I?"); fflush(sy_fp);
		do {
		    read(sy_fd, buf, 1);
		} while (buf[0] != 'I');
		read(sy_fd, buf, 3);
		value = buf[0] - '0' + ((buf[1] - '0') << 4);
		if (value < sync_mark_no) {
		    res = 1;
		    sync_mark_no = value;
		} else {
		    res = 0;
		}
	    } else {
		res = 2;	/* timeout */
		sync_mark_no--;
	    }
	    /*
	     * Wait a little bit to avoid cpu trashing.
	     */
	    if (res == 0)
		usleep(100000);
	} while (res == 0);
	break;
    default:			/* should not happen */
	res = -1;
	break;
    }
#endif
    return res;
}
#endif


/*
 * ----------------------------------------------------------------------
 *
 * ----------------------------------------------------------------------
 */
static int s_get_param(struct synth_struct *s, synth_par_t par, int *value)
{
    if (par >= 0 && par < S_MAX) {
	*value = s->state->param[par];
	return 0;
    } else
	return 1;
}


/*
 * ----------------------------------------------------------------------
 * Convert a parameter from Brass range (default 1000) to Dolphin Range.
 * ----------------------------------------------------------------------
 */
static int brass_to_dolphin(table_t table[], int value)
{
    int i;
    int size;

    /*
     * Get table size by searching for the -1 entry.
     */
    for (i = 0; table[i].d_val != -1; i++)
        ;
    size = i;

    for (i = 0; i < size; i++) {
        if (value < table[i].b_val)
            break;
    }
    if (i < size)
        return table[i].d_val;
    else
        return table[size - 1].d_val;
}


/*
 * ----------------------------------------------------------------------
 * Change a parameter. Only when the current synthesizer is the same
 * as the selected, the change is done imediately, otherwise only
 * the request is stored and executed later.
 * ----------------------------------------------------------------------
 */
static int s_set_param(struct synth_struct *s, synth_par_t par, int value)
{
    static char speed[] = "@W ,";
    static char pitch[] = "@F ,";
    static char volume[] = "@A ,";
    static char hex[] = "0123456789ABCDEF";
    int         r = 0;

    switch (par) {
    case S_SPEED:
	if (value < 600) value = 600;
	if (value > 4000) value = 4000;
	if (s->lang->lang == current_language) {
	    speed[2] = hex[brass_to_dolphin(speed_table, value)];
	    fwrite(speed, 1, strlen(speed), sy_fp);
	    fflush(sy_fp);
	    usleep(100000);	/* wait 0.1s */
	}
	s->state->param[par] = value;
	break;
    case S_PITCH:
	if (value < 200) value = 200;
	if (value > 4000) value = 4000;
	if (s->lang->lang == current_language) {
	    pitch[2] = hex[brass_to_dolphin(pitch_table, value)];
	    fwrite(pitch, 1, strlen(pitch), sy_fp);
	    fflush(sy_fp);
	    usleep(100000);	/* wait 0.1s */
	}
	s->state->param[par] = value;
	break;
    case S_VOLUME:
	if (value < 100) value = 100;
	if (value > 4000) value = 4000;
	if (s->lang->lang == current_language) {
	    volume[2] = hex[brass_to_dolphin(volume_table, value)];
	    fwrite(volume, 1, strlen(volume), sy_fp);
	    fflush(sy_fp);
	    usleep(100000);	/* wait 0.1s */
	}
	s->state->param[par] = value;
	break;
    default:
	r = 1;
    }

    return r;
}
