/*
 *  suse-blinux - Braille-display support for linux
 *  Author: Marco Skambraks <marco@suse.de>
 *  SuSE GmbH Nuernberg
 *
 *
 * suse-blinux based on brltty
 * special thanks to the Brltty-Team
 * Nicolas Pitre <nico@cam.org>
 * Stphane Doyon <s.doyon@videotron.ca>
 * Nikhil Nair <nn201@cus.cam.ac.uk>
 *
 * This is free software, placed under the terms of the
 * GNU General Public License, as published by the Free Software
 * Foundation.  Please see the file COPYING for details.
*/

#define BRL_C

#define __EXTENSIONS__		/* for termios.h */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <fcntl.h>
#include <sys/termios.h>
#include <string.h>

#include "../config.h"
#include "../brl.h"
#include "../brl_driver.h"
#include "../misc.h"
#include "../inskey.h"
#include "../message.h"
#include "brlconf.h"
#include "bindings.h"		/* for keybindings */

#define QSZ 256			/* size of internal input queue in bytes */
#define INT_CSR_SPEED 2		/* on/off time in cycles */
#define ACK_TIMEOUT 3000	/* timeout in ms for an ACK to come back */

#define MIN(a, b) (((a) < (b)) ? (a) : (b))
#define MAX(a, b) (((a) > (b)) ? (a) : (b))

int blite_fd = -1;		/* file descriptor for Braille display */

unsigned char highbits[4] = { 128, 64, 32, 16 };
unsigned char lowbits[4] = { 8, 4, 2, 1 };
static unsigned char blitetrans[256];	/* dot mapping table (output) */
static unsigned char revtrans[256];	/* mapping for reversed display */
static unsigned char *prevdata;	/* previously received data */
static unsigned char *rawdata;	/* writebrl() buffer for raw Braille data */
static struct termios oldtio;	/* old terminal settings */
static unsigned int blitesz;	/* set to 18 or 40 */
static int waiting_ack = 0;	/* waiting acknowledgement flag */
static int reverse_kbd = 0;	/* reverse keyboard flag */
static int intoverride = 0;	/* internal override flag -

				 * highly dubious behaviour ...
				 */
static int int_cursor = 0;	/* position of internal cursor: 0 = none */

/* The input queue is only manipulated by the qput() and qget()
 * functions.
 */
static unsigned char *qbase;	/* start of queue in memory */
static int qoff = 0;		/* offset of first byte */
static int qlen = 0;		/* number of items in the queue */
static int repeat = 0;		/* repeat count for next qget() */

/* Data type for a Braille Lite key, including translation into command
 * codes:
 */
typedef struct
{
  unsigned char raw;		/* raw value, after any keyboard reversal */
  unsigned char cmd;		/* command code */
  unsigned char asc;		/* ASCII translation of Braille keys */
  unsigned char spcbar;		/* 1 = on, 0 = off */
  unsigned char routing;	/* routing key number */
}
blkey;

static int devnr;
struct brlinfo blaziedevs[]=
{
{"brllite","Blazie Braillelite 40",40,0,0,0,B9600}
};
/* Local function prototypes: */
static void getbrlkeys (void);	/* process keystrokes from the Braille Lite */
static int qput (unsigned char);	/* add a byte to the input queue */
static int qget (blkey *);	/* get a byte from the input queue */

struct brlinfo identbrl (const char *name, const char *brldev)
{
int devcnt;
struct brlinfo retinfo;

devcnt=sizeof(blaziedevs)/sizeof(struct brlinfo);

for(devnr=0;devnr<devcnt;devnr++)
if(!strcmp(blaziedevs[devnr].name,name))
{
printf(" %s on %s\n",blaziedevs[devnr].fullname,brldev);
retinfo=blaziedevs[devnr];
break;
}

return retinfo;
}

void initbrl (brldim * brl, const char *brldev)
{
  brldim res;			/* return result */
  struct termios newtio;	/* new terminal settings */
  short i, n;			/* loop counters */
  static unsigned char standard[8] = { 0, 1, 2, 3, 4, 5, 6, 7 };	/* BRLTTY standard mapping */
  static unsigned char Blazie[8] = { 0, 3, 1, 4, 2, 5, 6, 7 };	/* Blazie standard */

  res.disp = prevdata = rawdata = qbase = NULL;	/* clear pointers */

  /* Open the Braille display device for random access */
  if (brldev == NULL)
    brldev = BRLDEV;
  blite_fd = open (brldev, O_RDWR | O_NOCTTY);
  if (blite_fd < 0)
   {
     LogPrint (LOG_ERR, "%s: %s\n", brldev, strerror (1));
     goto failure;
   }
  tcgetattr (blite_fd, &oldtio);	/* save current settings */

  /* Set bps, flow control and 8n1, enable reading */
  newtio.c_cflag = BAUDRATE | CRTSCTS | CS8 | CLOCAL | CREAD;

  /* Ignore bytes with parity errors and make terminal raw and dumb */
  newtio.c_iflag = IGNPAR;
  newtio.c_oflag = 0;		/* raw output */
  newtio.c_lflag = 0;		/* don't echo or generate signals */
  newtio.c_cc[VMIN] = 0;	/* set nonblocking read */
  newtio.c_cc[VTIME] = 0;
  tcflush (blite_fd, TCIFLUSH);	/* clean line */
  tcsetattr (blite_fd, TCSANOW, &newtio);	/* activate new settings */

  /* Braille Lite initialisation: leave this for a later version ... */

  blitesz = res.x = BLITE_SIZE;	/* initialise size of display - */
  res.y = 1;			/* Braille Lites are single line displays */
#if 0
  if ((brl->x = res.x) == -1)
    return;
#endif

  /* Allocate space for buffers */
  res.disp = (unsigned char *) malloc (res.x);
  prevdata = (unsigned char *) malloc (res.x);
  rawdata = (unsigned char *) malloc (res.x);
  qbase = (unsigned char *) malloc (QSZ);
  if (!res.disp || !prevdata || !rawdata || !qbase)
   {
     LogPrint (LOG_ERR, "can't allocate braille buffers\n");
     goto failure;
   }
  memset (prevdata, 0, res.x);

  /* Generate dot mapping tables: */
  memset (blitetrans, 0, 256);	/* ordinary dot mapping */
  memset (revtrans, 0, 256);	/* display reversal - extra mapping */
  for (n = 0; n < 256; n++)
   {
     for (i = 0; i < 8; i++)
       if (n & 1 << standard[i])
	 blitetrans[n] |= 1 << Blazie[i];
     for (i = 0; i < 8; i++)
      {
	revtrans[n] <<= 1;
	if (n & (1 << i))
	  revtrans[n] |= 1;
      }
   }
  *brl = res;
  return;

failure:;
  free (res.disp);
  free (prevdata);
  free (rawdata);
  free (qbase);
  if (blite_fd >= 0)
    close (blite_fd);
  brl->x = -1;
  return;
}

void closebrl (brldim * brl)
{
  /* We just clear the display, using writebrl(): */
  memset (brl->disp, 0, brl->x);
  writebrl (brl);

  free (brl->disp);
  free (prevdata);
  free (rawdata);
  free (qbase);

  tcsetattr (blite_fd, TCSANOW, &oldtio);	/* restore terminal settings */
  close (blite_fd);
}

void setbrlstat (const unsigned char *s)
{
}

void writebrl (brldim * brl)
{
  short i;			/* loop counter */
  static unsigned char prebrl[2] = { "\005D" };	/* code to send before Braille */
  static int timer = 0;		/* for internal cursor */
  int timeout;			/* while waiting for an ACK */

  /* If the intoverride flag is set, then calls to writebrl() from the main
   * module are ignored, because the display is in internal use.
   * This is highly antisocial behaviour!
   */
  if (intoverride)
    return;

  /* First, the internal cursor: */
  if (int_cursor)
   {
     timer = (timer + 1) % (INT_CSR_SPEED * 2);
     if (timer < INT_CSR_SPEED)
       brl->disp[int_cursor - 1] = 0x55;
     else
       brl->disp[int_cursor - 1] = 0xaa;
   }

  /* Next we must handle display reversal: */
  if (reverse_kbd)
    for (i = 0; i < blitesz; i++)
      rawdata[i] = revtrans[brl->disp[blitesz - 1 - i]];
  else
    memcpy (rawdata, brl->disp, blitesz);

  /* Only refresh display if the data has changed: */
  if (memcmp (rawdata, prevdata, blitesz))
   {
     /* Save new Braille data: */
     memcpy (prevdata, rawdata, blitesz);

     /* Dot mapping from standard to BrailleLite: */
     for (i = 0; i < blitesz; rawdata[i++] = blitetrans[rawdata[i]]);

     /* First we process any pending keystrokes, just in case any of them
      * are ^e ...
      */
     waiting_ack = 0;		/* Not really necessary, but ... */
     getbrlkeys ();

     /* Next we send the ^eD sequence, and wait for an ACK */
     waiting_ack = 1;
     do
      {
	/* send the ^ED... */
	write (blite_fd, prebrl, 2);

	/* Now we must wait for an acknowledgement ... */
	timeout = ACK_TIMEOUT / 10;
	getbrlkeys ();
	while (waiting_ack)
	 {
	   delay (10);		/* sleep for 10 ms */
	   getbrlkeys ();
	   if (--timeout < 0)
	     break;		/* we'll try to send ^ED again... */
	 }
      }
     while (waiting_ack);

     /* OK, now we'll suppose we're all clear to send Braille data. */
     write (blite_fd, rawdata, blitesz);

     /* And once again we wait for acknowledgement. */
     waiting_ack = 1;
     timeout = ACK_TIMEOUT / 10;
     getbrlkeys ();
     while (waiting_ack)
      {
	delay (10);		/* sleep for 10 ms */
	getbrlkeys ();
	if (--timeout < 0)
	  break;		/* just in case it'll never happen */
      }
   }
}

int readbrl (int *type)
{
  blkey key;

  /* Process any new keystrokes: */
  getbrlkeys ();
  if (qget (&key) == EOF)	/* no keys to process */
   {
     *type = 0;
     return EOF;
   }

  *type = 1;
  if (key.routing)
    return key.routing + 299;
  if (!key.asc)
    switch (key.cmd)
     {
     case 117:
       *type = 1;
       return 260;
     case 100:
       *type = 1;
       return 261;
     case 123:
       *type = 1;
       return 262;
     case 125:
       *type = 1;
       return 263;
     }

  if (key.spcbar)
    return 264;
  if (key.raw || key.asc)
    return key.raw;

  return EOF;

  *type = 0;
  return EOF;
}

static void getbrlkeys (void)
{
  unsigned char c;		/* character buffer */

  while (read (blite_fd, &c, 1))
   {
     if (waiting_ack && c == 5)	/* ^e is the acknowledgement character ... */
       waiting_ack = 0;
     else
       qput (c);
   }
}

static int qput (unsigned char c)
{
  if (qlen == QSZ)
    return EOF;
  qbase[(qoff + qlen++) % QSZ] = c;
  return 0;
}

static int qget (blkey * kp)
{
  unsigned char c, c2, c3;
  int ext;

  if (qlen == 0)
    return EOF;
  c = qbase[qoff];

  /* extended sequences start with a zero */
  ext = (c == 0);

  /* extended sequences requires 3 bytes */
  if (ext && qlen < 3)
    return EOF;

  memset (kp, 0, sizeof (*kp));

  if (!ext)
   {
     /* non-extended sequences (BL18) */

     /* We must deal with keyboard reversal here: */
     if (reverse_kbd)
      {
	if (c >= 0x80)		/* advance bar */
	  c ^= 0x03;
	else
	  c = (c & 0x40) | ((c & 0x38) >> 3) | ((c & 0x07) << 3);
      }

     /* Now we fill in all the info about the keypress: */
     if (c >= 0x80)		/* advance bar */
      {
	kp->raw = c;
	switch (c)
	 {
	 case 0x83:		/* left */
	   kp->cmd = BLT_BARLT;
	   break;
	 case 0x80:		/* right */
	   kp->cmd = BLT_BARRT;
	   break;
	 default:		/* unrecognised keypress */
	   kp->cmd = 0;
	 }
      }
     else
      {
	kp->spcbar = ((c & 0x40) ? 1 : 0);
	c &= 0x3f;		/* leave only dot key info */
	kp->raw = c;
	kp->cmd = cmdtrans[c];
	kp->asc = brltrans[c];
      }
   }
  else
   {
     /* extended sequences (BL40) */

     c2 = qbase[((qoff + 1) % QSZ)];
     c3 = qbase[((qoff + 2) % QSZ)];

     /* We must deal with keyboard reversal here: */
     if (reverse_kbd)
      {
	if (c2 == 0)
	 {			/* advance bars or routing keys */
	   if (c3 & 0x80)	/* advance bars */
	     c3 = ((c3 & 0xF0) |
		   ((c3 & 0x1) << 3) | ((c3 & 0x2) << 1) |
		   ((c3 & 0x4) >> 1) | ((c3 & 0x8) >> 3));
	   else if (c3 > 0 && c3 <= blitesz)
	     c3 = blitesz - c3 + 1;
	 }
	else
	  c2 = (((c2 & 0x38) >> 3) | ((c2 & 0x07) << 3) |
		((c2 & 0x40) << 1) | ((c2 & 0x80) >> 1));
	c3 = ((c3 & 0x40) | ((c3 & 0x38) >> 3) | ((c3 & 0x07) << 3));
      }

     /* Now we fill in all the info about the keypress: */
     if (c2 == 0)		/* advance bars or routing keys */
      {
	kp->raw = c3;
	if (c3 & 0x80)
	 {			/* advance bars */
	   switch (c3 & 0xF)
	    {
	    case 0x8:		/* left 1 */
	      kp->cmd = BLT_BARLT1;
	      break;
	    case 0x4:		/* right 1 */
	      kp->cmd = BLT_BARRT1;
	      break;
	    case 0x2:		/* left 2 */
	      kp->cmd = BLT_BARLT2;
	      break;
	    case 0x1:		/* right 2 */
	      kp->cmd = BLT_BARRT2;
	      break;
	    default:		/* unrecognised keypress */
	      kp->cmd = 0;
	    }
	 }
	else if (c3 > 0 && c3 <= blitesz)
	  kp->routing = c3;
      }
     else
      {
	kp->raw = c2;
	kp->spcbar = ((c3 & 0x40) ? 1 : 0);
	c3 &= 0x3f;		/* leave only dot key info */
	kp->cmd = cmdtrans[c3];
	kp->asc = brltrans[c3];
      }
   }

  if (repeat > 0)		/* we'll have to repeat this ... */
    repeat--;
  else
    /* adjust queue variables for next member */
   {
     qoff = (qoff + (ext ? 3 : 1)) % QSZ;
     qlen -= (ext ? 3 : 1);
   }

  return 0;
}
