# -*- coding: utf-8 -*-
# Copyright 2010, Google Inc.
# All rights reserved.
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions are
# met:
#
#     * Redistributions of source code must retain the above copyright
# notice, this list of conditions and the following disclaimer.
#     * Redistributions in binary form must reproduce the above
# copyright notice, this list of conditions and the following disclaimer
# in the documentation and/or other materials provided with the
# distribution.
#     * Neither the name of Google Inc. nor the names of its
# contributors may be used to endorse or promote products derived from
# this software without specific prior written permission.
#
# THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
# "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
# LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
# A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
# OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
# SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
# LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
# DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
# THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
# (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
# OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

"""Generate test set for stress test of Calculator."""

# usage:
# $ python gen_test.py --size=5000 > testset.txt
# $ python gen_test.py --size=5000 --scenario_test > scenario_test.txt
#
# |size| is number of lines to be generated.
# If |scenario_test| is True, key events scenario is output to file.  It can
# be tested by client/session_scenario_test_main.  If |verify| is also True,
# outputs ">>" lines to verify conversion result.
#
# Each test data is printed in each line with following format:
#   expression=answer
# Answer is suppressed if the expression is invalid, i.e. a false test case.
# Calculator should reject such expression.  There are two cases that the
# expression is invalid: syntactically incorrect expression, and expression
# that causes mathematical error such as infinity and NaN.
#
# Answers are generated by eval() of Python.  However, '%' operator behaves
# differently in C and Python, so '%' is treated exceptionally.  Though there
# is such difficulty, this script also generates test data which includes '%'
# operator.

import math
import optparse
import random

# Max number of operators generated by GenerateExpressionWithoutMod().
MAX_NUMBER_OF_OPERATORS = 10

# Max number of digits generated by GetRandomNumber()
MAX_LENGTH_OF_NUMBER = 20

# Max number of '%' operators generated by GenerateRandomExpression().
MAX_NUMBER_OF_MODS = 5

# Maps from operators of Calculator to corresponding Python operators
UNARY_OPERATORS = {
    "+": "+",
    "\xEF\xBC\x8B": "+",  # "＋"
    "-": "-",
    "\xE3\x83\xBC": "-",  # "−"
    "\xE2\x88\x92": "-",  # "ー", which is casually called 'nobashi-bo'.
}
BINARY_OPERATORS = {
    "+": "+",
    "\xEF\xBC\x8B": "+",  # "＋"
    "-": "-",
    "\xE3\x83\xBC": "-",  # "−"
    "\xE2\x88\x92": "-",  # "ー", which is casually called 'nobashi-bo'.
    "*": "*",
    "\xEF\xBC\x8A": "*",  # "＊"
    "/": "/",
    "\xE3\x83\xBB": "/",  # "・", which can be input by typing '/' key.
    "\xEF\xBC\x8F": "/",  # "／"
    "^": "**",
    "\xEF\xBC\xBE": "**",  # "＾"
}
MOD = ["%", "\xEF\xBC\x85"]  # "％"
LPAR = ["(", "\xEF\xBC\x88"]  # "（"
RPAR = [")", "\xEF\xBC\x89"]  # "）"

NUMBERS = {
    "0": "0",
    "\xEF\xBC\x90": "0",  # "０"
    "1": "1",
    "\xEF\xBC\x91": "1",  # "１"
    "2": "2",
    "\xEF\xBC\x92": "2",  # "２"
    "3": "3",
    "\xEF\xBC\x93": "3",  # "３"
    "4": "4",
    "\xEF\xBC\x94": "4",  # "４"
    "5": "5",
    "\xEF\xBC\x95": "5",  # "５"
    "6": "6",
    "\xEF\xBC\x96": "6",  # "６"
    "7": "7",
    "\xEF\xBC\x97": "7",  # "７"
    "8": "8",
    "\xEF\xBC\x98": "8",  # "８"
    "9": "9",
    "\xEF\xBC\x99": "9",  # "９"
}
ZERO = ["0", "\xEF\xBC\x90"]  # "０"
DOT = [".", "\xEF\xBC\x8E"]  # "．"

INVALID_MARK = ""


# Adds left and right parentheses to given string.
def AddParens(term):
  return random.choice(LPAR) + term + random.choice(RPAR)


# Returns True if value is NaN.
def IsNaN(value):
  return value != value


# Generates list of random integers with specified number and sum.
def GetRandomPartition(length, sum_value):
  result = []
  for i in range(length - 1):
    value = random.randint(0, sum_value)
    result.append(value)
    sum_value -= value
  result.append(sum_value)
  return result


# Returns a triple of strings.  First element is number for Calculator.  Second
# is number for Python, which is forced to be float (e.g. "1.0" rather than
# "1"), in order to avoid integer division (e.g. eval("3/2") == 1 in Python).
# Thrid is half-width version of first element.
# It actually generates each digit one by one randomly.
def GetRandomNumber(max_length):
  calc = ""
  pyth = ""
  dot_already = False
  length = random.randint(1, max_length)
  char = random.choice(NUMBERS.keys())
  calc += char
  pyth += NUMBERS[char]
  i = 1
  if NUMBERS[char] == "0":
    # If first character is zero, the next character should be '.'.
    dot_already = True
    calc += random.choice(DOT)
    pyth += "."
    # It does not count the first zero as one digit.
    i = 0
  while i < length:
    if dot_already:
      i += 1
      char = random.choice(NUMBERS.keys())
      calc += char
      pyth += NUMBERS[char]
      continue
    if random.randint(0, 3) == 0:
      dot_already = True
      calc += random.choice(DOT)
      pyth += "."
      continue
    i += 1
    char = NUMBERS.keys()[random.randint(0, len(NUMBERS.keys()) - 1)]
    calc += char
    pyth += NUMBERS[char]

  normal = pyth
  if not dot_already:
    pyth += ".0"
  return (calc, pyth, normal)


# Returns a triple of unary operators.  First element is of Calculator, second
# is of Python, and thrid is of key events list.
def GetRandomUnaryOperator():
  # Sometimes suppress unary operator
  if random.randint(0, 2) == 0:
    return ("", "", "")
  name = random.choice(UNARY_OPERATORS.keys())
  return (name, UNARY_OPERATORS[name], UNARY_OPERATORS[name])


# Returns a triple of binary operators.  First element is of Calculator, second
# is of Python, and thrid is of key events list.
def GetRandomBinaryOperator():
  name = random.choice(BINARY_OPERATORS.keys())
  pyth = BINARY_OPERATORS[name]
  if pyth == "**":
    return (name, pyth, "^")
  return (name, pyth, pyth)


# Returns a triple: Calculator expression, its evaluated number, and half-width
# strings used for key events list.  The expression does not include '%'
# operator.  It may generate an invalid expression, and in such case the second
# element is NaN.  |number_of_operators| does not count unary operators.
def GenerateExpressionWithoutMod(number_of_operators):
  depth = 0
  calc = ""  # output which is a Calculator expression
  pyth = ""  # output which will be evaluated by python
  normal = ""  # output which is used for key events list
  # Iterate for each number token.
  for i in range(number_of_operators + 1):
    # binary op.  Skip in the first iteration.
    if i > 0:
      binary_op = GetRandomBinaryOperator()
      calc += binary_op[0]
      pyth += binary_op[1]
      normal += binary_op[2]
    # unary op
    while random.randint(0, 1) == 0:
      unary_op = GetRandomUnaryOperator()
      calc += unary_op[0]
      pyth += unary_op[1]
      normal += unary_op[2]
    # left paren
    if random.randint(0, 1) == 1:
      depth += 1
      calc += random.choice(LPAR)
      pyth += "("
      normal += "("
    # unary op
    while random.randint(0, 1) == 0:
      unary_op = GetRandomUnaryOperator()
      calc += unary_op[0]
      pyth += unary_op[1]
      normal += unary_op[2]
    # number
    num_pair = GetRandomNumber(MAX_LENGTH_OF_NUMBER)
    calc += num_pair[0]
    pyth += num_pair[1]
    normal += num_pair[2]
    # right paren
    if (depth > 0) and (random.randint(0, 1) == 1):
      depth -= 1
      calc += random.choice(RPAR)
      pyth += ")"
      normal += ")"
  # Set |depth| number of RPARs.
  for i in range(depth):
    calc += random.choice(RPAR)
    pyth += ")"
    normal += ")"
  try:
    return (calc, eval(pyth), normal)
  except (ArithmeticError, SyntaxError, ValueError):
    # Return NaN if pyth is invalid expression.
    return (calc, float("nan"), normal)


# Returns a triple: Calculator expression, its evaluated number, and half-width
# string which is used for key events list. Expression may include '%'
# operator.  It may generate an invalid expression, and in such case the second
# element is NaN.  |max_number_of_operatos| does not count unary operators.
def GenerateRandomExpression(max_number_of_operators, max_number_of_mods):
  num_mods = random.randint(0, max_number_of_mods)
  num_other_ops = max(1, random.randint(0, max_number_of_operators - num_mods))
  num_ops_list = GetRandomPartition(num_mods + 1, num_other_ops)
  first_expr = GenerateExpressionWithoutMod(num_ops_list[0])
  calc = first_expr[0]
  value = first_expr[1]
  normal = first_expr[2]
  for i in range(num_mods):
    factor = GenerateExpressionWithoutMod(num_ops_list[i + 1])
    try:
      if random.randint(0, 1) == 0:
        # Add "(factor)%" to left side of calc.
        calc = AddParens(factor[0]) + random.choice(MOD) + AddParens(calc)
        normal = "(" + factor[2] + ")%(" + normal + ")"
        if not IsNaN(value):
          value = math.fmod(factor[1], value)  # it emulates C function fmod().
      else:
        # Add "%(factor)" to right side of calc.
        calc = AddParens(calc) + random.choice(MOD) + AddParens(factor[0])
        normal = "(" + normal + ")%(" + factor[2] + ")"
        if not IsNaN(value):
          value = math.fmod(value, factor[1])
    except (ArithmeticError, SyntaxError, ValueError):
      # Return NaN if the expression is invalid.
      value = float("nan")
  # Sometimes insert improper number of parentheses.
  switch_num = random.randint(0, 9)
  if switch_num == 0:
    calc = random.choice(LPAR) + calc
    normal = "(" + normal
    value = float("nan")
  elif switch_num == 1:
    calc += random.choice(RPAR)
    normal += ")"
    value = float("nan")
  return (calc, value, normal)


# Print key events list from half-width expression.
def PrintKeyEvents(expression):
  for c in expression:
    print c
  print "="
  print "SPACE"
  print "ENTER"


def main():
  parser = optparse.OptionParser()
  parser.add_option("--size", type="int", dest="size", default=5000)
  parser.add_option("--scenario_test",
                    action="store_true", dest="scenario_test", default=False)
  parser.add_option("--verify",
                    action="store_true", dest="verify", default=False)
  (options, unused_args) = parser.parse_args()

  if options.scenario_test:
    print "## This file is auto-generated key events data for Calculator."
    if options.verify:
      print "## Verification lines are inserted."

  random.seed(None)
  for i in range(options.size):
    expr_pair = GenerateRandomExpression(MAX_NUMBER_OF_OPERATORS,
                                         MAX_NUMBER_OF_MODS)
    calc = expr_pair[0]
    if options.scenario_test:
      PrintKeyEvents(expr_pair[2])
      if options.verify and not IsNaN(expr_pair[1]):
        # Insert a verification line.
        print ">> %.14g" % expr_pair[1]
      print ""
      continue
    if IsNaN(expr_pair[1]):
      print "%s=" % calc
      continue
    try:
      val_str = "%.14g" % expr_pair[1]
      if val_str.count("inf") > 0:
        print "%s=" % calc
      else:
        print "%(expr)s=%(val)s" % {"expr": calc, "val": val_str}
    except (ArithmeticError, SyntaxError, ValueError):
      print "%s=" % calc

if __name__ == "__main__":
  main()
