/*
 * Copyright (C) 2004, 2005, 2006 Nikolas Zimmermann <zimmermann@kde.org>
 * Copyright (C) 2004, 2005, 2006, 2007 Rob Buis <buis@kde.org>
 * Copyright (C) 2007 Apple Inc. All rights reserved.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public License
 * along with this library; see the file COPYING.LIB.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 */

#include "config.h"

#if ENABLE(SVG)
#include "SVGLength.h"

#include "CSSHelper.h"
#include "FloatConversion.h"
#include "FrameView.h"
#include "NotImplemented.h"
#include "RenderObject.h"
#include "RenderView.h"
#include "SVGParserUtilities.h"
#include "SVGSVGElement.h"

#include <math.h>
#include <wtf/Assertions.h>

namespace WebCore {

// Helper functions
static inline unsigned int storeUnit(SVGLengthMode mode, SVGLengthType type)
{
    return (mode << 4) | type;
}

static inline SVGLengthMode extractMode(unsigned int unit)
{
    unsigned int mode = unit >> 4;    
    return static_cast<SVGLengthMode>(mode);
}

static inline SVGLengthType extractType(unsigned int unit)
{
    unsigned int mode = unit >> 4;
    unsigned int type = unit ^ (mode << 4);
    return static_cast<SVGLengthType>(type);
}

static inline String lengthTypeToString(SVGLengthType type)
{
    switch (type) {
    case LengthTypeUnknown:
    case LengthTypeNumber:
        return "";    
    case LengthTypePercentage:
        return "%";
    case LengthTypeEMS:
        return "em";
    case LengthTypeEXS:
        return "ex";
    case LengthTypePX:
        return "px";
    case LengthTypeCM:
        return "cm";
    case LengthTypeMM:
        return "mm";
    case LengthTypeIN:
        return "in";
    case LengthTypePT:
        return "pt";
    case LengthTypePC:
        return "pc";
    }

    return String();
}

inline SVGLengthType stringToLengthType(const UChar*& ptr, const UChar* end)
{
    if (ptr == end)
        return LengthTypeNumber;

    const UChar firstChar = *ptr;
    ++ptr;

    if (firstChar == '%') {
        if (ptr == end)
            return LengthTypePercentage;
        return LengthTypeUnknown;
    }

    const UChar secondChar = *ptr;

    if (++ptr != end)
        return LengthTypeUnknown;

    if (firstChar == 'e' && secondChar == 'm')
        return LengthTypeEMS;
    if (firstChar == 'e' && secondChar == 'x')
        return LengthTypeEXS;
    if (firstChar == 'p' && secondChar == 'x')
        return LengthTypePX;
    if (firstChar == 'c' && secondChar == 'm')
        return LengthTypeCM;
    if (firstChar == 'm' && secondChar == 'm')
        return LengthTypeMM;
    if (firstChar == 'i' && secondChar == 'n')
        return LengthTypeIN;
    if (firstChar == 'p' && secondChar == 't')
        return LengthTypePT;
    if (firstChar == 'p' && secondChar == 'c')
        return LengthTypePC;

    return LengthTypeUnknown;
}

SVGLength::SVGLength(SVGLengthMode mode, const String& valueAsString)
    : m_valueInSpecifiedUnits(0.0f)
    , m_unit(storeUnit(mode, LengthTypeNumber))
{
    setValueAsString(valueAsString);
}

SVGLength::SVGLength(const SVGLength& other)
    : m_valueInSpecifiedUnits(other.m_valueInSpecifiedUnits)
    , m_unit(other.m_unit)
{
}

bool SVGLength::operator==(const SVGLength& other) const
{
    return m_unit == other.m_unit
        && m_valueInSpecifiedUnits == other.m_valueInSpecifiedUnits;
}

bool SVGLength::operator!=(const SVGLength& other) const
{
    return !operator==(other);
}

SVGLengthType SVGLength::unitType() const
{
    return extractType(m_unit);
}

float SVGLength::value(const SVGElement* context) const
{
    SVGLengthType type = extractType(m_unit);
    if (type == LengthTypeUnknown)
        return 0.0f;

    switch (type) {
    case LengthTypeNumber:
        return m_valueInSpecifiedUnits;
    case LengthTypePercentage:
        return SVGLength::PercentageOfViewport(m_valueInSpecifiedUnits / 100.0f, context, extractMode(m_unit));
    case LengthTypeEMS:
    case LengthTypeEXS:
    {
        RenderStyle* style = 0;
        if (context && context->renderer())
            style = context->renderer()->style();
        if (style) {
            float useSize = style->fontSize();
            ASSERT(useSize > 0);
            if (type == LengthTypeEMS)
                return m_valueInSpecifiedUnits * useSize;
            else {
                float xHeight = style->font().xHeight();
                // Use of ceil allows a pixel match to the W3Cs expected output of coords-units-03-b.svg
                // if this causes problems in real world cases maybe it would be best to remove this
                return m_valueInSpecifiedUnits * ceilf(xHeight);
            }
        }
        return 0.0f;
    }
    case LengthTypePX:
        return m_valueInSpecifiedUnits;
    case LengthTypeCM:
        return m_valueInSpecifiedUnits / 2.54f * cssPixelsPerInch;
    case LengthTypeMM:
        return m_valueInSpecifiedUnits / 25.4f * cssPixelsPerInch;
    case LengthTypeIN:
        return m_valueInSpecifiedUnits * cssPixelsPerInch;
    case LengthTypePT:
        return m_valueInSpecifiedUnits / 72.0f * cssPixelsPerInch;
    case LengthTypePC:
        return m_valueInSpecifiedUnits / 6.0f * cssPixelsPerInch;
    default:
        break;
    }

    ASSERT_NOT_REACHED();
    return 0.0f;
}

void SVGLength::setValue(float value)
{
    SVGLengthType type = extractType(m_unit);
    ASSERT(type != LengthTypeUnknown);

    switch (type) {
    case LengthTypeNumber:
        m_valueInSpecifiedUnits = value;
        break;
    case LengthTypePercentage:
    case LengthTypeEMS:
    case LengthTypeEXS:
        notImplemented();
        break;
    case LengthTypePX:
        m_valueInSpecifiedUnits = value;
        break;
    case LengthTypeCM:
        m_valueInSpecifiedUnits = value * 2.54f / cssPixelsPerInch;
        break;
    case LengthTypeMM:
        m_valueInSpecifiedUnits = value * 25.4f / cssPixelsPerInch;
        break;
    case LengthTypeIN:
        m_valueInSpecifiedUnits = value / cssPixelsPerInch;
        break;
    case LengthTypePT:
        m_valueInSpecifiedUnits = value * 72.0f / cssPixelsPerInch;
        break;
    case LengthTypePC:
        m_valueInSpecifiedUnits = value / 6.0f * cssPixelsPerInch;
        break;
    default:
        break;
    }
}

void SVGLength::setValueInSpecifiedUnits(float value)
{
    m_valueInSpecifiedUnits = value;
}

float SVGLength::valueInSpecifiedUnits() const
{
    return m_valueInSpecifiedUnits;
}

float SVGLength::valueAsPercentage() const
{
    // 100% = 100.0 instead of 1.0 for historical reasons, this could eventually be changed
    if (extractType(m_unit) == LengthTypePercentage)
        return valueInSpecifiedUnits() / 100.0f;

    return valueInSpecifiedUnits();
}

bool SVGLength::setValueAsString(const String& s)
{
    if (s.isEmpty())
        return false;

    float convertedNumber = 0.0f;
    const UChar* ptr = s.characters();
    const UChar* end = ptr + s.length();

    if (!parseNumber(ptr, end, convertedNumber, false))
        return false;

    SVGLengthType type = stringToLengthType(ptr, end);
    if (type == LengthTypeUnknown)
        return false;

    m_unit = storeUnit(extractMode(m_unit), type);
    m_valueInSpecifiedUnits = convertedNumber;
    return true;
}

String SVGLength::valueAsString() const
{
    return String::number(m_valueInSpecifiedUnits) + lengthTypeToString(extractType(m_unit));
}

void SVGLength::newValueSpecifiedUnits(unsigned short type, float value)
{
    if (type == LengthTypeUnknown || type > LengthTypePC)
        return;

    m_unit = storeUnit(extractMode(m_unit), (SVGLengthType) type);
    m_valueInSpecifiedUnits = value;
}

void SVGLength::convertToSpecifiedUnits(unsigned short type, const SVGElement* context)
{
    if (type == LengthTypeUnknown || type > LengthTypePC)
        return;

    float valueInUserUnits = value(context);
    m_unit = storeUnit(extractMode(m_unit), (SVGLengthType) type);
    setValue(valueInUserUnits);
}

float SVGLength::PercentageOfViewport(float value, const SVGElement* context, SVGLengthMode mode)
{
    ASSERT(context);

    float width = 0.0f, height = 0.0f;
    SVGElement* viewportElement = context->viewportElement();

    // PercentageOfViewport() is used to resolve all relative-positioned values within a SVG document (fragment)
    Document* doc = context->document();
    if (doc->documentElement() == context) {
        // Resolve value against outermost <svg> element
        if (RenderView* view = toRenderView(doc->renderer())) {
            width = view->viewWidth();
            height = view->viewHeight();
         }
    } else if (viewportElement && viewportElement->isSVG()) {
        // Resolve value against nearest viewport element (common case: inner <svg> elements)
        const SVGSVGElement* svg = static_cast<const SVGSVGElement*>(viewportElement);
        if (svg->hasAttribute(SVGNames::viewBoxAttr)) {
            width = svg->viewBox().width();
            height = svg->viewBox().height();
        } else {
            width = svg->width().value(svg);
            height = svg->height().value(svg);
        }
    } else if (context->parent() && !context->parent()->isSVGElement()) {
        // Resolve value against enclosing non-SVG RenderBox
        if (RenderObject* renderer = context->renderer()) {
            if (renderer->isBox()) {
                RenderBox* box = toRenderBox(renderer);
                width = box->width();
                height = box->height();
            }
        }
    }

    if (mode == LengthModeWidth)
        return value * width;
    else if (mode == LengthModeHeight)
        return value * height;
    else if (mode == LengthModeOther)
        return value * sqrtf(powf(width, 2) + powf(height, 2)) / sqrtf(2.0f);

    return 0.0f;
}

SVGLength SVGLength::fromCSSPrimitiveValue(CSSPrimitiveValue* value)
{
    ASSERT(value);

    SVGLengthType svgType;
    switch (value->primitiveType()) {
    case CSSPrimitiveValue::CSS_NUMBER:
        svgType = LengthTypeNumber;
        break;
    case CSSPrimitiveValue::CSS_PERCENTAGE:
        svgType = LengthTypePercentage;
        break;
    case CSSPrimitiveValue::CSS_EMS:
        svgType = LengthTypeEMS;
        break;
    case CSSPrimitiveValue::CSS_EXS:
        svgType = LengthTypeEXS;
        break;
    case CSSPrimitiveValue::CSS_PX:
        svgType = LengthTypePX;
        break;
    case CSSPrimitiveValue::CSS_CM:
        svgType = LengthTypeCM;
        break;
    case CSSPrimitiveValue::CSS_MM:
        svgType = LengthTypeMM;
        break;
    case CSSPrimitiveValue::CSS_IN:
        svgType = LengthTypeIN;
        break;
    case CSSPrimitiveValue::CSS_PT:
        svgType = LengthTypePT;
        break;
    case CSSPrimitiveValue::CSS_PC:
        svgType = LengthTypePC;
        break;
    case CSSPrimitiveValue::CSS_UNKNOWN:
    default:
        svgType = LengthTypeUnknown;
        break;
    };

    if (svgType == LengthTypeUnknown)
        return SVGLength();

    SVGLength length;
    length.newValueSpecifiedUnits(svgType, value->getFloatValue());
    return length;
}

PassRefPtr<CSSPrimitiveValue> SVGLength::toCSSPrimitiveValue(const SVGLength& length)
{
    CSSPrimitiveValue::UnitTypes cssType = CSSPrimitiveValue::CSS_UNKNOWN;
    switch (length.unitType()) {
    case LengthTypeUnknown:
        break;
    case LengthTypeNumber:
        cssType = CSSPrimitiveValue::CSS_NUMBER;
        break;
    case LengthTypePercentage:
        cssType = CSSPrimitiveValue::CSS_PERCENTAGE;
        break;
    case LengthTypeEMS:
        cssType = CSSPrimitiveValue::CSS_EMS;
        break;
    case LengthTypeEXS:
        cssType = CSSPrimitiveValue::CSS_EXS;
        break;
    case LengthTypePX:
        cssType = CSSPrimitiveValue::CSS_PX;
        break;
    case LengthTypeCM:
        cssType = CSSPrimitiveValue::CSS_CM;
        break;
    case LengthTypeMM:
        cssType = CSSPrimitiveValue::CSS_MM;
        break;
    case LengthTypeIN:
        cssType = CSSPrimitiveValue::CSS_IN;
        break;
    case LengthTypePT:
        cssType = CSSPrimitiveValue::CSS_PT;
        break;
    case LengthTypePC:
        cssType = CSSPrimitiveValue::CSS_PC;
        break;
    default:
        ASSERT_NOT_REACHED();
    };

    return CSSPrimitiveValue::create(length.valueInSpecifiedUnits(), cssType);
}

}

#endif
