/***************************************************************************
 *   Copyright (C) 2008 by S. MANKOWSKI / G. DE BURE skrooge@mankowski.fr  *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>  *
 ***************************************************************************/
/** @file
 * A skrooge plugin to search and process operations.
 *
 * @author Stephane MANKOWSKI
 */
#include "skgsearchpluginwidget.h"
#include "skgobjectmodel.h"
#include "skgmainpanel.h"
#include "skgtraces.h"
#include "skgdocument.h"
#include "skgbankincludes.h"
#include "skgservices.h"
#include "skgcategoryobject.h"
#include "skgruleobject.h"
#include "skgsortfilterproxymodel.h"

#include <kstandarddirs.h>

#include <QDomDocument>
#include <QDir>

SKGSearchPluginWidget::SKGSearchPluginWidget(SKGDocument* iDocument)
                :SKGTabPage(iDocument)
{
        SKGTRACEIN(1, "SKGSearchPluginWidget::SKGSearchPluginWidget");

        ui.setupUi(this);

        //Add Standard KDE Icons to buttons
        ui.kUpdate->setIcon ( KIcon ( "dialog-ok-apply" ) );
        ui.kAdd->setIcon ( KIcon ( "list-add" ) );
        ui.kApply->setIcon ( KIcon ( "system-run" ) );

        ui.kTopBtn->setIcon ( KIcon ( "arrow-up-double" ) );
        ui.kUpBtn->setIcon ( KIcon ( "arrow-up" ) );
        ui.kDownBtn->setIcon ( KIcon ( "arrow-down" ) );
        ui.kBottomBtn->setIcon ( KIcon ( "arrow-down-double" ) );

        ui.kQueryBtn->setIcon (KIcon ( "edit-find" ) );
        ui.kActionBtn->setIcon (KIcon ( "system-run" ) );
        ui.kExecutionBtn->setIcon (KIcon ( "configure" ) );

        //Set kProcessType
        ui.kProcessType->addItem(KIcon ( "view-refresh"), i18nc("Noun, a modification", "Update"));
        ui.kProcessType->addItem(KIcon ( "dialog-warning"), i18nc("Noun, an alarm", "Alarm"));
        ui.kAlarmFrm->hide();

        //Fill combo boxes
        QStringList attributeForQuery;
        attributeForQuery << "d_date" << "i_number" << "t_mode" << "t_payee" << "t_comment" << "t_REALCATEGORY" << "t_status" << "t_bookmarked" << "t_imported" << "t_ACCOUNT" << "f_REALCURRENTAMOUNT" << "t_REFUND";
        ui.kQueryCreator->setParameters(iDocument, "v_operation_consolidated", attributeForQuery);

        QStringList attributeForUpdate;
        attributeForUpdate << "i_number" << "t_mode" << "t_payee" << "t_comment" << "t_status" << "t_bookmarked" << "t_imported"<< "t_REALCATEGORY"<< "t_ACCOUNT" << "t_REFUND";
        ui.kActionCreator->setParameters(iDocument, "v_operation_consolidated", attributeForUpdate, true);

        //Bind operation view
        objectModel = new SKGObjectModel ( ( SKGDocumentBank* ) getDocument(), "v_rule", "1=1 ORDER BY f_sortorder", this, "", false );

        SKGSortFilterProxyModel* modelproxy = new SKGSortFilterProxyModel ( this );
        modelproxy->setSourceModel ( objectModel );
        modelproxy->setSortRole(99);
        modelproxy->setDynamicSortFilter ( true );
        ui.kFilterEdit->setProxy(modelproxy);

        ui.kView->setModel ( modelproxy );
        ui.kView->setWindowTitle ( i18n ( "Rules" ) );
        ui.kView->sortByColumn(0, Qt::AscendingOrder);

        //Add registered global action in contextual menu
        if (SKGMainPanel::getMainPanel()) {
                ui.kView->insertAction ( 0, SKGMainPanel::getMainPanel()->getGlobalAction ( "edit_delete" ) );
                QAction* sep=new QAction ( this );
                sep->setSeparator ( true );
                ui.kView->insertAction ( 0, sep );
                ui.kView->insertAction ( 0, SKGMainPanel::getMainPanel()->getGlobalAction ( "open_report" ) );
        }

        setCurrentMode(0);

        connect ( ui.kView, SIGNAL ( selectionChangedDelayed() ), this, SLOT ( onSelectionChanged() ) );
        connect ( objectModel, SIGNAL ( beforeReset() ), ui.kView, SLOT ( saveSelection() ) );
        connect ( objectModel, SIGNAL ( afterReset() ), ui.kView, SLOT ( resetSelection() ) );

        //Refresh
        connect ( ( const QObject* ) getDocument(), SIGNAL(tableModified(QString, int)), this, SLOT(dataModified(QString, int)), Qt::QueuedConnection );
        dataModified("", 0);

        onSelectionChanged();
}

SKGSearchPluginWidget::~SKGSearchPluginWidget()
{
        SKGTRACEIN(1, "SKGSearchPluginWidget::~SKGSearchPluginWidget");
        objectModel=NULL;
}

QString SKGSearchPluginWidget::getState()
{
        SKGTRACEIN(10, "SKGSearchPluginWidget::getState");
        QDomDocument doc("SKGML");
        QDomElement root = doc.createElement("parameters");
        doc.appendChild(root);

        root.setAttribute ( "filter", ui.kFilterEdit->lineEdit()->text() );
        root.setAttribute ( "currentPage", getCurrentMode() );

        //Memorize table settings
        root.setAttribute("view", ui.kView->getState());

        return doc.toString();
}

void SKGSearchPluginWidget::setState(const QString& iState )
{
        SKGTRACEIN(10, "SKGSearchPluginWidget::setState");
        QDomDocument doc("SKGML");
        doc.setContent(iState);
        QDomElement root = doc.documentElement();

        QString filter=root.attribute ( "filter" );
        QString currentPage=root.attribute ( "currentPage" );
        if ( !currentPage.isEmpty() ) setCurrentMode ( SKGServices::stringToInt ( currentPage ) );
        if ( !filter.isEmpty() ) ui.kFilterEdit->setText ( filter );

        ui.kView->setState(root.attribute ( "view"));
}

QString SKGSearchPluginWidget::getDefaultStateAttribute()
{
        return "SKGSEARCH_DEFAULT_PARAMETERS";
}

QWidget* SKGSearchPluginWidget::getWidgetForPrint()
{
        return this;
}

int SKGSearchPluginWidget::getCurrentMode()
{
        return mode;
}

void SKGSearchPluginWidget::onBtnModeClicked()
{
        QWidget* sender=static_cast<QWidget*> ( this->sender() );

        int currentMode=getCurrentMode();
        int newMode=0;

        if (sender==ui.kQueryBtn) newMode=0;
        else if (sender==ui.kActionBtn) newMode=1;
        else if (sender==ui.kExecutionBtn) newMode=2;

        if (currentMode==newMode) newMode=-1;
        setCurrentMode(newMode);
}

void SKGSearchPluginWidget::setCurrentMode(int iMode)
{
        mode=iMode;

        ui.kQueryGrp->hide();
        ui.kActionGrp->hide();
        ui.kExecGrp->hide();
        ui.kBtnFrm->setVisible(mode==0 || mode==1);

        ui.kQueryBtn->setChecked(false);
        ui.kActionBtn->setChecked(false);
        ui.kExecutionBtn->setChecked(false);

        if (mode==0) {
                ui.kQueryGrp->show();
                ui.kQueryBtn->setChecked(true);
        } else if (mode==1) {
                ui.kQueryGrp->show();
                ui.kActionGrp->show();
                ui.kActionBtn->setChecked(true);
        } else if (mode==2) {
                ui.kExecGrp->show();
                ui.kExecutionBtn->setChecked(true);
        }

        onEditorModified();
}

SKGObjectBase::SKGListSKGObjectBase SKGSearchPluginWidget::getSelectedObjects()
{
        SKGObjectBase::SKGListSKGObjectBase list=ui.kView->getSelectedObjects();

        //Sort selection by f_sortorder. It's mandatory for reorder functions
        qStableSort(list.begin(), list.end());

        return list;
}

int SKGSearchPluginWidget::getNbSelectedObjects()
{
        return ui.kView->getNbSelectedObjects();
}

void SKGSearchPluginWidget::dataModified(const QString& iTableName, int iIdTransaction)
{
        SKGTRACEIN(1, "SKGSearchPluginWidget::dataModified");
        Q_UNUSED(iIdTransaction);

        //Refresh account list
        if ( iTableName=="rule" || iTableName=="operation" || iTableName.isEmpty() ) {
                //Correction bug 2299394 vvv
                if ( ui.kView->isAutoResized() ) ui.kView->resizeColumnsToContents();
                //Correction bug 2299394 ^^^

		//Correction 215658: vvv because the table is modified, the selection is modified
		onSelectionChanged();
		//Correction 215658: ^^^
        }

        ui.kAlarmUnit->setText(((SKGDocumentBank*) getDocument())->getPrimaryUnit().Name);
}

void SKGSearchPluginWidget::onAddRule()
{
        SKGError err;
        SKGTRACEINRC(1, "SKGSearchPluginWidget::onAddRule", err);
        if (getCurrentMode()==0) {
                SKGRuleObject rule;
                {
                        SKGBEGINTRANSACTION ( *getDocument(), i18n ( "Search creation" ), err);
                        QApplication::setOverrideCursor ( QCursor ( Qt::WaitCursor ) );
                        rule=SKGRuleObject(getDocument());
                        if (err.isSucceeded()) err=rule.setXMLSearchDefinition(ui.kQueryCreator->getXMLCondition());
                        if (err.isSucceeded()) err=rule.setOrder(-1);
                        if (err.isSucceeded()) err=rule.setActionType(SKGRuleObject::SEARCH);
                        if (err.isSucceeded()) err=rule.save();
                        QApplication::restoreOverrideCursor();
                }

                //status bar
                if (err.isSucceeded()) {
                        err=SKGError ( 0, i18n ( "Search created" ) );
                        ui.kView->selectObject ( rule.getUniqueID() );
                } else err.addError ( ERR_FAIL, i18n ( "Search creation failed" ) );
        } else {
                SKGRuleObject rule;
                {
                        SKGBEGINTRANSACTION ( *getDocument(), i18n ( "Search and process creation" ), err);
                        QApplication::setOverrideCursor ( QCursor ( Qt::WaitCursor ) );
                        rule=SKGRuleObject(getDocument());
                        if (err.isSucceeded()) err=rule.setXMLSearchDefinition(ui.kQueryCreator->getXMLCondition());
                        if (err.isSucceeded()) err=rule.setOrder(-1);

                        QString xml=getXMLActionDefinition();
                        if (err.isSucceeded()) err=rule.setActionType(xml.isEmpty() ? SKGRuleObject::SEARCH : SKGRuleObject::ActionType (ui.kProcessType->currentIndex()+1));
                        if (err.isSucceeded()) err=rule.setXMLActionDefinition(xml);
                        if (err.isSucceeded()) err=rule.save();
                        QApplication::restoreOverrideCursor();
                }

                //status bar
                if (err.isSucceeded()) {
                        err=SKGError ( 0, i18n ( "Search and process created" ) );
                        ui.kView->selectObject ( rule.getUniqueID() );
                } else err.addError ( ERR_FAIL, i18n ( "Search and process creation failed" ) );
        }

        //Display error
        SKGMainPanel::getMainPanel()->displayErrorMessage ( err );
}

void SKGSearchPluginWidget::onModifyRule()
{
        SKGError err;
        SKGTRACEINRC(1, "SKGSearchPluginWidget::onModifyRule", err);
        if (getCurrentMode()==0) {
                SKGBEGINTRANSACTION ( *getDocument(), i18n ( "Search update" ), err);
                QApplication::setOverrideCursor ( QCursor ( Qt::WaitCursor ) );
                SKGObjectBase::SKGListSKGObjectBase selection=getSelectedObjects();
                if (selection.count()) {
                        SKGRuleObject rule=selection.at(0);
                        if (err.isSucceeded()) err=rule.setXMLSearchDefinition(ui.kQueryCreator->getXMLCondition());
                        if (err.isSucceeded()) err=rule.save();
                }
                QApplication::restoreOverrideCursor();

                //status bar
                if (err.isSucceeded()) err=SKGError ( 0, i18n ( "Search updated" ) );
                else err.addError ( ERR_FAIL, i18n ( "Search update failed" ) );
        } else {
                SKGBEGINTRANSACTION ( *getDocument(), i18n ( "Search and process update" ), err);
                QApplication::setOverrideCursor ( QCursor ( Qt::WaitCursor ) );
                SKGObjectBase::SKGListSKGObjectBase rules=getSelectedObjects();
                if (rules.count()==1) {
                        SKGRuleObject rule=rules.at(0);
                        if (err.isSucceeded()) err=rule.setXMLSearchDefinition(ui.kQueryCreator->getXMLCondition());
                        QString xml=getXMLActionDefinition();
                        if (err.isSucceeded()) err=rule.setActionType(xml.isEmpty() ? SKGRuleObject::SEARCH : SKGRuleObject::ActionType (ui.kProcessType->currentIndex()+1));
                        if (err.isSucceeded()) err=rule.setXMLActionDefinition(xml);
                        if (err.isSucceeded()) err=rule.save();
                }
                QApplication::restoreOverrideCursor();

                //status bar
                if (err.isSucceeded()) err=SKGError ( 0, i18n ( "Search and process updated" ) );
                else err.addError ( ERR_FAIL, i18n ( "Search and process update failed" ) );
        }

        //Display error
        SKGMainPanel::getMainPanel()->displayErrorMessage ( err );
}

void SKGSearchPluginWidget::onApply()
{
        SKGError err;
        SKGTRACEINRC(1, "SKGSearchPluginWidget::onApply", err);

        //Get rules
        SKGObjectBase::SKGListSKGObjectBase rules;
        if (ui.kSelectedActions->isChecked()) {
                //Get selection
                rules=getSelectedObjects();
        } else {
                //Get all rules
                SKGObjectBase::getObjects(getDocument(), "v_rule", "", rules);
        }

        int nb=rules.count();
        {
                SKGBEGINPROGRESSTRANSACTION ( *getDocument(), i18n ( "Process execution" ), err, nb);
                QApplication::setOverrideCursor ( QCursor ( Qt::WaitCursor ) );
                SKGRuleObject::ProcessMode mode=SKGRuleObject::ALL;
                if (ui.kAllImportedOperationBtn->isChecked()) mode=SKGRuleObject::IMPORTED;
                else if (ui.kNotValidatedBtn->isChecked()) mode=SKGRuleObject::IMPORTEDNOTVALIDATE;
                for (int i=0; err.isSucceeded() && i<nb; ++i) {
                        SKGRuleObject rule=rules.at(i);
                        err=rule.execute(mode);
                        if (err.isSucceeded()) err=getDocument()->stepForward (i+1);
                }
                QApplication::restoreOverrideCursor();
        }

        //status bar
        if (err.isSucceeded()) err=SKGError ( 0, i18n ( "Process executed" ) );
        else err.addError ( ERR_FAIL, i18n ( "Process execution failed" ) );

        //Display error
        SKGMainPanel::getMainPanel()->displayErrorMessage ( err );
}

void SKGSearchPluginWidget::onSelectionChanged()
{
        SKGObjectBase::SKGListSKGObjectBase selection=getSelectedObjects();
        int nbSel=selection.count();

        ui.kSelectedActions->setEnabled(nbSel>0);
        ui.kTopBtn->setEnabled(nbSel>0);
        ui.kUpBtn->setEnabled(nbSel>0);
        ui.kDownBtn->setEnabled(nbSel>0);
        ui.kBottomBtn->setEnabled(nbSel>0);
        ui.kUpdate->setEnabled(nbSel==1);

        if (nbSel==0) {
                ui.kAllActions->setChecked(true);
        } else {
                SKGRuleObject rule=selection.at(0);
                ui.kQueryCreator->setXMLCondition(rule.getXMLSearchDefinition());

                int index=qMax(0,((int) rule.getActionType())-1);
                ui.kProcessType->setCurrentIndex(index);
                if (index==0) {
                        //Set update mpde
                        ui.kActionCreator->setXMLCondition(rule.getXMLActionDefinition());
                } else {
                        //Set alarm mode
                        QDomDocument doc("SKGML");
                        doc.setContent(rule.getXMLActionDefinition());

                        QDomElement element = doc.documentElement();
                        QDomElement elementLine = element.firstChild().toElement();
                        QDomElement elementElement = elementLine.firstChild().toElement();
                        ui.kAlarmAmount->setValue(SKGServices::stringToDouble(elementElement.attribute("value")));
                        ui.kAlarmMessage->setText(elementElement.attribute("value2"));
                }
        }

        onEditorModified();

        emit selectionChanged();
}

void SKGSearchPluginWidget::onActionTypeChanged()
{
        ui.kAlarmFrm->setVisible(ui.kProcessType->currentIndex()==1);
        ui.kActionCreator->setVisible(ui.kProcessType->currentIndex()==0);
}

void SKGSearchPluginWidget::onTop()
{
        SKGError err;
        SKGTRACEINRC(1, "SKGSearchPluginWidget::onTop", err);

        //Get rules
        SKGObjectBase::SKGListSKGObjectBase  rules=getSelectedObjects();
        int nb=rules.count();
        {
                SKGBEGINPROGRESSTRANSACTION ( *getDocument(), i18n ( "Search update" ), err, nb);
                QApplication::setOverrideCursor ( QCursor ( Qt::WaitCursor ) );
                for (int i=nb-1; err.isSucceeded() && i>=0; --i) {
                        SKGRuleObject rule=rules.at(i);

                        double order=1;
                        SKGStringListList result;
                        err=SKGServices::executeSelectSqliteOrder(getDocument(), "SELECT min(f_sortorder) from rule", result);
                        if (err.isSucceeded() && result.count()==2)  order=SKGServices::stringToDouble(result.at(1).at(0))-1;

                        if (err.isSucceeded()) err=rule.setOrder(order);
                        if (err.isSucceeded()) err=rule.save();
                        if (err.isSucceeded()) err=getDocument()->stepForward ( i+1 );

                }
                QApplication::restoreOverrideCursor();
        }

        //status bar
        if (err.isSucceeded()) err=SKGError ( 0, i18n ( "Search updated" ) );
        else err.addError ( ERR_FAIL, i18n ( "Search update failed" ) );

        //Display error
        SKGMainPanel::getMainPanel()->displayErrorMessage ( err );
}

void SKGSearchPluginWidget::onUp()
{
        SKGError err;
        SKGTRACEINRC(1, "SKGSearchPluginWidget::onUp", err);

        //Get rules
        SKGObjectBase::SKGListSKGObjectBase  rules=getSelectedObjects();
        int nb=rules.count();
        {
                SKGBEGINPROGRESSTRANSACTION ( *getDocument(), i18n ( "Search update" ), err, nb);
                QApplication::setOverrideCursor ( QCursor ( Qt::WaitCursor ) );
                for (int i=0; err.isSucceeded() && i<nb; ++i) {
                        SKGRuleObject rule=rules.at(i);

                        double order=rule.getOrder();
                        SKGStringListList result;
                        err=SKGServices::executeSelectSqliteOrder(getDocument(), "SELECT f_sortorder from rule where f_sortorder<"+SKGServices::doubleToString(order)+" ORDER BY f_sortorder DESC", result);
                        if (err.isSucceeded()) {
                                if (result.count()==2) {
                                        order=SKGServices::stringToDouble(result.at(1).at(0))-1;
                                } else if (result.count()>=2) {
                                        order=(SKGServices::stringToDouble(result.at(1).at(0))+SKGServices::stringToDouble(result.at(2).at(0)))/2;
                                }
                        }

                        if (err.isSucceeded()) err=rule.setOrder(order);
                        if (err.isSucceeded()) err=rule.save();
                        if (err.isSucceeded()) err=getDocument()->stepForward ( i+1 );

                }
                QApplication::restoreOverrideCursor();
        }

        //status bar
        if (err.isSucceeded()) err=SKGError ( 0, i18n ( "Search updated" ) );
        else err.addError ( ERR_FAIL, i18n ( "Search update failed" ) );

        //Display error
        SKGMainPanel::getMainPanel()->displayErrorMessage ( err );
}

void SKGSearchPluginWidget::onDown()
{
        SKGError err;
        SKGTRACEINRC(1, "SKGSearchPluginWidget::onDown", err);

        //Get rules
        SKGObjectBase::SKGListSKGObjectBase  rules=getSelectedObjects();
        int nb=rules.count();
        {
                SKGBEGINPROGRESSTRANSACTION ( *getDocument(), i18n ( "Search update" ), err, nb);
                QApplication::setOverrideCursor ( QCursor ( Qt::WaitCursor ) );
                for (int i=nb-1; err.isSucceeded() && i>=0; --i) {
                        SKGRuleObject rule=rules.at(i);

                        double order=rule.getOrder();
                        SKGStringListList result;
                        err=SKGServices::executeSelectSqliteOrder(getDocument(), "SELECT f_sortorder from rule where f_sortorder>"+SKGServices::doubleToString(order)+" ORDER BY f_sortorder ASC", result);
                        if (err.isSucceeded()) {
                                if (result.count()==2) {
                                        order=SKGServices::stringToDouble(result.at(1).at(0))+1;
                                } else if (result.count()>=2) {
                                        order=(SKGServices::stringToDouble(result.at(1).at(0))+SKGServices::stringToDouble(result.at(2).at(0)))/2;
                                }
                        }

                        if (err.isSucceeded()) err=rule.setOrder(order);
                        if (err.isSucceeded()) err=rule.save();
                        if (err.isSucceeded()) err=getDocument()->stepForward ( i+1 );

                }
                QApplication::restoreOverrideCursor();
        }

        //status bar
        if (err.isSucceeded()) err=SKGError ( 0, i18n ( "Search updated" ) );
        else err.addError ( ERR_FAIL, i18n ( "Search update failed" ) );

        //Display error
        SKGMainPanel::getMainPanel()->displayErrorMessage ( err );
}

void SKGSearchPluginWidget::onBottom()
{
        SKGError err;
        SKGTRACEINRC(1, "SKGSearchPluginWidget::onBottom", err);

        //Get rules
        SKGObjectBase::SKGListSKGObjectBase  rules=getSelectedObjects();
        int nb=rules.count();
        {
                SKGBEGINPROGRESSTRANSACTION ( *getDocument(), i18n ( "Search update" ), err, nb);
                QApplication::setOverrideCursor ( QCursor ( Qt::WaitCursor ) );
                for (int i=0; err.isSucceeded() && i<nb; ++i) {
                        SKGRuleObject rule=rules.at(i);

                        double order=1;
                        SKGStringListList result;
                        err=SKGServices::executeSelectSqliteOrder(getDocument(), "SELECT max(f_sortorder) from rule", result);
                        if (err.isSucceeded() && result.count()==2) order=SKGServices::stringToDouble(result.at(1).at(0))+1;

                        if (err.isSucceeded()) err=rule.setOrder(order);
                        if (err.isSucceeded()) err=rule.save();
                        if (err.isSucceeded()) err=getDocument()->stepForward ( i+1 );

                }
                QApplication::restoreOverrideCursor();
        }

        //status bar
        if (err.isSucceeded()) err=SKGError ( 0, i18n ( "Search updated" ) );
        else err.addError ( ERR_FAIL, i18n ( "Search update failed" ) );

        //Display error
        SKGMainPanel::getMainPanel()->displayErrorMessage ( err );
}

void SKGSearchPluginWidget::onDoubleClick()
{
        _SKGTRACEIN(10, "SKGSearchPluginWidget::onDoubleClick");

        //Get selection
        SKGObjectBase::SKGListSKGObjectBase selection=getSelectedObjects();
        if (selection.count()==1) {
                SKGRuleObject rule(selection.at(0));

                QString addSql;
                if (ui.kProcessType->currentIndex()==0) {
                        if (ui.kAllImportedOperationBtn->isChecked()) addSql="t_imported!='N'";
                        else if (ui.kNotValidatedBtn->isChecked()) addSql="t_imported='P'";
                }

                //Build where clause and title
                QString wc=rule.getSelectSqlOrder(addSql);
                QString title=i18n("Sub operations corresponding to rule '%1'",  rule.getSearchDescription());

                //Open
                if (QApplication::keyboardModifiers()&Qt::ControlModifier && QApplication::keyboardModifiers()&Qt::ShiftModifier) {
                        //Call debug plugin
                        QDomDocument doc("SKGML");
                        QDomElement root = doc.createElement("parameters");
                        doc.appendChild(root);
                        root.setAttribute("sqlOrder", "SELECT * from v_operation_consolidated WHERE "+wc);

                        SKGMainPanel::getMainPanel()->setNewTabContent(SKGMainPanel::getMainPanel()->getPluginByName("Skrooge debug plugin"), -1, doc.toString());
                } else {
                        //Call operation plugin
                        QDomDocument doc("SKGML");
                        doc.setContent ( getDocument()->getParameter("SKGOPERATION_CONSOLIDATED_DEFAULT_PARAMETERS") );
                        QDomElement root = doc.documentElement();
                        if (root.isNull()) {
                                root=doc.createElement("parameters");
                                doc.appendChild(root);
                        }

                        root.setAttribute("operationTable", "v_operation_consolidated");
                        root.setAttribute("operationWhereClause", wc);
                        root.setAttribute("title", title);
                        root.setAttribute("title_icon", "edit-find");
                        root.setAttribute ( "currentPage", "-1" );

                        SKGMainPanel::getMainPanel()->setNewTabContent(SKGMainPanel::getMainPanel()->getPluginByName("Skrooge operation plugin"), -1, doc.toString());
                }
        }
}

void SKGSearchPluginWidget::onEditorModified()
{
        SKGObjectBase::SKGListSKGObjectBase selection=getSelectedObjects();
        int nbSelect=selection.count();

        int mode=getCurrentMode();
        ui.kUpdate->setEnabled(nbSelect==1);
        ui.kAdd->setEnabled(mode<=1);
        ui.kQueryInfo->setText("");

        if (nbSelect==1) {
                SKGRuleObject rule=selection.at(0);
                QString addSql;
                if (ui.kAllImportedOperationBtn->isChecked()) addSql="t_imported!='N'";
                else if (ui.kNotValidatedBtn->isChecked()) addSql="t_imported='P'";

                //Build where clause and title
                QString wc=rule.getSelectSqlOrder(addSql);

                QString sqlOrder="SELECT count(1) from v_operation_consolidated WHERE "+wc;
                SKGStringListList result;
                SKGServices::executeSelectSqliteOrder(getDocument(), sqlOrder, result);
                if (result.count()==2) {
                        ui.kQueryInfo->setText(i18np("%1 operation found.", "%1 operations found.", SKGServices::stringToInt(result.at(1).at(0))));
                }
        }
}

void SKGSearchPluginWidget::cleanEditor()
{
        if (getNbSelectedObjects()==0) {
                ui.kQueryCreator->clearContents();
                ui.kActionCreator->clearContents();
        }
}

QString SKGSearchPluginWidget::getXMLActionDefinition()
{
        QString output;
        if (ui.kProcessType->currentIndex()==0) {
                output=ui.kActionCreator->getXMLCondition();
        } else {
                QDomDocument doc("SKGML");
                QDomElement element = doc.createElement ( "element" );
                doc.appendChild ( element );

                QDomElement elementLine = doc.createElement ( "element" );
                element.appendChild ( elementLine );

                QDomElement elementElement = doc.createElement ( "element" );
                elementLine.appendChild ( elementElement );

                elementElement.setAttribute ( "attribute", "f_REALCURRENTAMOUNT");
                elementElement.setAttribute ( "operator", "ABS(TOTAL(#ATT#))#OP##V1#,ABS(TOTAL(#ATT#)), #V1#, '#V2S#'");
                elementElement.setAttribute ( "operator2", ">=");
                elementElement.setAttribute ( "value", SKGServices::doubleToString(ui.kAlarmAmount->value()));
                elementElement.setAttribute ( "value2", ui.kAlarmMessage->text() );

                output=doc.toString();
        }
        return output;
}
#include "skgsearchpluginwidget.moc"


