/***************************************************************************
 *   Copyright (C) 2008 by S. MANKOWSKI / G. DE BURE skrooge@mankowski.fr  *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>  *
 ***************************************************************************/
/** @file
 * A skrooge plugin to manage scheduled operations
 *
 * @author Stephane MANKOWSKI
 */
#include "skgscheduledplugin.h"
#include "skgscheduledpluginwidget.h"
#include "skgscheduledboardwidget.h"
#include "skgscheduled_settings.h"
#include "skgrecurrentoperationobject.h"
#include "skgtraces.h"
#include "skgmainpanel.h"
#include "skgtransactionmng.h"
#include "skgoperationobject.h"
#include "skgdocumentbank.h"

#include <kactioncollection.h>
#include <kstandardaction.h>

/**
 * This plugin factory.
 */
K_PLUGIN_FACTORY ( SKGScheduledPluginFactory, registerPlugin<SKGScheduledPlugin>(); )
/**
 * This plugin export.
 */
K_EXPORT_PLUGIN ( SKGScheduledPluginFactory ( "skrooge_scheduled", "skrooge_scheduled" ) )

SKGScheduledPlugin::SKGScheduledPlugin ( QObject* iParent, const QVariantList& /*iArg*/ ) : SKGInterfacePlugin ( iParent )
{
        SKGTRACEIN ( 10, "SKGScheduledPlugin::SKGScheduledPlugin" );
}

SKGScheduledPlugin::~SKGScheduledPlugin()
{
        SKGTRACEIN ( 10, "SKGScheduledPlugin::~SKGScheduledPlugin" );
        currentBankDocument=NULL;
        scheduleOperationAction=NULL;
}

void SKGScheduledPlugin::setupActions ( SKGDocument* iDocument, const QStringList& iArgument )
{
        SKGTRACEIN ( 10, "SKGScheduledPlugin::setupActions" );
        Q_UNUSED ( iArgument );

        currentBankDocument=iDocument;


        setComponentData ( SKGScheduledPluginFactory::componentData() );
        setXMLFile ( "skrooge_scheduled.rc" );

        //Create yours actions here
        scheduleOperationAction = new KAction ( KIcon ( "skrooge_schedule" ), i18n ( "Schedule" ), this );
        connect ( scheduleOperationAction, SIGNAL ( triggered ( bool ) ), this, SLOT ( actionScheduleOperation() ) );
        actionCollection()->addAction ( QLatin1String ( "schedule_operation" ), scheduleOperationAction );
        scheduleOperationAction->setShortcut ( Qt::CTRL+Qt::Key_I );

        if (SKGMainPanel::getMainPanel()) SKGMainPanel::getMainPanel()->registedGlobalAction ( "schedule_operation", scheduleOperationAction );
}

void SKGScheduledPlugin::refresh()
{
        SKGTRACEIN ( 10, "SKGScheduledPlugin::refresh" );
        if (SKGMainPanel::getMainPanel()) {
                SKGObjectBase::SKGListSKGObjectBase selection=SKGMainPanel::getMainPanel()->getSelectedObjects();

                if ( selection.count() >0 ) {
                        bool onOperation= ( selection.at ( 0 ).getRealTable() =="operation" );
                        if ( scheduleOperationAction ) scheduleOperationAction->setEnabled ( onOperation );
                } else {
                        if ( scheduleOperationAction ) scheduleOperationAction->setEnabled ( false );
                }

                //Automatic insert
                if ( currentBankDocument && currentBankDocument->getDatabase() !=NULL ) {
                        QString doc_id=currentBankDocument->getUniqueIdentifier();
                        if ( docUniqueIdentifier!=doc_id && currentBankDocument->getParameter ( "SKG_DB_BANK_VERSION" ) >="0.5" ) {
                                docUniqueIdentifier=doc_id;

                                SKGError err;
                                //Read Setting
                                bool check_on_open = skgscheduled_settings::check_on_open();

                                if ( check_on_open ) {
                                        SKGBEGINTRANSACTION ( *currentBankDocument, i18n ( "Insert recurrent operations" ), err );

                                        QApplication::setOverrideCursor ( QCursor ( Qt::WaitCursor ) );
                                        int nbi=0;
                                        err=SKGRecurrentOperationObject::process ( currentBankDocument, nbi );
                                        QApplication::restoreOverrideCursor();
                                }
                                //Display error
                                if (SKGMainPanel::getMainPanel()) SKGMainPanel::getMainPanel()->displayErrorMessage ( err );
                        }
                }
        }
}

void SKGScheduledPlugin::close()
{
        SKGTRACEIN ( 10, "SKGScheduledPlugin::close" );
}

int SKGScheduledPlugin::getNbDashboardWidgets()
{
        return 1;
}

QString SKGScheduledPlugin::getDashboardWidgetTitle(int iIndex)
{
        if (iIndex==0) return i18n("Scheduled operations");
        return "";
}

SKGWidget* SKGScheduledPlugin::getDashboardWidget(int iIndex)
{
        if (iIndex==0) return new SKGScheduledBoardWidget(currentBankDocument);
        return NULL;
}

SKGTabPage* SKGScheduledPlugin::getWidget()
{
        SKGTRACEIN ( 10, "SKGScheduledPlugin::getWidget" );
        return new SKGScheduledPluginWidget ( ( SKGDocumentBank* ) currentBankDocument );
}

QWidget* SKGScheduledPlugin::getPreferenceWidget()
{
        SKGTRACEIN ( 10, "SKGScheduledPlugin::getPreferenceWidget" );
        QWidget* widget=new QWidget();
        ui.setupUi ( widget );
        return widget;
}

KConfigSkeleton* SKGScheduledPlugin::getPreferenceSkeleton()
{
        return skgscheduled_settings::self();
}

QString SKGScheduledPlugin::title() const
{
        return i18n ( "Scheduled operations" );
}

QString SKGScheduledPlugin::icon() const
{
        return "chronometer";
}

QString SKGScheduledPlugin::toolTip () const
{
        return i18n ( "Operations scheduled management" );
}

int SKGScheduledPlugin::getOrder() const
{
        return 20;
}

QStringList SKGScheduledPlugin::tips() const
{
        QStringList output;
        output.push_back ( i18n ( "<p>... you can schedule operations or templates.</p>" ) );
        return output;
}

bool SKGScheduledPlugin::isInContext() const
{
        return true;
}

bool SKGScheduledPlugin::isEnabled() const
{
        return true;
}

SKGError SKGScheduledPlugin::savePreferences() const
{
        SKGError err;
        if ( currentBankDocument ) {
                //Read Setting
                if (skgscheduled_settings::create_template()) {
                        SKGObjectBase::SKGListSKGObjectBase recurrents;
                        err=SKGObjectBase::getObjects(currentBankDocument, "v_recurrentoperation", "(select count(1) from operation where operation.id=rd_operation_id and t_template='N')=1", recurrents);
                        int nb=recurrents.count();
                        if (nb) {
                                SKGBEGINPROGRESSTRANSACTION ( *currentBankDocument, i18n ( "Conversion schedule" ), err, nb );
                                for ( int i=0; err.isSucceeded() && i<nb; ++i ) {
                                        //Migration of existing schedule in template mode
                                        SKGRecurrentOperationObject recOp=recurrents.at(i);
                                        SKGOperationObject operationObj;

                                        if (err.isSucceeded()) recOp.getParentOperation(operationObj);

                                        SKGOperationObject operationObjOrig=operationObj;
                                        if (err.isSucceeded()) err=operationObjOrig.duplicate(operationObj , operationObjOrig.getDate(), true);

                                        if (err.isSucceeded()) err=recOp.setParentOperation(operationObj);
                                        if (err.isSucceeded()) err=recOp.save();
                                        if (err.isSucceeded()) err=recOp.load();

                                        if (err.isSucceeded()) err=operationObjOrig.setAttribute("r_recurrentoperation_id", SKGServices::intToString(recOp.getID()));
                                        if (err.isSucceeded()) err=operationObjOrig.save();

                                        if (err.isSucceeded()) err=currentBankDocument->stepForward ( i+1 );
                                }
                                if (err.isSucceeded()) currentBankDocument->sendMessage(i18n( "All scheduled operations have been converted in template"), true);
                        }
                }
        }
        return err;
}

void SKGScheduledPlugin::actionScheduleOperation()
{
        SKGError err;
        SKGTRACEINRC ( 10, "SKGScheduledPlugin::actionScheduleOperation",err );

        //Get Selection
        if (SKGMainPanel::getMainPanel()) {
                SKGObjectBase::SKGListSKGObjectBase selection=SKGMainPanel::getMainPanel()->getSelectedObjects();
                int nb=selection.count();
                QApplication::setOverrideCursor ( QCursor ( Qt::WaitCursor ) );
                if ( nb && currentBankDocument ) {
                        SKGBEGINPROGRESSTRANSACTION ( *currentBankDocument, i18n ( "Operation schedule" ), err, nb );
                        for ( int i=0; err.isSucceeded() && i<nb; ++i ) {
                                SKGOperationObject operationObj=selection.at ( i );
                                bool isTemplate=operationObj.isTemplate();

                                SKGOperationObject operationObjOrig;
                                if (!isTemplate && skgscheduled_settings::create_template()) {
                                        //The selected operation is not a template and settings is set to create one
                                        operationObjOrig=operationObj;
                                        if (err.isSucceeded()) err=operationObjOrig.duplicate(operationObj , operationObjOrig.getDate(), true);
                                        if (err.isSucceeded()) currentBankDocument->sendMessage(i18n( "A template has been created"), true);
                                }

                                SKGRecurrentOperationObject recOp;
                                err=operationObj.addRecurrentOperation ( recOp );
                                if (err.isSucceeded()) err=recOp.warnEnabled(skgscheduled_settings::remind_me());
                                if (err.isSucceeded()) err=recOp.setWarnDays(skgscheduled_settings::remind_me_days());
                                if (err.isSucceeded()) err=recOp.autoWriteEnabled(skgscheduled_settings::auto_write());
                                if (err.isSucceeded()) err=recOp.setAutoWriteDays(skgscheduled_settings::auto_write_days());
                                if (err.isSucceeded()) err=recOp.timeLimit(skgscheduled_settings::nb_times_val());
                                if (err.isSucceeded()) err=recOp.setTimeLimit(skgscheduled_settings::nb_times());
                                if (err.isSucceeded()) err=recOp.setPeriodIncrement(skgscheduled_settings::once_every());
                                if (err.isSucceeded()) err=recOp.setPeriodUnit((SKGRecurrentOperationObject::PeriodUnit) SKGServices::stringToInt(skgscheduled_settings::once_every_unit()));
                                if ( err.isSucceeded() && !isTemplate ) err=recOp.setDate(recOp.getNextDate());
                                if (err.isSucceeded()) err=recOp.save();
                                if (!isTemplate && skgscheduled_settings::create_template()) {
                                        if (err.isSucceeded()) err=recOp.load();
                                        if (err.isSucceeded()) err=operationObjOrig.setAttribute("r_recurrentoperation_id", SKGServices::intToString(recOp.getID()));
                                        if (err.isSucceeded()) err=operationObjOrig.save();
                                }
                                if (err.isSucceeded()) err=currentBankDocument->stepForward ( i+1 );
                        }
                }
                QApplication::restoreOverrideCursor();

                //status bar
                if (err.isSucceeded())  err=SKGError ( 0, i18n ( "Operation scheduled." ) );
                else err.addError ( ERR_FAIL, i18n ( "Operation schedule failed" ) );

                //Display error
                SKGMainPanel::getMainPanel()->displayErrorMessage ( err );
        }
}
#include "skgscheduledplugin.moc"
