/***************************************************************************
 *   Copyright (C) 2008 by S. MANKOWSKI / G. DE BURE skrooge@mankowski.fr  *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>  *
 ***************************************************************************/
/** @file
 * This file is Skrooge plugin for operation management.
 *
 * @author Stephane MANKOWSKI / Guillaume DE BURE
 */
#include "skgoperationpluginwidget.h"
#include "skgsortfilterproxymodel.h"
#include "skgmainpanel.h"
#include "skgobjectmodel.h"
#include "skgobjectbase.h"
#include "skgbankincludes.h"
#include "skgtraces.h"
#include "skgservices.h"
#include "skgsplittabledelegate.h"
#include "skgoperation_settings.h"

#include <kstandarddirs.h>
#include <kstandardaction.h>
#include <kmessagebox.h>
#include <kinputdialog.h>
#include <skgcalculatoredit.h>

#include <QtGui/QHeaderView>
#include <QDomDocument>
#include <QDir>
#include <QMap>
#include <QKeyEvent>
#include <QtCore/qfile.h>

/**
 * @def NOUPDATE
 * For a not modified field
 */
#define NOUPDATE "-------"

SKGOperationPluginWidget::SKGOperationPluginWidget ( SKGDocumentBank* iDocument )
        : SKGTabPage ( iDocument ), lastFastEditionOperationFound ( 0 ), showClosedAccounts ( false ),
        numberFieldIsNotUptodate ( true ),modeInfoZone ( 0 )
{
    SKGTRACEIN ( 1, "SKGOperationPluginWidget::SKGOperationPluginWidget" );

    ui.setupUi ( this );

    ui.kTitle->hide();
    ui.kResetInternalFilter->hide();
    ui.kReconciliatorFrame2->hide();

    QStringList listAtts;
    listAtts << "t_category" << "t_comment" << "f_value" << "t_refund";
    int nb=listAtts.count();
    ui.kSubOperationsTable->setColumnCount ( nb );
    for ( int i=0; i<nb; ++i )
    {
        QString att=listAtts.at ( i );
        QTableWidgetItem* item=new QTableWidgetItem ( iDocument->getIcon ( att ), iDocument->getDisplay ( att ) );
        ui.kSubOperationsTable->setHorizontalHeaderItem ( i, item );
    }

    {
        SKGTRACEIN ( 10, "SKGOperationPluginWidget::SKGOperationPluginWidget-model creation" );
        //Bind operation view
        objectModel = new SKGObjectModel ( ( SKGDocumentBank* ) getDocument(), "v_operation_display", "1=0", this, "", false );
        SKGSortFilterProxyModel* modelproxy = new SKGSortFilterProxyModel ( this );
        modelproxy->setSourceModel ( objectModel );
        modelproxy->setSortRole ( Qt::UserRole );
        modelproxy->setDynamicSortFilter ( true );

        ui.kFilterEdit->setProxy ( modelproxy );

        ui.kOperationView->setModel ( modelproxy );
        ui.kOperationView->setWindowTitle ( i18nc ( "Noun, financial operations such as paying your groceries","Operations" ) );

        //Add registered global action in contextual menu

        if ( SKGMainPanel::getMainPanel() )
        {
            fastEditionAction=SKGMainPanel::getMainPanel()->getGlobalAction ( "fast_edition" );

            ui.kOperationView->insertAction ( 0, SKGMainPanel::getMainPanel()->getGlobalAction ( "edit_delete" ) );
            {
                QAction* sep=new QAction ( this );
                sep->setSeparator ( true );
                ui.kOperationView->insertAction ( 0, sep );
            }
            ui.kOperationView->insertAction ( 0, SKGMainPanel::getMainPanel()->getGlobalAction ( "edit_switch_highlight" ) );
            ui.kOperationView->insertAction ( 0, SKGMainPanel::getMainPanel()->getGlobalAction ( "edit_point_selected_operation" ) );
            ui.kOperationView->insertAction ( 0, SKGMainPanel::getMainPanel()->getGlobalAction ( "edit_group_operation" ) );
            ui.kOperationView->insertAction ( 0, SKGMainPanel::getMainPanel()->getGlobalAction ( "edit_ungroup_operation" ) );
            ui.kOperationView->insertAction ( 0, SKGMainPanel::getMainPanel()->getGlobalAction ( "validate_imported_operation" ) );
            ui.kOperationView->insertAction ( 0, SKGMainPanel::getMainPanel()->getGlobalAction ( "merge_imported_operation" ) );
            {
                QAction* sep=new QAction ( this );
                sep->setSeparator ( true );
                ui.kOperationView->insertAction ( 0, sep );
            }
            ui.kOperationView->insertAction ( 0, SKGMainPanel::getMainPanel()->getGlobalAction ( "edit_duplicate_operation" ) );
            ui.kOperationView->insertAction ( 0, SKGMainPanel::getMainPanel()->getGlobalAction ( "edit_template_operation" ) );
            ui.kOperationView->insertAction ( 0, SKGMainPanel::getMainPanel()->getGlobalAction ( "schedule_operation" ) );
            {
                QAction* sep=new QAction ( this );
                sep->setSeparator ( true );
                ui.kOperationView->insertAction ( 0, sep );
            }
            ui.kOperationView->insertAction ( 0, SKGMainPanel::getMainPanel()->getGlobalAction ( "open_report" ) );
        }

        connect ( ui.kOperationView, SIGNAL ( selectionChangedDelayed() ), this, SLOT ( onSelectionChanged() ) );
        connect ( objectModel, SIGNAL ( beforeReset() ), ui.kOperationView, SLOT ( saveSelection() ) );
        connect ( objectModel, SIGNAL ( afterReset() ), ui.kOperationView, SLOT ( resetSelection() ) );
    }

    //Add Standard KDE Icons to buttons to Operations
    ui.kModifyOperationBtn->setIcon ( KIcon ( "dialog-ok-apply" ) );
    ui.kAddOperationBtn->setIcon ( KIcon ( "list-add" ) );
    ui.kCleanBtn->setIcon ( KIcon ( "edit-clear" ) );
    ui.kResetInternalFilter->setIcon ( KIcon ( "dialog-cancel" ) );
    ui.kRemoveSubOperation->setIcon ( KIcon ( "edit-delete" ) );
    ui.kReconciliatorButton->setIcon ( KIcon ( "object-rotate-left" ) );
    ui.kValidate->setIcon ( KIcon ( "dialog-ok-apply" ) );

    ui.kStandardBtn->setIcon ( KIcon ( "dialog-ok-apply" ) );
    ui.kTransferBtn->setIcon ( KIcon ( "skrooge_transfer" ) );
    ui.kSplitBtn->setIcon ( KIcon ( "skrooge_split" ) );
    ui.kShareBtn->setIcon ( KIcon ( "skrooge_unit" ) );

    ui.kOperationBtn->setIcon ( KIcon ( "view-pim-tasks" ) );

    QStringList overlay;
    overlay.push_back ( "skrooge_template" );
    ui.kTemplateBtn->setIcon ( KIcon ( "view-pim-tasks", NULL, overlay ) );

    //Refresh
    connect ( ui.kHideUselessOperation, SIGNAL ( stateChanged ( int ) ), this, SLOT ( onFilterChanged() ), Qt::QueuedConnection );

    //Fast edition
    connect ( qApp, SIGNAL ( focusChanged ( QWidget*, QWidget* ) ), this, SLOT ( onFocusChanged() ) );
    connect ( fastEditionAction, SIGNAL ( triggered ( bool ) ), this, SLOT ( onFastEdition() ) );

    // SubOperations
    connect ( ui.kAmountEdit,SIGNAL ( textChanged ( const QString& ) ),this,SLOT ( onQuantityChanged() ) );
    connect ( ui.kSubOperationsTable,SIGNAL ( cellChanged ( int,int ) ),this,SLOT ( onSubopCellChanged ( int,int ) ) );
    connect ( ui.kRemoveSubOperation,SIGNAL ( clicked ( bool ) ),this,SLOT ( onRemoveSubOperation() ) );

    connect ( (SKGTabWidget*) SKGMainPanel::getMainPanel()->centralWidget(), SIGNAL ( currentChanged ( int ) ), this, SLOT ( refreshDelShortcut() ) );


    ui.kSubOperationsTable->verticalHeader()->setDefaultSectionSize ( ui.kSubOperationsTable->verticalHeader()->minimumSectionSize() );
    ui.kSubOperationsTable->horizontalHeader()->setResizeMode ( QHeaderView::Interactive );
    ui.kSubOperationsTable->setWordWrap ( false );
    ui.kSubOperationsTable->setItemDelegate ( new SKGSplitTableDelegate ( ui.kSubOperationsTable, getDocument() ) );

    ui.kTransferFrm->hide();
    ui.kSplitFrm->hide();
    ui.kShareFrm->hide();

    setCurrentMode ( 0 );
    ui.kOperationBtn->setChecked ( true );
    modeTemplate=false;

    //Set Event filters for locking widgets
    ui.kTypeEdit->lineEdit()->installEventFilter ( this );
    ui.kTypeEdit->installEventFilter ( this );
    ui.kUnitEdit->lineEdit()->installEventFilter ( this );
    ui.kUnitEdit->installEventFilter ( this );
    ui.kCategoryEdit->lineEdit()->installEventFilter ( this );
    ui.kCategoryEdit->installEventFilter ( this );
    ui.kCommentEdit->installEventFilter ( this );
    ui.kPayeeEdit->lineEdit()->installEventFilter ( this );
    ui.kPayeeEdit->installEventFilter ( this );
    ui.kTrackerEdit->lineEdit()->installEventFilter ( this );
    ui.kTrackerEdit->installEventFilter ( this );
    ui.kAccountEdit->installEventFilter ( this );
    ui.kAmountEdit->installEventFilter ( this );
    ui.kNumberEdit->installEventFilter ( this );

    frozen_logo = KStandardDirs::locate("data",QString::fromLatin1("skrooge/images/operations/skrooge_frozen.png"));

    connect ( ( const QObject* ) getDocument(), SIGNAL(tableModified(QString, int)), this, SLOT(dataModified(QString, int)), Qt::QueuedConnection );

    fillComboBoxes();
    onOperationCreatorModified();
}

SKGOperationPluginWidget::~SKGOperationPluginWidget()
{
    SKGTRACEIN ( 1, "SKGOperationPluginWidget::~SKGOperationPluginWidget" );
    objectModel=NULL;
    fastEditionAction=NULL;

    //To be sure that delete shortcut is restored
    mode=0;
    refreshDelShortcut();
}

bool SKGOperationPluginWidget::isWidgetEditionEnabled(QWidget* iWidget)
{
    return !iWidget->property("frozen").toBool();
}

void SKGOperationPluginWidget::setWidgetEditionEnabled(QWidget* iWidget, bool iEnabled)
{
  if(iWidget && isWidgetEditionEnabled(iWidget)!=iEnabled)
  {
    KLineEdit* line=dynamic_cast<KLineEdit*>(iWidget);
    if(iEnabled)
    {
        iWidget->setStyleSheet("");
        iWidget->setProperty("frozen",false);
	if(line && iWidget->property("clearButtonShown").toBool()==true) line->setClearButtonShown(true);
    }
    else
    {
        QString align="right";
	if(line)
	{
	  if(line->alignment()&Qt::AlignRight) align="left";

	  if(line->isClearButtonShown())
	  {
	    iWidget->setProperty("clearButtonShown",true);
	    line->setClearButtonShown(false);
	  }
	}
        iWidget->setStyleSheet("background-image:url("+frozen_logo+");background-repeat:no-repeat;background-clip: padding; background-position: top "+align+"; background-origin: content;");
        iWidget->setProperty("frozen",true);
    }

    QString addOn=i18n("This field is frozen (it will not be affected by Fast Edition). Double click to unfreeze it");
    QString t=iWidget->toolTip().remove('\n'+addOn).remove(addOn);
    if(!iEnabled)
    {
        t=iWidget->toolTip();
        if(!t.isEmpty()) t+='\n';
        t+=addOn;
    }
    iWidget->setToolTip(t);
  }
}

bool SKGOperationPluginWidget::eventFilter ( QObject *object, QEvent *event )
{
    if ( event->type() == QEvent::MouseButtonDblClick )
    {
      KLineEdit* line=dynamic_cast<KLineEdit*>(object);
      if(line)
      {
	setWidgetEditionEnabled(line, !isWidgetEditionEnabled(line));
      }
    }
    else if ( event->type() == QEvent::FocusIn )
    {
      KLineEdit* line=dynamic_cast<KLineEdit*>(object);
      if(line) previousValue=line->text();
      else
      {
	SKGComboBox* cmb=dynamic_cast<SKGComboBox*>(object);
	if(cmb) previousValue=cmb->text();
      }
    }
    else if ( event->type() == QEvent::FocusOut )
    {
      KLineEdit* line=dynamic_cast<KLineEdit*>(object);
      if(line)
      {
	if(previousValue!=line->text()) setWidgetEditionEnabled(line, false);
      }
      else
      {
	SKGComboBox* cmb=dynamic_cast<SKGComboBox*>(object);
	if(cmb)
	{
	  if(previousValue!=cmb->text()) setWidgetEditionEnabled(cmb->lineEdit(), false);
	}
      }
    }

    return false;
}

void SKGOperationPluginWidget::setAllWidgetsEnabled()
{
    //Enable widgets
    setWidgetEditionEnabled(ui.kTypeEdit->lineEdit(), true);
    setWidgetEditionEnabled(ui.kUnitEdit->lineEdit(), true);
    setWidgetEditionEnabled(ui.kCategoryEdit->lineEdit(), true);
    setWidgetEditionEnabled(ui.kCommentEdit, true);
    setWidgetEditionEnabled(ui.kPayeeEdit->lineEdit(), true);
    setWidgetEditionEnabled(ui.kTrackerEdit->lineEdit(), true);
    setWidgetEditionEnabled(ui.kAccountEdit, true);
    setWidgetEditionEnabled(ui.kAmountEdit, true);
    setWidgetEditionEnabled(ui.kNumberEdit, true);
}

void SKGOperationPluginWidget::onSelectionChanged()
{
    SKGTRACEIN ( 10, "SKGOperationPluginWidget::onSelectionChanged" );

    int mode=getCurrentMode();

    //Enable widgets
    setAllWidgetsEnabled();

    //Mapping
    QItemSelectionModel *selModel=ui.kOperationView->selectionModel();
    if ( selModel )
    {
        QModelIndexList indexes=selModel->selectedRows();
        int nbSelect=indexes.count();
        bool onConsolidatedTable=false;
        if ( nbSelect && objectModel )
        {
            QModelIndex idx=indexes[indexes.count()-1];

            QSortFilterProxyModel* proxyModel= ( QSortFilterProxyModel* ) ui.kOperationView->model();
            QModelIndex idxs=proxyModel->mapToSource ( idx );

            SKGOperationObject obj ( objectModel->getObject ( idxs ) );
            onConsolidatedTable= ( obj.getTable() =="v_operation_consolidated" );

            ui.kDateEdit->setDate ( SKGServices::stringToTime ( obj.getAttribute ( "d_date" ) ).date() );
            ui.kCommentEdit->setText ( obj.getAttribute ( obj.getTable() =="v_operation_consolidated" ? "t_REALCOMMENT" : "t_comment" ) );
            QString number=obj.getAttribute ( "i_number" );
            if ( number=="0" ) number="";
            ui.kNumberEdit->setText ( number );
            ui.kAccountEdit->setText ( obj.getAttribute ( "t_ACCOUNT" ) );
            ui.kPayeeEdit->setText ( obj.getAttribute ( "t_payee" ) );
            ui.kTypeEdit->setText ( obj.getAttribute ( "t_mode" ) );
            ui.kUnitEdit->setText ( obj.getAttribute ( "t_UNIT" ) );
            QString cat=obj.getAttribute ( "t_REALCATEGORY" );
            if ( cat.isEmpty() ) cat=obj.getAttribute ( "t_CATEGORY" );
            ui.kCategoryEdit->setText ( cat );
            ui.kTrackerEdit->setText ( obj.getAttribute ( "t_REFUND" ) );
            QString quantity=obj.getAttribute ( "f_REALQUANTITY" );
            if ( quantity.isEmpty() ) quantity=obj.getAttribute ( "f_QUANTITY" );
            if ( SKGServices::stringToDouble ( quantity ) >0 ) quantity='+'+quantity;
            ui.kAmountEdit->setText ( quantity );

            if ( nbSelect>1 )
            {
	        //In case of multi selection
                if ( mode>=0 ) setCurrentMode ( 0 );
                ui.kAccountEdit->setText ( NOUPDATE );
                ui.kTypeEdit->setText ( NOUPDATE );
                ui.kUnitEdit->setText ( NOUPDATE );
                ui.kCategoryEdit->setText ( NOUPDATE );
                ui.kTrackerEdit->setText ( NOUPDATE );
                ui.kCommentEdit->setText ( NOUPDATE );
                ui.kPayeeEdit->setText ( NOUPDATE );
            }
            else
            {
	        //It is a single selection
	        //Is it a split ?
                int nbSubOperations = obj.getNbSubOperations();
                if ( nbSubOperations > 1 && !onConsolidatedTable )
                {
		    //yes, it is a split
                    if ( mode>=0 ) setCurrentMode ( 1 );;

                    displaySubOperations();
                }
                else
                {
		    //Is it a transfer ?
		    SKGOperationObject op2;
		    if(obj.isTransfer(op2))
		    {
		      //yes it is a transfer
		      SKGAccountObject account2;
		      op2.getParentAccount(account2);
		      ui.kTargetAccountEdit->setText(account2.getName ());
		      if ( mode>=0 ) setCurrentMode ( 2 );
		    }
		    else
		    {
		      if ( mode>=0 ) setCurrentMode ( 0 );
		    }
                }
            }
        }

        ui.kDateEdit->setEnabled ( nbSelect<=1 );
        ui.kAmountEdit->setEnabled ( nbSelect<=1 );
        ui.kNumberEdit->setEnabled ( nbSelect<=1 );

        bool splitTest=nbSelect<=1 && !onConsolidatedTable;
        ui.kSplitBtn->setEnabled ( splitTest );
        if ( !splitTest && mode==1 ) setCurrentMode ( 0 );

        onOperationCreatorModified();

        emit selectionChanged();
    }

    if ( modeInfoZone==2 ) onRefreshInformationZone();
}

void SKGOperationPluginWidget::onOperationCreatorModified()
{
    SKGTRACEIN ( 10, "SKGOperationPluginWidget::onOperationCreatorModified" );

    int mode=getCurrentMode();

    //Is it an existing unit ?
    QString unitName=ui.kUnitEdit->currentText();
    SKGUnitObject unit ( getDocument() );
    unit.setName ( unitName );
    unit.setSymbol ( unitName );
    if ( unit.load().isSucceeded() )
    {
        ui.kShareBtn->setEnabled ( true );
        if ( mode==3 && unit.getType() ==SKGUnitObject::SHARE )
        {
            //Update units
            SKGServices::SKGUnitInfo unitOfUnitName= ( ( SKGDocumentBank* ) getDocument() )->getPrimaryUnit();
            SKGUnitObject unitOfUnit;
            unit.getUnit ( unitOfUnit );
            if ( unitOfUnit.exist() )
            {
                unitOfUnitName.Name=unitOfUnit.getSymbol();
                unitOfUnitName.NbDecimal=unitOfUnit.getNumberDecimal();
            }
            ui.kUnitShare->setText ( unitOfUnitName.Name );
            ui.kUnitCommission->setText ( unitOfUnitName.Name );
            ui.kUnitTax->setText ( unitOfUnitName.Name );

            //Update total in "purchase / sale share" page
            double total=ui.kAmountSharesEdit->value() + ( ui.kCommissionEdit->value() +ui.kTaxEdit->value() ) * ( ui.kAmountEdit->value() >0 ? 1 : -1 );
            ui.KTotal->setText ( KGlobal::locale()->formatMoney ( total, unitOfUnitName.Name, unitOfUnitName.NbDecimal ) );
        }
        else
        {
            //BUG 2692665
            ui.kUnitShare->setText ( unitName );
            ui.kUnitCommission->setText ( unitName );
            ui.kUnitTax->setText ( unitName );
        }

    }
    else
    {
        ui.kShareBtn->setEnabled ( false );
        if ( mode==3 ) setCurrentMode ( 0 );
    }

    bool activated=ui.kAccountEdit->currentText().length() >0 &&
                   ui.kAmountEdit->text().length() >0 &&
                   unitName.length() >0 &&
                   ( mode!=3 || ui.kAmountSharesEdit->text().length() >0 );

    int nbSelect=getNbSelectedObjects();

    ui.kAddOperationBtn->setEnabled ( activated );
    ui.kModifyOperationBtn->setEnabled ( activated && nbSelect>0 && ( getCurrentMode() ==0 ||  getCurrentMode() ==1 ||  getCurrentMode() ==2 ) );

    QList<QTableWidgetItem*> selectedItems = ui.kSubOperationsTable->selectedItems();
    ui.kRemoveSubOperation->setEnabled ( selectedItems.count() >0 );

    refreshDelShortcut();
}

void SKGOperationPluginWidget::refreshDelShortcut()
{
    //Get delete action
    QAction* delAction=SKGMainPanel::getMainPanel()->getGlobalAction ( "edit_delete" );
    if(delAction)
    {
      //Save previous short cut
      if(deleteShortcut.isEmpty()) deleteShortcut=delAction->shortcut ();

      //Set new short cut
      delAction->setShortcut(SKGMainPanel::getMainPanel()->currentTab()==this && getCurrentMode()==1 && ui.kRemoveSubOperation->isEnabled() ? QKeySequence () : deleteShortcut);
    }

}

void SKGOperationPluginWidget::onUpdateOperationClicked()
{
    SKGError err;
    SKGTRACEINRC ( 10, "SKGOperationPluginWidget::onUpdateOperationClicked",err );
    bool confirmed=true;
    //Get Selection
    SKGObjectBase::SKGListSKGObjectBase selection=getSelectedObjects();

    int nb=selection.count();
    {
        SKGBEGINPROGRESSTRANSACTION ( *getDocument(), i18n ( "Operation update" ), err, nb );

        QApplication::setOverrideCursor ( QCursor ( Qt::WaitCursor ) );
        err=updateSelection ( selection );
        QApplication::restoreOverrideCursor();
    }

    //status bar
    if ( err.isSucceeded() )
    {
        if ( confirmed ) err=SKGError ( 0, i18n ( "Operation updated" ) );
        else err=SKGError ( 0, i18n ( "Operation canceled" ) );
    }
    else err.addError ( ERR_FAIL, i18n ( "Operation update failed" ) );

    //Display error
    SKGMainPanel::getMainPanel()->displayErrorMessage ( err );
}

SKGError SKGOperationPluginWidget::updateSelection ( const SKGObjectBase::SKGListSKGObjectBase& iSelection, bool forceCreation )
{
    SKGError err;
    SKGTRACEINRC ( 10, "SKGOperationPluginWidget::updateSelection",err );

    //Get Selection
    SKGObjectBase::SKGListSKGObjectBase selection=getSelectedObjects();
    int nb=iSelection.count();

    for ( int i=0; err.isSucceeded() && i<nb; ++i )
    {
        SKGObjectBase obj=iSelection.at ( i );
        SKGOperationObject operationObj=SKGOperationObject ( obj.getDocument(), obj.getID() );

	SKGObjectBase::SKGListSKGObjectBase gops;
	if ( err.isSucceeded() ) err=operationObj.getGroupedOperations(gops);
	if ( gops.count()==2 &&  getCurrentMode()<2 )
	{
	  getDocument()->sendMessage(i18n("You modified one part of a transfer"));
	}

	//Update operation if single selection
        if ( getCurrentMode() ==0 )
        {
            //Get subop
            SKGSubOperationObject subOp;
            int nbSubop=0;
            if ( obj.getTable() =="v_operation_consolidated" )
            {
                //It's a sub operation
                subOp=SKGSubOperationObject ( obj.getDocument(), SKGServices::stringToInt ( obj.getAttribute ( "i_SUBOPID" ) ) );
                nbSubop=1;
            }
            else
            {
                //It's a real operation, we take the first one
                SKGObjectBase::SKGListSKGObjectBase subOps;
                if ( err.isSucceeded() ) err=operationObj.getSubOperations ( subOps );
                nbSubop=subOps.count();
                if ( nbSubop ) subOp=subOps[0];
            }

            QString trackerName=ui.kTrackerEdit->text();
            if ( err.isSucceeded() && trackerName !=NOUPDATE )
            {
                if ( trackerName.isEmpty() ) err=subOp.removeTracker();
                else
                {
                    SKGTrackerObject tracker;
                    err=SKGTrackerObject::createTracker ( ( SKGDocumentBank* ) getDocument(), trackerName, tracker, true );
                    if ( err.isSucceeded() ) err=subOp.setTracker ( tracker );
                }
            }

            SKGCategoryObject cat;
            if ( err.isSucceeded() && ui.kCategoryEdit->text() !=NOUPDATE )
            {
                if ( err.isSucceeded() && nbSubop>1 )
                {
                    err=SKGError ( 25, i18n ( "Cannot update a split operation" ) );
                }
                else
                {
                    err=SKGCategoryObject::createPathCategory ( ( SKGDocumentBank* ) getDocument(), ui.kCategoryEdit->text(), cat, true );
                    if ( err.isSucceeded() ) err=subOp.setCategory ( cat );
                }
            }
            else
            {
                //Get current category to be able to
                subOp.getCategory ( cat );
            }

            if ( err.isSucceeded() && nb==1 )
            {
                if ( getCurrentMode() ==0 && nbSubop>1 )
                {
                    err=SKGError ( 25, i18n ( "Cannot update a split operation" ) );
                }
                else
                {
                    double val=ui.kAmountEdit->value();

                    //Is the sign forced ?
                    if ( ui.kAmountEdit->sign() ==0 )
                    {
		        //No
                        SKGObjectBase cat2 ( cat.getDocument(), "v_category_display", cat.getID() );

                        //Are we able to find to sign with the category ?
                        if ( cat2.getAttribute ( "t_TYPEEXPENSE" ) == "-" ) val=-val;
                    }
                    err=subOp.setQuantity ( val );
                }
            }

            if ( err.isSucceeded() && ui.kCommentEdit->text() !=NOUPDATE )
            {
                if ( obj.getTable() =="v_operation_consolidated" ) err=subOp.setComment ( ui.kCommentEdit->text() );
            }
            if ( err.isSucceeded() ) err=subOp.save();
        }
        else if ( getCurrentMode() ==1 )
        {
            int nbsubop=ui.kSubOperationsTable->rowCount();
            QList<int> listIdSubOp;
            for ( int j=0; err.isSucceeded() && j<nbsubop; ++j )
            {
                //Get values
                QTableWidgetItem* item=ui.kSubOperationsTable->item ( j,0 );
                int id= ( forceCreation ? 0 : item->data ( Qt::UserRole ).toInt() );
                QString catName=item->text();
                item=ui.kSubOperationsTable->item ( j,1 );
                QString comment=item->text();
                item=ui.kSubOperationsTable->item ( j,2 );
                double val=SKGServices::stringToDouble ( item->text() );
                item=ui.kSubOperationsTable->item ( j,3 );
                QString trackerName=item->text();

                SKGSubOperationObject subOperation;
                if ( id )
                {
                    //Update existing sub op
                    subOperation=SKGSubOperationObject ( ( SKGDocumentBank* ) getDocument(), id );
                }
                else
                {
                    //Create new sub op
                    err=operationObj.addSubOperation ( subOperation );
                }

                //Create sub operation object
                SKGCategoryObject cat;
                if ( err.isSucceeded() && !catName.isEmpty() )
                {
                    err=SKGCategoryObject::createPathCategory ( ( SKGDocumentBank* ) getDocument(), catName, cat, true );
                    if ( err.isSucceeded() ) err=subOperation.setCategory ( cat );
                }
                if ( err.isSucceeded() )
                {
                    if ( trackerName.isEmpty() ) err=subOperation.removeTracker();
                    else
                    {
                        SKGTrackerObject tracker;
                        err=SKGTrackerObject::createTracker ( ( SKGDocumentBank* ) getDocument(), trackerName, tracker, true );
                        if ( err.isSucceeded() ) err=subOperation.setTracker ( tracker );
                    }
                }
                if ( err.isSucceeded() ) err=subOperation.setComment ( comment );
                if ( err.isSucceeded() ) err=subOperation.setQuantity ( val );
                if ( err.isSucceeded() ) err=subOperation.save();

                //The sub operation created or updated mustn't be removed
                listIdSubOp.push_back ( subOperation.getID() );
            }

            //Remove useless subop
            if ( err.isSucceeded() )
            {
                SKGObjectBase::SKGListSKGObjectBase subOps;
                err=operationObj.getSubOperations ( subOps );
                int nbsubop=subOps.count();
                for ( int j=0; err.isSucceeded() && j<nbsubop; ++j )
                {
                    SKGObjectBase sop=subOps.at ( j );
                    if ( !listIdSubOp.contains ( sop.getID() ) )
                    {
                        err=sop.remove ( false );
                    }
                }
            }
        }
	else if ( getCurrentMode() ==2 )
        {
	    //Create sub operation object
            double operationQuantity=ui.kAmountEdit->value();

	    if(ui.kAmountEdit->sign()!=0) operationQuantity=-operationQuantity;
	    else
	    {
	      //Bug 192832 – Do not take amount sign into account vvv
	      if ( abs ( operationQuantity ) !=operationQuantity )
	      {
		  operationQuantity=-operationQuantity;
		  err=getDocument()->sendMessage ( i18n ( "Absolute value has been used for transfer creation." ) );
	      }
	      //Bug 192832 – Do not take amount sign into account ^^^
	    }

            SKGSubOperationObject subOperation;
	    SKGObjectBase::SKGListSKGObjectBase subOps;
            if ( err.isSucceeded() ) err=operationObj.getSubOperations ( subOps );
            if ( err.isSucceeded() )
	    {
	      subOperation=subOps.at(0);
	      err=subOperation.setQuantity ( -operationQuantity );
	    }
            if ( err.isSucceeded() ) err=subOperation.save();

            //Get account
            SKGAccountObject accountObj2 ( getDocument() );
            if ( err.isSucceeded() ) err=accountObj2.setName ( ui.kTargetAccountEdit->currentText() );
            if ( err.isSucceeded() ) err=accountObj2.load();

            //Check unit of target account
            SKGUnitObject unit;
            if ( err.isSucceeded() ) err=operationObj.getUnit(unit);

	    //Correction bug 2299303 vvv
            SKGUnitObject unitTargetAccount;
            if ( err.isSucceeded() ) err=accountObj2.getUnit ( unitTargetAccount );
            if ( err.isSucceeded() && unitTargetAccount.exist() && unit!=unitTargetAccount )
            {
                //The unit of the operation is not compliant with the unit of the target account
                //We ask to the user if he wants to continue or convert into the target account
                bool ok=false;
                QApplication::setOverrideCursor ( QCursor ( Qt::ArrowCursor ) );
                double newval=KInputDialog::getDouble ( i18n ( "Confirmation" ),
                                                        i18n ( "The operation's unit is not compatible with the target account.\n"
                                                               "Click Cancel if you want to continue anyway; "
                                                               "otherwise, enter the value in the target account's unit (%1):", unitTargetAccount.getSymbol() ),
                                                        SKGUnitObject::convert ( operationQuantity, unit, unitTargetAccount ),
                                                        -DBL_MAX, DBL_MAX, 2, &ok, this );
                QApplication::restoreOverrideCursor();
                if ( ok )
                {
                    operationQuantity=newval;
                    unit=unitTargetAccount;
                }

            }
            //Correction bug 2299303 ^^^

            //create transferred operation
            SKGOperationObject operation2;

            if ( err.isSucceeded() )
	    {
	      if ( gops.count()==2 ) operation2=(obj==SKGOperationObject(gops.at(0)) ? gops.at(1) : gops.at(0));
	      else err=accountObj2.addOperation ( operation2 );
	    }
            if ( err.isSucceeded() ) err=operation2.setMode ( ui.kTypeEdit->currentText() );
            if ( err.isSucceeded() ) err=operation2.setPayee ( ui.kPayeeEdit->currentText() );
            if ( err.isSucceeded() ) err=operation2.setNumber ( SKGServices::stringToInt ( ui.kNumberEdit->text() ) );
            if ( err.isSucceeded() ) err=operation2.setComment ( ui.kCommentEdit->text() );
            if ( err.isSucceeded() ) err=operation2.setDate ( ui.kDateEdit->date() );
            if ( err.isSucceeded() ) err=operation2.setUnit ( unit );
            if ( err.isSucceeded() ) err=operation2.setGroupOperation ( operationObj );
            if ( err.isSucceeded() ) err=operationObj.load ();
            if ( err.isSucceeded() ) err=operation2.setTemplate ( isTemplateMode() );
            if ( err.isSucceeded() ) err=operation2.save();

            //Create sub operation object
            SKGSubOperationObject subOperation2;

	    SKGObjectBase::SKGListSKGObjectBase subops;
	    if ( err.isSucceeded() ) err=operation2.getSubOperations(subops);
            if ( err.isSucceeded() )
	    {
	      if ( subops.count() ) subOperation2=subops.at(0);
	      else err=operation2.addSubOperation ( subOperation2 );
	    }
            if ( err.isSucceeded() ) err=subOperation2.setQuantity ( operationQuantity );
            if ( err.isSucceeded() ) err=subOperation2.save();

	}

	if ( err.isSucceeded() ) err=operationObj.setTemplate ( isTemplateMode() );

        if ( nb==1 )
        {
            if ( err.isSucceeded() ) err=operationObj.setNumber ( SKGServices::stringToInt ( ui.kNumberEdit->text() ) );
            if ( err.isSucceeded() ) err=operationObj.setDate ( ui.kDateEdit->date() );
        }
        if ( err.isSucceeded() && ui.kCommentEdit->text() !=NOUPDATE )
        {
            if ( obj.getTable() !="v_operation_consolidated" )  err=operationObj.setComment ( ui.kCommentEdit->text() );
        }

        if ( err.isSucceeded() && ui.kAccountEdit->text() !=NOUPDATE )
        {
            SKGAccountObject account ( getDocument() );
            err=account.setName ( ui.kAccountEdit->text() );
            if ( err.isSucceeded() ) err=account.load();
            if ( err.isSucceeded() ) err=operationObj.setParentAccount ( account );
        }
        if ( err.isSucceeded() && ui.kTypeEdit->text() !=NOUPDATE ) err=operationObj.setMode ( ui.kTypeEdit->text() );
        if ( err.isSucceeded() && ui.kPayeeEdit->text() !=NOUPDATE ) err=operationObj.setPayee ( ui.kPayeeEdit->text() );
        if ( err.isSucceeded() && ui.kUnitEdit->text() !=NOUPDATE )
        {
            SKGUnitObject unit ( getDocument() );
            err=unit.setSymbol ( ui.kUnitEdit->text() );
            if ( !unit.exist() )
            {
                if ( err.isSucceeded() ) err= unit.setName ( ui.kUnitEdit->text() );
                if ( err.isSucceeded() ) err = unit.save();

                SKGUnitValueObject unitVal;
                if ( err.isSucceeded() ) err=unit.addUnitValue ( unitVal );
                if ( err.isSucceeded() ) err=unitVal.setDate ( QDate::currentDate() );
                if ( err.isSucceeded() ) err=unitVal.setQuantity ( 1 );
                if ( err.isSucceeded() ) err=unitVal.save();

                if ( err.isSucceeded() ) getDocument()->sendMessage ( i18n ( "Unit [%1] has been created" , ui.kUnitEdit->text() ), true );
            }
            else err=unit.load();

            if ( err.isSucceeded() ) err=operationObj.setUnit ( unit );
        }

        //Save
        if ( err.isSucceeded() ) err=operationObj.save();

        if ( err.isSucceeded() ) err=getDocument()->stepForward ( i+1 );
    }

    return err;
}

void SKGOperationPluginWidget::onAddOperationClicked()
{
    SKGError err;
    SKGTRACEIN ( 10, "SKGOperationPluginWidget::onAddOperationClicked" );

    //Get parameters
    QString accountName=ui.kAccountEdit->currentText();
    QString catName=ui.kCategoryEdit->currentText();
    QString trackerName=ui.kTrackerEdit->currentText();

    bool confirmed=true;
    SKGOperationObject operation;
    {
        SKGBEGINTRANSACTION ( *getDocument(), i18n ( "Operation creation" ), err );

        QApplication::setOverrideCursor ( QCursor ( Qt::WaitCursor ) );

        //Get account
        SKGAccountObject accountObj ( getDocument() );
        if ( err.isSucceeded() ) err=accountObj.setName ( accountName );
        if ( err.isSucceeded() ) err=accountObj.load();

        //Create operation object
        if ( err.isSucceeded() ) err=accountObj.addOperation ( operation );
        if ( err.isSucceeded() ) err=operation.setMode ( ui.kTypeEdit->currentText() );
        if ( err.isSucceeded() ) err=operation.setPayee ( ui.kPayeeEdit->currentText() );
        if ( err.isSucceeded() ) err=operation.setNumber ( SKGServices::stringToInt ( ui.kNumberEdit->text() ) );
        if ( err.isSucceeded() ) err=operation.setComment ( ui.kCommentEdit->text() );
        if ( err.isSucceeded() ) err=operation.setDate ( ui.kDateEdit->date() );
        if ( err.isSucceeded() ) err=operation.setTemplate ( isTemplateMode() );

        SKGUnitObject unit ( getDocument() );
        err=unit.setSymbol ( ui.kUnitEdit->text() );
        if ( !unit.exist() )
        {
            if ( err.isSucceeded() ) err= unit.setName ( ui.kUnitEdit->text() );
            if ( err.isSucceeded() ) err = unit.save();

            SKGUnitValueObject unitVal;
            if ( err.isSucceeded() ) err=unit.addUnitValue ( unitVal );
            if ( err.isSucceeded() ) err=unitVal.setDate ( QDate::currentDate() );
            if ( err.isSucceeded() ) err=unitVal.setQuantity ( 1 );
            if ( err.isSucceeded() ) err=unitVal.save();

            if ( err.isSucceeded() ) getDocument()->sendMessage ( i18n ( "Unit [%1] has been created" , ui.kUnitEdit->text() ), true );
        }
        else err=unit.load();

        if ( err.isSucceeded() ) err=operation.setUnit ( unit );
        if ( err.isSucceeded() ) err=operation.save();

        if ( getCurrentMode() <=2 )
        {
            //STD OPERATION (SPLIT , TRANSFER OR NOT)
            //We must create a suboperation, just be sure to be able to update it
            SKGSubOperationObject subOperation;
            if ( err.isSucceeded() ) err=operation.addSubOperation ( subOperation );
            if ( err.isSucceeded() ) err=subOperation.setQuantity ( 0 );
            if ( err.isSucceeded() ) err=subOperation.save();

            SKGObjectBase::SKGListSKGObjectBase list;
            list << operation;
            if ( err.isSucceeded() ) err=updateSelection ( list, true );

        }
        else if ( getCurrentMode() ==3 )
        {
            //PURCHASE OR SALE SHARE
            //Create sub operation object
            SKGSubOperationObject subOperation;
            double val=ui.kAmountEdit->value();
            if ( err.isSucceeded() ) err=operation.addSubOperation ( subOperation );
            if ( err.isSucceeded() ) err=subOperation.setQuantity ( val );
            if ( err.isSucceeded() ) err=subOperation.save();

            if ( err.isSucceeded() && val>0 ) err=operation.setProperty ( "SKG_OP_ORIGINAL_AMOUNT", SKGServices::doubleToString ( ui.kAmountSharesEdit->value() ) );
            if ( err.isSucceeded() ) err=operation.save();

            //Get account
            SKGAccountObject accountObj2 ( getDocument() );
            if ( err.isSucceeded() ) err=accountObj2.setName ( ui.kPaymentAccountEdit->currentText() );
            if ( err.isSucceeded() ) err=accountObj2.load();

            //create paiement operation for shares
            SKGOperationObject operation2;
            if ( err.isSucceeded() ) err=accountObj2.addOperation ( operation2 );
            if ( err.isSucceeded() ) err=operation2.setMode ( ui.kTypeEdit->currentText() );
            if ( err.isSucceeded() ) err=operation2.setPayee ( ui.kPayeeEdit->currentText() );
            if ( err.isSucceeded() ) err=operation2.setNumber ( SKGServices::stringToInt ( ui.kNumberEdit->text() ) );
            if ( err.isSucceeded() ) err=operation2.setComment ( ui.kCommentEdit->text() );
            if ( err.isSucceeded() ) err=operation2.setDate ( ui.kDateEdit->date() );
            if ( err.isSucceeded() )
            {
                SKGUnitObject unitOfUnit;
                err=unit.getUnit ( unitOfUnit );
                if ( err.isFailed() )
                {
                    unitOfUnit=SKGUnitObject ( getDocument() );
                    err=unitOfUnit.setSymbol ( ( ( SKGDocumentBank* ) getDocument() )->getPrimaryUnit().Name );
                    if ( err.isSucceeded() ) err=unitOfUnit.load();
                }

                if ( err.isSucceeded() ) err=operation2.setUnit ( unitOfUnit );
            }
            if ( err.isSucceeded() ) err=operation2.setGroupOperation ( operation );
            if ( err.isSucceeded() ) err=operation2.setTemplate ( isTemplateMode() );
            if ( err.isSucceeded() ) err=operation2.save();

            //Create main sub operation
            SKGSubOperationObject subOperation2;
            if ( err.isSucceeded() ) err=operation2.addSubOperation ( subOperation2 );
            if ( err.isSucceeded() ) err=subOperation2.setComment ( i18n ( "Shares" ) );
            if ( err.isSucceeded() ) err=subOperation2.setQuantity ( ui.kAmountSharesEdit->value() * ( val>0 ? -1 : 1 ) );
            if ( err.isSucceeded() ) err=subOperation2.save();

            //Create commission sub operation
            if ( ui.kAmountSharesEdit->value() )
            {
                SKGSubOperationObject subOperation2;
                if ( err.isSucceeded() ) err=operation2.addSubOperation ( subOperation2 );
                if ( err.isSucceeded() ) err=subOperation2.setComment ( i18n ( "Commission" ) );
                if ( err.isSucceeded() ) err=subOperation2.setQuantity ( -ui.kCommissionEdit->value() );
                if ( err.isSucceeded() ) err=subOperation2.save();
            }

            //Create tax sub operation
            if ( ui.kTaxEdit->value() )
            {
                SKGSubOperationObject subOperation2;
                if ( err.isSucceeded() ) err=operation2.addSubOperation ( subOperation2 );
                if ( err.isSucceeded() ) err=subOperation2.setComment ( i18n ( "Tax" ) );
                if ( err.isSucceeded() ) err=subOperation2.setQuantity ( -ui.kTaxEdit->value() );
                if ( err.isSucceeded() ) err=subOperation2.save();
            }
        }

        QApplication::restoreOverrideCursor();
    }

    //status bar
    if ( err.isSucceeded() )
    {
        if ( confirmed )
        {
            err=SKGError ( 0, i18n ( "Operation created" ) );
            ui.kOperationView->selectObject ( operation.getUniqueID() );
        }
        else err=SKGError ( 0, i18n ( "Operation canceled" ) );
    }
    else err.addError ( ERR_FAIL, i18n ( "Operation creation failed" ) );

    //Display error
    SKGMainPanel::getMainPanel()->displayErrorMessage ( err );
}

SKGObjectBase::SKGListSKGObjectBase SKGOperationPluginWidget::getSelectedObjects()
{
    return ui.kOperationView->getSelectedObjects();
}

int SKGOperationPluginWidget::getNbSelectedObjects()
{
    return ui.kOperationView->getNbSelectedObjects();
}

SKGTableView* SKGOperationPluginWidget::getTableView()
{
    return ui.kOperationView;
}

QString SKGOperationPluginWidget::getState()
{
    SKGTRACEIN ( 10, "SKGOperationPluginWidget::getState" );
    QDomDocument doc ( "SKGML" );
    QDomElement root;
    if ( lastState.hasChildNodes() )
    {
        doc=lastState;
        root = doc.documentElement();
    }
    else
    {
        root = doc.createElement ( "parameters" );
        doc.appendChild ( root );
    }
    QString account=root.attribute ( "account" );
    QString currentPage=root.attribute ( "currentPage" );
    QString modeTemplate=root.attribute ( "modeTemplate" );
    QString hideUselessOperation=root.attribute ( "hideUselessOperation" );
    QString filter=root.attribute ( "filter" );
    QString modeInfoZoneS=root.attribute ( "modeInfoZone" );
    QString reconcilitorAmountS=root.attribute ( "reconcilitorAmount" );

    if ( account.isEmpty() ) root.setAttribute ( "account", ui.kDisplayAccountCombo->currentText() );
    if ( currentPage.isEmpty() ) root.setAttribute ( "currentPage", getCurrentMode() );
    if ( modeTemplate.isEmpty() ) root.setAttribute ( "modeTemplate", isTemplateMode() ? "Y" : "N" );
    if ( hideUselessOperation.isEmpty() ) root.setAttribute ( "hideUselessOperation", ui.kHideUselessOperation->checkState() ==Qt::Checked ? "Y" : "N" );
    if ( filter.isEmpty() ) root.setAttribute ( "filter", ui.kFilterEdit->lineEdit()->text() );
    if ( modeInfoZoneS.isEmpty() ) root.setAttribute ( "modeInfoZone", SKGServices::intToString ( modeInfoZone ) );
    if ( reconcilitorAmountS.isEmpty() ) root.setAttribute ( "reconcilitorAmount", ui.kReconcilitorAmountEdit->text() );

    //Memorize table settings
    root.setAttribute ( "view", ui.kOperationView->getState() );

    return doc.toString();
}

void SKGOperationPluginWidget::setState ( const QString& iState )
{
    SKGTRACEIN ( 10, "SKGOperationPluginWidget::setState" );
    QDomDocument doc ( "SKGML" );
    doc.setContent ( iState );
    QDomElement root = doc.documentElement();

    QString account=root.attribute ( "account" );
    QString currentPage=root.attribute ( "currentPage" );
    QString modeTemplate=root.attribute ( "modeTemplate" );
    QString hideUselessOperation=root.attribute ( "hideUselessOperation" );
    QString filter=root.attribute ( "filter" );
    QString title=root.attribute ( "title" );
    QString title_icon=root.attribute ( "title_icon" );
    QString modeInfoZoneS=root.attribute ( "modeInfoZone" );
    QString reconcilitorAmountS=root.attribute ( "reconcilitorAmount" );

    if ( !account.isEmpty() )
    {
        SKGAccountObject acc;
        SKGNamedObject::getObjectByName ( getDocument(), "v_account", account, acc );
        if ( acc.isClosed() && showClosedAccounts==false )
        {
            showClosedAccounts=true;
            dataModified("", 0);
        }
        bool previous=ui.kDisplayAccountCombo->blockSignals ( true );
        ui.kDisplayAccountCombo->setText ( account );
        ui.kDisplayAccountCombo->blockSignals ( previous );
    }

    if ( !reconcilitorAmountS.isEmpty() )
    {
        bool previous=ui.kReconcilitorAmountEdit->blockSignals ( true );
        ui.kReconcilitorAmountEdit->setText ( reconcilitorAmountS );
        ui.kReconcilitorAmountEdit->blockSignals ( previous );
    }
    if ( !currentPage.isEmpty() ) setCurrentMode ( SKGServices::stringToInt ( currentPage ) );
    if ( !modeTemplate.isEmpty() ) setTemplateMode ( modeTemplate==QString ( 'Y' ) );
    if ( !hideUselessOperation.isEmpty() )
    {
        bool previous=ui.kHideUselessOperation->blockSignals ( true );
        ui.kHideUselessOperation->setCheckState ( hideUselessOperation=="Y" ? Qt::Checked : Qt::Unchecked );
        ui.kHideUselessOperation->blockSignals ( previous );
    }
    if ( !filter.isEmpty() ) ui.kFilterEdit->setText ( filter );
    if ( !title.isEmpty() )
    {
        ui.kTitle->setComment ( "<html><body><b>"+SKGServices::stringToHtml ( title ) +"</b></body></html>" );
        ui.kTitle->show();
    }
    else
    {
        ui.kTitle->hide();
    }
    if ( !title_icon.isEmpty() ) ui.kTitle->setPixmap ( KIcon ( title_icon ).pixmap ( 22, 22 ), KTitleWidget::ImageLeft );

    QString operationTable=root.attribute ( "operationTable" );
    operationWhereClause=root.attribute ( "operationWhereClause" );
    if ( objectModel && !operationTable.isEmpty() )  objectModel->setTable ( operationTable );
    if ( objectModel && !operationWhereClause.isEmpty() )  objectModel->setFilter ( operationWhereClause );
    if ( !operationTable.isEmpty() || !operationWhereClause.isEmpty() )
    {
        //We keep a copy of given state in case of bookmark
        lastState=doc;
    }

    //Update model
    if ( objectModel )
    {
        bool previous=objectModel->blockRefresh ( true );
        onFilterChanged();
        objectModel->blockRefresh ( previous );
    }

    //Due to the refresh of the information zone, this must be done here
    if ( !modeInfoZoneS.isEmpty() ) modeInfoZone=SKGServices::stringToInt ( modeInfoZoneS )-1;
    else modeInfoZone=-1;
    onRotateAccountTools();

    //!!! Must be done here after onFilterChanged
    ui.kOperationView->setState ( root.attribute ( "view" ) );
}

QString SKGOperationPluginWidget::getDefaultStateAttribute()
{
    if ( objectModel && objectModel->getTable() =="v_operation_consolidated" ) return "SKGOPERATION_CONSOLIDATED_DEFAULT_PARAMETERS";
    if ( ui.kTitle->isVisible() ) return "";
    return "SKGOPERATION_DEFAULT_PARAMETERS";
}

void SKGOperationPluginWidget::dataModified(const QString& iTableName, int iIdTransaction)
{
    SKGTRACEIN ( 10, "SKGBankPluginWidget::dataModified" );
    Q_UNUSED(iIdTransaction);

    //Refresh widgets
    QSqlDatabase* db = getDocument()->getDatabase();
    setEnabled ( db!=NULL );
    if ( db!=NULL && (iTableName=="operation" || iTableName.isEmpty()))
    {

        //Correction bug 2299394 vvv
        if ( ui.kOperationView->isAutoResized() ) ui.kOperationView->resizeColumnsToContents();
        //Correction bug 2299394 ^^^

	//Fill comboboxes
        fillComboBoxes();

	//Correction 215658: vvv because the table is modified, the selection is modified
	onSelectionChanged();
	//Correction 215658: ^^^

	if ( modeInfoZone!=2 ) onRefreshInformationZone(); //Because mode 2 is linked to selection (see onSelectionChanged)
    }
}

void SKGOperationPluginWidget::fillComboBoxes()
{
    SKGTRACEIN ( 1, "SKGOperationPluginWidget::fillComboBoxes" );

    //Refresh account summary
    QSqlDatabase* db = getDocument()->getDatabase();
    setEnabled ( db!=NULL );
    if ( db!=NULL )
    {
        //Disconnect combo filter account
        disconnect ( ui.kDisplayAccountCombo, SIGNAL ( currentIndexChanged ( int ) ), this, SLOT ( onFilterChanged() ) );
        disconnect ( ui.kDisplayAccountCombo, SIGNAL ( currentIndexChanged ( int ) ), this, SLOT ( onRefreshInformationZone() ) );
        disconnect ( ui.kDisplayAccountCombo, SIGNAL ( textChanged ( QString ) ), this, SLOT ( onOperationCreatorModified() ) );
        disconnect ( ui.kUnitEdit, SIGNAL ( textChanged ( QString ) ), this, SLOT ( onOperationCreatorModified() ) );
        disconnect ( ui.kAmountEdit, SIGNAL ( textChanged ( QString ) ), this, SLOT ( onOperationCreatorModified() ) );
        disconnect ( ui.kAmountSharesEdit, SIGNAL ( textChanged ( QString ) ), this, SLOT ( onOperationCreatorModified() ) );
        disconnect ( ui.kCommissionEdit, SIGNAL ( textChanged ( QString ) ), this, SLOT ( onOperationCreatorModified() ) );
        disconnect ( ui.kTaxEdit, SIGNAL ( textChanged ( QString ) ), this, SLOT ( onOperationCreatorModified() ) );

        //Set account combos
        QString current=ui.kDisplayAccountCombo->currentText();

        //Clear
        ui.kDisplayAccountCombo->clear();
        ui.kAccountEdit->clear();
        ui.kPaymentAccountEdit->clear();
        ui.kTargetAccountEdit->clear();

        ui.kDisplayAccountCombo->addItem ( i18n ( "All" ) );

        SKGStringListList listAccount;
        SKGServices::executeSelectSqliteOrder ( getDocument(), QString ( "SELECT t_ICON, t_name from v_account_display " ) + ( showClosedAccounts ? "" :"where t_close='N' order by t_name" ), listAccount );

        int nbAccounts=listAccount.count();
        if ( nbAccounts ==0 )
        {
            ui.kTitle->setText ( i18n ( "First you have to create an account." ) );
            ui.kTitle->setPixmap ( KIcon ( "dialog-information" ).pixmap ( 22, 22 ) , KTitleWidget::ImageLeft );
            ui.kTitle->show();
        }
        else
        {
            if ( !lastState.hasChildNodes() ) ui.kTitle->hide();

            QDir dirLogo ( KStandardDirs::locate ( "data", QString::fromLatin1 ( "skrooge/images/logo/" ) ) );
            for ( int i=1; i<nbAccounts; ++i ) //Ignore header
            {
                QIcon icon ( dirLogo.absoluteFilePath ( listAccount.at ( i ).at ( 0 ) ) );
                QString text=listAccount.at ( i ).at ( 1 );
                ui.kDisplayAccountCombo->addItem ( icon, text );
                ui.kAccountEdit->addItem ( icon, text );
                ui.kPaymentAccountEdit->addItem ( icon, text );
                ui.kTargetAccountEdit->addItem ( icon, text );
            }
        }

        int pos=ui.kDisplayAccountCombo->findText ( current );
        if ( pos==-1 ) pos=0;
        ui.kDisplayAccountCombo->setCurrentIndex ( pos );

        //Set type combo
        SKGMainPanel::fillWithDistinctValue ( ui.kTypeEdit, getDocument(), "operation", "t_mode", "" );

        //Set type unit
        SKGMainPanel::fillWithDistinctValue ( ui.kUnitEdit, getDocument(), "unit", "ifnull(t_symbol,t_name)", "t_type!='I'" );
        SKGServices::SKGUnitInfo primary= ( ( SKGDocumentBank* ) getDocument() )->getPrimaryUnit();
        if ( !primary.Name.isEmpty() ) ui.kUnitEdit->setText ( primary.Name );

        //Set type category
        SKGMainPanel::fillWithDistinctValue ( ui.kCategoryEdit, getDocument(), "category", "t_fullname", "" );

        //Set type tracker
        SKGMainPanel::fillWithDistinctValue ( ui.kTrackerEdit, getDocument(), "refund", "t_name", "t_close='N'" );

        //Set type payee
        SKGMainPanel::fillWithDistinctValue ( ui.kPayeeEdit, getDocument(), "operation", "t_payee", "" );

        //Set type number
        numberFieldIsNotUptodate=true;

        //Connect combo filter account
        connect ( ui.kDisplayAccountCombo, SIGNAL ( currentIndexChanged ( int ) ), this, SLOT ( onFilterChanged() ), Qt::QueuedConnection );
        connect ( ui.kDisplayAccountCombo, SIGNAL ( currentIndexChanged ( int ) ), this, SLOT ( onRefreshInformationZone() ), Qt::QueuedConnection );
        connect ( ui.kDisplayAccountCombo, SIGNAL ( textChanged ( QString ) ), this, SLOT ( onOperationCreatorModified() ), Qt::QueuedConnection );
        connect ( ui.kUnitEdit, SIGNAL ( textChanged ( QString ) ), this, SLOT ( onOperationCreatorModified() ), Qt::QueuedConnection );
        connect ( ui.kAmountEdit, SIGNAL ( textChanged ( QString ) ), this, SLOT ( onOperationCreatorModified() ), Qt::QueuedConnection );
        connect ( ui.kAmountSharesEdit, SIGNAL ( textChanged ( QString ) ), this, SLOT ( onOperationCreatorModified() ), Qt::QueuedConnection );
        connect ( ui.kCommissionEdit, SIGNAL ( textChanged ( QString ) ), this, SLOT ( onOperationCreatorModified() ), Qt::QueuedConnection );
        connect ( ui.kTaxEdit, SIGNAL ( textChanged ( QString ) ), this, SLOT ( onOperationCreatorModified() ), Qt::QueuedConnection );
    }
}
void SKGOperationPluginWidget::openOperation ( const SKGOperationObject& iOperation )
{
    _SKGTRACEIN ( 10, "SKGOperationPluginWidget::openOperation" );

    //Build where clause and title
    int opid=iOperation.getID();
    QString wc="id="+SKGServices::intToString ( opid );

    opid=SKGServices::stringToInt ( iOperation.getAttribute ( "i_group_id" ) );
    if ( opid!=0 ) wc+=" or i_group_id="+SKGServices::intToString ( opid );
    wc='('+wc+')';
    QString title;
    QString targetTable;
    QString parametersTable;
    if ( iOperation.getTable() =="v_operation_display" )
    {
        title=i18n ( "Sub operations grouped or split" );
        targetTable="v_operation_consolidated";
        parametersTable="SKGOPERATION_CONSOLIDATED_DEFAULT_PARAMETERS";
    }
    else
    {
        title=i18n ( "Operations grouped" );
        targetTable="v_operation_display";
        parametersTable="SKGOPERATION_DEFAULT_PARAMETERS";
    }

    //Open
    if ( QApplication::keyboardModifiers() &Qt::ControlModifier && QApplication::keyboardModifiers() &Qt::ShiftModifier )
    {
        //Call debug plugin
        QDomDocument doc ( "SKGML" );
        QDomElement root = doc.createElement ( "parameters" );
        doc.appendChild ( root );
        root.setAttribute ( "sqlOrder", "SELECT * from "+targetTable+" WHERE "+wc );

        if ( SKGMainPanel::getMainPanel() ) SKGMainPanel::getMainPanel()->setNewTabContent ( SKGMainPanel::getMainPanel()->getPluginByName ( "Skrooge debug plugin" ), -1, doc.toString() );
    }
    else
    {
        //Call operation plugin
        QDomDocument doc ( "SKGML" );
        doc.setContent ( iOperation.getDocument()->getParameter ( parametersTable ) );
        QDomElement root = doc.documentElement();
        if ( root.isNull() )
        {
            root=doc.createElement ( "parameters" );
            doc.appendChild ( root );
        }

        root.setAttribute ( "account", i18n("All") );
	root.setAttribute ( "operationTable", targetTable );
        root.setAttribute ( "operationWhereClause", wc );
        root.setAttribute ( "title", title );
        root.setAttribute ( "title_icon", "view-pim-tasks" );
        root.setAttribute ( "currentPage", "-1" );

        if ( SKGMainPanel::getMainPanel() ) SKGMainPanel::getMainPanel()->setNewTabContent ( SKGMainPanel::getMainPanel()->getPluginByName ( "Skrooge operation plugin" ), -1, doc.toString() );
    }
}

void SKGOperationPluginWidget::onDoubleClick()
{
    _SKGTRACEIN ( 10, "SKGOperationPluginWidget::onDoubleClick" );

    //Get selection
    SKGObjectBase::SKGListSKGObjectBase selection=getSelectedObjects();
    if ( selection.count() ==1 )
    {
        SKGOperationObject op ( selection.at ( 0 ) );

        if ( op.isTemplate() )
        {
            //this is a template, we have to create an operation
            SKGError err;
            SKGBEGINTRANSACTION ( *getDocument(), i18n ( "Operation creation" ), err );
            QApplication::setOverrideCursor ( QCursor ( Qt::WaitCursor ) );
            SKGOperationObject operation;
            err=op.duplicate ( operation );
            QApplication::restoreOverrideCursor();

            //status bar
            if ( err.isSucceeded() )
            {
                setTemplateMode ( false );
                err=SKGError ( 0, i18n ( "Operation created" ) );
                ui.kOperationView->selectObject ( operation.getUniqueID() );
            }
            else err.addError ( ERR_FAIL, i18n ( "Operation creation failed" ) );

            //Display error
            SKGMainPanel::getMainPanel()->displayErrorMessage ( err );

        }
        else
        {
            //This is not a template, we have to open it
            openOperation ( op );
        }
    }
}

void SKGOperationPluginWidget::onResetInternalFilter()
{
    SKGTRACEIN ( 10, "SKGOperationPluginWidget::onResetInternalFilter" );

    //Initialisation
    lastState.clear();

    if ( objectModel ) objectModel->setTable ( "v_operation_display" );

    setState ( getDocument()->getParameter ( getDefaultStateAttribute() ) );

    onFilterChanged();
}

void SKGOperationPluginWidget::onRefreshInformationZone()
{
    SKGTRACEIN ( 1, "SKGOperationPluginWidget::onRefreshInformationZone" );
    QApplication::setOverrideCursor ( QCursor ( Qt::WaitCursor ) );
    KLocale* locale=KGlobal::locale();
    SKGDocumentBank* doc= ( SKGDocumentBank* ) getDocument();
    SKGServices::SKGUnitInfo unit1=doc->getPrimaryUnit();
    SKGServices::SKGUnitInfo unit2=doc->getSecondaryUnit();
    if ( modeInfoZone==0 )
    {
        //Refresh info area
        //Compute where clause
        QString filter="1=1";
        if ( ui.kDisplayAccountCombo->currentIndex() >0 )
        {
            filter="t_name='"+SKGServices::stringToSqlString ( ui.kDisplayAccountCombo->currentText() ) +'\'';
        }
        SKGStringListList listTmp;
        SKGServices::executeSelectSqliteOrder ( getDocument(),
                                                "SELECT TOTAL(f_CURRENTAMOUNT), TOTAL(f_CHECKED), TOTAL(f_COMING_SOON) from v_account_display WHERE "+filter,
                                                listTmp );
        if ( listTmp.count() ==2 )
        {
            if ( ui.kDisplayAccountCombo->currentIndex() >0 )
            {
                SKGUnitObject unitAccount;
                SKGAccountObject account ( getDocument() );
                if ( account.setName ( ui.kDisplayAccountCombo->currentText() ).isSucceeded() )
                {
                    if ( account.load().isSucceeded() )
                    {
                        if ( account.getUnit ( unitAccount ).isSucceeded() )
                        {
                            if ( !unitAccount.getSymbol().isEmpty() )
                            {
                                unit1.Name=unitAccount.getSymbol();
                                unit1.Value=SKGServices::stringToDouble ( unitAccount.getAttribute ( "f_CURRENTAMOUNT" ) );

                                if ( unit1.Name != ( ( SKGDocumentBank* ) getDocument() )->getPrimaryUnit().Name )
                                {
                                    unit2= ( ( SKGDocumentBank* ) getDocument() )->getPrimaryUnit();
                                }
                            }
                        }
                    }
                }
            }

            double v1=SKGServices::stringToDouble ( listTmp.at ( 1 ).at ( 0 ) );
            double v2=SKGServices::stringToDouble ( listTmp.at ( 1 ).at ( 1 ) );
            double v3=SKGServices::stringToDouble ( listTmp.at ( 1 ).at ( 2 ) );
            QString s1=doc->formatMoney ( v1, unit1 );
            QString s2=doc->formatMoney ( v2, unit1 );
            QString s3=doc->formatMoney ( v3, unit1 );
            ui.kInfo->setText ( i18n ( "Balance: %1     Cleared: %2     In Transit: %3", s1, s2, s3 ) );
            if ( !unit2.Name.isEmpty() && unit2.Value )
            {
                s1=doc->formatMoney ( v1, unit2 );
                s2=doc->formatMoney ( v2, unit2 );
                s3=doc->formatMoney ( v3, unit2 );
            }
            ui.kInfo->setToolTip ( i18n ( "<p>Balance: %1</p><p>Cleared: %2</p><p>In Transit: %3</p>", s1, s2, s3 ) );
        }
    }
    else if ( modeInfoZone==1 )
    {
        //Refresh reconciliation area
        //Compute where clause
        QString filter='\''+SKGServices::stringToSqlString ( ui.kDisplayAccountCombo->currentText() ) +'\'';
        SKGStringListList listTmp;
        SKGServices::executeSelectSqliteOrder ( getDocument(),
                                                "SELECT ABS(TOTAL(f_CURRENTAMOUNT_EXPENSE)),TOTAL(f_CURRENTAMOUNT_INCOME) FROM v_operation_display WHERE t_status='P' AND t_ACCOUNT="+filter,
                                                listTmp );
        if ( listTmp.count() ==2 )
        {
            if ( ui.kDisplayAccountCombo->currentIndex() >0 )
            {
                SKGUnitObject unitAccount;
                SKGAccountObject account ( getDocument() );
                if ( account.setName ( ui.kDisplayAccountCombo->currentText() ).isSucceeded() )
                {
                    if ( account.load().isSucceeded() )
                    {
                        if ( account.getUnit ( unitAccount ).isSucceeded() )
                        {
                            if ( !unitAccount.getSymbol().isEmpty() )
                            {
                                unit1.Name=unitAccount.getSymbol();
                                unit1.Value=SKGServices::stringToDouble ( unitAccount.getAttribute ( "f_CURRENTAMOUNT" ) );

                                if ( unit1.Name!= ( ( SKGDocumentBank* ) getDocument() )->getPrimaryUnit().Name )
                                {
                                    unit2= ( ( SKGDocumentBank* ) getDocument() )->getPrimaryUnit();
                                }
                            }
                        }
                    }
                }
            }

            SKGStringListList listTmp2;
            double diff=0;
            SKGServices::executeSelectSqliteOrder ( getDocument(),
                                                    "SELECT TOTAL(f_CHECKED) from v_account_display WHERE t_name="+filter,
                                                    listTmp2 );
            if ( listTmp.count() ==2 )
            {
                diff=SKGServices::stringToDouble ( listTmp2.at ( 1 ).at ( 0 ) ) /unit1.Value-ui.kReconcilitorAmountEdit->value();
            }

            //Set tooltip
	    bool onOneAccount= ( ui.kDisplayAccountCombo->currentIndex() >0 );
	    if(!onOneAccount)
	    {
	      ui.kReconciliatorInfo->setText(i18n("You must select only one account to use reconciliation."));
	      ui.kReconciliatorInfo->setToolTip(ui.kReconciliatorInfo->text());
	    }
	    else
	    {
	      double v1=SKGServices::stringToDouble ( listTmp.at ( 1 ).at ( 0 ) );
	      double v2=SKGServices::stringToDouble ( listTmp.at ( 1 ).at ( 1 ) );
	      QString sdiff=doc->formatMoney ( diff, unit1 );
	      QString s1=doc->formatMoney ( v1, unit1 );
	      QString s2=doc->formatMoney ( v2, unit1 );
	      ui.kReconciliatorInfo->setText ( i18n ( "%1 - Delta: %2     Expenditure: %3     Income: %4", unit1.Name, sdiff, s1, s2 ) );

	      //Comparison
	      QString zero=doc->formatMoney ( 0, unit1 );
	      QString negativezero=doc->formatMoney ( -0.00001, unit1 );
	      ui.kValidate->setEnabled ( sdiff==zero || sdiff==negativezero );

	      if ( !unit2.Name.isEmpty() && unit2.Value )
	      {
		  sdiff=doc->formatMoney ( diff, unit2 );
		  s1=doc->formatMoney ( v1, unit2 );
		  s2=doc->formatMoney ( v2, unit2 );
	      }
	      ui.kReconciliatorInfo->setToolTip ( i18n ( "<p>Delta: %1</p><p>Expenditure: %2</p><p>Income: %3</p>", sdiff, s1, s2 ) );
		}
	}
    }
    else if ( modeInfoZone==2 )
    {
        //Refresh info area with selection
        SKGObjectBase::SKGListSKGObjectBase selection=getSelectedObjects();
        double amount=0;
        int nb=selection.count();
        QMap<QString, double> amountPerUnit;
        QMap<QString, int> decimalPerUnit;
        for ( int i=0; i<nb; ++i )
        {
            SKGOperationObject obj=selection.at ( i );
            amount+=obj.getCurrentAmount();

            SKGUnitObject unit;
            obj.getUnit ( unit );
            amountPerUnit[unit.getSymbol() ]+=SKGServices::stringToDouble ( obj.getAttribute ( "f_QUANTITY" ) );
            decimalPerUnit[unit.getSymbol() ]=unit.getNumberDecimal();
        }

        QString addition;
        QMapIterator<QString, double> i ( amountPerUnit );
        while ( i.hasNext() )
        {
            i.next();

            double amo=i.value();
            if ( !addition.isEmpty() && amo>0 ) addition+='+';
            addition+=QString ( "<font color=\"" ) + ( amo<0 ? "red" : "black" ) +"\">"+
                      locale->formatMoney ( amo, i.key(), decimalPerUnit[i.key() ] ) +
                      "</font>";
        }

        QString v2=QString ( "<font color=\"" )  + ( amount<0 ? "red" : "black" ) +"\">"+
                   locale->formatMoney ( amount/unit1.Value, unit1.Name, unit1.NbDecimal ) +
                   "</font>";
        if ( nb==0 || v2==addition )
        {
            addition="";
        }
        else
        {
            v2+='=';
            v2+=addition;
        }
        if ( nb )
        {
            ui.kInfo->setText ( i18np ( "Selection: %1 operation for %2", "Selection: %1 operations for %2", nb, v2 ) );
            if ( !unit2.Name.isEmpty() && unit2.Value )
            {
                v2=addition+"<font color=\""  + ( amount<0 ? "red" : "black" ) +"\">"+locale->formatMoney ( amount/unit2.Value, unit2.Name, unit2.NbDecimal ) +"</font>";
            }
            ui.kInfo->setToolTip ( i18np ( "Selection: %1 operation for %2", "Selection: %1 operations for %2", nb, v2 ) );
        }
        else
        {
            ui.kInfo->setText ( i18n ( "Selection: none" ) );
            ui.kInfo->setToolTip ( i18n ( "Selection: none" ) );
        }
    }
    QApplication::restoreOverrideCursor();
}

void SKGOperationPluginWidget::onFilterChanged()
{
    SKGTRACEIN ( 1, "SKGOperationPluginWidget::onFilterChanged" );
    if ( !isEnabled() ) return;
    QApplication::setOverrideCursor ( QCursor ( Qt::WaitCursor ) );

    //Enable/ disable widgets
    bool onOneAccount= ( ui.kDisplayAccountCombo->currentIndex() >0 );
    ui.kAccountEdit->setEnabled ( !onOneAccount );
    ui.kReconciliatorFrame2->setEnabled ( onOneAccount );
    if ( !onOneAccount && modeInfoZone==1 )
    {
        ui.kReconciliatorFrame2->hide();
        ui.kInfo->show();
        modeInfoZone=0;
    }

    ui.kAccountLabel->setEnabled ( operationWhereClause.isEmpty() );
    ui.kDisplayAccountCombo->setEnabled ( operationWhereClause.isEmpty() );
    ui.kHideUselessOperation->setEnabled ( operationWhereClause.isEmpty() );
    if ( operationWhereClause.isEmpty() ) ui.kResetInternalFilter->hide();
    else ui.kResetInternalFilter->show();

    //Compute where clause
    QString filter2=operationWhereClause;
    if ( onOneAccount )
    {
        QString account=ui.kDisplayAccountCombo->currentText();
        if ( operationWhereClause.isEmpty() ) filter2="t_ACCOUNT='"+SKGServices::stringToSqlString ( account ) +'\'';

        ui.kAccountEdit->setText ( account );
    }

    //Add status filter
    if ( operationWhereClause.isEmpty() && ui.kHideUselessOperation->checkState() ==Qt::Checked )
    {
        if ( !filter2.isEmpty() ) filter2+=" AND ";
        filter2+="t_status!='Y'";
    }

    //Update model
    if ( objectModel )
    {
        objectModel->setFilter ( filter2 );
        ui.kOperationView->setState ( ui.kOperationView->getState() );
    }
    QApplication::restoreOverrideCursor();
}

void SKGOperationPluginWidget::fillNumber()
{
    SKGTRACEIN ( 10, "SKGOperationPluginWidget::fillNumber" );
    QStringList list;
    SKGServices::getDistinctValues ( getDocument(), "v_operation_next_numbers", "i_number", "", list );

    //Fill completion
    KCompletion *comp = ui.kNumberEdit->completionObject();
    comp->clear ();
    comp->insertItems ( list );

    numberFieldIsNotUptodate=false;
}

void SKGOperationPluginWidget::onFocusChanged()
{
    _SKGTRACEIN ( 10, "SKGOperationPluginWidget::onFocusChanged" );

    if ( numberFieldIsNotUptodate && ui.kNumberEdit->hasFocus() )
    {
        fillNumber();
    }

    bool test=ui.kTypeEdit->hasFocus() ||
//                  ui.kAmountEdit->hasFocus() ||
//  		    ui.kNumberEdit->hasFocus() ||
              ui.kUnitEdit->hasFocus() ||
              ui.kCategoryEdit->hasFocus() ||
              ui.kTrackerEdit->hasFocus() ||
              ui.kCommentEdit->hasFocus() ||
              ui.kPayeeEdit->hasFocus()
              ;
    if ( fastEditionAction ) fastEditionAction->setEnabled ( test );
}

void SKGOperationPluginWidget::onFastEdition()
{
    SKGTRACEIN ( 10, "SKGOperationPluginWidget::onFocusChanged" );
    QApplication::setOverrideCursor ( QCursor ( Qt::WaitCursor ) );
    SKGError err;

    //Get widget item
    QWidget* w=QApplication::focusWidget();

    //Build the where clause
    QString wc;
    if ( ui.kTypeEdit->hasFocus() ) wc="t_mode LIKE '"+SKGServices::stringToSqlString ( ui.kTypeEdit->text() ) +"%'";
    else if ( ui.kUnitEdit->hasFocus() ) wc="t_UNIT LIKE '"+SKGServices::stringToSqlString ( ui.kUnitEdit->text() ) +"%'";
    else if ( ui.kCategoryEdit->hasFocus() ) wc="t_CATEGORY LIKE '"+SKGServices::stringToSqlString ( ui.kCategoryEdit->text() ) +"%'";
    else if ( ui.kTrackerEdit->hasFocus() ) wc="t_REFUND LIKE '"+SKGServices::stringToSqlString ( ui.kTrackerEdit->text() ) +"%'";
    else if ( ui.kCommentEdit->hasFocus() ) wc="t_comment LIKE '"+SKGServices::stringToSqlString ( ui.kCommentEdit->text() ) +"%'";
    else if ( ui.kPayeeEdit->hasFocus() ) wc="t_payee LIKE '"+SKGServices::stringToSqlString ( ui.kPayeeEdit->text() ) +"%'";

    if ( !wc.isEmpty() )
    {
        //Read Setting
        QString fasteditmode = skgoperation_settings::fasteditmode();

        /*
        0-Search in templates only
        1-Search first in templates and after in operations
        2-Search in operations only
        3-Search first in operations and after in templates
        */
        if ( fasteditmode=="0" ) wc+=" AND t_template='Y'";
        else if ( fasteditmode=="2" ) wc+=" AND t_template='N'";

        if ( wc!=lastFastEditionWhereClause )
        {
            lastFastEditionWhereClause=wc;
            lastFastEditionOperationFound=0;
        }

        //Look for last operation
        if ( lastFastEditionOperationFound!=0 )
        {
            wc+=" AND id<"+SKGServices::intToString ( lastFastEditionOperationFound );
        }

        //Add order by
        wc+=" ORDER BY ";
        if ( fasteditmode=="1" ) wc+=" t_template DESC, ";
        else if ( fasteditmode=="3" ) wc+=" t_template ASC, ";
        wc+="d_date DESC, id DESC LIMIT 1";

        SKGObjectBase::SKGListSKGObjectBase operations;
        err=SKGObjectBase::getObjects ( getDocument(), "v_operation_display_all", wc, operations );
        if ( err.isSucceeded() && operations.count() )
        {
            SKGOperationObject op=operations.at ( 0 );

            lastFastEditionOperationFound=op.getID();
            if(isWidgetEditionEnabled(ui.kTypeEdit->lineEdit())) ui.kTypeEdit->setText ( op.getMode() );
            if(isWidgetEditionEnabled(ui.kUnitEdit->lineEdit())) ui.kUnitEdit->setText ( op.getAttribute ( "t_UNIT" ) );
            if(isWidgetEditionEnabled(ui.kCategoryEdit->lineEdit())) ui.kCategoryEdit->setText ( op.getAttribute ( "t_CATEGORY" ) );
            if(isWidgetEditionEnabled(ui.kCommentEdit)) ui.kCommentEdit->setText ( op.getComment() );
            if(isWidgetEditionEnabled(ui.kPayeeEdit->lineEdit())) ui.kPayeeEdit->setText ( op.getPayee() );
            if(isWidgetEditionEnabled(ui.kTrackerEdit->lineEdit())) ui.kTrackerEdit->setText ( op.getAttribute ( "t_REFUND" ) );
            if(ui.kAccountEdit->isEnabled() ) ui.kAccountEdit->setText ( op.getAttribute ( "t_ACCOUNT" ) );
            if(isWidgetEditionEnabled(ui.kAmountEdit)) ui.kAmountEdit->setValue ( SKGServices::stringToDouble ( op.getAttribute ( "f_QUANTITY" ) ) );

            //set next number
	    if(isWidgetEditionEnabled(ui.kNumberEdit))
	    {
	      int number=op.getNumber();
	      if ( number==0 )
	      {
		  ui.kNumberEdit->setText ( "" );
	      }
	      else
	      {
		  if ( numberFieldIsNotUptodate )
		  {
		      fillNumber();
		  }

		  KCompletion *comp = ui.kNumberEdit->completionObject();
		  if ( comp )
		  {
		      QStringList list=comp->items();
		      int nb=list.count();
		      int cpt=0;
		      while ( nb>=0 && cpt>=0 && cpt<1000 )
		      {
			  ++number;

			  if ( list.contains ( SKGServices::intToString ( number ) ) )
			  {
			      cpt=-2;
			  }
			  ++cpt;
		      }

		      if ( cpt<0 ) ui.kNumberEdit->setText ( SKGServices::intToString ( number ) );
		  }
	      }
	    }

            //Get nb operation linked
            SKGObjectBase::SKGListSKGObjectBase groupedOperations;
            op.getGroupedOperations ( groupedOperations );
            int nbGroupedOp=groupedOperations.count();

            //Get nb sub op
            SKGObjectBase::SKGListSKGObjectBase subOperations;
            op.getSubOperations ( subOperations );
            int nbSupOp=subOperations.count();

            if ( nbSupOp>1 )
            {
                //It is a SPLIT operation
                setCurrentMode ( 1 );;
                displaySubOperations ( op );
            }
            else
            {
                if ( nbGroupedOp>1 )
                {
                    //It is a TRANSFER
                    SKGOperationObject op2=groupedOperations.at ( 0 );
                    if ( op2==op ) op2=groupedOperations.at ( 1 );

                    SKGAccountObject targetAccount;
                    op2.getParentAccount ( targetAccount );

                    ui.kTargetAccountEdit->setText ( targetAccount.getName() );
                }
                else setCurrentMode ( 0 );
            }

        }
        else
        {
            lastFastEditionWhereClause="";
            lastFastEditionOperationFound=0;
        }
    }

    if ( w ) w->setFocus ( Qt::OtherFocusReason );
    QApplication::restoreOverrideCursor();

    //Display error
    SKGMainPanel::getMainPanel()->displayErrorMessage ( err );

}

void SKGOperationPluginWidget::onTemplateModeClicked()
{
    QToolButton* sender=static_cast<QToolButton*> ( this->sender() );
    setTemplateMode ( sender==ui.kTemplateBtn );
}

bool SKGOperationPluginWidget::isTemplateMode()
{
    return ( modeTemplate );
}

void SKGOperationPluginWidget::setTemplateMode ( bool iTemplate )
{
    SKGTRACEIN ( 10, "SKGOperationPluginWidget::setTemplateMode" );

    if ( iTemplate!=isTemplateMode() )
    {
        modeTemplate=iTemplate;

        //Set icons
        if ( !iTemplate )
        {
            ui.kModifyOperationBtn->setIcon ( KIcon ( "dialog-ok-apply" ) );
            ui.kAddOperationBtn->setIcon ( KIcon ( "list-add" ) );
        }
        else
        {
            QStringList overlay;
            overlay.push_back ( "skrooge_template" );
            ui.kModifyOperationBtn->setIcon ( KIcon ( "dialog-ok-apply", NULL, overlay ) );
            ui.kAddOperationBtn->setIcon ( KIcon ( "list-add", NULL, overlay ) );
        }

        //Initialisation
        lastState.clear();
        if ( objectModel ) objectModel->setTable ( !iTemplate ? "v_operation_display" :"v_operation_template_display" );
        onFilterChanged();
    }

    //Set states
    ui.kOperationBtn->setChecked ( !iTemplate );
    ui.kTemplateBtn->setChecked ( iTemplate );
}

int SKGOperationPluginWidget::getCurrentMode()
{
    return mode;
}

void SKGOperationPluginWidget::onBtnModeClicked()
{
    SKGTRACEIN ( 10, "SKGOperationPluginWidget::onBtnModeClicked" );
    QWidget* sender=static_cast<QWidget*> ( this->sender() );

    int currentMode=getCurrentMode();
    int newMode=0;

    if ( sender==ui.kStandardBtn ) newMode=0;
    else if ( sender==ui.kSplitBtn ) newMode=1;
    else if ( sender==ui.kTransferBtn ) newMode=2;
    else if ( sender==ui.kShareBtn ) newMode=3;

    if ( currentMode==newMode ) newMode=-1;
    setCurrentMode ( newMode );
}

void SKGOperationPluginWidget::setCurrentMode ( int iMode )
{
    SKGTRACEIN ( 10, "SKGOperationPluginWidget::setCurrentMode" );
    mode=iMode;

    ui.kStandardFrm->hide();
    ui.kSplitFrm->hide();
    ui.kTransferFrm->hide();
    ui.kShareFrm->hide();
    ui.kCommunFrm->setVisible ( mode>=0 );
    ui.kBtnFrm->setVisible ( mode>=0 );

    ui.kStandardBtn->setChecked ( false );
    ui.kSplitBtn->setChecked ( false );
    ui.kTransferBtn->setChecked ( false );
    ui.kShareBtn->setChecked ( false );

    if ( mode!=1 && mode!=-1 )
    {
        ui.kSubOperationsTable->setRowCount ( 0 );
        ui.kSubOperationsTable->clearContents();
    }

    if ( mode==0 )
    {
        ui.kStandardFrm->show();
        ui.kStandardBtn->setChecked ( true );
    }
    else if ( mode==1 )
    {
        if ( ui.kSubOperationsTable->rowCount() ==0 )
        {
            addSubOperationLine ( 0, ui.kCategoryEdit->text(), ui.kTrackerEdit->text(), ui.kCommentEdit->text(), ui.kAmountEdit->value(), 0 );
        }
        ui.kSplitFrm->show();
        ui.kSplitBtn->setChecked ( true );
        //ui.kStandardBtn->setChecked(true);
    }
    else if ( mode==2 )
    {
        ui.kTransferFrm->show();
        ui.kTransferBtn->setChecked ( true );
    }
    else if ( mode==3 )
    {
        ui.kShareFrm->show();
        ui.kShareBtn->setChecked ( true );
    }

    onOperationCreatorModified();
}

void SKGOperationPluginWidget::displaySubOperations ( const SKGOperationObject& iOperation )
{
    SKGTRACEIN ( 10, "SKGOperationPluginWidget::displaySubOperations(list)" );
    ui.kSubOperationsTable->setRowCount ( 0 );
    ui.kSubOperationsTable->clearContents();

    int nbSubOperations=0;

    QList<SKGObjectBase> subOperations;
    SKGError err =  iOperation.getSubOperations ( subOperations );
    nbSubOperations = subOperations.count();
    for ( int i = 0; i < nbSubOperations; ++i )
    {
        SKGSubOperationObject subOperation = subOperations.at ( i );

        SKGCategoryObject category;
        subOperation.getCategory ( category );

        SKGTrackerObject tracker;
        subOperation.getTracker ( tracker );

        addSubOperationLine ( i, category.getFullName(), tracker.getName(),
                              subOperation.getComment(), subOperation.getQuantity(),
                              subOperation.getID() );
    }

    onQuantityChanged();
}

void SKGOperationPluginWidget::displaySubOperations()
{
    SKGTRACEIN ( 10, "SKGOperationPluginWidget::displaySubOperations" );
    SKGOperationObject operation;
    if ( getSelectedOperation ( operation ).isSucceeded() ) displaySubOperations ( operation );
}

double SKGOperationPluginWidget::getRemainingQuantity()
{
    SKGTRACEIN ( 10, "SKGOperationPluginWidget::getRemainingQuantity" );
    double sumQuantities = 0;
    int nbSubOperations = ui.kSubOperationsTable->rowCount();

    for ( int i = 0; i < nbSubOperations ; ++i )
    {
        QTableWidgetItem* quantityItem = ui.kSubOperationsTable->item ( i,2 );
        sumQuantities = sumQuantities + SKGServices::stringToDouble ( quantityItem->text() );
    }

    return ui.kAmountEdit->value() - sumQuantities;
}

void SKGOperationPluginWidget::onQuantityChanged()
{
    SKGTRACEIN ( 10, "SKGOperationPluginWidget::onQuantityChanged" );
    int nbSubOperations = ui.kSubOperationsTable->rowCount();

    QTableWidgetItem* remainingQuantityItem = ui.kSubOperationsTable->item ( nbSubOperations-1,2 );
    if ( remainingQuantityItem )
    {
        bool previous=ui.kSubOperationsTable->blockSignals ( true ); // Disable signals so that filling cell doesn't create new lines
        remainingQuantityItem->setText ( SKGServices::doubleToString ( SKGServices::stringToDouble ( remainingQuantityItem->text() ) +getRemainingQuantity() ) );
        ui.kSubOperationsTable->blockSignals ( previous ); // Reenable signals
    }
}

void SKGOperationPluginWidget::onSubopCellChanged ( int row, int column )
{
    SKGTRACEIN ( 10, "SKGOperationPluginWidget::onSubopCellChanged" );
    QTableWidgetItem* subop_cell = ui.kSubOperationsTable->item ( row,column );
    QBrush base_brush = ui.kSubOperationsTable->item ( row,0 )->foreground();

    int nbSubOperations = ui.kSubOperationsTable->rowCount();
    if ( row == nbSubOperations-1 && column == 2 )
    {
        // If the quantity in the last line is edited, we add a new
        // line with the new remaining quantity
        addSubOperationLine ( nbSubOperations,"","","",0 );
    }
    if ( column == 2 )
    {

        if ( subop_cell->text().toDouble() != 0 )
        {
            onQuantityChanged();
        }
        else
        {
            base_brush.setColor ( Qt::red );
        }
        subop_cell->setForeground ( base_brush );
    }



}

void SKGOperationPluginWidget::onRemoveSubOperation()
{
    SKGTRACEIN ( 10, "SKGOperationPluginWidget::onRemoveSubOperation" );
    QList<int> rowsToRemove;
    QList<QTableWidgetItem*> selectedItems = ui.kSubOperationsTable->selectedItems();
    int nb=selectedItems.count();
    for ( int i = 0; i < nb; ++i )
    {
        QTableWidgetItem* item = selectedItems.at ( i );
        int row = item->row();
        if ( !rowsToRemove.contains ( row ) )
        {
            rowsToRemove.append ( row );
        }
    }
    for ( int j = rowsToRemove.count()-1; j >=0 ; --j )
    {
        ui.kSubOperationsTable->removeRow ( rowsToRemove.at ( j ) );
    }

    // If all rows removed, add an empty line
    if ( ui.kSubOperationsTable->rowCount() == 0 )
    {
        addSubOperationLine ( 0,"","","",0 );
    }

    onQuantityChanged();
}

void SKGOperationPluginWidget::onRotateAccountTools()
{
    SKGTRACEIN ( 10, "SKGOperationPluginWidget::onRotateAccountTools" );
    if ( modeInfoZone==0 )
    {
        //Switch to reconciliation mode
        ui.kReconciliatorFrame2->show();
        ui.kInfo->hide();
        modeInfoZone=1;
    }
    else if ( modeInfoZone==1 )
    {
        //Switch to information on selection mode
        ui.kReconciliatorFrame2->hide();
        ui.kInfo->show();
        modeInfoZone=2;
    }
    else
    {
        //Switch to information mode
        ui.kReconciliatorFrame2->hide();
        ui.kInfo->show();
        modeInfoZone=0;
    }
    onRefreshInformationZone();
}

void SKGOperationPluginWidget::onValidatePointedOperations()
{
    SKGError err;
    SKGTRACEINRC ( 10, "SKGOperationPluginWidget::onValidatePointedOperations",err );

    QApplication::setOverrideCursor ( QCursor ( Qt::WaitCursor ) );
    QString account=ui.kDisplayAccountCombo->currentText();
    SKGObjectBase::SKGListSKGObjectBase list;
    err=SKGObjectBase::getObjects ( getDocument(), "v_operation_display", "t_status='P' AND t_ACCOUNT='"+SKGServices::stringToSqlString ( account ) +'\'', list );
    int nb=list.count();
    if ( err.isSucceeded() )
    {
        SKGBEGINPROGRESSTRANSACTION ( *getDocument(), i18n ( "Switch to checked" ), err, nb );
        for ( int i=0; err.isSucceeded() && i<nb; ++i )
        {
            SKGOperationObject op=list[i];
            err=op.setStatus ( SKGOperationObject::CHECKED );
            if ( err.isSucceeded() ) err=op.save();
            if ( err.isSucceeded() ) err=getDocument()->stepForward ( i+1 );
        }
    }
    //status bar
    if ( err.isSucceeded() ) err=SKGError ( 0, i18n ( "Operation checked." ) );
    else err.addError ( ERR_FAIL, i18n ( "Switch failed" ) );
    QApplication::restoreOverrideCursor();

    //Display error
    SKGMainPanel::getMainPanel()->displayErrorMessage ( err );
}

void SKGOperationPluginWidget::addSubOperationLine ( int row, const QString& category, const QString& tracker, const QString& comment, double quantity, int id )
{
    SKGTRACEIN ( 10, "SKGOperationPluginWidget::addSubOperationLine" );
    bool previous=ui.kSubOperationsTable->blockSignals ( true );

    ui.kSubOperationsTable->insertRow ( row );

    // Category
    QTableWidgetItem* categoryItem = new QTableWidgetItem ( category );
    categoryItem->setData ( Qt::UserRole, id );
    ui.kSubOperationsTable->setItem ( row,0,categoryItem );

    // Comment
    ui.kSubOperationsTable->setItem ( row,1,new QTableWidgetItem ( comment ) );

    // Quantity
    QTableWidgetItem* quantityItem = new QTableWidgetItem ( SKGServices::doubleToString ( quantity ) );
    ui.kSubOperationsTable->setItem ( row,2,quantityItem );
    quantityItem->setTextAlignment ( Qt::AlignRight );

    // Refund
    QTableWidgetItem* trackerItem = new QTableWidgetItem ( tracker );
    categoryItem->setData ( Qt::UserRole, id );
    ui.kSubOperationsTable->setItem ( row,3,trackerItem );

    ui.kSubOperationsTable->blockSignals ( previous );

    ui.kSubOperationsTable->resizeColumnsToContents();
}

QWidget* SKGOperationPluginWidget::getWidgetForPrint()
{
    return ui.kOperationView;
}

SKGError SKGOperationPluginWidget::getSelectedOperation ( SKGOperationObject& operation )
{
    SKGError err;
    SKGObjectBase::SKGListSKGObjectBase selectedOperations = getSelectedObjects();
    if ( selectedOperations.count() > 0 )
    {
        operation = selectedOperations.at ( 0 );
        err.setReturnCode ( 0 );
    }
    else
    {
        err.setReturnCode ( 1 );
        err.setMessage ( i18n ( "No Operation Selected" ) );
    }
    return err;
}

void SKGOperationPluginWidget::cleanEditor()
{
    if ( getNbSelectedObjects() ==0 || sender()==ui.kCleanBtn )
    {
        ui.kDateEdit->setDate ( QDate::currentDate() );
        ui.kPayeeEdit->setText ( "" );
        ui.kCategoryEdit->setText ( "" );
        ui.kTrackerEdit->setText ( "" );
        ui.kAmountEdit->setText ( "" );
        ui.kTypeEdit->setText ( "" );
        ui.kCommentEdit->setText ( "" );
        ui.kNumberEdit->setText ( "" );

	setAllWidgetsEnabled();
    }
    if (sender()==ui.kCleanBtn) setCurrentMode(0);
}
#include "skgoperationpluginwidget.moc"
