/***************************************************************************
 *   Copyright (C) 2008 by S. MANKOWSKI / G. DE BURE skrooge@mankowski.fr  *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>  *
 ***************************************************************************/
/** @file
 * This file is Skrooge plugin for operation management.
 *
 * @author Stephane MANKOWSKI / Guillaume DE BURE
 */
#include "skgoperationplugin.h"
#include "skgoperationpluginwidget.h"
#include "skgoperationboardwidget.h"
#include "skgbookmarkboardwidget.h"
#include "skgtraces.h"
#include "skgoperationobject.h"
#include "skgsuboperationobject.h"
#include "skgtransactionmng.h"
#include "skgmainpanel.h"
#include "skgtableview.h"
#include "skgoperation_settings.h"

#include <kactioncollection.h>
#include <kstandardaction.h>

#include <QDomDocument>

/**
 * This plugin factory.
 */
K_PLUGIN_FACTORY ( SKGOperationPluginFactory, registerPlugin<SKGOperationPlugin>(); )
/**
 * This plugin export.
 */
K_EXPORT_PLUGIN ( SKGOperationPluginFactory ( "skrooge_operation", "skrooge_operation" ) )

SKGOperationPlugin::SKGOperationPlugin ( QObject* iParent, const QVariantList& /*iArg*/ ) : SKGInterfacePlugin ( iParent )
{
        SKGTRACEIN ( 10, "SKGOperationPlugin::SKGOperationPlugin" );
}

SKGOperationPlugin::~SKGOperationPlugin()
{
        SKGTRACEIN ( 10, "SKGOperationPlugin::~SKGOperationPlugin" );
        currentBankDocument=NULL;

        duplicateAction=NULL;
        switchToPointedAction=NULL;
        switchHighLight=NULL;
        openHighLights=NULL;
        openLastModified=NULL;
        groupOperation=NULL;
        ungroupOperation=NULL;
}

void SKGOperationPlugin::setupActions ( SKGDocument* iDocument, const QStringList& iArgument )
{
        SKGTRACEIN ( 10, "SKGOperationPlugin::setupActions" );
        Q_UNUSED ( iArgument );

        currentBankDocument=iDocument        ;


        setComponentData ( SKGOperationPluginFactory::componentData() );
        setXMLFile ( "skrooge_operation.rc" );

        //Menu
        duplicateAction = new KAction ( KIcon ( "skrooge_duplicate" ), i18n ( "Duplicate" ), this );
        connect ( duplicateAction, SIGNAL ( triggered ( bool ) ), this, SLOT ( actionDuplicate() ) );
        actionCollection()->addAction ( QLatin1String ( "edit_duplicate_operation" ), duplicateAction );
        duplicateAction->setShortcut ( Qt::CTRL+Qt::Key_D );

        if (SKGMainPanel::getMainPanel()) SKGMainPanel::getMainPanel()->registedGlobalAction ( "edit_duplicate_operation", duplicateAction );

        createTemplateAction = new KAction ( KIcon ( "skrooge_template" ), i18n ( "Create template" ), this );
        connect ( createTemplateAction, SIGNAL ( triggered ( bool ) ), this, SLOT ( actionCreateTemplate() ) );
        actionCollection()->addAction ( QLatin1String ( "edit_template_operation" ), createTemplateAction );
        createTemplateAction->setShortcut ( Qt::CTRL+Qt::SHIFT+Qt::Key_T );

        if (SKGMainPanel::getMainPanel()) SKGMainPanel::getMainPanel()->registedGlobalAction ( "edit_template_operation", createTemplateAction );

        switchToPointedAction = new KAction ( KIcon ( "dialog-ok" ), i18n ( "Point" ), this );
        connect ( switchToPointedAction, SIGNAL ( triggered ( bool ) ), this, SLOT ( actionSwitchToPointed() ) );
        actionCollection()->addAction ( QLatin1String ( "edit_point_selected_operation" ), switchToPointedAction );
        switchToPointedAction->setShortcut ( Qt::CTRL+Qt::Key_R );

        if (SKGMainPanel::getMainPanel()) SKGMainPanel::getMainPanel()->registedGlobalAction ( "edit_point_selected_operation", switchToPointedAction );

        KAction* fastEditionAction = new KAction ( KIcon ( "games-solve" ), i18n ( "Fast edit" ), this );
        actionCollection()->addAction ( QLatin1String ( "fast_edition" ), fastEditionAction );
        fastEditionAction->setEnabled ( false );
        fastEditionAction->setShortcut ( Qt::Key_F10 );

        if (SKGMainPanel::getMainPanel()) SKGMainPanel::getMainPanel()->registedGlobalAction ( "fast_edition", fastEditionAction );

        switchHighLight = new KAction ( KIcon ( "rating" ), i18n ( "Switch highlight" ), this );
        connect ( switchHighLight, SIGNAL ( triggered ( bool ) ), this, SLOT ( actionSwitchHighLight() ) );
        actionCollection()->addAction ( QLatin1String ( "edit_switch_highlight" ), switchHighLight );
        switchHighLight->setShortcut ( Qt::CTRL+Qt::Key_H );

        if (SKGMainPanel::getMainPanel()) SKGMainPanel::getMainPanel()->registedGlobalAction ( "edit_switch_highlight", switchHighLight );

        QStringList overlayopen;
        overlayopen.push_back("skg_open");
        openHighLights = new KAction(KIcon( "rating", NULL, overlayopen), i18n ( "Open highlights..." ), this );
        connect ( openHighLights, SIGNAL ( triggered ( bool ) ), this, SLOT ( actionOpenHighLights() ) );
        actionCollection()->addAction ( QLatin1String ( "view_open_highlight" ), openHighLights );
        openHighLights->setShortcut ( Qt::META+Qt::Key_H );

        if (SKGMainPanel::getMainPanel()) SKGMainPanel::getMainPanel()->registedGlobalAction ( "view_open_highlight", openHighLights );

        openLastModified = new KAction ( KIcon("view-refresh", NULL, overlayopen), i18n ( "Open last modified..." ), this);
        connect ( openLastModified, SIGNAL ( triggered ( bool ) ), this, SLOT ( actionOpenLastModified() ) );
        actionCollection()->addAction ( QLatin1String ( "view_open_last_modified" ), openLastModified );
        openLastModified->setShortcut ( Qt::META+Qt::Key_L );

        if (SKGMainPanel::getMainPanel()) SKGMainPanel::getMainPanel()->registedGlobalAction ( "view_open_last_modified", openLastModified );

        openSuboperations = new KAction ( KIcon("skrooge_split", NULL, overlayopen), i18n ( "Open sub operations..." ), this);
        connect ( openSuboperations, SIGNAL ( triggered ( bool ) ), this, SLOT ( actionOpenSubOperations() ) );
        actionCollection()->addAction ( QLatin1String ( "view_open_suboperations" ), openSuboperations );
        openSuboperations->setShortcut ( Qt::META+Qt::Key_S );

        if (SKGMainPanel::getMainPanel()) SKGMainPanel::getMainPanel()->registedGlobalAction ( "view_open_suboperations", openSuboperations );

        groupOperation = new KAction ( KIcon ( "skrooge_transfer" ), i18n ( "Group operations" ), this );
        connect ( groupOperation, SIGNAL ( triggered ( bool ) ), this, SLOT ( actionGroupOperation() ) );
        actionCollection()->addAction ( QLatin1String ( "edit_group_operation" ), groupOperation );
        groupOperation->setShortcut ( Qt::CTRL+Qt::Key_G );

        if (SKGMainPanel::getMainPanel()) SKGMainPanel::getMainPanel()->registedGlobalAction ( "edit_group_operation", groupOperation );

        QStringList overlay;
        overlay.push_back("edit-delete");
        ungroupOperation = new KAction ( KIcon ( "skrooge_transfer", NULL, overlay ), i18n ( "Ungroup operations" ), this );
        connect ( ungroupOperation, SIGNAL ( triggered ( bool ) ), this, SLOT ( actionUngroupOperation() ) );
        actionCollection()->addAction ( QLatin1String ( "edit_ungroup_operation" ), ungroupOperation );
        ungroupOperation->setShortcut ( Qt::CTRL+Qt::SHIFT+Qt::Key_G );

        if (SKGMainPanel::getMainPanel()) SKGMainPanel::getMainPanel()->registedGlobalAction ( "edit_ungroup_operation", ungroupOperation );
}

void SKGOperationPlugin::refresh()
{
        SKGTRACEIN ( 10, "SKGOperationPlugin::refresh" );
        if ( currentBankDocument && SKGMainPanel::getMainPanel() ) {
                bool test=(currentBankDocument->getDatabase()!=NULL);
                if ( openHighLights ) openHighLights->setEnabled ( test );
                if ( openLastModified ) openLastModified->setEnabled ( test && currentBankDocument->getNbTransaction()>0 );

                SKGObjectBase::SKGListSKGObjectBase selection=SKGMainPanel::getMainPanel()->getSelectedObjects();
                if ( selection.count() >0 ) {
                        bool onOperation= ( selection.at ( 0 ).getRealTable() =="operation" );
                        if ( duplicateAction ) duplicateAction->setEnabled ( onOperation );
                        if ( createTemplateAction ) createTemplateAction->setEnabled ( onOperation );
                        if ( switchHighLight ) switchHighLight->setEnabled ( onOperation );
                        if ( switchToPointedAction ) switchToPointedAction->setEnabled ( onOperation );
                        if ( groupOperation ) groupOperation->setEnabled ( onOperation && selection.count()>=2);
                        if ( ungroupOperation ) ungroupOperation->setEnabled ( onOperation );
                } else {
                        if ( duplicateAction ) duplicateAction->setEnabled ( false );
                        if ( createTemplateAction ) createTemplateAction->setEnabled ( false );
                        if ( switchHighLight ) switchHighLight->setEnabled ( false );
                        if ( switchToPointedAction ) switchToPointedAction->setEnabled ( false );
                        if ( groupOperation ) groupOperation->setEnabled ( false );
                        if ( ungroupOperation ) ungroupOperation->setEnabled ( false );
                }
        }
}

void SKGOperationPlugin::close()
{
        SKGTRACEIN ( 10, "SKGOperationPlugin::close" );
}

int SKGOperationPlugin::getNbDashboardWidgets()
{
        return 2;
}

QString SKGOperationPlugin::getDashboardWidgetTitle(int iIndex)
{
        if (iIndex==0) return i18n("Income && Expenditure");
        else if (iIndex==1) return i18n("Highlighted operations");
        return "";
}

SKGWidget* SKGOperationPlugin::getDashboardWidget(int iIndex)
{
        if (iIndex==0) return new SKGOperationBoardWidget(currentBankDocument);
        else if (iIndex==1) return new SKGBookmarkBoardWidget(currentBankDocument);
        return NULL;
}

SKGTabPage* SKGOperationPlugin::getWidget()
{
        SKGTRACEIN ( 10, "SKGOperationPlugin::getWidget" );
        return new SKGOperationPluginWidget(( SKGDocumentBank* ) currentBankDocument);
}

QWidget* SKGOperationPlugin::getPreferenceWidget()
{
        SKGTRACEIN ( 10, "SKGMonthlyPlugin::getPreferenceWidget" );
        QWidget* widget=new QWidget();
        ui.setupUi ( widget );
        return widget;
}

KConfigSkeleton* SKGOperationPlugin::getPreferenceSkeleton()
{
        return skgoperation_settings::self();
}

QString SKGOperationPlugin::title() const
{
        return i18n ( "Operations" );
}

QString SKGOperationPlugin::icon() const
{
        return "view-pim-tasks";
}

QString SKGOperationPlugin::toolTip () const
{
        return i18n ( "Operation management" );
}


int SKGOperationPlugin::getOrder() const
{
        return 15;
}

QStringList SKGOperationPlugin::tips() const
{
        QStringList output;
        output.push_back ( i18n ( "<p>... you can press <strong>+</strong>, <strong>-</strong>, <strong>CTRL +</strong> or <strong>CTRL -</strong> to quickly change dates.</p>" ) );
        output.push_back ( i18n ( "<p>... you can update many operations in one shot.</p>" ) );
        output.push_back ( i18n ( "<p>... you can double click on an operation to show or edit sub operations.</p>" ) );
        output.push_back ( i18n ( "<p>... you can duplicate an operation including complex operations (split, grouped, ...).</p>" ) );
        output.push_back ( i18n ( "<p>... you can create template of operations.</p>" ) );
        output.push_back ( i18n ( "<p>... you can group and ungroup operations.</p>" ) );
        return output;
}

bool SKGOperationPlugin::isInContext() const
{
        return true;
}

void SKGOperationPlugin::actionGroupOperation()
{
        SKGError err;
        SKGTRACEINRC ( 10, "SKGOperationPlugin::actionGroupOperation",err );
        //Get Selection
        if ( SKGMainPanel::getMainPanel() && currentBankDocument ) {
                SKGObjectBase::SKGListSKGObjectBase selection=SKGMainPanel::getMainPanel()->getSelectedObjects();
                int nb=selection.count();
                if (nb>=2) {
                        SKGBEGINPROGRESSTRANSACTION ( *currentBankDocument, i18n ( "Group operations" ), err, nb );
                        QApplication::setOverrideCursor ( QCursor ( Qt::WaitCursor ) );
                        SKGOperationObject main=selection.at(0);
                        if ( err.isSucceeded() ) err=currentBankDocument->stepForward(1);
                        for ( int i=1; err.isSucceeded() && i<nb; ++i ) {
                                SKGOperationObject operationObj=selection.at(i);
                                if ( err.isSucceeded() ) err=operationObj.setGroupOperation(main);
                                if ( err.isSucceeded() ) err=operationObj.save();

                                if ( err.isSucceeded() ) err=currentBankDocument->stepForward ( i+1 );
                        }
                        QApplication::restoreOverrideCursor();
                }

                //status bar
                if ( err.isSucceeded() ) err=SKGError ( 0, i18n ( "Operations grouped." ) );
                else err.addError ( ERR_FAIL, i18n ( "Group creation failed" ) );

                //Display error
                SKGMainPanel::getMainPanel()->displayErrorMessage ( err );
        }
}

void SKGOperationPlugin::actionUngroupOperation()
{
        SKGError err;
        SKGTRACEINRC ( 10, "SKGOperationPlugin::actionUngroupOperation",err );
        //Get Selection
        if ( SKGMainPanel::getMainPanel() && currentBankDocument ) {
                SKGObjectBase::SKGListSKGObjectBase selection=SKGMainPanel::getMainPanel()->getSelectedObjects();
                int nb=selection.count();
                {
                        SKGBEGINPROGRESSTRANSACTION ( *currentBankDocument, i18n ( "Ungroup operation" ), err, nb );

                        QApplication::setOverrideCursor ( QCursor ( Qt::WaitCursor ) );
                        for ( int i=0; err.isSucceeded() && i<nb; ++i ) {
                                SKGOperationObject operationObj=selection.at ( i );
                                if ( err.isSucceeded() ) err=operationObj.setGroupOperation(operationObj);
                                if ( err.isSucceeded() ) err=operationObj.save();

                                if ( err.isSucceeded() ) err=currentBankDocument->stepForward ( i+1 );
                        }
                        QApplication::restoreOverrideCursor();
                }

                //status bar
                if ( err.isSucceeded() ) err=SKGError ( 0, i18n ( "Operation ungrouped." ) );
                else err.addError ( ERR_FAIL, i18n ( "Group deletion failed" ) );

                //Display error
                SKGMainPanel::getMainPanel()->displayErrorMessage ( err );
        }
}

void SKGOperationPlugin::actionSwitchToPointed()
{
        SKGError err;
        SKGTRACEINRC ( 10, "SKGOperationPlugin::actionSwitchToPointed",err );
        //Get Selection
        if ( SKGMainPanel::getMainPanel() && currentBankDocument ) {
                SKGObjectBase::SKGListSKGObjectBase selection=SKGMainPanel::getMainPanel()->getSelectedObjects();
                int nb=selection.count();
                {
                        SKGBEGINPROGRESSTRANSACTION ( *currentBankDocument, i18n ( "Switch to pointed" ), err, nb );

                        QApplication::setOverrideCursor ( QCursor ( Qt::WaitCursor ) );
                        for ( int i=0; err.isSucceeded() && i<nb; ++i ) {
                                SKGOperationObject operationObj=selection.at ( i );
                                if ( err.isSucceeded() ) err=operationObj.setStatus ( SKGOperationObject::POINTED );
                                if ( err.isSucceeded() ) err=operationObj.save();

                                if ( err.isSucceeded() ) err=currentBankDocument->stepForward ( i+1 );
                        }
                        QApplication::restoreOverrideCursor();
                }

                //status bar
                if ( err.isSucceeded() ) err=SKGError ( 0, i18n ( "Operation pointed." ) );
                else err.addError ( ERR_FAIL, i18n ( "Switch failed" ) );

                //Display error
                SKGMainPanel::getMainPanel()->displayErrorMessage ( err );
        }
}

void SKGOperationPlugin::actionSwitchHighLight()
{
        SKGError err;
        SKGTRACEINRC ( 10, "SKGOperationPlugin::actionSwitchHighLight",err );
        //Get Selection
        if ( SKGMainPanel::getMainPanel() && currentBankDocument ) {
                SKGObjectBase::SKGListSKGObjectBase selection=SKGMainPanel::getMainPanel()->getSelectedObjects();
                int nb=selection.count();
                {
                        SKGBEGINPROGRESSTRANSACTION ( *currentBankDocument, i18n ( "Operation highlight" ), err, nb );

                        QApplication::setOverrideCursor ( QCursor ( Qt::WaitCursor ) );
                        for ( int i=0; err.isSucceeded() && i<nb; ++i ) {
                                SKGOperationObject operationObj=selection.at ( i );
                                if ( err.isSucceeded() ) err=operationObj.bookmark ( !operationObj.isBookmarked() );
                                if ( err.isSucceeded() ) err=operationObj.save();

                                if ( err.isSucceeded() ) err=currentBankDocument->stepForward ( i+1 );
                        }
                        QApplication::restoreOverrideCursor();
                }

                //status bar
                if ( err.isSucceeded() )  err=SKGError ( 0, i18n ( "Operation highlighted." ) );
                else err.addError ( ERR_FAIL, i18n ( "Operation highlight failed" ) );

                //Display error
                SKGMainPanel::getMainPanel()->displayErrorMessage ( err );
        }
}

void SKGOperationPlugin::actionDuplicate()
{
        SKGError err;
        SKGTRACEINRC ( 10, "SKGOperationPlugin::actionDuplicate",err );
        //Get Selection
        if ( SKGMainPanel::getMainPanel() && currentBankDocument ) {
                QStringList listUUID;
                SKGObjectBase::SKGListSKGObjectBase selection=SKGMainPanel::getMainPanel()->getSelectedObjects();
                int nb=selection.count();
                {
                        SKGBEGINPROGRESSTRANSACTION ( *currentBankDocument, i18n ( "Duplicate operation" ), err, nb );

                        QApplication::setOverrideCursor ( QCursor ( Qt::WaitCursor ) );
                        for ( int i=0; err.isSucceeded() && i<nb; ++i ) {
                                SKGOperationObject operationObj=selection.at ( i );
                                SKGOperationObject dup;
                                if ( err.isSucceeded() ) err=operationObj.duplicate ( dup );
                                if ( err.isSucceeded() ) err=currentBankDocument->stepForward ( i+1 );

                                listUUID.push_back ( dup.getUniqueID() );
                        }
                        QApplication::restoreOverrideCursor();
                }

                //status bar
                if ( err.isSucceeded() ) {
                        err=SKGError ( 0, i18n ( "Operation duplicated." ) );
                        SKGOperationPluginWidget* w=dynamic_cast<SKGOperationPluginWidget*>(SKGMainPanel::getMainPanel()->currentTab());
                        if (w) w->getTableView()->selectObjects(listUUID, true);
                } else err.addError ( ERR_FAIL, i18n ( "Duplicate operation failed" ) );

                //Display error
                SKGMainPanel::getMainPanel()->displayErrorMessage ( err );
        }
}

void SKGOperationPlugin::actionCreateTemplate()
{
        SKGError err;
        SKGTRACEINRC ( 10, "SKGOperationPlugin::actionCreateTemplate",err );
        //Get Selection
        if ( SKGMainPanel::getMainPanel() && currentBankDocument ) {
                QStringList listUUID;
                SKGObjectBase::SKGListSKGObjectBase selection=SKGMainPanel::getMainPanel()->getSelectedObjects();
                int nb=selection.count();
                {
                        SKGBEGINPROGRESSTRANSACTION ( *currentBankDocument, i18n ( "Create template" ), err, nb );

                        QApplication::setOverrideCursor ( QCursor ( Qt::WaitCursor ) );
                        for ( int i=0; err.isSucceeded() && i<nb; ++i ) {
                                SKGOperationObject operationObj=selection.at ( i );
                                SKGOperationObject dup;
                                if ( err.isSucceeded() ) err=operationObj.duplicate ( dup , QDate::currentDate(), true );
                                if ( err.isSucceeded() ) err=currentBankDocument->stepForward ( i+1 );

                                listUUID.push_back ( dup.getUniqueID() );
                        }
                        QApplication::restoreOverrideCursor();
                }

                //status bar
                if ( err.isSucceeded() ) {
                        err=SKGError ( 0, i18n ( "Template created." ) );
                        SKGOperationPluginWidget* w=dynamic_cast<SKGOperationPluginWidget*>(SKGMainPanel::getMainPanel()->currentTab());
                        if (w) {
                                w->setTemplateMode(true);
                                w->getTableView()->selectObjects(listUUID, true);
                        }
                } else err.addError ( ERR_FAIL, i18n ( "Creation template failed" ) );

                //Display error
                SKGMainPanel::getMainPanel()->displayErrorMessage ( err );
        }
}

void SKGOperationPlugin::actionOpenHighLights()
{
        SKGTRACEIN ( 10, "SKGOperationPlugin::actionOpenHighLights");
        QString wc="t_bookmarked='Y'";
        QString title=i18n ( "Highlighted operations" );

        //Call operation plugin
        QDomDocument doc("SKGML");
        doc.setContent ( currentBankDocument->getParameter("SKGOPERATION_DEFAULT_PARAMETERS") );
        QDomElement root = doc.documentElement();
        if (root.isNull()) {
                root=doc.createElement("parameters");
                doc.appendChild(root);
        }

        root.setAttribute ( "account", i18n("All") );
        root.setAttribute ( "operationTable", "v_operation_display" );
        root.setAttribute ( "operationWhereClause", wc );
        root.setAttribute ( "title", title );
        root.setAttribute ( "title_icon", "rating" );

        if (SKGMainPanel::getMainPanel()) SKGMainPanel::getMainPanel()->setNewTabContent ( SKGMainPanel::getMainPanel()->getPluginByName ( "Skrooge operation plugin" ), -1, doc.toString() );
}

void SKGOperationPlugin::actionOpenLastModified()
{
        SKGTRACEIN ( 10, "SKGOperationPlugin::actionOpenLastModified");
        QString wc="id in (SELECT i_object_id FROM doctransactionitem di, doctransaction dt WHERE dt.t_mode='U' AND dt.id=di.rd_doctransaction_id AND di.t_object_table='operation'AND (select count(1) from doctransaction B where B.i_parent=dt.id)=0)";
        QString title=i18n ( "Operations modified or created during last action" );

        //Call operation plugin
        QDomDocument doc("SKGML");
        doc.setContent ( currentBankDocument->getParameter("SKGOPERATION_DEFAULT_PARAMETERS") );
        QDomElement root = doc.documentElement();
        if (root.isNull()) {
                root=doc.createElement("parameters");
                doc.appendChild(root);
        }

        root.setAttribute ( "account", i18n("All") );
	root.setAttribute ( "operationTable", "v_operation_display" );
        root.setAttribute ( "operationWhereClause", wc );
        root.setAttribute ( "title", title );
        root.setAttribute ( "title_icon", "rating" );

        if (SKGMainPanel::getMainPanel()) SKGMainPanel::getMainPanel()->setNewTabContent ( SKGMainPanel::getMainPanel()->getPluginByName ( "Skrooge operation plugin" ), -1, doc.toString() );
}

void SKGOperationPlugin::actionOpenSubOperations()
{
        SKGTRACEIN ( 10, "SKGOperationPlugin::actionOpenLastModified");
        //Call operation plugin
        QDomDocument doc("SKGML");
        doc.setContent ( currentBankDocument->getParameter("SKGOPERATION_CONSOLIDATED_DEFAULT_PARAMETERS") );
        QDomElement root = doc.documentElement();
        if (root.isNull()) {
                root=doc.createElement("parameters");
                doc.appendChild(root);
        }

        root.setAttribute ( "account", i18n("All") );
	root.setAttribute ( "operationTable", "v_operation_consolidated" );
        root.setAttribute ( "operationWhereClause", "" );
        root.setAttribute ( "title", i18n("Sub operations") );
        root.setAttribute ( "title_icon", icon() );

        if (SKGMainPanel::getMainPanel()) SKGMainPanel::getMainPanel()->setNewTabContent ( SKGMainPanel::getMainPanel()->getPluginByName ( "Skrooge operation plugin" ), -1, doc.toString() );
}
#include "skgoperationplugin.moc"
