/***************************************************************************
 *   Copyright (C) 2008 by S. MANKOWSKI / G. DE BURE skrooge@mankowski.fr  *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>  *
 ***************************************************************************/
/** @file
 * A table with graph with more features.
 *
 * @author Stephane MANKOWSKI / Guillaume DE BURE
 */
#include "skgtablewithgraph.h"
#include "skgservices.h"
#include "skgtraces.h"
#include "skgmainpanel.h"
#include "skggraphicsscene.h"

#include <kmenu.h>
#include <kfiledialog.h>
#include <kstringhandler.h>

#include <QtGui/QHeaderView>
#include <QtGui/QGraphicsLineItem>
#include <QDomDocument>
#include <QRegExp>
#include <QPrinter>
#include <QPainter>
#include <QTableWidgetItem>
#include <QDesktopServices>
#include <QTimer>

#include <math.h>

/**
  * Data identifier for value
  */
#define DATA_VALUE 12
/**
  * Data identifier for color
  */
#define DATA_COLOR_H 11
/**
  * Data identifier for color
  */
#define DATA_COLOR_S 12
/**
  * Data identifier for color
  */
#define DATA_COLOR_V 13
/**
  * Data identifier for color
  */
#define DATA_COLOR_A 14
/**
  * Box size
  */
#define BOX_SIZE 120
/**
  * Box margin
  */
#define BOX_MARGIN 10

Qt::SortOrder SKGTableWithGraph::sortOrder=Qt::AscendingOrder;
int SKGTableWithGraph::sortColumn=0;

SKGTableWithGraph::SKGTableWithGraph ( QWidget *parent )
        : QWidget ( parent ), scene ( NULL ), nbLevel ( 0 ), showSumAndAverage ( true ), selectable(true), mainMenu ( NULL )
{
    ui.setupUi ( this );
    ui.kLineUp->setIcon ( KIcon ( "arrow-right" ) );
    ui.kLineDown->setIcon ( KIcon ( "arrow-left" ) );

    timer = new QTimer ( this );
    timer->setSingleShot ( true );
    connect ( timer, SIGNAL ( timeout() ), this, SLOT ( refresh() ) );

    //Build contextual menu
    ui.kTable->setContextMenuPolicy ( Qt::CustomContextMenu );
    connect ( ui.kTable, SIGNAL ( customContextMenuRequested ( const QPoint & ) ),this,SLOT ( showMenu ( const QPoint& ) ) );

    mainMenu = new KMenu ( ui.kTable );

    QMenu* exp=mainMenu->addMenu ( i18nc ( "Verb, action to export items in another format","Export" ) );

    QAction* actCSV = exp->addAction ( KIcon ( "text-csv" ), i18n ( "Export CSV..." ) );
    connect ( actCSV, SIGNAL ( triggered ( bool ) ), this, SLOT ( onExportCSV() ) );

    QAction* actTXT = exp->addAction ( KIcon ( "text-plain" ), i18n ( "Export TXT..." ) );
    connect ( actTXT, SIGNAL ( triggered ( bool ) ), this, SLOT ( onExportTXT() ) );

    //Set headers parameters
    QHeaderView* verticalHeader=ui.kTable->verticalHeader();
    if ( verticalHeader )  verticalHeader->hide();

    ui.kTable->setSortingEnabled ( false ); //sort must be enable for refresh method
    QHeaderView* horizontalHeader=ui.kTable->horizontalHeader();
    if ( horizontalHeader )
    {
        horizontalHeader->setResizeMode ( QHeaderView::ResizeToContents );
        horizontalHeader->show();
        horizontalHeader->setSortIndicatorShown ( true );
        horizontalHeader->setSortIndicator ( sortColumn, sortOrder );
        connect ( horizontalHeader, SIGNAL ( sortIndicatorChanged ( int, Qt::SortOrder ) ), this, SLOT ( refresh() ) );
    }
    ui.kTable->verticalHeader()->setDefaultSectionSize ( ui.kTable->verticalHeader()->minimumSectionSize() );

    ui.kDisplayMode->addItem ( KIcon ( "skg-chart-bar-stacked" ), i18nc ( "Noun, a type of graph, with bars stacked upon each other","Stack" ) );
    ui.kDisplayMode->addItem ( KIcon ( "skg-chart-bar" ), i18nc ( "Noun, a type of graph, with bars placed besides each other","Histogram" ) );
    ui.kDisplayMode->addItem ( KIcon ( "skg-chart-pie" ), i18nc ( "Noun, a type of graph that looks like a sliced pie","Pie" ) );
    ui.kDisplayMode->addItem ( KIcon ( "skg-chart-ring" ), i18nc ( "Noun, a type of graph that looks like concentric slices of a pie (a la filelight)","Concentric pie" ) );
    ui.kDisplayMode->addItem ( KIcon ( "skg-chart-scatter" ), i18nc ( "Noun, a type of graph with only points","Point" ) );
    ui.kDisplayMode->addItem ( KIcon ( "skg-chart-line" ), i18nc ( "Noun, a type of graph with only lines","Line" ) );

    connect ( ui.kDisplayMode, SIGNAL ( currentIndexChanged ( int ) ), this, SLOT ( redrawGraph() ), Qt::QueuedConnection );
    connect ( ui.kAllPositive, SIGNAL ( stateChanged ( int ) ), this, SLOT ( redrawGraph() ), Qt::QueuedConnection );

}


SKGTableWithGraph::~SKGTableWithGraph()
{
    if ( scene )
    {
        delete scene;
        scene=NULL;
    }

    mainMenu=NULL;
    timer=NULL;
}

KMenu* SKGTableWithGraph::getTableContextualMenu() const
{
    return mainMenu;
}

KMenu* SKGTableWithGraph::getGraphContextualMenu() const
{
    return ui.graphicView->getContextualMenu();
}

void SKGTableWithGraph::showMenu ( const QPoint& pos )
{
    if ( mainMenu ) mainMenu->popup ( ui.kTable->mapToGlobal ( pos ) );
}

QTableWidget* SKGTableWithGraph::table() const
{
    return ui.kTable;
}

SKGStringListList SKGTableWithGraph::getTable()
{
    //Build table
    SKGStringListList table;

    //Get header names
    int nb=ui.kTable->columnCount();
    QStringList cols;
    for ( int i=0; i<nb; i++ )
    {
        cols.append ( ui.kTable->horizontalHeaderItem ( i )->text() );
    }
    table.append ( cols );

    //Get content
    int nb2=ui.kTable->rowCount();
    for ( int i=0; i<nb2; i++ )
    {
        QStringList row;
        for ( int j=0; j<nb; j++ )
        {
            row.append ( ui.kTable->item ( i,j )->text() );
        }
        table.append ( row );
    }
    return table;
}
QString SKGTableWithGraph::getState()
{
    SKGTRACEIN ( 10, "SKGTableWithGraph::getState" );
    QDomDocument doc ( "SKGML" );
    QDomElement root = doc.createElement ( "parameters" );
    doc.appendChild ( root );

    if(isGraphVisible() && isTableVisible()) root.setAttribute ( "splitterState", QString ( ui.splitter->saveState().toHex() ) );
    root.setAttribute ( "graphMode", SKGServices::intToString ( ( int ) getGraphType() ) );
    root.setAttribute ( "nbLevel", SKGServices::intToString ( nbLevel ) );
    root.setAttribute ( "allPositive", ui.kAllPositive->checkState() ==Qt::Checked ? "Y" : "N" );
    root.setAttribute ( "filter", ui.kFilterEdit->text() );

    QHeaderView* horizontalHeader=ui.kTable->horizontalHeader();
    root.setAttribute ( "sortOrder", SKGServices::intToString ( horizontalHeader->sortIndicatorOrder() ) );
    root.setAttribute ( "sortColumn", SKGServices::intToString ( horizontalHeader->sortIndicatorSection() ) );
    root.setAttribute ( "graphicViewState", ui.graphicView->getState() );

    //TODO: save graphic zoom and sroll position
    return doc.toString();
}

void SKGTableWithGraph::setState ( const QString& iState )
{
    SKGTRACEIN ( 10, "SKGTableWithGraph::setState" );
    QDomDocument doc ( "SKGML" );
    if ( doc.setContent ( iState ) )
    {
        QDomElement root = doc.documentElement();

        QString splitterState=root.attribute ( "splitterState" );
        if ( !splitterState.isEmpty() ) ui.splitter->restoreState ( QByteArray::fromHex ( splitterState.toAscii() ) );
        QString graphMode=root.attribute ( "graphMode" );
        if ( !graphMode.isEmpty() )
        {
            setGraphType ( SKGTableWithGraph::HISTOGRAM );
            ui.kDisplayMode->setCurrentIndex ( 1 );
            setGraphType ( ( SKGTableWithGraph::GraphType ) SKGServices::stringToInt ( graphMode ) );
        }
        QString nbLevelString=root.attribute ( "nbLevel" );
        if ( !nbLevelString.isEmpty() ) nbLevel=SKGServices::stringToInt ( nbLevelString );
        QString allPositive=root.attribute ( "allPositive" );
        if ( !allPositive.isEmpty() ) ui.kAllPositive->setCheckState ( ( allPositive=="Y" ? Qt::Checked : Qt::Unchecked ) );
        ui.kFilterEdit->setText ( root.attribute ( "filter" ) );

        QString sortOrder=root.attribute ( "sortOrder" );
        QString sortColumn=root.attribute ( "sortColumn" );
        if ( sortOrder.isEmpty() ) sortOrder='0';
        if ( sortColumn.isEmpty() ) sortColumn='0';
        ui.kTable->horizontalHeader()->setSortIndicator ( SKGServices::stringToInt ( sortColumn ), ( Qt::SortOrder ) SKGServices::stringToInt ( sortOrder ) );

        QString graphicViewState=root.attribute ( "graphicViewState" );
        if ( !graphicViewState.isEmpty() ) ui.graphicView->setState ( graphicViewState );
    }
    else
    {
        //vvv Correction bug 2278703:Zoom fit when opening report tab
        setGraphType ( SKGTableWithGraph::HISTOGRAM );
        setGraphType ( SKGTableWithGraph::STACK );
        //^^^ Correction bug 2278703:Zoom fit when opening report tab
    }
}

void SKGTableWithGraph::onOneLevelMore()
{
    _SKGTRACEIN ( 10, "SKGTableWithGraph::onOneLevelMore" );
    ++nbLevel;
    refresh();
}

void SKGTableWithGraph::onOneLevelLess()
{
    _SKGTRACEIN ( 10, "SKGTableWithGraph::onOneLevelLess" );
    --nbLevel;
    refresh();
}

void SKGTableWithGraph::onFilterModified()
{
    if ( timer ) timer->start ( 300 );
}

void SKGTableWithGraph::setData ( const SKGStringListList& iData, SKGServices::SKGUnitInfo iPrimaryUnit,
                                  SKGServices::SKGUnitInfo iSecondaryUnit,
                                  bool iShowSumAndAverage )
{
    SKGTRACEIN ( 10, "SKGTableWithGraph::setData" );
    data=iData;
    primaryUnit=iPrimaryUnit;
    secondaryUnit=iSecondaryUnit;
    showSumAndAverage=iShowSumAndAverage;

    refresh();

}
bool SKGTableWithGraph::listSort ( const QStringList &s1, const QStringList &s2 )
{
    if ( sortColumn>=s1.count() ) sortColumn=s1.count()-1;
    if ( sortColumn>=0 )
    {
        QString v1=s1.at ( sortColumn );
        QString v2=s2.at ( sortColumn );
        if ( sortColumn==0 )
        {
            int v=KStringHandler::naturalCompare ( v1, v2, Qt::CaseInsensitive );
            return ( sortOrder ? v>0 : v<0 );
        }

        double vd1=SKGServices::stringToDouble ( v1 );
        double vd2=SKGServices::stringToDouble ( v2 );
        return ( sortOrder ? vd1>vd2 : vd1<vd2 );
    }
    else return false;
}

void SKGTableWithGraph::refresh()
{
    SKGTRACEIN ( 10, "SKGTableWithGraph::refresh" );
    QApplication::setOverrideCursor ( QCursor ( Qt::WaitCursor ) );

    //Create grouped table
    int nbLevelMax=0;
    int nbCols=-1;
    SKGStringListList groupedTable=data;
    {
        QString previousLineName="###";
        if ( groupedTable.count() )
        {
            nbCols=groupedTable.at ( 0 ).count();
        }
        for ( int i=1; nbCols && i<groupedTable.count(); ++i ) //Dynamically modified
        {
            QStringList line=groupedTable.at ( i );
            QString val=line.at ( 0 );

            //Rebuild val for the number of level
            QStringList vals=val.split ( OBJECTSEPARATOR );
            int nbvals=vals.count();
            if ( nbvals>nbLevel+1 )
            {
                //Rebuild val
                val="";
                for ( int k=0; k<=nbLevel; ++k )
                {
                    val+=vals[k];
                    if ( k!=nbLevel ) val+=OBJECTSEPARATOR;
                }
            }
            nbLevelMax=qMax ( nbLevelMax, nbvals-1 );

            if ( val==previousLineName )
            {
                //Current line is merged with previous one
                QStringList newLine;
                newLine.push_back ( val );
                for ( int k=1; k<nbCols; ++k )
                {
                    QString valstring1=line.at ( k );
                    QString valstring2=groupedTable.at ( i-1 ).at ( k );

                    double sum2=0;
                    if ( !valstring1.isEmpty() || !valstring2.isEmpty() )
                    {
                        if ( !valstring1.isEmpty() ) sum2+=SKGServices::stringToDouble ( valstring1 );
                        if ( !valstring2.isEmpty() ) sum2+=SKGServices::stringToDouble ( valstring2 );
                        newLine.push_back ( SKGServices::doubleToString ( sum2 ) );
                    }
                    else
                    {
                        newLine.push_back ( "" );
                    }
                }

                groupedTable.replace ( i-1, newLine );

                //Remove current line
                groupedTable.removeAt ( i );
                --i;
            }
            else
            {
                //Current line is just modified
                QStringList newLine;
                newLine.push_back ( val );
                for ( int k=1; k<nbCols; ++k )
                {
                    newLine.push_back ( line.at ( k ) );
                }
                groupedTable.replace ( i, newLine );

                previousLineName=val;
            }
        }
    }

    //Create filtered table
    {
        QStringList words=SKGServices::splitCSVLine ( ui.kFilterEdit->text(), ' ', true );
	int nbwords=words.count();

        for ( int i=groupedTable.count()-1; i>=1; --i ) //The first line is not filtered because it's the title
        {
            QStringList line=groupedTable.at ( i );
            if ( nbCols )
            {
                QString val=line.at ( 0 );

                //Filtered
		bool ok=true;
		for ( int w=0; ok && w<nbwords; ++w )
		{
		    QString word=words.at ( w );
		    ok=val.contains ( word , Qt::CaseInsensitive );

		}
                if ( !ok ) groupedTable.removeAt ( i );
            }
        }
    }

    //Compute sums line
    int nblines=groupedTable.count();
    if ( showSumAndAverage )
    {
        QStringList sums;
        sums.push_back ( i18nc ( "Noun, the numerical sum of a list of values","Sum" ) );
        for ( int i=1; i<nbCols; ++i )
        {
            double sum=0;
            for ( int j=1; j<nblines; ++j )
            {
                QString valstring=groupedTable.at ( j ).at ( i );
                if ( !valstring.isEmpty() ) sum+=SKGServices::stringToDouble ( valstring );
            }
            sums.push_back ( SKGServices::doubleToString ( sum ) );
        }
        groupedTable.push_back ( sums );
        nblines++;
    }

    //Compute sum and average column
    if ( nbCols>2 && showSumAndAverage )
    {
        //Add title
        QStringList newLine=groupedTable.at ( 0 );
        newLine.push_back ( i18nc ( "Noun, the numerical sum of a list of values","Sum" ) );
        newLine.push_back ( i18nc ( "Noun, the numerical average of a list of values","Average" ) );
        groupedTable.replace ( 0, newLine );

        for ( int i=1; i<nblines; ++i )
        {
            QStringList newLine=groupedTable.at ( i );
            double sum=0;
            for ( int j=1; j<nbCols; ++j )
            {
                QString valstring=groupedTable.at ( i ).at ( j );
                if ( !valstring.isEmpty() ) sum+=SKGServices::stringToDouble ( valstring );
            }
            newLine.push_back ( SKGServices::doubleToString ( sum ) );
            newLine.push_back ( SKGServices::doubleToString ( sum/ ( nbCols-1 ) ) );
            groupedTable.replace ( i, newLine );
        }
        nbCols+=2;
    }

    //Sort lines
    {
        //Set parameters for static method
        QHeaderView* horizontalHeader=ui.kTable->horizontalHeader();
        sortOrder=horizontalHeader->sortIndicatorOrder();
        sortColumn=horizontalHeader->sortIndicatorSection();

        //Extract title and sums
        QStringList fist=groupedTable.takeFirst();
        QStringList last;
        if ( showSumAndAverage && groupedTable.count() ) last=groupedTable.takeLast();

        //Sort
        qSort ( groupedTable.begin(), groupedTable.end(), listSort );

        //Add title and sums
        groupedTable.insert ( 0, fist );
        if ( showSumAndAverage ) groupedTable.push_back ( last );
    }

    IFSKGTRACEL ( 10 )
    {
        QStringList dump=SKGServices::tableToDump ( groupedTable, SKGServices::DUMP_TEXT );
        int nbl=dump.count();
        SKGTRACE << "nbLevelMax=" << nbLevelMax << endl;
        for ( int i=0; i<nbl; ++i )
        {
            SKGTRACE << dump[i] << endl;
        }
    }


    //Fill table
    KLocale* locale=KGlobal::locale();
    ui.kTable->hide();
    ui.kTable->clear();
    ui.kTable->setRowCount ( nblines-1 );
    ui.kTable->setColumnCount ( nbCols );
    for ( int i=0; i<nblines; ++i )
    {
        QStringList line=groupedTable.at ( i );
        if ( i==0 )
        {
            //Set header
            ui.kTable->setHorizontalHeaderLabels ( line );
        }
        else
        {
            for ( int j=0; j<nbCols; ++j )
            {
                QString val=line.at ( j );

                if ( j==0 )
                {
                    ui.kTable->setItem ( i-1, 0, new QTableWidgetItem ( val ) );
                }
                else
                {
                    QTableWidgetItem* item;
                    if ( !val.isEmpty() )
                    {
                        double vald=SKGServices::stringToDouble ( val );

                        item=new QTableWidgetItem ( locale->formatMoney ( vald, primaryUnit.Name, primaryUnit.NbDecimal ) );
                        if ( !secondaryUnit.Name.isEmpty() && secondaryUnit.Value )
                        {
                            item->setToolTip ( locale->formatMoney ( vald/secondaryUnit.Value, secondaryUnit.Name, secondaryUnit.NbDecimal ) );
                        }

                        item->setData ( DATA_VALUE, vald );
                        item->setTextAlignment ( Qt::AlignRight );
                        if ( vald<0 ) item->setForeground ( QBrush ( QColor ( Qt::red ) ) );
                    }
                    else
                    {
                        item=new QTableWidgetItem ( "" );
                        item->setData ( DATA_VALUE, "" );
                    }
                    ui.kTable->setItem ( i-1, j, item );
                }
            }
        }
    }

    //Enable/Disable buttons
    nbLevel=qMin ( nbLevelMax, nbLevel );
    ui.kLineDown->setEnabled ( nbLevel>0 );
    ui.kLineUp->setEnabled ( nbLevel<nbLevelMax );

    //Refresh graphic view
    redrawGraph();

    ui.kTable->show();

    QApplication::restoreOverrideCursor();
}

void SKGTableWithGraph::onSelectionChanged ( QTableWidgetItem* current, QTableWidgetItem* previous )
{
    _SKGTRACEIN ( 10, "SKGTableWithGraph::onSelectionChanged" );

    //Unset color on previous selection
    if ( previous )
    {
        QAbstractGraphicsShapeItem * graphicItem=dynamic_cast<QAbstractGraphicsShapeItem *> ( ( QGraphicsItem* ) previous->data ( 1 ).toLongLong() );
        if ( graphicItem )
        {
            graphicItem->setBrush ( QBrush ( QColor::fromHsv ( graphicItem->data ( DATA_COLOR_H ).toInt(),
                                             graphicItem->data ( DATA_COLOR_S ).toInt(),
                                             graphicItem->data ( DATA_COLOR_V ).toInt(),
                                             graphicItem->data ( DATA_COLOR_A ).toInt() ) ) );
            graphicItem->setSelected ( false );
        }
        else
        {
            QGraphicsLineItem * graphicItem=dynamic_cast<QGraphicsLineItem *> ( ( QGraphicsItem* ) previous->data ( 1 ).toLongLong() );
            if ( graphicItem )
            {
                QPen pen=graphicItem->pen();
                pen.setColor ( QColor::fromHsv ( graphicItem->data ( DATA_COLOR_H ).toInt(),
                                                 graphicItem->data ( DATA_COLOR_S ).toInt(),
                                                 graphicItem->data ( DATA_COLOR_V ).toInt(),
                                                 graphicItem->data ( DATA_COLOR_A ).toInt() ) );
                graphicItem->setPen ( pen );
                graphicItem->setSelected ( false );
            }
        }
    }

    //Set highlight color on current selection
    if ( current )
    {
        QAbstractGraphicsShapeItem * graphicItem=dynamic_cast<QAbstractGraphicsShapeItem *> ( ( QGraphicsItem* ) current->data ( 1 ).toLongLong() );
        if ( graphicItem )
        {
            graphicItem->setBrush ( QBrush ( QApplication::palette().color ( QPalette::Highlight ) ) );
            graphicItem->setSelected ( true );
        }
        else
        {
            QGraphicsLineItem * graphicItem=dynamic_cast<QGraphicsLineItem *> ( ( QGraphicsItem* ) current->data ( 1 ).toLongLong() );
            if ( graphicItem )
            {
                QPen pen=graphicItem->pen();
                pen.setColor ( QApplication::palette().color ( QPalette::Highlight ) );
                graphicItem->setPen ( pen );
                graphicItem->setSelected ( false );
            }
        }
    }
}

void SKGTableWithGraph::onDoubleClickGraph()
{
    if ( scene )
    {
        //Get selection
        QList<QGraphicsItem *> selectedGraphItems=scene->selectedItems();
        if ( selectedGraphItems.count() )
        {
            emit cellDoubleClicked ( selectedGraphItems[0]->data ( 1 ).toInt(), selectedGraphItems[0]->data ( 2 ).toInt() );
        }
    }
}

void SKGTableWithGraph::onDoubleClick ( int row, int column )
{
    emit cellDoubleClicked ( row, column );
}

void SKGTableWithGraph::onSelectionChangedInGraph()
{
    _SKGTRACEIN ( 10, "SKGTableWithGraph::onSelectionChangedInGraph" );
    if ( scene )
    {
        //Get selection
        QList<QGraphicsItem *> selectedGraphItems=scene->selectedItems();
        if ( selectedGraphItems.count() )
        {
            ui.kTable->setCurrentCell ( selectedGraphItems[0]->data ( 1 ).toInt(), selectedGraphItems[0]->data ( 2 ).toInt() );
        }
    }
}

void SKGTableWithGraph::redrawGraph()
{
    SKGTRACEIN ( 10, "SKGTableWithGraph::redrawGraph" );
    QApplication::setOverrideCursor ( QCursor ( Qt::WaitCursor ) );

    ui.graphicView->hide();
    ui.kTable->hide();

    //Recreate scene
    if ( scene )
    {
        SKGTRACEIN ( 10, "SKGTableWithGraph::redrawGraph-remove scene" );
        scene->clear();
        delete scene;
    }


    scene=new SKGGraphicsScene();
    {
        SKGTRACEIN ( 10, "SKGTableWithGraph::redrawGraph-create scene" );
        //Get current selection
        int crow=ui.kTable->currentRow();
        int ccolumn=ui.kTable->currentColumn();

        int nbRows=ui.kTable->rowCount()- ( showSumAndAverage ? 1 : 0 ); //Sum
        int nbColumns=ui.kTable->columnCount();
        int sumColumnIndex=nbColumns-1;
        if ( nbColumns>2 && showSumAndAverage )
        {
            nbColumns=nbColumns-2; //Sum and average
            --sumColumnIndex;
        }
        QPen dotPen=QPen ( Qt::SolidLine ); //WARNING: Qt::DotLine is very bad for performances
        dotPen.setColor ( Qt::gray );

        GraphType mode=(GraphType) ui.kDisplayMode->currentIndex();
        bool inPositive=false;

        if ( mode==STACK || mode==HISTOGRAM || mode==POINT || mode==LINE )
        {
            ui.kAllPositive->show();
            inPositive= ( ui.kAllPositive->checkState() ==Qt::Checked );
        }
        else ui.kAllPositive->hide();

        //Compute y limits
        double minLimit=0;
        double maxLimit=0;
        SKGTRACEL ( 30 ) << "mode=" << mode << endl;
        SKGTRACEL ( 30 ) << "crow=" << crow << endl;
        SKGTRACEL ( 30 ) << "ccolumn=" << ccolumn << endl;
        SKGTRACEL ( 30 ) << "nbRows=" << nbRows << endl;
        SKGTRACEL ( 30 ) << "nbColumns=" << nbColumns << endl;
        SKGTRACEL ( 30 ) << "sumColumnIndex=" << sumColumnIndex << endl;
        if ( mode==STACK )
        {
            //STACK
            for ( int x=0; x<nbRows; ++x )
            {
                QTableWidgetItem* tableItem=ui.kTable->item ( x,sumColumnIndex );
                if ( tableItem )
                {
                    double val=tableItem->data ( DATA_VALUE ).toDouble();
                    if ( inPositive )
                    {
                        maxLimit=qMax ( maxLimit, fabs ( val ) );
                    }
                    else
                    {
                        minLimit=qMin ( minLimit, val );
                        maxLimit=qMax ( maxLimit, val );
                    }
                }
            }
        }
        else if ( mode==HISTOGRAM || mode==POINT || mode==LINE )
        {
            //HISTOGRAM or POINTS or LINES
            for ( int x=0; x<nbRows; ++x )
            {
                for ( int y=0; y<nbColumns; ++y )
                {
                    QTableWidgetItem* tableItem=ui.kTable->item ( x,y );
                    if ( tableItem )
                    {
                        double val=tableItem->data ( DATA_VALUE ).toDouble();
                        if ( inPositive )
                        {
                            maxLimit=qMax ( maxLimit, fabs ( val ) );
                        }
                        else
                        {
                            minLimit=qMin ( minLimit, val );
                            maxLimit=qMax ( maxLimit, val );
                        }
                    }
                }
            }
        }
        SKGTRACEL ( 30 ) << "minLimit=" << minLimit << endl;
        SKGTRACEL ( 30 ) << "maxLimit=" << maxLimit << endl;

        //Compute
        double width=10;
        double maxX=0;
        if ( nbRows )
        {
            if ( mode==STACK )
            {
                width= ( maxLimit-minLimit ) /nbRows;
                maxX=width*nbRows+width;
            }
            else if ( mode==HISTOGRAM || mode==POINT || mode==LINE )
            {
                width= ( maxLimit-minLimit ) / ( nbRows* ( nbColumns-1 ) );
                maxX=width* ( nbColumns-1 ) * ( nbRows+1 ) +width;
            }
        }
        SKGTRACEL ( 30 ) << "width=" << width << endl;
        SKGTRACEL ( 30 ) << "maxX=" << maxX << endl;

        double margin=width;
        int nbColInMode2= ( nbColumns>2 ? sqrt ( nbColumns-1 ) +.5 : 1 );
        if ( mode!=PIE && mode!=CONCENTRICPIE )
        {
            scene->setSceneRect ( -margin, -margin-maxLimit, maxX+margin, maxLimit-minLimit+2*margin );
            scene->setItemIndexMethod ( QGraphicsScene::NoIndex );

            SKGTRACEL ( 10 ) << "Scene rect:" <<scene->sceneRect().x() << "," <<   scene->sceneRect().y() << "," <<  scene->sceneRect().width() << "," <<  scene->sceneRect().height() << endl;
            SKGTRACEL ( 10 ) << "Items:" <<nbRows << "x" << ( nbColumns-1 ) << "=" << nbRows* ( nbColumns-1 ) << endl;
        }
        else
        {
            int nbRowInMode2= ( ( int ) ( ( nbColumns-1 ) /nbColInMode2 ) );
            if ( nbRowInMode2*nbColInMode2<nbColumns-1 ) ++nbRowInMode2;

            scene->setSceneRect ( 0, 0, BOX_SIZE*nbColInMode2, BOX_SIZE*nbRowInMode2 );
            scene->setItemIndexMethod ( QGraphicsScene::NoIndex );
        }

        //Redraw scene
        double ymin=0;
        double ymax=0;
        for ( int x=0; x<nbRows; ++x )
        {
            QString xname=ui.kTable->item ( x, 0 )->text();
            double yPlus=0;
            double yMoins=0;
	    int jprevious=-1;
            for ( int j=1; j<nbColumns; ++j )
            {
                //Get column name
                QString yname=ui.kTable->horizontalHeaderItem ( j )->text();

                //Get cell value
                QTableWidgetItem* tableItem=ui.kTable->item ( x,j );
                if ( tableItem )
                {
                    double val=tableItem->data ( DATA_VALUE ).toDouble();
                    QString valstring=tableItem->text();

                    int color_h=359*x/nbRows;
                    int color_s=255-155*j/nbColumns;
                    int color_v=255-155*j/nbColumns;
                    int color_a=200;

                    QColor color;
                    if ( x==crow && j==ccolumn ) color=QApplication::palette().color ( QPalette::Highlight );
                    else color=QColor::fromHsv ( color_h,color_s,color_v );

                    color.setAlpha ( color_a );

                    QBrush brush ( color );

                    QGraphicsItem* graphItem=NULL;
                    if ( mode==STACK )
                    {
                        //STACK
                        if ( val>0 || inPositive )
                        {
                            graphItem=scene->addRect ( width*x, -yPlus, width, -fabs ( val ), QPen(), brush );
                            yPlus+=fabs ( val );
                            if ( yPlus>ymax ) ymax=yPlus;
                        }
                        else
                        {
                            graphItem=scene->addRect ( width*x, -yMoins, width, -val, QPen(), brush );
                            yMoins+=val;
                            if ( yMoins<ymin ) ymin=yMoins;
                        }
                    }
                    else if ( mode==HISTOGRAM )
                    {
                        //HISTOGRAM
                        if ( inPositive )
                        {
                            graphItem=scene->addRect ( width* ( ( j-1 ) * ( nbRows+1 ) +x ), 0, width, -fabs ( val ), QPen(), brush );
                            if ( fabs ( val ) >ymax ) ymax=fabs ( val );
                        }
                        else
                        {
                            graphItem=scene->addRect ( width* ( ( j-1 ) * ( nbRows+1 ) +x ), 0, width, -val, QPen(), brush );
                            if ( val>ymax ) ymax=val;
                            if ( val<ymin ) ymin=val;
                        }
                    }
                    else if ( mode==POINT )
                    {
                        //POINTS
                        double radius=width/4;
                        double xmin=width* ( ( j-1 ) * ( nbRows+1 ) + nbRows/2 ) +width/2-radius;
                        if ( inPositive )
                        {
                            switch ( x%3 )
                            {
                            case 0:
                                graphItem=scene->addEllipse ( xmin, -fabs ( val )-radius, 2*radius, 2*radius, QPen(), brush );
                                break;
                            case 1:
                                graphItem=scene->addRect ( xmin, -fabs ( val )-radius, 2*radius, 2*radius, QPen(), brush );
                                break;
                            case 2:
                                QPolygonF polygon;
                                polygon << QPointF ( xmin, -fabs ( val )-radius ) << QPointF ( xmin+2*radius, -fabs ( val ) +radius )
                                << QPointF ( xmin+2*radius, -fabs ( val )-radius ) << QPointF ( xmin, -fabs ( val ) +radius );
                                graphItem=scene->addPolygon ( polygon, QPen(), brush );
                                break;
                            }
                            if ( fabs ( val ) >ymax ) ymax=fabs ( val );
                        }
                        else
                        {
                            switch ( x%3 )
                            {
                            case 0:
                                graphItem=scene->addEllipse ( xmin, -val-radius, 2*radius, 2*radius, QPen(), brush );
                                break;
                            case 1:
                                graphItem=scene->addRect ( xmin, -val-radius, 2*radius, 2*radius, QPen(), brush );
                                break;
                            case 2:
                                QPolygonF polygon;
                                polygon << QPointF ( xmin, -val-radius ) << QPointF ( xmin+2*radius, -val+radius )
                                << QPointF ( xmin+2*radius, -val-radius ) << QPointF ( xmin, -val+radius );
                                graphItem=scene->addPolygon ( polygon, QPen(), brush );
                                break;
                            }
                            if ( val>ymax ) ymax=val;
                            if ( val<ymin ) ymin=val;
                        }

                    }
                    else if ( mode==LINE )
                    {
                        //LINES
			//Empty cells are ignored
			if(!valstring.isEmpty())
			{
			  if(jprevious==-1) jprevious=j;

			  //Create pen
			  QPen pen=QPen ( color );
			  pen.setWidth ( qMax(width/2, ( maxLimit-minLimit )/100) );
			  pen.setCapStyle ( Qt::RoundCap );
			  pen.setJoinStyle ( Qt::RoundJoin );

			  QTableWidgetItem* tableItem2=ui.kTable->item ( x, jprevious );
			  if ( tableItem2 )
			  {
			      QString val2string=tableItem->text();
			      if(!val2string.isEmpty())
			      {
				double val2=tableItem2->data ( DATA_VALUE ).toDouble();

				if ( inPositive )
				{
				    graphItem=scene->addLine ( width* ( ( jprevious-1 ) * ( nbRows+1 ) + nbRows/2 ) +width/2- ( jprevious==j ? width/20 : 0 ), -fabs ( val2 ), width* ( ( j-1 ) * ( nbRows+1 ) + nbRows/2 ) +width/2, -fabs ( val ), pen );
				    if ( fabs ( val ) >ymax ) ymax=fabs ( val );
				}
				else
				{
				    graphItem=scene->addLine ( width* ( ( jprevious-1 ) * ( nbRows+1 ) + nbRows/2 ) +width/2- ( jprevious==j ? width/20 : 0 ), -val2, width* ( ( j-1 ) * ( nbRows+1 ) + nbRows/2 ) +width/2, -val, pen );
				    if ( val>ymax ) ymax=val;
				    if ( val<ymin ) ymin=val;
				}
				jprevious=j;
			      }
			  }
			}
                    }
                    else if ( mode==PIE || mode==CONCENTRICPIE )
                    {
                        //PIE
                        //Compute absolute sum of the column
                        val=abs ( val );
                        double sumColumn=0;
                        double previousSum=0;
                        for ( int x2=0; x2<nbRows; ++x2 )
                        {
                            QTableWidgetItem* tableItem=ui.kTable->item ( x2,j );
                            if ( tableItem )
                            {
                                double absVal=abs ( tableItem->data ( DATA_VALUE ).toDouble() );
                                sumColumn+=absVal;
                                if ( x2<x ) previousSum+=absVal;
                            }
                        }

                        if ( sumColumn )
                        {
                            int nbvals=xname.split ( OBJECTSEPARATOR ).count();
                            double pas=0;
                            double p=0;
                            if ( mode==CONCENTRICPIE )
                            {
                                pas=100/ ( nbLevel+1 );
                                p=pas* ( nbLevel+1-nbvals );
                            }

                            QPainterPath path;
                            path.moveTo ( BOX_SIZE* ( ( j-1 ) %nbColInMode2 ) +BOX_SIZE/2, BOX_SIZE*floor ( ( j-1 ) /nbColInMode2 ) +BOX_SIZE/2 );
                            path.arcTo ( BOX_SIZE* ( ( j-1 ) %nbColInMode2 ) +BOX_MARGIN+p/2, BOX_SIZE*floor ( ( j-1 ) /nbColInMode2 ) +BOX_MARGIN+p/2, BOX_SIZE-2*BOX_MARGIN-p, BOX_SIZE-2*BOX_MARGIN-p, 360*previousSum/sumColumn, 360*val/sumColumn );
                            if ( mode==CONCENTRICPIE && nbvals<=nbLevel+1 && nbvals>1 )
                            {
                                p=pas* ( nbLevel+1-nbvals+1 );
                                path.moveTo ( BOX_SIZE* ( ( j-1 ) %nbColInMode2 ) +BOX_SIZE/2, BOX_SIZE*floor ( ( j-1 ) /nbColInMode2 ) +BOX_SIZE/2 );
                                path.arcTo ( BOX_SIZE* ( ( j-1 ) %nbColInMode2 ) +BOX_MARGIN+p/2, BOX_SIZE*floor ( ( j-1 ) /nbColInMode2 ) +BOX_MARGIN+p/2, BOX_SIZE-2*BOX_MARGIN-p, BOX_SIZE-2*BOX_MARGIN-p, 360*previousSum/sumColumn, 360*val/sumColumn );
                            }
                            path.closeSubpath();
                            graphItem=scene->addPath ( path, mode==CONCENTRICPIE ? QPen ( Qt::white ) : QPen(), brush );
                        }
                    }

                    if ( graphItem )
                    {
                        graphItem->setZValue ( 5 );
                        graphItem->setToolTip ( xname+'\n'+yname+'\n'+valstring+'\n'+tableItem->toolTip() );
                        graphItem->setFlag ( QGraphicsItem::ItemIsSelectable, isSelectable() );
                        if(isSelectable()) graphItem->setCursor ( Qt::PointingHandCursor );
                        graphItem->setData ( 1, x );
                        graphItem->setData ( 2, j );
                        graphItem->setData ( DATA_COLOR_H, color_h );
                        graphItem->setData ( DATA_COLOR_S, color_s );
                        graphItem->setData ( DATA_COLOR_V, color_v );
                        graphItem->setData ( DATA_COLOR_A, color_a );

                        tableItem->setData ( 1, ( qlonglong )  graphItem );
                    }
                }
                else
                {
                    SKGTRACE << "WARNING: cell " << x << "," << j << " null" << endl;
                }
            }
        }

        //Draw axis
        if ( nbRows )
        {
            if ( mode==PIE || mode==CONCENTRICPIE )
            {
                //PIE
                int nbRowInMode2= ( ( int ) ( ( nbColumns-1 ) /nbColInMode2 ) );
                if ( nbRowInMode2*nbColInMode2<nbColumns-1 ) ++nbRowInMode2;
                for ( int i=0; i<=nbColInMode2; ++i )
                {
                    QGraphicsLineItem* item=scene->addLine ( BOX_SIZE*i, 0, BOX_SIZE*i, BOX_SIZE*nbRowInMode2 );
                    item->setPen ( dotPen );
                    item->setFlag ( QGraphicsItem::ItemIsSelectable, false );
                }
                for ( int i=0; i<=nbRowInMode2; ++i )
                {
                    QGraphicsLineItem* item=scene->addLine ( 0, BOX_SIZE*i, BOX_SIZE*nbColInMode2, BOX_SIZE*i );
                    item->setPen ( dotPen );
                    item->setFlag ( QGraphicsItem::ItemIsSelectable, false );
                }

                for ( int j=1; j<nbColumns; ++j )
                {
                    //Get column name
                    QString yname=ui.kTable->horizontalHeaderItem ( j )->text();

                    QGraphicsTextItem* textItem=scene->addText ( yname );
                    textItem->setPos ( BOX_SIZE* ( ( j-1 ) %nbColInMode2 ) +2, BOX_SIZE*floor ( ( j-1 ) /nbColInMode2 ) +2 );
                    textItem->scale ( .5, .5 );
                    textItem->setFlag ( QGraphicsItem::ItemIsSelectable, false );

                }
            }
            else
            {
                //STACK & HISTOGRAMM
                double scaleText=0;
                QGraphicsLineItem* item;

                //Lines x
                int ystep=computeStepSize ( ymax-ymin, 10 );
                if ( ystep>0 )
                {
                    for ( int y=ystep; y<ymax+margin; y=y+ystep )
                    {
                        item=scene->addLine ( -margin, -y, maxX, -y );
                        item->setPen ( dotPen );
                        item->setFlag ( QGraphicsItem::ItemIsSelectable, false );
                        item->setZValue ( 1 );

                        QGraphicsTextItem* textItem=scene->addText ( SKGServices::intToString ( y ) );
                        textItem->setPos ( -margin, -y );
                        if ( scaleText==0 )
                        {
                            QRectF rect=textItem->boundingRect();
                            scaleText=qMin ( width/rect.width(), ystep/rect.height() );
                        }
                        textItem->scale ( scaleText, scaleText );
                        textItem->setFlag ( QGraphicsItem::ItemIsSelectable, false );
                    }
                    for ( int y=-ystep; y>ymin-margin; y=y-ystep )
                    {
                        item=scene->addLine ( -margin, -y, maxX, -y );
                        item->setPen ( dotPen );
                        item->setFlag ( QGraphicsItem::ItemIsSelectable, false );
                        item->setZValue ( 1 );

                        QGraphicsTextItem* textItem=scene->addText ( SKGServices::intToString ( y ) );
                        textItem->setPos ( -margin, -y );
                        textItem->scale ( scaleText, scaleText );
                        textItem->setFlag ( QGraphicsItem::ItemIsSelectable, false );
                    }
                }

                if ( mode==HISTOGRAM || mode==POINT || mode==LINE )
                {
                    //Line y
                    double xstep=width* ( nbRows+1 );
		    if(maxX/xstep<=100)
		    {
		      int j=1;
		      for ( double x=xstep-width/2; x<maxX; x=x+xstep )
		      {
			  QGraphicsLineItem* item=scene->addLine ( x, -ymin+margin, x, -ymax-margin );
			  item->setPen ( dotPen );
			  item->setFlag ( QGraphicsItem::ItemIsSelectable, false );

			  QString yname=ui.kTable->horizontalHeaderItem ( j )->text();
			  if ( j<nbColumns-1 ) ++j;

			  QGraphicsTextItem* textItem=scene->addText ( yname );
			  textItem->setPos ( x-xstep+(j<=2 ? width/2 : 0), 0 );
			  textItem->scale ( scaleText, scaleText );
			  textItem->setFlag ( QGraphicsItem::ItemIsSelectable, false );
		      }
		    }
                }

                //Axis y
                item=scene->addLine ( 0, -ymin+margin, 0, -ymax-margin );
                item->setFlag ( QGraphicsItem::ItemIsSelectable, false );
                addArrow ( QPointF ( 0, -ymax-margin ), margin/2, 45, 90 );
                item->setZValue ( 2 );

                //Axis x
                item=scene->addLine ( -margin, 0, maxX, 0 );
                item->setFlag ( QGraphicsItem::ItemIsSelectable, false );
                addArrow ( QPointF ( maxX, 0 ), margin/2, 45, 180 );
                item->setZValue ( 2 );
            }
        }
    }
    {
        SKGTRACEIN ( 10, "SKGTableWithGraph::redrawGraph-add scene to view" );
        ui.graphicView->setScene ( scene );
        ui.graphicView->show();
        ui.kTable->show();
        ui.graphicView->onZoomOriginal();

        //Add selection event on scene
        connect ( scene, SIGNAL ( selectionChanged() ), this, SLOT ( onSelectionChangedInGraph() ), Qt::QueuedConnection );
        connect ( scene, SIGNAL ( doubleClicked() ), this, SLOT ( onDoubleClickGraph() ), Qt::QueuedConnection );
    }
    QApplication::restoreOverrideCursor();
}

double SKGTableWithGraph::computeStepSize ( double iRange, double iTargetSteps )
{
    // Calculate an initial guess at step size
    double tempStep = iRange / iTargetSteps;
    // Get the magnitude of the step size
    double mag = floor ( log10 ( tempStep ) );
    double magPow = pow ( ( double ) 10.0, mag );
    // Calculate most significant digit of the new step size
    double magMsd = ( ( int ) ( tempStep / magPow + .5 ) );
    // promote the MSD to either 1, 2, or 5
    if ( magMsd > 5.0 )
        magMsd = 10.0;
    else if ( magMsd > 2.0 )
        magMsd = 5.0;
    else if ( magMsd > 1.0 )
        magMsd = 2.0;
    return magMsd * magPow;
}

void SKGTableWithGraph::addArrow ( const QPointF& iPeak, double iSize, double iArrowAngle, double iDegree )
{
    if ( scene )
    {
        QPolygonF pol;
        double radian=3.14*iArrowAngle/360;
        pol << QPointF ( 0, 0 ) << QPointF ( iSize*cos ( radian ), iSize*sin ( radian ) ) << QPointF ( iSize*cos ( radian ), -iSize*sin ( radian ) ) << QPointF ( 0, 0 );
        QGraphicsPolygonItem * item=scene->addPolygon ( pol, QPen ( Qt::black ), QBrush ( Qt::black ) );
        item->rotate ( iDegree );
        item->moveBy ( iPeak.x(), iPeak.y() );
        item->setFlag ( QGraphicsItem::ItemIsSelectable, false );
        item->setZValue ( 2 );
    }
}

void SKGTableWithGraph::onExportCSV()
{
    _SKGTRACEIN ( 10, "SKGTableWithGraph::onExportCSV" );
    QString fileName=SKGMainPanel::getSaveFileName ( KUrl ( "kfiledialog:///IMPEXP" ), "*.csv|"+i18n ( "CSV Files" ) , this );
    if ( fileName.isEmpty() ) return;

    {
        SKGError err;

        //Write file
        QFile file ( fileName );
        if ( !file.open ( QIODevice::WriteOnly | QIODevice::Text ) )
        {
            err.setReturnCode ( ERR_INVALIDARG );
            err.setMessage ( i18n ( "Save file [%1] failed",fileName ) );
        }
        else
        {
            QTextStream out ( &file );
            QStringList dump=SKGServices::tableToDump ( getTable(), SKGServices::DUMP_CSV );
            int nbl=dump.count();
            for ( int i=0; i<nbl; ++i )
            {
                out << dump[i] << endl;
            }
        }

        //Close file
        file.flush();
        file.close();

    }
    QDesktopServices::openUrl ( QUrl ( fileName ) );
}

void SKGTableWithGraph::onExportTXT()
{
    _SKGTRACEIN ( 10, "SKGTableWithGraph::onExportTXT" );
    QString fileName=SKGMainPanel::getSaveFileName ( KUrl ( "kfiledialog:///IMPEXP" ), "*.txt|"+i18n ( "Text document" ) , this );
    if ( fileName.isEmpty() ) return;

    {
        SKGError err;

        //Write file
        QFile file ( fileName );
        if ( !file.open ( QIODevice::WriteOnly | QIODevice::Text ) )
        {
            err.setReturnCode ( ERR_INVALIDARG );
            err.setMessage ( i18n ( "Save file [%1] failed",fileName ) );
        }
        else
        {
            QTextStream out ( &file );
            QStringList dump=SKGServices::tableToDump ( getTable(), SKGServices::DUMP_TEXT );
            int nbl=dump.count();
            for ( int i=0; i<nbl; ++i )
            {
                out << dump[i] << endl;
            }
        }

        //Close file
        file.flush();
        file.close();

    }
    QDesktopServices::openUrl ( QUrl ( fileName ) );
}

void SKGTableWithGraph::setGraphType ( SKGTableWithGraph::GraphType iType ) const
{
    ui.kDisplayMode->setCurrentIndex ( ( int ) iType );
}

SKGTableWithGraph::GraphType SKGTableWithGraph::getGraphType () const
{
    return ( SKGTableWithGraph::GraphType ) ui.kDisplayMode->currentIndex();
}

void SKGTableWithGraph::setGraphVisible ( bool iVisible ) const
{
    ui.graph_widget->setVisible ( iVisible );
}

bool SKGTableWithGraph::isGraphVisible() const
{
    return ui.graph_widget->isVisible();
}

void SKGTableWithGraph::setTableVisible ( bool iVisible ) const
{
    ui.table_widget->setVisible ( iVisible );
}

bool SKGTableWithGraph::isTableVisible() const
{
    return ui.table_widget->isVisible();
}

void SKGTableWithGraph::setGraphTypeSelectorVisible ( bool iVisible ) const
{
    ui.kGraphTypeFrm->setVisible ( iVisible );
}

bool SKGTableWithGraph::isGraphTypeSelectorVisible() const
{
    return ui.kGraphTypeFrm->isVisible();
}

void SKGTableWithGraph::showTable()
{
    setTableVisible ( true );
    setGraphVisible ( false );
}


void SKGTableWithGraph::showGraph()
{
    setGraphVisible ( true );
    setTableVisible ( false );
}


void SKGTableWithGraph::showTableAndGraph()
{
    setGraphVisible ( true );
    setTableVisible ( true );
}

void SKGTableWithGraph::setSelectable(bool iSelectable)
{
  selectable=iSelectable;
}

bool SKGTableWithGraph::isSelectable() const
{
  return selectable;
}

#include "skgtablewithgraph.moc"

