/***************************************************************************
 *   Copyright (C) 2008 by S. MANKOWSKI / G. DE BURE skrooge@mankowski.fr  *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>  *
 ***************************************************************************/
#ifndef SKGTABLEVIEW_H
#define SKGTABLEVIEW_H
/** @file
 * A table view with more features.
 *
 * @author Stephane MANKOWSKI / Guillaume DE BURE
 */
#include "skgservices.h"
#include "skgobjectbase.h"
#include "skgbasegui_export.h"

#include <kglobal.h>
#include <QStringList>
#include <QTableView>
#include <QtDesigner/QDesignerExportWidget>
#include <QBasicTimer>

#include <kmenu.h>

class SKGDocument;
class QTimer;

/**
 * This file is a tab widget used by plugins
 */
class SKGBASEGUI_EXPORT SKGTableView : public QTableView
{
        Q_OBJECT
        /**
         * State of the view
         */
        Q_PROPERTY( QString state READ getState WRITE setState USER true)
        /**
         * Auto resize mode of the view
         */
        Q_PROPERTY( bool autoResized READ isAutoResized)
        /**
         * Smooth scrolling mode of the view
         */
        Q_PROPERTY( bool smoothScrolling READ isSmoothScrolling)

public:
        /**
         * Default Constructor
         * @param parent the parent
         */
        SKGTableView(QWidget *parent = 0);

        /**
         * Default Destructor
         */
        virtual ~SKGTableView();

        /**
         * Get the current state
         * @return a string containing all information needed to set the same state.
         * Could be an XML stream
         */
        virtual QString getState();

        /**
         * Set the current state
         * MUST BE OVERWRITTEN
         * @param iState must be interpreted to set the state of the widget
         */
        virtual void setState(const QString& iState);

        /**
         * To know if the autoresized mode is enable
         */
        virtual bool isAutoResized();

        /**
         * To know if the smoothscrolling mode is enable
         */
        virtual bool isSmoothScrolling();

        /**
        * Set parameter to activate and save default state of this table
        * @param iDocument document pointer
        * @param iParameterName parameter name in this document
        */
        virtual void setDefaultSaveParameters(const SKGDocument* iDocument, const QString& iParameterName);

        /**
         * Get the table content
         * @return the table content
         */
        virtual SKGStringListList getTable();

        /**
         * Get the current selection
         * @return selected objects
         */
        virtual SKGObjectBase::SKGListSKGObjectBase getSelectedObjects();

        /**
         * Get the number of seleted object
         * @return number of selected objects
         */
        virtual int getNbSelectedObjects();

        /**
         * Sets the current selection model to the given selectionModel.
         * @param selectionModel the selection model
         */
        virtual void setSelectionModel ( QItemSelectionModel * selectionModel );

signals:
        /**
         * Emitted when the empty area
         */
        void clickEmptyArea();

        /**
         * Emitted 300ms after selection changed
         */
        void selectionChangedDelayed();

protected:

        /**
          * This function is called with the given event when a mouse button is pressed while the cursor is inside the widget.
         * If a valid item is pressed on it is made into the current item. This function emits the pressed() signal.
         * @param event the event
         */
        virtual void mousePressEvent ( QMouseEvent * event );

public slots:

        /**
         * Save the selection
         */
        virtual void saveSelection();

        /**
         * Reset the selection
         */
        virtual void resetSelection();

        /**
         * Select an object and focus on it
         * @param iUniqueID unique ID of the object
         */
        virtual void selectObject(const QString& iUniqueID);

        /**
         * Select objects and focus on the first one
         * @param iUniqueIDs unique IDs of objects
         * @param iFocusOnFirstOne set the focus on the first one
         */
        virtual void selectObjects(const QStringList& iUniqueIDs, bool iFocusOnFirstOne=false);

        /**
         * This property holds whether to draw the background using alternating colors.
         * @param enable true of false
         */
        virtual void setAlternatingRowColors(bool enable);

        /**
         * Swith the smooth Scrolling mode
         */
        virtual void switchSmoothScrolling();

        /**
         * Reset columns order
         */
        virtual void resetColumnsOrder();

        /**
         * Resizes all columns based on the size hints of the delegate used to render each item in the columns.
         */
        virtual void resizeColumnsToContents();

        /**
         * Resizes all columns based on the size hints of the delegate used to render each item in the columns.
         */
        virtual void resizeColumnsToContentsDelayed();

private slots:
        void onExportPDF();
        void onExportCSV();
        void onExportTXT();

        void moveSection();
        void setupHeaderMenu(bool iRefreshColumnSize=true);
        void showHeaderMenu(const QPoint& pos);
        void showHideColumn();
        void switchAutoResize();
        void changeSchema();
        void saveDefaultClicked();

        void onSelectionChanged();

private:
        Q_DISABLE_COPY(SKGTableView);

        void timerEvent(QTimerEvent *event);
        bool eventFilter(QObject *object, QEvent *event);

        KMenu* header_menu;
        bool autoResize;
        QAction* actAutoResize;
        bool smoothScrolling;
        QAction* actSmoothScrolling;
        SKGDocument* document;
        QString parameterName;
        QStringList selection;


        typedef enum { Steady, Pressed, ManualScroll, AutoScroll, Stop } State;
        State state;
        QPoint pressPos;
        QPoint offset;
        QPoint dragPos;
        QPoint speed;
        QList<QEvent*> ignored;
        QBasicTimer ticker;
        QTimer* timerDelayedResize;
        QTimer* timerSelectionChanged;
};

#endif // SKGTABLEVIEW_H
