
/*
 *  devicelisting.cpp
 *
 *  Copyright (C) 2009 David Hubner <hubnerd@ntlworld.com>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 *
 */

#include "devicelisting.h"

DeviceListing::DeviceListing(QWidget *parent, InfoPanel *info) : 
  QTreeWidget(parent), iPanel(info) {
    
    // Check if clicked
    connect(this,SIGNAL(itemClicked(QTreeWidgetItem *,int)),this,SLOT(deviceItemClicked(QTreeWidgetItem *,int)));
    
    // Check if item is added
    connect(Solid::DeviceNotifier::instance(),SIGNAL(deviceAdded(QString)),this,SLOT(deviceAddedSlot(QString)));
  
    // Check if item is removed
    connect(Solid::DeviceNotifier::instance(),SIGNAL(deviceRemoved(QString)),this,SLOT(deviceRemovedSlot(QString)));
    
    createMenuActions();
    setHeaderLabels(QStringList(i18n("Devices")));
    populateListing();   
}

DeviceListing::~DeviceListing() {
  
  clear();
}

void DeviceListing::createMenuActions() {
 
  colAct = new QAction(i18n("Collapse All"), this);
  connect(colAct, SIGNAL(triggered()), this, SLOT(collapseAllDevices()));
  
  expAct = new QAction(i18n("Expand All"), this);
  connect(expAct, SIGNAL(triggered()), this, SLOT(expandAllDevices()));
  
  allAct = new QAction(i18n("Show All Devices"), this);
  connect(allAct, SIGNAL(triggered()), this, SLOT(showAllDevices()));
  
  relAct = new QAction(i18n("Show Relevent Devices"), this);
  connect(relAct, SIGNAL(triggered()), this, SLOT(showReleventDevices()));
}

void DeviceListing::contextMenuEvent(QContextMenuEvent *event) {
  
  QMenu menu(this);
  
  menu.addAction(colAct);
  menu.addAction(expAct);
  menu.addAction(allAct);
  menu.addAction(relAct);
  menu.exec(event->globalPos());
}

QTreeWidgetItem *DeviceListing::createListItems(Solid::DeviceInterface::Type type) {

      switch(type) {
	case Solid::DeviceInterface::Processor: 
	  return new SolProcessorDevice(type);
	case Solid::DeviceInterface::StorageDrive:
	  return new SolStorageDevice(type);
	case Solid::DeviceInterface::NetworkInterface:
	  return new SolNetworkDevice(type);
	case Solid::DeviceInterface::AudioInterface:
	  return new SolAudioDevice(type);
	case Solid::DeviceInterface::Camera:
	  return new SolCameraDevice(type);
	case Solid::DeviceInterface::PortableMediaPlayer:
	  return new SolMediaPlayerDevice(type);
	case Solid::DeviceInterface::Button:
	  return new SolButtonDevice(type);
	case Solid::DeviceInterface::Battery:
	  return new SolBatteryDevice(type);
	case Solid::DeviceInterface::AcAdapter:
	  return new SolAcAdapterDevice(type);
	case Solid::DeviceInterface::DvbInterface:
	  return new SolDvbDevice(type);
	case Solid::DeviceInterface::SerialInterface:
	  return new SolSerialDevice(type);
	case Solid::DeviceInterface::SmartCardReader:
	  return new SolSmartCardDevice(type);
	case Solid::DeviceInterface::Video:
	  return new SolVideoDevice(type);
	default:
	  return new SolDevice(type,i18n("Unknown"));
      }
}

void DeviceListing::populateListing(show showStatus) {
 
  Solid::DeviceInterface::Type needHardWare[] = { 
    Solid::DeviceInterface::Processor,
    Solid::DeviceInterface::StorageDrive,
    Solid::DeviceInterface::NetworkInterface,
    Solid::DeviceInterface::AudioInterface,
    Solid::DeviceInterface::Video,
    Solid::DeviceInterface::SerialInterface,
    Solid::DeviceInterface::SmartCardReader,
    Solid::DeviceInterface::DvbInterface,
    Solid::DeviceInterface::Button,
    Solid::DeviceInterface::Battery, 
    Solid::DeviceInterface::AcAdapter,
    Solid::DeviceInterface::PortableMediaPlayer,
    Solid::DeviceInterface::Camera
  };
  
  clear();
 
  for(unsigned int i=0;i<(sizeof(needHardWare)/sizeof(Solid::DeviceInterface::Type));i++) { 
    QTreeWidgetItem *tmpDevice = createListItems(needHardWare[i]);
    if((tmpDevice->childCount() > 0) || (showStatus==ALL)) addTopLevelItem(tmpDevice);
  }
}

void DeviceListing::deviceItemClicked(QTreeWidgetItem *listItemIn ,int columnIn) {

  Q_UNUSED(columnIn);
  
  SolDevice *listItem = static_cast<SolDevice *>(listItemIn);
  if(listItem->isDeviceSet()) {
    
    iPanel->setTopInfo(listItem->getDeviceIcon(),listItem->getDevice());
    
    QVListLayout *bottomLay = listItem->getInfoPanelLayout();
    if(!bottomLay) return;
    
    iPanel->setBottomInfo(bottomLay);
  }
}

void DeviceListing::deviceAddedSlot(QString udi) {

  //FIXME: Need to add some kind of add without reseting the list
  
  if(udi.isEmpty()) return;
  populateListing();
}

void DeviceListing::deviceRemovedSlot(QString udi) {

  if(udi.isEmpty()) return;
  populateListing();
}

void DeviceListing::collapseAllDevices() {
  
  collapseAll();
}

void DeviceListing::expandAllDevices() {
 
  expandAll();
}

void DeviceListing::showAllDevices() {
 
  populateListing(ALL);
}

void DeviceListing::showReleventDevices() {
 
  populateListing(RELEVENT);
}