/***************************************************************************
 *   Copyright (C) 2008 by S. MANKOWSKI / G. DE BURE skrooge@miraks.com    *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>  *
 ***************************************************************************/
/** @file
 * This file is Skrooge plugin for operation management.
 *
 * @author Stephane MANKOWSKI / Guillaume DE BURE
 */
#include "skgoperationplugin.h"
#include "skgoperationpluginwidget.h"
#include "skgoperationboardwidget.h"
#include "skgtraces.h"
#include "skgoperationobject.h"
#include "skgsuboperationobject.h"
#include "skgtransactionmng.h"
#include "skgmainpanel.h"
#include "skgtableview.h"
#include "skgoperation_settings.h"

#include <kactioncollection.h>
#include <kstandardaction.h>

#include <QDomDocument>

K_PLUGIN_FACTORY ( SKGOperationPluginFactory, registerPlugin<SKGOperationPlugin>(); )
K_EXPORT_PLUGIN ( SKGOperationPluginFactory ( "skrooge_operation", "skrooge_operation" ) )

SKGOperationPlugin::SKGOperationPlugin ( QObject* iParent, const QVariantList& /*iArg*/ ) : SKGInterfacePlugin ( iParent )
{
        SKGTRACEIN ( 10, "SKGOperationPlugin::SKGOperationPlugin" );
}

SKGOperationPlugin::~SKGOperationPlugin()
{
        SKGTRACEIN ( 10, "SKGOperationPlugin::~SKGOperationPlugin" );
        currentBankDocument=NULL;

        duplicateAction=NULL;
        switchToPointedAction=NULL;
        switchHighLight=NULL;
        openHighLights=NULL;
}

void SKGOperationPlugin::setupActions ( SKGDocument* iDocument, const QStringList& iArgument )
{
        SKGTRACEIN ( 10, "SKGOperationPlugin::setupActions" );
        Q_UNUSED ( iArgument );

        currentBankDocument=iDocument        ;


        setComponentData ( SKGOperationPluginFactory::componentData() );
        setXMLFile ( "skrooge_operation.rc" );

        //Menu
        duplicateAction = new KAction ( KIcon ( "skrooge_duplicate" ), i18n ( "&Duplicate" ), this );
        connect ( duplicateAction, SIGNAL ( triggered ( bool ) ), this, SLOT ( actionDuplicate() ) );
        actionCollection()->addAction ( QLatin1String ( "edit_duplicate_operation" ), duplicateAction );
        duplicateAction->setShortcut ( Qt::CTRL+Qt::Key_D );

        if (SKGMainPanel::getMainPanel()) SKGMainPanel::getMainPanel()->registedGlobalAction ( "edit_duplicate_operation", duplicateAction );

        createTemplateAction = new KAction ( KIcon ( "skrooge_template" ), i18n ( "&Create template" ), this );
        connect ( createTemplateAction, SIGNAL ( triggered ( bool ) ), this, SLOT ( actionCreateTemplate() ) );
        actionCollection()->addAction ( QLatin1String ( "edit_template_operation" ), createTemplateAction );
        createTemplateAction->setShortcut ( Qt::CTRL+Qt::SHIFT+Qt::Key_T );

        if (SKGMainPanel::getMainPanel()) SKGMainPanel::getMainPanel()->registedGlobalAction ( "edit_template_operation", createTemplateAction );

        switchToPointedAction = new KAction ( KIcon ( "dialog-ok" ), i18n ( "&Point" ), this );
        connect ( switchToPointedAction, SIGNAL ( triggered ( bool ) ), this, SLOT ( actionSwitchToPointed() ) );
        actionCollection()->addAction ( QLatin1String ( "edit_point_selected_operation" ), switchToPointedAction );
        switchToPointedAction->setShortcut ( Qt::CTRL+Qt::Key_R );

        if (SKGMainPanel::getMainPanel()) SKGMainPanel::getMainPanel()->registedGlobalAction ( "edit_point_selected_operation", switchToPointedAction );

        KAction* fastEditionAction = new KAction ( KIcon ( "games-solve" ), i18n ( "&Fast edit" ), this );
        actionCollection()->addAction ( QLatin1String ( "fast_edition" ), fastEditionAction );
        fastEditionAction->setEnabled ( false );
        fastEditionAction->setShortcut ( Qt::Key_F10 );

        if (SKGMainPanel::getMainPanel()) SKGMainPanel::getMainPanel()->registedGlobalAction ( "fast_edition", fastEditionAction );

        KAction* fastEditionWithoutAmountAction = new KAction ( KIcon ( "games-solve" ), i18n ( "&Fast edit without amount" ), this );
        actionCollection()->addAction ( QLatin1String ( "fast_edition_without_amount" ), fastEditionWithoutAmountAction );
        fastEditionWithoutAmountAction->setEnabled ( false );
        fastEditionWithoutAmountAction->setShortcut ( Qt::CTRL+Qt::Key_F10 );

        if (SKGMainPanel::getMainPanel()) SKGMainPanel::getMainPanel()->registedGlobalAction ( "fast_edition_without_amount", fastEditionWithoutAmountAction );

        switchHighLight = new KAction ( KIcon ( "rating" ), i18n ( "Switch &highlight" ), this );
        connect ( switchHighLight, SIGNAL ( triggered ( bool ) ), this, SLOT ( actionSwitchHighLight() ) );
        actionCollection()->addAction ( QLatin1String ( "edit_switch_highlight" ), switchHighLight );
        switchHighLight->setShortcut ( Qt::CTRL+Qt::Key_H );

        if (SKGMainPanel::getMainPanel()) SKGMainPanel::getMainPanel()->registedGlobalAction ( "edit_switch_highlight", switchHighLight );

        openHighLights = new KAction ( KIcon ( "skrooge_open_highlight" ), i18n ( "&Open highlights" ), this );
        connect ( openHighLights, SIGNAL ( triggered ( bool ) ), this, SLOT ( actionOpenHighLights() ) );
        actionCollection()->addAction ( QLatin1String ( "view_open_highlight" ), openHighLights );
        openHighLights->setShortcut ( Qt::CTRL+Qt::SHIFT+Qt::Key_H );

        if (SKGMainPanel::getMainPanel()) SKGMainPanel::getMainPanel()->registedGlobalAction ( "view_open_highlight", openHighLights );
}

void SKGOperationPlugin::refresh()
{
        SKGTRACEIN ( 10, "SKGOperationPlugin::refresh" );
        if ( currentBankDocument && SKGMainPanel::getMainPanel() ) {
                bool test=(currentBankDocument->getDatabase()!=NULL);
                if ( openHighLights ) openHighLights->setEnabled ( test );

                SKGObjectBase::SKGListSKGObjectBase selection=SKGMainPanel::getMainPanel()->getSelectedObjects();
                if ( selection.count() >0 ) {
                        bool onOperation= ( selection.at ( 0 ).getRealTable() =="operation" );
                        if ( duplicateAction ) duplicateAction->setEnabled ( onOperation );
                        if ( createTemplateAction ) createTemplateAction->setEnabled ( onOperation );
                        if ( switchHighLight ) switchHighLight->setEnabled ( onOperation );
                        if ( switchToPointedAction ) switchToPointedAction->setEnabled ( onOperation );
                } else {
                        if ( duplicateAction ) duplicateAction->setEnabled ( false );
                        if ( createTemplateAction ) createTemplateAction->setEnabled ( false );
                        if ( switchHighLight ) switchHighLight->setEnabled ( false );
                        if ( switchToPointedAction ) switchToPointedAction->setEnabled ( false );
                }
        }
}

void SKGOperationPlugin::close()
{
        SKGTRACEIN ( 10, "SKGOperationPlugin::close" );
}

int SKGOperationPlugin::getNbDashboardWidgets()
{
        return 1;
}

QString SKGOperationPlugin::getDashboardWidgetTitle(int iIndex)
{
        if (iIndex==0) return i18n("Income && Expenditure");
        return "";
}

SKGWidget* SKGOperationPlugin::getDashboardWidget(int iIndex)
{
        if (iIndex==0) return new SKGOperationBoardWidget(currentBankDocument);
        return NULL;
}

SKGTabWidget* SKGOperationPlugin::getWidget()
{
        SKGTRACEIN ( 10, "SKGOperationPlugin::getWidget" );
        return new SKGOperationPluginWidget(( SKGDocumentBank* ) currentBankDocument);
}

QWidget* SKGOperationPlugin::getPreferenceWidget()
{
        SKGTRACEIN ( 10, "SKGMonthlyPlugin::getPreferenceWidget" );
        QWidget* widget=new QWidget();
        ui.setupUi ( widget );
        return widget;
}

KConfigSkeleton* SKGOperationPlugin::getPreferenceSkeleton()
{
        return skgoperation_settings::self();
}

QString SKGOperationPlugin::title() const
{
        return i18n ( "Operations" );
}

QString SKGOperationPlugin::icon() const
{
        return "view-pim-tasks";
}

QString SKGOperationPlugin::statusTip () const
{
        return i18n ( "Operation management (creation, update, ...)" );
}

QString SKGOperationPlugin::toolTip () const
{
        return i18n ( "Operation management" );
}


int SKGOperationPlugin::getOrder() const
{
        return 15;
}

QStringList SKGOperationPlugin::tips() const
{
        QStringList output;
        output.push_back ( i18n ( "<p>... you can press <strong>+</strong>, <strong>-</strong>, <strong>CTRL +</strong> or <strong>CTRL -</strong> to quickly change dates.</p>" ) );
        output.push_back ( i18n ( "<p>... you can update many operations in one shot.</p>" ) );
        output.push_back ( i18n ( "<p>... you can double click on an operation to show or edit sub operations.</p>" ) );
        output.push_back ( i18n ( "<p>... you can duplicate an operation including complex operations (split, grouped, ...).</p>" ) );
        output.push_back ( i18n ( "<p>... you can create template of operations.</p>" ) );
        return output;
}

bool SKGOperationPlugin::isInContext() const
{
        return true;
}

void SKGOperationPlugin::actionSwitchToPointed()
{
        SKGError err;
        SKGTRACEINRC ( 10, "SKGOperationPlugin::actionSwitchToPointed",err );
        //Get Selection
        if ( SKGMainPanel::getMainPanel() && currentBankDocument ) {
                SKGObjectBase::SKGListSKGObjectBase selection=SKGMainPanel::getMainPanel()->getSelectedObjects();
                int nb=selection.count();
                {
                        SKGBEGINPROGRESSTRANSACTION ( *currentBankDocument, i18n ( "Switch to pointed" ), err, nb );

                        QApplication::setOverrideCursor ( QCursor ( Qt::WaitCursor ) );
                        for ( int i=0; err.isSucceeded() && i<nb; ++i ) {
                                SKGOperationObject operationObj=selection.at ( i );
                                if ( err.isSucceeded() ) err=operationObj.setStatus ( SKGOperationObject::POINTED );
                                if ( err.isSucceeded() ) err=operationObj.save();

                                if ( err.isSucceeded() ) err=currentBankDocument->stepForward ( i+1 );
                        }
                        QApplication::restoreOverrideCursor();
                }

                //status bar
                if ( err.isSucceeded() ) err=SKGError ( 0, i18n ( "Operation pointed." ) );
                else err.addError ( ERR_FAIL, i18n ( "Switch failed" ) );

                //Display error
                SKGMainPanel::getMainPanel()->displayErrorMessage ( err );
        }
}

void SKGOperationPlugin::actionSwitchHighLight()
{
        SKGError err;
        SKGTRACEINRC ( 10, "SKGOperationPlugin::actionSwitchHighLight",err );
        //Get Selection
        if ( SKGMainPanel::getMainPanel() && currentBankDocument ) {
                SKGObjectBase::SKGListSKGObjectBase selection=SKGMainPanel::getMainPanel()->getSelectedObjects();
                int nb=selection.count();
                {
                        SKGBEGINPROGRESSTRANSACTION ( *currentBankDocument, i18n ( "Operation highlight" ), err, nb );

                        QApplication::setOverrideCursor ( QCursor ( Qt::WaitCursor ) );
                        for ( int i=0; err.isSucceeded() && i<nb; ++i ) {
                                SKGOperationObject operationObj=selection.at ( i );
                                if ( err.isSucceeded() ) err=operationObj.bookmark ( !operationObj.isBookmarked() );
                                if ( err.isSucceeded() ) err=operationObj.save();

                                if ( err.isSucceeded() ) err=currentBankDocument->stepForward ( i+1 );
                        }
                        QApplication::restoreOverrideCursor();
                }

                //status bar
                if ( err.isSucceeded() )  err=SKGError ( 0, i18n ( "Operation highlighted." ) );
                else err.addError ( ERR_FAIL, i18n ( "Operation highlight failed" ) );

                //Display error
                SKGMainPanel::getMainPanel()->displayErrorMessage ( err );
        }
}

void SKGOperationPlugin::actionDuplicate()
{
        SKGError err;
        SKGTRACEINRC ( 10, "SKGOperationPlugin::actionDuplicate",err );
        //Get Selection
        if ( SKGMainPanel::getMainPanel() && currentBankDocument ) {
                QStringList listUUID;
                SKGObjectBase::SKGListSKGObjectBase selection=SKGMainPanel::getMainPanel()->getSelectedObjects();
                int nb=selection.count();
                {
                        SKGBEGINPROGRESSTRANSACTION ( *currentBankDocument, i18n ( "Duplicate operation" ), err, nb );

                        QApplication::setOverrideCursor ( QCursor ( Qt::WaitCursor ) );
                        for ( int i=0; err.isSucceeded() && i<nb; ++i ) {
                                SKGOperationObject operationObj=selection.at ( i );
                                SKGOperationObject dup;
                                if ( err.isSucceeded() ) err=operationObj.duplicate ( dup );
                                if ( err.isSucceeded() ) err=currentBankDocument->stepForward ( i+1 );

                                listUUID.push_back ( dup.getUniqueID() );
                        }
                        QApplication::restoreOverrideCursor();
                }

                //status bar
                if ( err.isSucceeded() ) {
                        err=SKGError ( 0, i18n ( "Operation duplicated." ) );
                        SKGOperationPluginWidget* w=dynamic_cast<SKGOperationPluginWidget*>(SKGMainPanel::getMainPanel()->currentTab());
                        if (w) w->getTableView()->selectObjects(listUUID, true);
                } else err.addError ( ERR_FAIL, i18n ( "Duplicate operation failed" ) );

                //Display error
                SKGMainPanel::getMainPanel()->displayErrorMessage ( err );
        }
}

void SKGOperationPlugin::actionCreateTemplate()
{
        SKGError err;
        SKGTRACEINRC ( 10, "SKGOperationPlugin::actionCreateTemplate",err );
        //Get Selection
        if ( SKGMainPanel::getMainPanel() && currentBankDocument ) {
                QStringList listUUID;
                SKGObjectBase::SKGListSKGObjectBase selection=SKGMainPanel::getMainPanel()->getSelectedObjects();
                int nb=selection.count();
                {
                        SKGBEGINPROGRESSTRANSACTION ( *currentBankDocument, i18n ( "Create template" ), err, nb );

                        QApplication::setOverrideCursor ( QCursor ( Qt::WaitCursor ) );
                        for ( int i=0; err.isSucceeded() && i<nb; ++i ) {
                                SKGOperationObject operationObj=selection.at ( i );
                                SKGOperationObject dup;
                                if ( err.isSucceeded() ) err=operationObj.duplicate ( dup , QDate::currentDate(), true );
                                if ( err.isSucceeded() ) err=currentBankDocument->stepForward ( i+1 );

                                listUUID.push_back ( dup.getUniqueID() );
                        }
                        QApplication::restoreOverrideCursor();
                }

                //status bar
                if ( err.isSucceeded() ) {
                        err=SKGError ( 0, i18n ( "Template created." ) );
                        SKGOperationPluginWidget* w=dynamic_cast<SKGOperationPluginWidget*>(SKGMainPanel::getMainPanel()->currentTab());
                        if (w) {
                                w->setTemplateMode(true);
                                w->getTableView()->selectObjects(listUUID, true);
                        }
                } else err.addError ( ERR_FAIL, i18n ( "Creation template failed" ) );

                //Display error
                SKGMainPanel::getMainPanel()->displayErrorMessage ( err );
        }
}

void SKGOperationPlugin::actionOpenHighLights()
{
        SKGTRACEIN ( 10, "SKGOperationPlugin::actionOpenHighLights");
        QString wc="t_bookmarked='Y'";
        QString title=i18n ( "Highlighted operations" );

        //Call operation plugin
        QDomDocument doc("SKGML");
        doc.setContent ( currentBankDocument->getParameter("SKGOPERATION_DEFAULT_PARAMETERS") );
        QDomElement root = doc.documentElement();
        if (root.isNull()) {
                root=doc.createElement("parameters");
                doc.appendChild(root);
        }

        root.setAttribute ( "operationTable", "v_operation_display" );
        root.setAttribute ( "operationWhereClause", wc );
        root.setAttribute ( "title", title );
        root.setAttribute ( "title_icon", "rating" );

        if (SKGMainPanel::getMainPanel()) SKGMainPanel::getMainPanel()->setNewTabContent ( SKGMainPanel::getMainPanel()->getPluginByName ( "Skrooge operation plugin" ), -1, doc.toString() );
}

#include "skgoperationplugin.moc"
