/***************************************************************************
 *   Copyright (C) 2008 by S. MANKOWSKI / G. DE BURE skrooge@miraks.com    *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>  *
 ***************************************************************************/
/** @file
 * This file is Skrooge plugin for debug.
 *
 * @author Stephane MANKOWSKI / Guillaume DE BURE
 */
#include "skgdebugpluginwidget.h"
#include "skgmainpanel.h"
#include "skgdocument.h"
#include "skgtransactionmng.h"
#include "skgservices.h"
#include "skgtraces.h"

#include <QDomDocument>

SKGDebugPluginWidget::SKGDebugPluginWidget(SKGDocument* iDocument)
                :SKGTabWidget(iDocument)
{
        SKGTRACEIN(10, "SKGDebugPluginWidget::SKGDebugPluginWidget");
        ui.setupUi(this);

        //Set icons
        ui.kSQLPushButton->setIcon(KIcon("system-run"));
        ui.kSQLTransactionPushButton->setIcon(KIcon("system-run"));
        ui.kRefreshViewsAndIndexes->setIcon(KIcon("view-refresh"));

        //Fill combo box
        ui.kExplainCmb->addItem (KIcon("system-run"), i18nc("Verb, execute the SQL query", "Execute"));
        ui.kExplainCmb->addItem (KIcon("help-hint"), i18nc("Verb, explain the query","Explain"));
        ui.kExplainCmb->addItem (KIcon("games-hint"), i18n("Explain query plan"));

        //Set level trace
        ui.kTraceLevel->setValue(SKGTraces::SKGLevelTrace);

        //Set profiling mode
        ui.kEnableProfilingChk->setCheckState ( SKGTraces::SKGPerfo ? Qt::Checked : Qt::Unchecked);

        //Init debug page
        QStringList tables;;
        ui.kSQLInput->addItem("SELECT * FROM sqlite_master;");
        SKGServices::getDistinctValues(iDocument, "sqlite_master", "name", "type in ('table', 'view')", tables);
        int nb=tables.count();
        for (int i=0; i<nb; ++i) {
                ui.kSQLInput->addItem("SELECT * FROM "+tables[i]+';');
        }
        ui.kSQLInput->addItem("ANALYZE;");
        ui.kSQLInput->addItem("PRAGMA integrity_check;");
        for (int i=0; i<nb; ++i) {
                ui.kSQLInput->addItem("PRAGMA table_info("+tables[i]+");");
                ui.kSQLInput->addItem("PRAGMA index_list("+tables[i]+");");
        }

        SKGServices::getDistinctValues(iDocument, "sqlite_master", "name", "type='index'", tables);
        nb=tables.count();
        for (int i=0; i<nb; ++i) {
                ui.kSQLInput->addItem("PRAGMA index_info("+tables[i]+");");
        }
}

SKGDebugPluginWidget::~SKGDebugPluginWidget()
{
        SKGTRACEIN(10, "SKGDebugPluginWidget::~SKGDebugPluginWidget");
}

QString SKGDebugPluginWidget::getState()
{
        SKGTRACEIN(10, "SKGDebugPluginWidget::getState");
        QDomDocument doc("SKGML");
        QDomElement root = doc.createElement("parameters");
        doc.appendChild(root);

        root.setAttribute("explain", ui.kExplainCmb->currentIndex());
        root.setAttribute("enableProfiling", ui.kEnableProfilingChk->checkState()==Qt::Checked ? "Y" : "N");
        root.setAttribute("levelTraces", ui.kTraceLevel->value());
        root.setAttribute("sqlOrder", ui.kSQLInput->currentText());

        return doc.toString();
}

void SKGDebugPluginWidget::setState(const QString& iState )
{
        SKGTRACEIN(10, "SKGDebugPluginWidget::setState");
        QDomDocument doc("SKGML");
        doc.setContent(iState);
        QDomElement root = doc.documentElement();

        QString explain=root.attribute ( "explain");
        QString enableProfiling=root.attribute ( "enableProfiling");
        QString levelTraces=root.attribute ( "levelTraces");
        QString sqlOrder=root.attribute ( "sqlOrder");
        QString sqlResult=root.attribute ( "sqlResult");

        if (!explain.isEmpty()) ui.kExplainCmb->setCurrentIndex(SKGServices::stringToInt(explain=="Y" ? "1" : explain));
        if (!enableProfiling.isEmpty()) ui.kEnableProfilingChk->setCheckState (enableProfiling=="Y" ? Qt::Checked : Qt::Unchecked);
        if (!levelTraces.isEmpty()) ui.kTraceLevel->setValue(SKGServices::stringToInt(levelTraces));
        if (!sqlOrder.isEmpty()) ui.kSQLInput->setText(sqlOrder);
        if (!sqlResult.isEmpty()) ui.kSQLResult->setPlainText(sqlResult);
}

void SKGDebugPluginWidget::onExecuteSqlOrderInTransaction()
{
        onExecuteSqlOrder(true);
}

void SKGDebugPluginWidget::onExecuteSqlOrder(bool iInTransaction)
{
        SKGTRACEIN(10, "SKGDebugPluginWidget::onExecuteSqlOrder");
        QApplication::setOverrideCursor(QCursor(Qt::WaitCursor));
        SKGError err;
        QString text=ui.kSQLInput->currentText();
        int exp=ui.kExplainCmb->currentIndex();
        if (exp==1)  text="EXPLAIN "+text;
        else if (exp==2)  text="EXPLAIN QUERY PLAN "+text;
        QString oResult;
        double time=SKGServices::getMicroTime();
        if (iInTransaction) {
                SKGBEGINTRANSACTION(*getDocument(), i18n("SQL command from debug plugin"), err);
                err=SKGServices::dumpSelectSqliteOrder(getDocument(), text, oResult);
        } else {
                err=SKGServices::dumpSelectSqliteOrder(getDocument(), text, oResult);
        }
        time=SKGServices::getMicroTime()-time;

        oResult+=i18n("\nExecution time: %1 ms",SKGServices::doubleToString(time));

        if (err.isSucceeded()) {
                ui.kSQLResult->setPlainText(oResult);
        } else {
                ui.kSQLResult->setPlainText(err.getFullMessageWithHistorical());
        }
        QApplication::restoreOverrideCursor();
}

void SKGDebugPluginWidget::onTraceLevelModified()
{
        SKGTRACEIN(10, "SKGDebugPluginWidget::onTraceLevelModified");
        SKGTraces::SKGLevelTrace=ui.kTraceLevel->value();
}

void SKGDebugPluginWidget::onProfilingModeChanged()
{
        SKGTRACEIN(10, "SKGDebugPluginWidget::onProfilingModeChanged");
        SKGTraces::SKGPerfo=(ui.kEnableProfilingChk->checkState()==Qt::Checked);
}

void SKGDebugPluginWidget::onRefreshViewsAndIndexes()
{
        SKGTRACEIN(10, "SKGDebugPluginWidget::onProfilingModeChanged");
        QApplication::setOverrideCursor(QCursor(Qt::WaitCursor));
        SKGError err;
        err=getDocument()->refreshViewsIndexesAndTriggers();
        if (err.isFailed()) {
                ui.kSQLResult->setPlainText(err.getFullMessageWithHistorical());
        }
        QApplication::restoreOverrideCursor();
}

#include "skgdebugpluginwidget.moc"
