/***************************************************************************
 *   Copyright (C) 2008 by S. MANKOWSKI / G. DE BURE skrooge@miraks.com    *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>  *
 ***************************************************************************/
/** @file
 * This file is Skrooge plugin for bookmarks management.
 *
 * @author Stephane MANKOWSKI / Guillaume DE BURE
 */
#include "skgbookmarkplugin.h"
#include "skgtraces.h"
#include "skgerror.h"
#include "skgmainpanel.h"
#include "skgservices.h"
#include "skgnodeobject.h"
#include "skgtransactionmng.h"
#include "skgbookmarkplugindockwidget.h"

#include <QtGui/QWidget>
#include <QDir>
#include <QFile>

#include <kactioncollection.h>
#include <kstandarddirs.h>

K_PLUGIN_FACTORY(SKGBookmarkPluginFactory, registerPlugin<SKGBookmarkPlugin>();)
K_EXPORT_PLUGIN(SKGBookmarkPluginFactory("skrooge_bookmark", "skrooge_bookmark"))

SKGBookmarkPlugin::SKGBookmarkPlugin(QObject* iParent, const QVariantList& iArg) :
                SKGInterfacePlugin(iParent), dockWidget(NULL)
{
        Q_UNUSED(iArg);
        SKGTRACEIN(10, "SKGBookmarkPlugin::SKGBookmarkPlugin");
}

SKGBookmarkPlugin::~SKGBookmarkPlugin()
{
        SKGTRACEIN(10, "SKGBookmarkPlugin::~SKGBookmarkPlugin");
        currentBankDocument=NULL;
        dockWidget=NULL;
        importStdCatAction=NULL;
}

void SKGBookmarkPlugin::setupActions(SKGDocument* iDocument, const QStringList& iArgument)
{
        SKGTRACEIN(10, "SKGBookmarkPlugin::setupActions");
        Q_UNUSED(iArgument);

        currentBankDocument=iDocument;


        setComponentData( SKGBookmarkPluginFactory::componentData() );
        setXMLFile("skrooge_bookmark.rc");

        dockWidget = new QDockWidget(SKGMainPanel::getMainPanel());
        dockWidget->setObjectName(QString::fromUtf8("skrooge_bookmark_docwidget"));
        dockWidget->setAllowedAreas(Qt::LeftDockWidgetArea|Qt::RightDockWidgetArea);
        dockWidget->setWindowTitle(title());
        dockWidget->setWidget(new SKGBookmarkPluginDockWidget(currentBankDocument));

        // add action to control hide / display of Bookmarks
        dockWidget->toggleViewAction()->setShortcut(Qt::SHIFT+Qt::Key_F10);
        actionCollection()->addAction("view_bookmarks", dockWidget->toggleViewAction());

        //Import bookmarks
        QStringList overlaybookmarks;
        overlaybookmarks.push_back("rating");

        importStdCatAction = new KAction(KIcon("document-import", NULL, overlaybookmarks), i18n("Import standard &bookmarks"), this);
        connect(importStdCatAction, SIGNAL(triggered(bool)), SLOT(importStandardBookmarks()));
        actionCollection()->addAction( QLatin1String("import_standard_bookmarks"), importStdCatAction );

        if (SKGMainPanel::getMainPanel()) SKGMainPanel::getMainPanel()->registedGlobalAction("import_standard_bookmarks", importStdCatAction);
}

void SKGBookmarkPlugin::refresh()
{
        SKGTRACEIN(10, "SKGBookmarkPlugin::refresh");
        if (dockWidget) {
                SKGBookmarkPluginDockWidget* p=(SKGBookmarkPluginDockWidget*) dockWidget->widget();
                if (p) p->refresh();
        }

        if (currentBankDocument) {
                bool test=(currentBankDocument->getDatabase()!=NULL);
                if (importStdCatAction) importStdCatAction->setEnabled(test);

                //Automatic bookmarks creation
                if (currentBankDocument->getDatabase()!=NULL) {
                        QString doc_id=currentBankDocument->getUniqueIdentifier();
                        if (docUniqueIdentifier!=doc_id) {
                                docUniqueIdentifier=doc_id;

                                int nbNode=0;
                                SKGError err=SKGObjectBase::getNbObjects(currentBankDocument, "node", "", nbNode);
                                if (err.isSucceeded() && nbNode==0) {
                                        importStandardBookmarks();
                                }
                        }
                }
        }
}

QString SKGBookmarkPlugin::title() const
{
        return i18nc("Noun, a bookmark as in a webbrowser bookmark","Bookmark");
}

QString SKGBookmarkPlugin::icon() const
{
        return "rating";
}

QString SKGBookmarkPlugin::statusTip () const
{
        return i18nc("Noun, a bookmark as in a webbrowser bookmark","Bookmark");
}

QString SKGBookmarkPlugin::toolTip () const
{
        return i18nc("Noun, a bookmark as in a webbrowser bookmark","Bookmark");
}

QStringList SKGBookmarkPlugin::tips() const
{
        QStringList output;
        QString autostart=i18nc("Verb, automatically load when skrooge is started","autostart");
        output.push_back(i18n("<p>... some bookmarks can be opened automatically when Skrooge is launched."));
        output.push_back(i18n("<p>... bookmarks can be reorganized by drag & drop.</p>"));
        output.push_back(i18n("<p>... a double click on a folder of bookmarks will open all the bookmarks it contains.</p>"));
        output.push_back(i18n("<p>... you can import standard bookmarks.</p>"));
        return output;
}

int SKGBookmarkPlugin::getOrder() const
{
        return 3;
}

void SKGBookmarkPlugin::close()
{
        SKGTRACEIN(10, "SKGBookmarkPlugin::close");
}

QDockWidget* SKGBookmarkPlugin::getDockWidget()
{
        return dockWidget;
}

void SKGBookmarkPlugin::importStandardBookmarks()
{
        SKGTRACEIN(10, "SKGBookmarkPlugin::importStandardBookmarks");
        SKGError err;
        QApplication::setOverrideCursor(QCursor(Qt::WaitCursor));
        QFile file(KStandardDirs::locate("data", QString::fromLatin1("skrooge/resources/default_bookmarks.txt")));
        if (file.open(QIODevice::ReadOnly | QIODevice::Text)) {

                SKGBEGINTRANSACTION(*currentBankDocument, i18n("Import standard bookmarks"), err);
                QString style;
                QTextStream in(&file);
                while (!in.atEnd() && err.isSucceeded()) {
                        QStringList line=SKGServices::splitCSVLine(in.readLine(), '|', false);
                        if (line.count()==2) {
                                SKGNodeObject node;
                                err=SKGNodeObject::createPathNode(currentBankDocument, line.at(0), node);
                                if (err.isSucceeded()) err=node.setData(line.at(1));
                                if (err.isSucceeded()) err=node.save();
                        }
                }

                file.close();
        }
        QApplication::restoreOverrideCursor();

        //status
        if (err.isSucceeded())  err=SKGError(0, i18n("Standard bookmarks imported."));
        else err.addError(ERR_FAIL, i18n("Import standard bookmarks failed"));

        //Display error
        if (SKGMainPanel::getMainPanel()) SKGMainPanel::getMainPanel()->displayErrorMessage(err);
}
#include "skgbookmarkplugin.moc"
