/***************************************************************************
 *   Copyright (C) 2008 by S. MANKOWSKI / G. DE BURE skrooge@miraks.com    *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>  *
 ***************************************************************************/
/** @file
* This file is part of Skrooge and defines classes SKGObjectModelBase.
*
* @author Stephane MANKOWSKI / Guillaume DE BURE
 */
#include "skgobjectmodelbase.h"
#include "skgtraces.h"
#include "skgdocument.h"
#include "skgnodeobject.h"
#include "skgtransactionmng.h"
#include "skgmainpanel.h"

#include <kstandarddirs.h>
#include <klocalizedstring.h>
#include <klocale.h>
#include <kicon.h>

#include <QApplication>
#include <QMimeData>
#include <qcolor.h>
#include <assert.h>

SKGObjectModelBase::SKGObjectModelBase(SKGDocument* iDocument,
                                       const QString& iTable,
                                       const QString& iWhereClause,
                                       QObject *parent,
                                       const QString& iParentAttribute,
                                       bool iResetOnCreation)
                : QAbstractItemModel(parent), isResetRealyNeeded(iResetOnCreation), document(iDocument),
                whereClause(iWhereClause),
                parentAttribute(iParentAttribute),refreshBlocked(false)
{
        SKGTRACEIN(1, "SKGObjectModelBase::SKGObjectModelBase");

        setTable(iTable);

        //Build schemas
        if (getRealTable()=="doctransaction") {
                SKGModelTemplate def;
                def.id="default";
                def.name=i18nc("Noun, the default value of an item","Default");
                def.icon="edit-undo";
                def.schema="t_name;t_value;d_lastmodifdate;t_savestep";
                listSchema.push_back(def);

                SKGModelTemplate minimum;
                minimum.id="minimum";
                minimum.name=i18nc("Noun, the minimum value of an item","Minimum");
                minimum.icon="";
                minimum.schema="t_name;t_value;d_lastmodifdate|N;t_savestep|N";
                listSchema.push_back(minimum);
        } else if (getRealTable()=="parameters") {
                SKGModelTemplate def;
                def.id="default";
                def.name=i18nc("Noun, the default value of an item","Default");
                def.icon="edit-undo";
                def.schema="t_name;t_value";
                listSchema.push_back(def);
        } else if (getRealTable()=="node") {
                SKGModelTemplate def;
                def.id="default";
                def.name=i18nc("Noun, the default value of an item","Default");
                def.icon="edit-undo";
                def.schema="t_name";
                listSchema.push_back(def);
        }

        connect((const QObject*) document, SIGNAL(tableModified(QString, int)), this, SLOT(dataModified(QString, int)));
        refresh();
}

SKGObjectModelBase::~SKGObjectModelBase()
{
        SKGTRACEIN(1, "SKGObjectModelBase::~SKGObjectModelBase");
        clear();
        document=NULL;
}

void SKGObjectModelBase::clear()
{
        SKGTRACEIN(1, "SKGObjectModelBase::clear");
        QHashIterator<int, SKGObjectBase*> i(objectsHashTable);
        while (i.hasNext()) {
                i.next();
                SKGObjectBase* val=i.value();
                if (val) {
                        delete val;
                        val=NULL;
                }
        }

        listObjects.clear();
        parentChildRelations.clear();
        childParentRelations.clear();
        objectsHashTable.clear();
        objectsHashTableRows.clear();
}

int SKGObjectModelBase::getID(const SKGObjectBase* obj, int row) const
{
        if (!parentAttribute.isEmpty()) return (obj ? obj->getID() : 0);
        return (obj ? 100*(obj->getID())+row : 0);
}

QList<SKGObjectModelBase::SKGModelTemplate> SKGObjectModelBase::getSchemas() const
{
        return listSchema;
}

void SKGObjectModelBase::setSupportedAttributes ( const QStringList & iListAttribute )
{
        SKGTRACEIN(1, "SKGObjectModelBase::setSupportedAttributes");
        listSupported.clear();
        listVisibility.clear();
        listSize.clear();

        QStringList l=iListAttribute;
        if (l.count()==0 && listSchema.count())  l=SKGServices::splitCSVLine(listSchema.at(0).schema);

        int nb=l.count();
        for (int i=0; i<nb; ++i) {
                QStringList values = l.at(i).split('|');
                int nbValues=values.count();

                if (nbValues>0) {
                        listSupported.push_back(values.at(0));
                        bool visible=true;
                        if (nbValues>1) visible=(values.at(1)=="Y");
                        listVisibility.push_back(visible);
                        if (nbValues>2) listSize.push_back(SKGServices::stringToInt(values.at(2)));
                        else listSize.push_back(-1);
                }
        }
        isResetRealyNeeded=true;
}

void SKGObjectModelBase::setFilter ( const QString & iWhereClause )
{
        if (iWhereClause!=whereClause) isResetRealyNeeded=true;
        whereClause=iWhereClause;
}

void SKGObjectModelBase::setTable ( const QString & iTable )
{
        if (iTable!=table && !table.isEmpty()) isResetRealyNeeded=true;
        table=iTable;
        realTable=SKGServices::getRealTable(table);
}

QString SKGObjectModelBase::getTable() const
{
        return table;
}

QString SKGObjectModelBase::getRealTable() const
{
        return realTable;
}

QString SKGObjectModelBase::getWhereClause() const
{
        return whereClause;
}

SKGDocument* SKGObjectModelBase::getDocument() const
{
        return document;
}

void SKGObjectModelBase::buidCache()
{
        SKGTRACEIN(1, "SKGObjectModelBase::buidCache");
        doctransactionTable=(getRealTable()=="doctransaction");
        nodeTable=(getRealTable()=="node");
}

bool SKGObjectModelBase::blockRefresh(bool iBlocked)
{
        bool previous=refreshBlocked;
        refreshBlocked=iBlocked;
        return previous;
}

void SKGObjectModelBase::refresh()
{
        if (!isResetRealyNeeded || refreshBlocked) return;
        SKGTRACEIN(1, "SKGObjectModelBase::refresh");
        emit beforeReset();
        {
                _SKGTRACEIN(10, "QAbstractItemModel::refresh-remove");
                clear();
                listAttibutes.clear();
                listAttibuteTypes.clear();
                /*beginRemoveRows(QModelIndex(), 0, rowCount(QModelIndex())-1);
                endRemoveRows();*/
        }

        {
                _SKGTRACEIN(10, "QAbstractItemModel::refresh-fill");
                QStringList listAttibutesTmp;
                if (SKGServices::getAttributesList(document, table, listAttibutesTmp).isSucceeded()) {
                        isResetRealyNeeded=false;
                        if (listAttibutesTmp.count()) {
                                //Filter attributes
                                int nb=listSupported.count();
                                if (nb==0) {
                                        setSupportedAttributes(QStringList());
                                        nb=listSupported.count();
                                }
                                for (int i=0 ; i<nb ; ++i) {
                                        QString att=listSupported[i];
                                        if (listAttibutesTmp.contains(att)) {
                                                listAttibutes.push_back(att);
                                                if (att.startsWith(QLatin1String("t_"))) listAttibuteTypes.push_back(SKGObjectModelBase::STRING);
                                                else if (att.startsWith(QLatin1String("f_"))) listAttibuteTypes.push_back(SKGObjectModelBase::DOUBLE);
                                                else if (att.startsWith(QLatin1String("i_"))) listAttibuteTypes.push_back(SKGObjectModelBase::INTEGER);
                                                else if (att.startsWith(QLatin1String("d_"))) listAttibuteTypes.push_back(SKGObjectModelBase::DATE);
                                                else listAttibuteTypes.push_back(SKGObjectModelBase::OTHER);
                                        }
                                }

                                //Remove double
                                nb=listAttibutes.count();
                                for (int i=nb-1 ; i>=0 ; --i) {
                                        QString att=listAttibutes[i];
                                        if (att.contains("_REAL")) {
                                                att.replace("_REAL", "_");
                                                int p=listAttibutes.indexOf(att);
                                                if (p==-1) {
                                                        att=att.toLower();
                                                        p=listAttibutes.indexOf(att);
                                                }
                                                if (p!=-1) {
                                                        ;
                                                        listAttibutes.removeAt(p);
                                                        listAttibuteTypes.removeAt(p);
                                                        if (p<i) --i;
                                                }
                                        }
                                }
                        }

                        //Get objects
                        SKGObjectBase::getObjects(document, table, whereClause, listObjects);

                        QList<SKGObjectBase*> listObjectToTreat;
                        QList<int> listObjectToTreatId;

                        //Initialize object to treat
                        int nb=listObjects.count();
                        SKGTRACEL(1) << nb << " objects found" << endl;
                        for (int t=0; t<nb; ++t) {
                                SKGObjectBase* c=new SKGObjectBase(listObjects.at(t));
                                int id=getID(c, t);

                                int idparent=0;
                                if (!parentAttribute.isEmpty()) {
                                        int idp=SKGServices::stringToInt(c->getAttribute(parentAttribute));
                                        if (idp>0) idparent=idp;
                                }

                                listObjectToTreat.push_back(c);
                                listObjectToTreatId.push_back(id);

                                childParentRelations.insert(id, idparent);

                                QList<int> childrensids=parentChildRelations.value(idparent);
                                childrensids.push_back(id);

                                parentChildRelations.insert(idparent, childrensids);
                                objectsHashTableRows.insert(id, childrensids.count()-1);
                                objectsHashTable.insert(id, c);
                        }
                }

                //Build cache
                buidCache();

                {
                        SKGTRACEIN(1, "SKGObjectModelBase::refresh-reset");
                        QAbstractItemModel::reset();
                }
        }
        emit afterReset();
}

int SKGObjectModelBase::rowCount(const QModelIndex &parent ) const
{
        if (parent.column()>0) return 0;
        _SKGTRACEIN(10, "SKGObjectModelBase::rowCount");

        int idParent=0;
        if (parent.isValid()) idParent = parent.internalId();

        QList<int> children=parentChildRelations.value(idParent);
        //SKGTRACE << table << "-rowCount(" << idParent << " )=" << children.count() << endl;
        return children.count();

}

int SKGObjectModelBase::columnCount(const QModelIndex &parent ) const
{
        Q_UNUSED(parent);
        return listAttibutes.count();
}

QModelIndex SKGObjectModelBase::index ( int row, int column, const QModelIndex & parent) const
{
        if (!hasIndex(row, column, parent)) return QModelIndex();
        _SKGTRACEIN(10, "SKGObjectModelBase::index");

        int idParent=0;
        if (parent.isValid())  idParent = parent.internalId();

        int idChild=parentChildRelations.value(idParent).at(row);

        //SKGTRACE << table << "-" << idParent << "(" << row << ") ==> " << idChild << endl;
        return (idChild ? createIndex(row, column, idChild) : QModelIndex());
}

QModelIndex SKGObjectModelBase::parent ( const QModelIndex & index ) const
{
        if (!index.isValid()) return QModelIndex();
        _SKGTRACEIN(10, "SKGObjectModelBase::parent");

        int idChild=0;
        if (index.isValid()) idChild=index.internalId();

        int idParent=childParentRelations.value(idChild);
        int row=objectsHashTableRows.value(idParent);
        //SKGTRACE << table << "-" << idChild << "(" << row << ") <== " << idParent << endl;
        return idParent ? createIndex(row, 0, idParent) : QModelIndex();
}

int SKGObjectModelBase::getIndexAttribute (const QString& iAttributeName ) const
{
        int output=listAttibutes.indexOf(iAttributeName);
        if (output==-1) {
                SKGTRACE << "[" << iAttributeName << "] not found in [" << getRealTable() << "]" << endl;
        }
        return output;
}

QString SKGObjectModelBase::getAttribute (int iIndex ) const
{
        return listAttibutes.value(iIndex);
}

SKGObjectModelBase::AttributeType SKGObjectModelBase::getAttributeType (int iIndex ) const
{
        return listAttibuteTypes.value(iIndex);
}

QVariant SKGObjectModelBase::headerData ( int section, Qt::Orientation orientation, int role ) const
{
        _SKGTRACEIN(10, "SKGObjectModelBase::headerData");

        if (orientation==Qt::Horizontal) {
                if (role == Qt::DisplayRole) {
                        QString att;
                        if (section>=0 && section< listAttibutes.count()) att=listAttibutes[section];
                        else att=SKGServices::intToString(section);

                        return getDocument()->getDisplay(getTable()+'.'+att);
                } else if (role == Qt::UserRole ) {
                        QString att;
                        if (section>=0 && section< listAttibutes.count()) att=listAttibutes.at(section);
                        else att=SKGServices::intToString(section);

                        int indexAtt=listSupported.indexOf(att);

                        att=getDocument()->getDisplay(getTable()+'.'+att);


                        if (indexAtt>=0 && indexAtt< listVisibility.count()) {
                                bool visible=listVisibility.at(indexAtt);
                                att+=QString("|")+(visible ? "Y" : "N");
                                if (indexAtt>=0 && indexAtt< listSize.count()) att+='|'+SKGServices::intToString(listSize.at(indexAtt));
                        }
                        return att;
                } else if ( role == Qt::DecorationRole ) {
                        QString att;
                        if ( section>=0 && section< listAttibutes.count() ) att=listAttibutes[section];
                        else att=SKGServices::intToString ( section );

                        return getDocument()->getIcon(getTable()+'.'+att);
                }
        }
        return QVariant();
}

SKGObjectBase SKGObjectModelBase::getObject(const QModelIndex &index) const
{
        SKGObjectBase* obj = getObjectPointer(index);
        SKGObjectBase output;
        if (obj!=NULL) output=*obj;
        return output;
}

SKGObjectBase* SKGObjectModelBase::getObjectPointer(const QModelIndex &index) const
{
        _SKGTRACEIN(10, "SKGObjectModelBase::getObject");
        //SKGTRACE << "getObject:" << index.internalId() << endl;
        return objectsHashTable.value(index.internalId());
}

QVariant SKGObjectModelBase::data(const QModelIndex &index, int role) const
{
        if (!index.isValid()) return QVariant();
        _SKGTRACEIN(10, "SKGObjectModelBase::data");

        switch ( role ) {
        case Qt::DisplayRole:
        case Qt::EditRole:
        case Qt::UserRole: {
                SKGObjectBase* obj = getObjectPointer(index);
                QString att=listAttibutes[index.column()];
                QString val=obj->getAttribute(att);

                switch ( getAttributeType(index.column()) ) {
                case SKGObjectModelBase::DOUBLE: {
                        double dval=SKGServices::stringToDouble(val);
                        return dval;
                }
                case SKGObjectModelBase::INTEGER: {
                        return SKGServices::stringToInt(val);
                }
                case SKGObjectModelBase::DATE: {
                        QDate dval=SKGServices::stringToTime(val).date();
                        if (role == Qt::DisplayRole) {
                                return KGlobal::locale()->formatDate (dval, KLocale::FancyShortDate);
                        } else {
                                return dval;
                        }
                }
                default: {}
                }

                if (doctransactionTable && att=="t_savestep") return "";

                //return val+"("+SKGServices::intToString(rowCount(index))+")";
                return val;
        }

        case Qt::TextColorRole: {
                //Text color
                if (getAttributeType(index.column())==SKGObjectModelBase::DOUBLE) {
                        QVariant value_displayed = SKGObjectModelBase::data(index, Qt::UserRole);
                        bool ok=false;
                        double value_double=value_displayed.toDouble(&ok);
                        if (ok && value_double<0) return qVariantFromValue(QColor(Qt::red));
                }
                break;
        }
        case Qt::TextAlignmentRole: {
                //Text alignment
                SKGObjectModelBase::AttributeType attType=getAttributeType(index.column());
                return (int)(Qt::AlignVCenter|(attType==SKGObjectModelBase::DOUBLE || attType==SKGObjectModelBase::INTEGER ? Qt::AlignRight : Qt::AlignLeft));
        }
        case Qt::DecorationRole: {
                //Decoration
                SKGObjectBase* obj = getObjectPointer(index);
                if (index.column() == 0 && nodeTable) {
                        SKGNodeObject node =*obj;
                        QStringList data=SKGServices::splitCSVLine(node.getData());
                        QIcon icon=KIcon("folder");
                        if (data.count()>2)  icon=(QIcon) KIcon(data.at(2));

                        if (node.isAutoStart()) {
                                //Modify icon
                                QStringList overlay;
                                overlay.push_back("user-online");
                                if (data.count()>2)  icon=(QIcon) KIcon(data.at(2), NULL, overlay);
                                else icon=(QIcon) KIcon("folder", NULL, overlay);
                        }
                        return qVariantFromValue(icon);
                } else if (index.column() == 0 && doctransactionTable) {
                        return qVariantFromValue((QIcon) KIcon(obj->getAttribute("t_mode")=="U" ? "edit-undo" :"edit-redo"));
                } else if (doctransactionTable && listAttibutes[index.column()]=="t_savestep") {
                        if (obj->getAttribute("t_savestep")=="Y") return qVariantFromValue((QIcon) KIcon("document-save"));
                }
                break;
        }
        case Qt::ToolTipRole: {
                //Tooltip
                if (doctransactionTable) {
                        SKGObjectBase* obj = getObjectPointer(index);
                        if (obj && document) {
                                QStringList msg;
                                document->getMessages(obj->getID(), msg);
                                int nbMessages=msg.count();
                                if (nbMessages) {
                                        QString message;
                                        for (int i=0; i<nbMessages; ++i) {
                                                if (i!=0) message+='\n';
                                                message+=msg.at(i);
                                        }

                                        return message;
                                }
                        }
                } else if (getAttributeType(index.column())==SKGObjectModelBase::DATE) {
                        SKGObjectBase* obj = getObjectPointer(index);
                        if (obj) {
                                QString att=listAttibutes[index.column()];
                                QString val=obj->getAttribute(att);

                                QDate dval=SKGServices::stringToTime(val).date();
                                QString fancyDate=KGlobal::locale()->formatDate (dval, KLocale::FancyShortDate);
                                QString shortDate=KGlobal::locale()->formatDate (dval, KLocale::ShortDate);


                                if (shortDate!=fancyDate) return shortDate;
                        }
                }
                IFSKGTRACEL(1) {
                        SKGObjectBase* obj = getObjectPointer(index);
                        if (obj) {
                                return obj->getUniqueID();
                        }
                }
                break;
        }
        case 99: {
                SKGObjectBase* obj = getObjectPointer(index);
                if (obj) return SKGServices::stringToDouble(obj->getAttribute("f_sortorder"));

                break;
        }
        default : {}
        }

        return QVariant();
}

Qt::ItemFlags SKGObjectModelBase::flags(const QModelIndex &index) const
{
        _SKGTRACEIN(10, "SKGObjectModelBase::flags");

        Qt::ItemFlags flags=QAbstractItemModel::flags(index);

        if (nodeTable) {
                flags|=Qt::ItemIsEditable;
                if (index.isValid()) flags|=Qt::ItemIsDragEnabled | Qt::ItemIsDropEnabled;
                else flags|=Qt::ItemIsDropEnabled;
        }
        if (index.column() == 0) flags|=Qt::ItemIsUserCheckable;
        return flags;

}

bool SKGObjectModelBase::setData ( const QModelIndex & index, const QVariant& value, int role )
{
        if ( !index.isValid() ) return false;

        if ( role == Qt::EditRole ) {
                SKGError err;
                if ( nodeTable ) {
                        SKGNodeObject obj=getObject ( index );
                        QString name=value.toString();
                        SKGBEGINTRANSACTION(*getDocument(), i18n("Bookmark update [%1]", name), err);
                        err=obj.setName(name);
                        if (err.isSucceeded()) err=obj.save();
                }

                SKGMainPanel::displayErrorMessage ( err );
                return err.isSucceeded();
        }
        return QAbstractItemModel::setData ( index, value, role );
}

Qt::DropActions SKGObjectModelBase::supportedDragActions() const
{
        return (nodeTable ? Qt::MoveAction : Qt::IgnoreAction);
}

Qt::DropActions SKGObjectModelBase::supportedDropActions() const
{
        return (nodeTable ? Qt::MoveAction : Qt::IgnoreAction);
}

QStringList SKGObjectModelBase::mimeTypes() const
{
        QStringList types;
        types << "application/skrooge."+getRealTable()+".ids";
        return types;
}

QMimeData* SKGObjectModelBase::mimeData(const QModelIndexList &indexes) const
{
        QMimeData *mimeData = new QMimeData();
        QByteArray encodedData;

        QDataStream stream(&encodedData, QIODevice::WriteOnly);

        QString t=getTable();
        foreach (const QModelIndex& index, indexes) {
                if (index.isValid()) {
                        SKGObjectBase obj=getObject(index);
                        t=obj.getRealTable();
                        stream << t;
                        stream << obj.getID();
                }
        }

        mimeData->setData("application/skrooge."+t+".ids", encodedData);
        return mimeData;
}

bool SKGObjectModelBase::dropMimeData(const QMimeData *data,
                                      Qt::DropAction action,
                                      int row, int column,
                                      const QModelIndex &parent)
{
        Q_UNUSED(row);
        if (action == Qt::IgnoreAction) return true;
        if (!data || !data->hasFormat("application/skrooge.node.ids")) return false;
        if (column > 0) return false;

        QByteArray encodedData = data->data("application/skrooge.node.ids");
        QDataStream stream(&encodedData, QIODevice::ReadOnly);
        QStringList newItems;

        SKGError err;
        SKGNodeObject parentNode;
        if (parent.isValid()) parentNode=getObject(parent);
        {
                SKGBEGINTRANSACTION(*getDocument(), i18n("Move bookmark"), err);

                double min=0;
                double max=0;
                if (row>=1) {
                        QModelIndex previousIndex=SKGObjectModelBase::index (row-1, 0, parent);
                        SKGNodeObject previousObject = getObject(previousIndex);
                        min=previousObject.getOrder();
                }

                if (row>=rowCount(parent)) max=min+1;
                else {
                        QModelIndex nextIndex=SKGObjectModelBase::index (row, 0, parent);
                        SKGNodeObject nextObject = getObject(nextIndex);
                        max=nextObject.getOrder();
                }
                if (max<=min) max=min+1;

                while (!stream.atEnd() && err.isSucceeded()) {
                        int o_id;
                        QString o_table;
                        stream >> o_table;
                        stream >> o_id;

                        //Set parent
                        SKGNodeObject child(getDocument(), o_id);
                        err=child.load();
                        if (err.isSucceeded()) {
                                if (parent.isValid()) err=child.setParentNode(parentNode);
                                else err=child.removeParentNode();
                        }

                        //Set order
                        if (err.isSucceeded())  err=child.setOrder((min+max)/2.0);

                        //Save
                        if (err.isSucceeded()) err=child.save();
                }
        }

        SKGMainPanel::displayErrorMessage(err);
        return err.isSucceeded();
}

bool SKGObjectModelBase::canFetchMore ( const QModelIndex & parent ) const
{
        Q_UNUSED(parent);
        _SKGTRACEIN(10, "SKGObjectModelBase::canFetchMore");
        return false;
}

void SKGObjectModelBase::dataModified(const QString& iTableName, int iIdTransaction)
{
        if (getRealTable()==iTableName || iTableName.isEmpty()) {
                SKGTRACEIN(1, "SKGObjectModelBase::dataModified");
                SKGTRACEL(1) << "getTable=" << getRealTable() << endl;
                SKGTRACEL(1) << "Parameters=" << iTableName << " , " << iIdTransaction << endl;

                //Full refresh
                isResetRealyNeeded=true;

                //Refresh model
                refresh();
        }
}


#include "skgobjectmodelbase.moc"

