/***************************************************************************
 *   Copyright (C) 2008 by S. MANKOWSKI / G. DE BURE skrooge@miraks.com    *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>  *
 ***************************************************************************/
#ifndef SKGCATEGORYOBJECT_H
#define SKGCATEGORYOBJECT_H
/** @file
 * This file is part of Skrooge and defines classes SKGCategoryObject.
 *
 * @author Stephane MANKOWSKI / Guillaume DE BURE
 */

#include "skgnamedobject.h"
#include "skgbankmodeler_export.h"
class SKGDocumentBank;
/**
 * This class manages category object
 */
class SKGBANKMODELER_EXPORT SKGCategoryObject : public SKGNamedObject
{
        Q_OBJECT;
public:
        /**
         * Constructor
         * @param iDocument the document containing the object
         * @param iID the identifier in @p iTable of the object
         */
        explicit SKGCategoryObject(const SKGDocument* iDocument=NULL, int iID=0);

        /**
         * Copy constructor
         * @param iObject the object to copy
         */
        SKGCategoryObject(const SKGCategoryObject& iObject);

        /**
         * Copy constructor
         * @param iObject the object to copy
         */
        SKGCategoryObject(const SKGObjectBase& iObject);

        /**
         * Operator affectation
         * @param iObject the object to copy
         */
        virtual const SKGCategoryObject& operator= (const SKGObjectBase &iObject);

        /**
         * Destructor
         */
        virtual ~SKGCategoryObject();

        /**
         * Create a category branch if needed and return the leaf of the branch
         * @param iDocument the document where to create
         * @param iFullPath the full path. Example: cat1 > cat2 > cat3
         * @param oCategory the leaf of the branch
         * @param iSendPopupMessageOnCreation to send a creation message if the category is created
         * @return an object managing the error.
         *   @see SKGError
         */
        static SKGError createPathCategory(const SKGDocumentBank* iDocument,
                                           const QString& iFullPath,
                                           SKGCategoryObject& oCategory,
                                           bool iSendPopupMessageOnCreation=false);

        /**
         * Set the name of this object
         * @param iName the name
         * @return an object managing the error
         *   @see SKGError
         */
        virtual SKGError setName(const QString& iName);

        /**
         * Get the full name of this category.
         * The full name is the unique name of the category.
         * It is computed by the concatenation of names for all
         * the fathers of this category.
         * @return the full name
         */
        virtual QString getFullName() const;

        /**
         * Add a category
         * @param oCategory the created category
         * @return an object managing the error.
         *   @see SKGError
         */
        virtual SKGError addCategory(SKGCategoryObject& oCategory);

        /**
         * Move the category by changing the parent
         * @param iCategory the parent category
         * @return an object managing the error.
         *   @see SKGError
         */
        virtual SKGError setParentCategory(const SKGCategoryObject& iCategory);

        /**
         * Remove the parent category. The category will be a root.
         * @return an object managing the error.
         *   @see SKGError
         */
        virtual SKGError removeParentCategory();

        /**
         * Get the parent category
         * @param oCategory the parent category
         * @return an object managing the error.
         *   @see SKGError
         */
        virtual SKGError getParentCategory(SKGCategoryObject& oCategory) const;

        /**
         * Get the root category
         * @param oCategory the root category
         * @return an object managing the error.
         *   @see SKGError
         */
        virtual SKGError getRootCategory(SKGCategoryObject& oCategory) const;

        /**
         * Get categories
         * @param oCategoryList the list of categories under the current one
         * @return an object managing the error
         *   @see SKGError
         */
        virtual SKGError getCategories(SKGListSKGObjectBase& oCategoryList) const;

protected:
        /**
         * Get where clause needed to identify objects.
         * For this class, the whereclause is based on name + r_category_id
         * @return the where clause
         */
        virtual QString getWhereclauseId() const;
};

#endif
