/***************************************************************************
 *   Copyright (C) 2007 by Jean-Baptiste Mardelle (jb@kdenlive.org)        *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA          *
 ***************************************************************************/

#include "projectlist.h"
#include "projectitem.h"
#include "addfoldercommand.h"
#include "kdenlivesettings.h"
#include "slideshowclip.h"
#include "ui_colorclip_ui.h"
#include "titlewidget.h"
#include "definitions.h"
#include "clipmanager.h"
#include "docclipbase.h"
#include "kdenlivedoc.h"
#include "renderer.h"
#include "kthumb.h"
#include "projectlistview.h"
#include "editclipcommand.h"
#include "editfoldercommand.h"
#include "ui_templateclip_ui.h"

#include <KDebug>
#include <KAction>
#include <KLocale>
#include <KFileDialog>
#include <KInputDialog>
#include <KMessageBox>
#include <KIO/NetAccess>
#include <KFileItem>

#include <nepomuk/global.h>
#include <nepomuk/resourcemanager.h>
//#include <nepomuk/tag.h>

#include <QMouseEvent>
#include <QStylePainter>
#include <QPixmap>
#include <QIcon>
#include <QMenu>
#include <QProcess>
#include <QHeaderView>

ProjectList::ProjectList(QWidget *parent) :
        QWidget(parent),
        m_render(NULL),
        m_fps(-1),
        m_commandStack(NULL),
        m_editAction(NULL),
        m_deleteAction(NULL),
        m_openAction(NULL),
        m_reloadAction(NULL),
        m_transcodeAction(NULL),
        m_selectedItem(NULL),
        m_refreshed(false),
        m_infoQueue(),
        m_thumbnailQueue()
{

    m_listView = new ProjectListView(this);;
    QVBoxLayout *layout = new QVBoxLayout;
    layout->setContentsMargins(0, 0, 0, 0);

    // setup toolbar
    KTreeWidgetSearchLine *searchView = new KTreeWidgetSearchLine(this);
    m_toolbar = new QToolBar("projectToolBar", this);
    m_toolbar->addWidget(searchView);
    searchView->setTreeWidget(m_listView);

    m_addButton = new QToolButton(m_toolbar);
    m_addButton->setPopupMode(QToolButton::MenuButtonPopup);
    m_toolbar->addWidget(m_addButton);

    layout->addWidget(m_toolbar);
    layout->addWidget(m_listView);
    setLayout(layout);

    m_queueTimer.setInterval(100);
    connect(&m_queueTimer, SIGNAL(timeout()), this, SLOT(slotProcessNextClipInQueue()));
    m_queueTimer.setSingleShot(true);



    connect(m_listView, SIGNAL(itemSelectionChanged()), this, SLOT(slotClipSelected()));
    connect(m_listView, SIGNAL(focusMonitor()), this, SLOT(slotClipSelected()));
    connect(m_listView, SIGNAL(pauseMonitor()), this, SLOT(slotPauseMonitor()));
    connect(m_listView, SIGNAL(requestMenu(const QPoint &, QTreeWidgetItem *)), this, SLOT(slotContextMenu(const QPoint &, QTreeWidgetItem *)));
    connect(m_listView, SIGNAL(addClip()), this, SLOT(slotAddClip()));
    connect(m_listView, SIGNAL(addClip(const QList <QUrl>, const QString &, const QString &)), this, SLOT(slotAddClip(const QList <QUrl>, const QString &, const QString &)));
    connect(m_listView, SIGNAL(itemChanged(QTreeWidgetItem *, int)), this, SLOT(slotItemEdited(QTreeWidgetItem *, int)));
    connect(m_listView, SIGNAL(showProperties(DocClipBase *)), this, SIGNAL(showClipProperties(DocClipBase *)));

    m_listViewDelegate = new ItemDelegate(m_listView);
    m_listView->setItemDelegate(m_listViewDelegate);

    if (KdenliveSettings::activate_nepomuk()) {
        Nepomuk::ResourceManager::instance()->init();
        if (!Nepomuk::ResourceManager::instance()->initialized()) {
            kDebug() << "Cannot communicate with Nepomuk, DISABLING it";
            KdenliveSettings::setActivate_nepomuk(false);
        }
    }
}

ProjectList::~ProjectList()
{
    delete m_menu;
    delete m_toolbar;
    m_listView->blockSignals(true);
    m_listView->clear();
    delete m_listViewDelegate;
}

void ProjectList::setupMenu(QMenu *addMenu, QAction *defaultAction)
{
    QList <QAction *> actions = addMenu->actions();
    for (int i = 0; i < actions.count(); i++) {
        if (actions.at(i)->data().toString() == "clip_properties") {
            m_editAction = actions.at(i);
            m_toolbar->addAction(m_editAction);
            actions.removeAt(i);
            i--;
        } else if (actions.at(i)->data().toString() == "delete_clip") {
            m_deleteAction = actions.at(i);
            m_toolbar->addAction(m_deleteAction);
            actions.removeAt(i);
            i--;
        } else if (actions.at(i)->data().toString() == "edit_clip") {
            m_openAction = actions.at(i);
            actions.removeAt(i);
            i--;
        } else if (actions.at(i)->data().toString() == "reload_clip") {
            m_reloadAction = actions.at(i);
            actions.removeAt(i);
            i--;
        }
    }

    QMenu *m = new QMenu();
    m->addActions(actions);
    m_addButton->setMenu(m);
    m_addButton->setDefaultAction(defaultAction);
    m_menu = new QMenu();
    m_menu->addActions(addMenu->actions());
}

void ProjectList::setupGeneratorMenu(QMenu *addMenu, QMenu *transcodeMenu)
{
    if (!addMenu) return;
    QMenu *menu = m_addButton->menu();
    menu->addMenu(addMenu);
    m_addButton->setMenu(menu);

    m_menu->addMenu(addMenu);
    if (addMenu->isEmpty()) addMenu->setEnabled(false);
    m_menu->addMenu(transcodeMenu);
    if (transcodeMenu->isEmpty()) transcodeMenu->setEnabled(false);
    m_transcodeAction = transcodeMenu;
    m_menu->addAction(m_reloadAction);
    m_menu->addAction(m_editAction);
    m_menu->addAction(m_openAction);
    m_menu->addAction(m_deleteAction);
    m_menu->insertSeparator(m_deleteAction);
}


QByteArray ProjectList::headerInfo() const
{
    return m_listView->header()->saveState();
}

void ProjectList::setHeaderInfo(const QByteArray &state)
{
    m_listView->header()->restoreState(state);
}

void ProjectList::slotEditClip()
{
    ProjectItem *item = static_cast <ProjectItem*>(m_listView->currentItem());
    if (!(item->flags() & Qt::ItemIsDragEnabled)) return;
    if (item && !item->isGroup()) {
        emit clipSelected(item->referencedClip());
        emit showClipProperties(item->referencedClip());
    }
}

void ProjectList::slotOpenClip()
{
    ProjectItem *item = static_cast <ProjectItem*>(m_listView->currentItem());
    if (item && !item->isGroup()) {
        if (item->clipType() == IMAGE) {
            if (KdenliveSettings::defaultimageapp().isEmpty()) KMessageBox::sorry(this, i18n("Please set a default application to open images in the Settings dialog"));
            else QProcess::startDetached(KdenliveSettings::defaultimageapp(), QStringList() << item->clipUrl().path());
        }
        if (item->clipType() == AUDIO) {
            if (KdenliveSettings::defaultaudioapp().isEmpty()) KMessageBox::sorry(this, i18n("Please set a default application to open audio files in the Settings dialog"));
            else QProcess::startDetached(KdenliveSettings::defaultaudioapp(), QStringList() << item->clipUrl().path());
        }
    }
}

void ProjectList::cleanup()
{
    m_listView->clearSelection();
    QTreeWidgetItemIterator it(m_listView);
    ProjectItem *item;
    while (*it) {
        item = static_cast <ProjectItem *>(*it);
        if (item->numReferences() == 0) item->setSelected(true);
        it++;
    }
    slotRemoveClip();
}

void ProjectList::trashUnusedClips()
{
    QTreeWidgetItemIterator it(m_listView);
    ProjectItem *item;
    QStringList ids;
    KUrl::List urls;
    while (*it) {
        item = static_cast <ProjectItem *>(*it);
        if (item->numReferences() == 0) {
            ids << item->clipId();
            KUrl url = item->clipUrl();
            if (!url.isEmpty()) urls << url;
        }
        it++;
    }
    m_doc->deleteProjectClip(ids);
    for (int i = 0; i < urls.count(); i++) {
        KIO::NetAccess::del(urls.at(i), this);
    }
}

void ProjectList::slotReloadClip(const QString &id)
{
    QList<QTreeWidgetItem *> selected;
    if (id.isEmpty()) selected = m_listView->selectedItems();
    else selected.append(getItemById(id));
    ProjectItem *item;
    for (int i = 0; i < selected.count(); i++) {
        item = static_cast <ProjectItem *>(selected.at(i));
        if (item && !item->isGroup()) {
            if (item->clipType() == IMAGE) {
                item->referencedClip()->producer()->set("force_reload", 1);
            } else if (item->clipType() == TEXT) {
                if (!item->referencedClip()->getProperty("xmltemplate").isEmpty()) regenerateTemplate(item);
            }
            //requestClipInfo(item->toXml(), item->clipId(), true);
            // Clear the file_hash value, which will cause a complete reload of the clip
            emit getFileProperties(item->toXml(), item->clipId(), true);
        }
    }
}

void ProjectList::setRenderer(Render *projectRender)
{
    m_render = projectRender;
    m_listView->setIconSize(QSize(40 * m_render->dar(), 40));
}

void ProjectList::slotClipSelected()
{
    if (m_listView->currentItem()) {
        ProjectItem *clip = static_cast <ProjectItem*>(m_listView->currentItem());
        if (!clip->isGroup()) {
            m_selectedItem = clip;
            emit clipSelected(clip->referencedClip());
        }
        m_editAction->setEnabled(true);
        m_deleteAction->setEnabled(true);
        m_reloadAction->setEnabled(true);
        m_transcodeAction->setEnabled(true);
        if (clip->clipType() == IMAGE && !KdenliveSettings::defaultimageapp().isEmpty()) {
            m_openAction->setIcon(KIcon(KdenliveSettings::defaultimageapp()));
            m_openAction->setEnabled(true);
        } else if (clip->clipType() == AUDIO && !KdenliveSettings::defaultaudioapp().isEmpty()) {
            m_openAction->setIcon(KIcon(KdenliveSettings::defaultaudioapp()));
            m_openAction->setEnabled(true);
        } else m_openAction->setEnabled(false);
    } else {
        emit clipSelected(NULL);
        m_editAction->setEnabled(false);
        m_deleteAction->setEnabled(false);
        m_openAction->setEnabled(false);
        m_reloadAction->setEnabled(false);
        m_transcodeAction->setEnabled(false);
    }
}

void ProjectList::slotPauseMonitor()
{
    if (m_render) m_render->pause();
}

void ProjectList::slotUpdateClipProperties(const QString &id, QMap <QString, QString> properties)
{
    ProjectItem *item = getItemById(id);
    if (item) {
        slotUpdateClipProperties(item, properties);
        if (properties.contains("out")) {
            slotReloadClip(id);
        } else if (properties.contains("colour") || properties.contains("resource") || properties.contains("xmldata") || properties.contains("force_aspect_ratio") || properties.contains("templatetext")) {
            slotRefreshClipThumbnail(item);
            emit refreshClip();
        }
    }
}

void ProjectList::slotUpdateClipProperties(ProjectItem *clip, QMap <QString, QString> properties)
{
    if (!clip) return;
    if (!clip->isGroup()) clip->setProperties(properties);
    if (properties.contains("name")) {
        m_listView->blockSignals(true);
        clip->setText(1, properties.value("name"));
        m_listView->blockSignals(false);
        emit clipNameChanged(clip->clipId(), properties.value("name"));
    }
    if (properties.contains("description")) {
        CLIPTYPE type = clip->clipType();
        m_listView->blockSignals(true);
        clip->setText(2, properties.value("description"));
        m_listView->blockSignals(false);
        if (KdenliveSettings::activate_nepomuk() && (type == AUDIO || type == VIDEO || type == AV || type == IMAGE || type == PLAYLIST)) {
            // Use Nepomuk system to store clip description
            Nepomuk::Resource f(clip->clipUrl().path());
            f.setDescription(properties.value("description"));
        }
        emit projectModified();
    }
}

void ProjectList::slotItemEdited(QTreeWidgetItem *item, int column)
{
    ProjectItem *clip = static_cast <ProjectItem*>(item);
    if (column == 2) {
        if (clip->referencedClip()) {
            QMap <QString, QString> oldprops;
            QMap <QString, QString> newprops;
            oldprops["description"] = clip->referencedClip()->getProperty("description");
            newprops["description"] = item->text(2);

            if (clip->clipType() == TEXT) {
                // This is a text template clip, update the image
                /*oldprops.insert("xmldata", clip->referencedClip()->getProperty("xmldata"));
                newprops.insert("xmldata", generateTemplateXml(clip->referencedClip()->getProperty("xmltemplate"), item->text(2)).toString());*/
                oldprops.insert("templatetext", clip->referencedClip()->getProperty("templatetext"));
                newprops.insert("templatetext", item->text(2));
            }
            slotUpdateClipProperties(clip->clipId(), newprops);
            EditClipCommand *command = new EditClipCommand(this, clip->clipId(), oldprops, newprops, false);
            m_commandStack->push(command);
        }
    } else if (column == 1) {
        if (clip->isGroup()) {
            editFolder(item->text(1), clip->groupName(), clip->clipId());
            clip->setGroupName(item->text(1));
            m_doc->clipManager()->addFolder(clip->clipId(), item->text(1));
            const int children = item->childCount();
            for (int i = 0; i < children; i++) {
                ProjectItem *child = static_cast <ProjectItem *>(item->child(i));
                child->setProperty("groupname", item->text(1));
            }
        } else {
            if (clip->referencedClip()) {
                QMap <QString, QString> oldprops;
                QMap <QString, QString> newprops;
                oldprops["name"] = clip->referencedClip()->getProperty("name");
                newprops["name"] = item->text(1);
                slotUpdateClipProperties(clip, newprops);
                emit projectModified();
                EditClipCommand *command = new EditClipCommand(this, clip->clipId(), oldprops, newprops, false);
                m_commandStack->push(command);
            }
        }
    }
}

void ProjectList::slotContextMenu(const QPoint &pos, QTreeWidgetItem *item)
{
    bool enable = false;
    if (item) {
        enable = true;
    }
    m_editAction->setEnabled(enable);
    m_deleteAction->setEnabled(enable);
    m_reloadAction->setEnabled(enable);
    m_transcodeAction->setEnabled(enable);
    if (enable) {
        ProjectItem *clip = static_cast <ProjectItem*>(item);
        if (clip->clipType() == IMAGE && !KdenliveSettings::defaultimageapp().isEmpty()) {
            m_openAction->setIcon(KIcon(KdenliveSettings::defaultimageapp()));
            m_openAction->setEnabled(true);
        } else if (clip->clipType() == AUDIO && !KdenliveSettings::defaultaudioapp().isEmpty()) {
            m_openAction->setIcon(KIcon(KdenliveSettings::defaultaudioapp()));
            m_openAction->setEnabled(true);
        } else m_openAction->setEnabled(false);
    } else m_openAction->setEnabled(false);
    m_menu->popup(pos);
}

void ProjectList::slotRemoveClip()
{
    if (!m_listView->currentItem()) return;
    QStringList ids;
    QMap <QString, QString> folderids;
    QList<QTreeWidgetItem *> selected = m_listView->selectedItems();
    ProjectItem *item;
    for (int i = 0; i < selected.count(); i++) {
        item = static_cast <ProjectItem *>(selected.at(i));
        if (item->isGroup()) folderids[item->groupName()] = item->clipId();
        else ids << item->clipId();
        if (item->numReferences() > 0) {
            if (KMessageBox::questionYesNo(this, i18np("Delete clip <b>%2</b>?<br>This will also remove the clip in timeline", "Delete clip <b>%2</b>?<br>This will also remove its %1 clips in timeline", item->numReferences(), item->names().at(1)), i18n("Delete Clip")) != KMessageBox::Yes) return;
        } else if (item->isGroup() && item->childCount() > 0) {
            int children = item->childCount();
            if (KMessageBox::questionYesNo(this, i18np("Delete folder <b>%2</b>?<br>This will also remove the clip in that folder", "Delete folder <b>%2</b>?<br>This will also remove the %1 clips in that folder",  children, item->names().at(1)), i18n("Delete Folder")) != KMessageBox::Yes) return;
            for (int i = 0; i < children; ++i) {
                ProjectItem *child = static_cast <ProjectItem *>(item->child(i));
                ids << child->clipId();
            }
        }
    }
    if (!ids.isEmpty()) m_doc->deleteProjectClip(ids);
    if (!folderids.isEmpty()) deleteProjectFolder(folderids);
    if (m_listView->topLevelItemCount() == 0) {
        m_editAction->setEnabled(false);
        m_deleteAction->setEnabled(false);
        m_openAction->setEnabled(false);
        m_reloadAction->setEnabled(false);
        m_transcodeAction->setEnabled(false);
    }
}

void ProjectList::selectItemById(const QString &clipId)
{
    ProjectItem *item = getItemById(clipId);
    if (item) m_listView->setCurrentItem(item);
}


void ProjectList::slotDeleteClip(const QString &clipId)
{
    ProjectItem *item = getItemById(clipId);
    if (!item) {
        kDebug() << "/// Cannot find clip to delete";
        return;
    }
    m_listView->blockSignals(true);
    delete item;
    m_doc->clipManager()->deleteClip(clipId);
    m_listView->blockSignals(false);
    slotClipSelected();
}


void ProjectList::editFolder(const QString folderName, const QString oldfolderName, const QString &clipId)
{
    EditFolderCommand *command = new EditFolderCommand(this, folderName, oldfolderName, clipId, false);
    m_commandStack->push(command);
    m_doc->setModified(true);
}

void ProjectList::slotAddFolder()
{
    AddFolderCommand *command = new AddFolderCommand(this, i18n("Folder"), QString::number(m_doc->clipManager()->getFreeFolderId()), true);
    m_commandStack->push(command);
}

void ProjectList::slotAddFolder(const QString foldername, const QString &clipId, bool remove, bool edit)
{
    if (remove) {
        ProjectItem *item = getFolderItemById(clipId);
        if (item) {
            m_doc->clipManager()->deleteFolder(clipId);
            delete item;
        }
    } else {
        if (edit) {
            ProjectItem *item = getFolderItemById(clipId);
            if (item) {
                m_listView->blockSignals(true);
                item->setGroupName(foldername);
                m_listView->blockSignals(false);
                m_doc->clipManager()->addFolder(clipId, foldername);
                const int children = item->childCount();
                for (int i = 0; i < children; i++) {
                    ProjectItem *child = static_cast <ProjectItem *>(item->child(i));
                    child->setProperty("groupname", foldername);
                }
            }
        } else {
            QStringList text;
            text << QString() << foldername;
            m_listView->blockSignals(true);
            m_listView->setCurrentItem(new ProjectItem(m_listView, text, clipId));
            m_doc->clipManager()->addFolder(clipId, foldername);
            m_listView->blockSignals(false);
        }
    }
}



void ProjectList::deleteProjectFolder(QMap <QString, QString> map)
{
    QMapIterator<QString, QString> i(map);
    QUndoCommand *delCommand = new QUndoCommand();
    delCommand->setText(i18n("Delete Folder"));
    while (i.hasNext()) {
        i.next();
        new AddFolderCommand(this, i.key(), i.value(), false, delCommand);
    }
    m_commandStack->push(delCommand);
    m_doc->setModified(true);
}

void ProjectList::slotAddClip(DocClipBase *clip, bool getProperties)
{
    m_listView->setEnabled(false);
    if (getProperties) {
        m_listView->blockSignals(true);
        m_refreshed = false;
        // remove file_hash so that we load all properties for the clip
        QDomElement e = clip->toXML().cloneNode().toElement();
        e.removeAttribute("file_hash");
        m_infoQueue.insert(clip->getId(), e);
        //m_render->getFileProperties(clip->toXML(), clip->getId(), true);
    }
    const QString parent = clip->getProperty("groupid");
    ProjectItem *item = NULL;
    if (!parent.isEmpty()) {
        ProjectItem *parentitem = getFolderItemById(parent);
        if (!parentitem) {
            QStringList text;
            QString groupName = clip->getProperty("groupname");
            //kDebug() << "Adding clip to new group: " << groupName;
            if (groupName.isEmpty()) groupName = i18n("Folder");
            text << QString() << groupName;
            parentitem = new ProjectItem(m_listView, text, parent);
        } else {
            //kDebug() << "Adding clip to existing group: " << parentitem->groupName();
        }
        if (parentitem) item = new ProjectItem(parentitem, clip);
    }
    if (item == NULL) item = new ProjectItem(m_listView, clip);
    KUrl url = clip->fileURL();

    if (getProperties == false && !clip->getClipHash().isEmpty()) {
        QString cachedPixmap = m_doc->projectFolder().path(KUrl::AddTrailingSlash) + "thumbs/" + clip->getClipHash() + ".png";
        if (QFile::exists(cachedPixmap)) {
            item->setIcon(0, QPixmap(cachedPixmap));
        }
    }

    if (!url.isEmpty() && KdenliveSettings::activate_nepomuk()) {
        // if file has Nepomuk comment, use it
        Nepomuk::Resource f(url.path());
        QString annotation = f.description();
        if (!annotation.isEmpty()) item->setText(2, annotation);
        item->setText(3, QString::number(f.rating()));
    }
    if (getProperties && m_listView->isEnabled()) m_listView->blockSignals(false);
    if (getProperties && !m_queueTimer.isActive()) m_queueTimer.start();
}

void ProjectList::slotResetProjectList()
{
    m_listView->clear();
    emit clipSelected(NULL);
    m_thumbnailQueue.clear();
    m_infoQueue.clear();
    m_refreshed = false;
}

void ProjectList::requestClipInfo(const QDomElement xml, const QString id)
{
    m_refreshed = false;
    m_infoQueue.insert(id, xml);
    //if (m_infoQueue.count() == 1 || ) QTimer::singleShot(300, this, SLOT(slotProcessNextClipInQueue()));
}

void ProjectList::slotProcessNextClipInQueue()
{
    if (m_infoQueue.isEmpty()) {
        slotProcessNextThumbnail();
        return;
    }

    QMap<QString, QDomElement>::const_iterator j = m_infoQueue.constBegin();
    if (j != m_infoQueue.constEnd()) {
        const QDomElement dom = j.value();
        const QString id = j.key();
        m_infoQueue.remove(j.key());
        emit getFileProperties(dom, id, false);
    }
    //if (!m_infoQueue.isEmpty() && !m_queueTimer.isActive()) m_queueTimer.start();
}

void ProjectList::slotUpdateClip(const QString &id)
{
    ProjectItem *item = getItemById(id);
    m_listView->blockSignals(true);
    if (item) item->setData(1, UsageRole, QString::number(item->numReferences()));
    m_listView->blockSignals(false);
}

void ProjectList::updateAllClips()
{
    m_listView->setSortingEnabled(false);

    QTreeWidgetItemIterator it(m_listView);
    DocClipBase *clip;
    ProjectItem *item;
    m_listView->blockSignals(true);
    while (*it) {
        item = static_cast <ProjectItem *>(*it);
        if (!item->isGroup()) {
            clip = item->referencedClip();
            if (item->referencedClip()->producer() == NULL) {
                if (clip->isPlaceHolder() == false) {
                    requestClipInfo(clip->toXML(), clip->getId());
                } else item->setFlags(Qt::ItemIsSelectable | Qt::ItemIsEnabled);
            } else {
                if (item->icon(0).isNull()) {
                    requestClipThumbnail(clip->getId());
                }
                if (item->data(1, DurationRole).toString().isEmpty()) {
                    item->changeDuration(item->referencedClip()->producer()->get_playtime());
                }
            }
            item->setData(1, UsageRole, QString::number(item->numReferences()));
            //qApp->processEvents();
        }
        ++it;
    }
    qApp->processEvents();
    if (!m_queueTimer.isActive()) m_queueTimer.start();

    if (m_listView->isEnabled()) m_listView->blockSignals(false);
    m_listView->setSortingEnabled(true);
    if (m_infoQueue.isEmpty()) slotProcessNextThumbnail();
}

void ProjectList::slotAddClip(const QList <QUrl> givenList, const QString &groupName, const QString &groupId)
{
    if (!m_commandStack) {
        kDebug() << "!!!!!!!!!!!!!!!! NO CMD STK";
    }
    KUrl::List list;
    if (givenList.isEmpty()) {
        // Build list of mime types
        QStringList mimeTypes = QStringList() << "application/x-kdenlive" << "application/x-kdenlivetitle" << "video/x-flv" << "application/vnd.rn-realmedia" << "video/x-dv" << "video/dv" << "video/x-msvideo" << "video/x-matroska" << "video/mlt-playlist" << "video/mpeg" << "video/ogg" << "video/x-ms-wmv" << "audio/x-flac" << "audio/x-matroska" << "audio/mp4" << "audio/mpeg" << "audio/x-mp3" << "audio/ogg" << "audio/x-wav" << "application/ogg" << "video/mp4" << "video/quicktime" << "image/gif" << "image/jpeg" << "image/png" << "image/x-tga" << "image/x-bmp" << "image/svg+xml" << "image/tiff" << "image/x-xcf-gimp" << "image/x-vnd.adobe.photoshop" << "image/x-pcx" << "image/x-exr";

        QString allExtensions;
        foreach(const QString& mimeType, mimeTypes) {
            KMimeType::Ptr mime(KMimeType::mimeType(mimeType));
            if (mime) {
                allExtensions.append(mime->patterns().join(" "));
                allExtensions.append(' ');
            }
        }
        const QString dialogFilter = allExtensions.simplified() + ' ' + QLatin1Char('|') + i18n("All Supported Files") + "\n* " + QLatin1Char('|') + i18n("All Files");
        list = KFileDialog::getOpenUrls(KUrl("kfiledialog:///clipfolder"), dialogFilter, this);

    } else {
        for (int i = 0; i < givenList.count(); i++)
            list << givenList.at(i);
    }
    if (list.isEmpty()) return;

    if (givenList.isEmpty()) {
        QStringList groupInfo = getGroup();
        m_doc->slotAddClipList(list, groupInfo.at(0), groupInfo.at(1));
    } else m_doc->slotAddClipList(list, groupName, groupId);
}

void ProjectList::slotRemoveInvalidClip(const QString &id, bool replace)
{
    ProjectItem *item = getItemById(id);
    QTimer::singleShot(300, this, SLOT(slotProcessNextClipInQueue()));
    if (item) {
        const QString path = item->referencedClip()->fileURL().path();
        if (!path.isEmpty()) {
            if (replace) KMessageBox::sorry(this, i18n("Clip <b>%1</b><br>is invalid, will be removed from project.", path));
            else {
                if (KMessageBox::questionYesNo(this, i18n("Clip <b>%1</b><br>is missing or invalid. Remove it from project?", path), i18n("Invalid clip")) == KMessageBox::Yes) replace = true;
            }
        }
        QStringList ids;
        ids << id;
        if (replace) m_doc->deleteProjectClip(ids);
    }
}

void ProjectList::slotAddColorClip()
{
    if (!m_commandStack) {
        kDebug() << "!!!!!!!!!!!!!!!! NO CMD STK";
    }
    QDialog *dia = new QDialog(this);
    Ui::ColorClip_UI dia_ui;
    dia_ui.setupUi(dia);
    dia_ui.clip_name->setText(i18n("Color Clip"));
    dia_ui.clip_duration->setText(KdenliveSettings::color_duration());
    if (dia->exec() == QDialog::Accepted) {
        QString color = dia_ui.clip_color->color().name();
        color = color.replace(0, 1, "0x") + "ff";
        QStringList groupInfo = getGroup();
        m_doc->slotCreateColorClip(dia_ui.clip_name->text(), color, dia_ui.clip_duration->text(), groupInfo.at(0), groupInfo.at(1));
    }
    delete dia;
}


void ProjectList::slotAddSlideshowClip()
{
    if (!m_commandStack) {
        kDebug() << "!!!!!!!!!!!!!!!! NO CMD STK";
    }
    SlideshowClip *dia = new SlideshowClip(m_timecode, this);

    if (dia->exec() == QDialog::Accepted) {
        QStringList groupInfo = getGroup();
        m_doc->slotCreateSlideshowClipFile(dia->clipName(), dia->selectedPath(), dia->imageCount(), dia->clipDuration(), dia->loop(), dia->fade(), dia->lumaDuration(), dia->lumaFile(), dia->softness(), groupInfo.at(0), groupInfo.at(1));
    }
    delete dia;
}

void ProjectList::slotAddTitleClip()
{
    QStringList groupInfo = getGroup();
    m_doc->slotCreateTextClip(groupInfo.at(0), groupInfo.at(1));
}

void ProjectList::slotAddTitleTemplateClip()
{
    QStringList groupInfo = getGroup();
    if (!m_commandStack) {
        kDebug() << "!!!!!!!!!!!!!!!! NO CMD STK";
    }

    // Get the list of existing templates
    QStringList filter;
    filter << "*.kdenlivetitle";
    const QString path = m_doc->projectFolder().path(KUrl::AddTrailingSlash) + "titles/";
    QStringList templateFiles = QDir(path).entryList(filter, QDir::Files);

    QDialog *dia = new QDialog(this);
    Ui::TemplateClip_UI dia_ui;
    dia_ui.setupUi(dia);
    for (int i = 0; i < templateFiles.size(); ++i) {
        dia_ui.template_list->comboBox()->addItem(templateFiles.at(i), path + templateFiles.at(i));
    }
    dia_ui.template_list->fileDialog()->setFilter("*.kdenlivetitle");
    //warning: setting base directory doesn't work??
    KUrl startDir(path);
    dia_ui.template_list->fileDialog()->setUrl(startDir);
    dia_ui.text_box->setHidden(true);
    if (dia->exec() == QDialog::Accepted) {
        QString textTemplate = dia_ui.template_list->comboBox()->itemData(dia_ui.template_list->comboBox()->currentIndex()).toString();
        if (textTemplate.isEmpty()) textTemplate = dia_ui.template_list->comboBox()->currentText();
        // Create a cloned template clip
        m_doc->slotCreateTextTemplateClip(groupInfo.at(0), groupInfo.at(1), KUrl(textTemplate));
    }
    delete dia;
}

QStringList ProjectList::getGroup() const
{
    QStringList result;
    ProjectItem *item = static_cast <ProjectItem*>(m_listView->currentItem());
    if (item && !item->isGroup()) {
        while (item->parent()) {
            item = static_cast <ProjectItem*>(item->parent());
            if (item->isGroup()) break;
        }
    }
    if (item && item->isGroup()) {
        result << item->groupName();
        result << item->clipId();
    } else result << QString() << QString();
    return result;
}

void ProjectList::setDocument(KdenliveDoc *doc)
{
    m_listView->blockSignals(true);
    m_listView->clear();
    m_listView->setSortingEnabled(false);
    emit clipSelected(NULL);
    m_thumbnailQueue.clear();
    m_infoQueue.clear();
    m_refreshed = false;
    m_fps = doc->fps();
    m_timecode = doc->timecode();
    m_commandStack = doc->commandStack();
    m_doc = doc;

    QMap <QString, QString> flist = doc->clipManager()->documentFolderList();
    QMapIterator<QString, QString> f(flist);
    while (f.hasNext()) {
        f.next();
        (void) new ProjectItem(m_listView, QStringList() << QString() << f.value(), f.key());
    }

    QList <DocClipBase*> list = doc->clipManager()->documentClipList();
    for (int i = 0; i < list.count(); i++) {
        slotAddClip(list.at(i), false);
    }

    m_listView->blockSignals(false);
    m_toolbar->setEnabled(true);
    connect(m_doc->clipManager(), SIGNAL(reloadClip(const QString &)), this, SLOT(slotReloadClip(const QString &)));
    connect(m_doc->clipManager(), SIGNAL(checkAllClips()), this, SLOT(updateAllClips()));
}

QDomElement ProjectList::producersList()
{
    QDomDocument doc;
    QDomElement prods = doc.createElement("producerlist");
    doc.appendChild(prods);
    kDebug() << "////////////  PRO LIST BUILD PRDSLIST ";
    QTreeWidgetItemIterator it(m_listView);
    while (*it) {
        if (!((ProjectItem *)(*it))->isGroup())
            prods.appendChild(doc.importNode(((ProjectItem *)(*it))->toXml(), true));
        ++it;
    }
    return prods;
}

void ProjectList::slotCheckForEmptyQueue()
{
    if (!m_refreshed && m_thumbnailQueue.isEmpty() && m_infoQueue.isEmpty()) {
        m_refreshed = true;
        emit loadingIsOver();
        emit displayMessage(QString(), -1);
        m_listView->blockSignals(false);
        m_listView->setEnabled(true);
    } else if (!m_refreshed) QTimer::singleShot(300, this, SLOT(slotCheckForEmptyQueue()));
}

void ProjectList::reloadClipThumbnails()
{
    kDebug() << "//////////////  RELOAD CLIPS THUMBNAILS!!!";
    m_thumbnailQueue.clear();
    QTreeWidgetItemIterator it(m_listView);
    while (*it) {
        if (!((ProjectItem *)(*it))->isGroup())
            m_thumbnailQueue << ((ProjectItem *)(*it))->clipId();
        ++it;
    }
    QTimer::singleShot(300, this, SLOT(slotProcessNextThumbnail()));
}

void ProjectList::requestClipThumbnail(const QString id)
{
    if (!m_thumbnailQueue.contains(id)) m_thumbnailQueue.append(id);
}

void ProjectList::slotProcessNextThumbnail()
{
    if (m_thumbnailQueue.isEmpty() && m_infoQueue.isEmpty()) {
        slotCheckForEmptyQueue();
        return;
    }
    if (!m_infoQueue.isEmpty()) {
        //QTimer::singleShot(300, this, SLOT(slotProcessNextThumbnail()));
        return;
    }
    if (m_thumbnailQueue.count() > 1) {
        int max = m_doc->clipManager()->clipsCount();
        emit displayMessage(i18n("Loading thumbnails"), (int)(100 * (max - m_thumbnailQueue.count()) / max));
    }
    slotRefreshClipThumbnail(m_thumbnailQueue.takeFirst(), false);
}

void ProjectList::slotRefreshClipThumbnail(const QString &clipId, bool update)
{
    ProjectItem *item = getItemById(clipId);
    if (item) slotRefreshClipThumbnail(item, update);
    else slotProcessNextThumbnail();
}

void ProjectList::slotRefreshClipThumbnail(ProjectItem *item, bool update)
{
    if (item) {
        DocClipBase *clip = item->referencedClip();
        if (!clip) {
            slotProcessNextThumbnail();
            return;
        }
        QPixmap pix;
        int height = 50;
        int width = (int)(height  * m_render->dar());
        if (clip->clipType() == AUDIO) pix = KIcon("audio-x-generic").pixmap(QSize(width, height));
        else if (clip->clipType() == IMAGE) pix = QPixmap::fromImage(KThumb::getFrame(item->referencedClip()->producer(), 0, width, height));
        else pix = item->referencedClip()->thumbProducer()->extractImage(item->referencedClip()->getClipThumbFrame(), width, height);
        if (!pix.isNull()) {
            m_listView->blockSignals(true);
            item->setIcon(0, pix);
            if (m_listView->isEnabled()) m_listView->blockSignals(false);
            m_doc->cachePixmap(item->getClipHash(), pix);
        }
        if (update) emit projectModified();
        QTimer::singleShot(30, this, SLOT(slotProcessNextThumbnail()));
    }
}

void ProjectList::slotReplyGetFileProperties(const QString &clipId, Mlt::Producer *producer, const QMap < QString, QString > &properties, const QMap < QString, QString > &metadata, bool replace)
{
    ProjectItem *item = getItemById(clipId);
    if (item && producer) {
        m_listView->blockSignals(true);
        item->setProperties(properties, metadata);
        Q_ASSERT_X(item->referencedClip(), "void ProjectList::slotReplyGetFileProperties", QString("Item with groupName %1 does not have a clip associated").arg(item->groupName()).toLatin1());
        item->referencedClip()->setProducer(producer, replace);
        //emit receivedClipDuration(clipId);
        if (m_listView->isEnabled() && replace) {
            // update clip in clip monitor
            emit clipSelected(NULL);
            emit clipSelected(item->referencedClip());
        }
        /*else {
            // Check if duration changed.
            emit receivedClipDuration(clipId);
            delete producer;
        }*/
        if (m_listView->isEnabled()) m_listView->blockSignals(false);
        if (item->icon(0).isNull()) {
            requestClipThumbnail(clipId);
        }
    } else kDebug() << "////////  COULD NOT FIND CLIP TO UPDATE PRPS...";
    int max = m_doc->clipManager()->clipsCount();
    emit displayMessage(i18n("Loading clips"), (int)(100 * (max - m_infoQueue.count()) / max));
    // small delay so that the app can display the progress info
    QTimer::singleShot(30, this, SLOT(slotProcessNextClipInQueue()));
}

void ProjectList::slotReplyGetImage(const QString &clipId, const QPixmap &pix)
{
    ProjectItem *item = getItemById(clipId);
    if (item && !pix.isNull()) {
        m_listView->blockSignals(true);
        item->setIcon(0, pix);
        m_doc->cachePixmap(item->getClipHash(), pix);
        if (m_listView->isEnabled()) m_listView->blockSignals(false);
    }
}

ProjectItem *ProjectList::getItemById(const QString &id)
{
    ProjectItem *item;
    QTreeWidgetItemIterator it(m_listView);
    while (*it) {
        item = static_cast<ProjectItem *>(*it);
        if (item->clipId() == id && item->clipType() != FOLDER)
            return item;
        ++it;
    }
    return NULL;
}

ProjectItem *ProjectList::getFolderItemById(const QString &id)
{
    ProjectItem *item;
    QTreeWidgetItemIterator it(m_listView);
    while (*it) {
        item = static_cast<ProjectItem *>(*it);
        if (item->clipId() == id && item->clipType() == FOLDER)
            return item;
        ++it;
    }
    return NULL;
}

void ProjectList::slotSelectClip(const QString &ix)
{
    ProjectItem *clip = getItemById(ix);
    if (clip) {
        m_listView->setCurrentItem(clip);
        m_listView->scrollToItem(clip);
        m_editAction->setEnabled(true);
        m_deleteAction->setEnabled(true);
        m_reloadAction->setEnabled(true);
        m_transcodeAction->setEnabled(true);
        if (clip->clipType() == IMAGE && !KdenliveSettings::defaultimageapp().isEmpty()) {
            m_openAction->setIcon(KIcon(KdenliveSettings::defaultimageapp()));
            m_openAction->setEnabled(true);
        } else if (clip->clipType() == AUDIO && !KdenliveSettings::defaultaudioapp().isEmpty()) {
            m_openAction->setIcon(KIcon(KdenliveSettings::defaultaudioapp()));
            m_openAction->setEnabled(true);
        } else m_openAction->setEnabled(false);
    }
}

QString ProjectList::currentClipUrl() const
{
    ProjectItem *item = static_cast <ProjectItem*>(m_listView->currentItem());
    if (item == NULL) return QString();
    return item->clipUrl().path();
}

void ProjectList::regenerateTemplate(const QString &id)
{
    ProjectItem *clip = getItemById(id);
    if (clip) regenerateTemplate(clip);
}

void ProjectList::regenerateTemplate(ProjectItem *clip)
{
    //TODO: remove this unused method, only force_reload is necessary
    clip->referencedClip()->producer()->set("force_reload", 1);
}

QDomDocument ProjectList::generateTemplateXml(QString path, const QString &replaceString)
{
    QDomDocument doc;
    QFile file(path);
    if (!file.open(QIODevice::ReadOnly)) {
        kWarning() << "ERROR, CANNOT READ: " << path;
        return doc;
    }
    if (!doc.setContent(&file)) {
        kWarning() << "ERROR, CANNOT READ: " << path;
        file.close();
        return doc;
    }
    file.close();
    QDomNodeList texts = doc.elementsByTagName("content");
    for (int i = 0; i < texts.count(); i++) {
        QString data = texts.item(i).firstChild().nodeValue();
        data.replace("%s", replaceString);
        texts.item(i).firstChild().setNodeValue(data);
    }
    return doc;
}

#include "projectlist.moc"
