/**

	playlist.cpp

	Playlists' engine source file

**/

#ifndef PLAYLIST_H

#include <qfile.h>

#include <cstdlib>

#include "playlist.h"

Playlist::Playlist()
{
	mainIndex = NULL;
	mainGuard = NULL;

	count = 0;
	playlistCount = 0;
}

Playlist::~Playlist()
{
	while (getPlaylistCount())
		delPlaylist(getPlaylist(0));
}

pListIndex * Playlist::newPlaylist(const QString & name, const QString & descr)
{
	if (playlistCount == INT_MAX)
		return 0;

	pListIndex * newLst = new pListIndex;

	if (!newLst)
		return NULL;

	newLst->name = name;
	newLst->description = descr;
	newLst->previous = mainGuard;
	
	if (!mainIndex)
	{
		mainIndex = newLst;
		mainGuard = mainIndex;
		mainGuard->previous = NULL;
	}

	if (mainGuard)
		mainGuard->next = newLst;

	mainGuard = newLst;

	playlistCount++;

	return newLst;
}

void Playlist::setPlaylistName(pListIndex * const index, const QString & name)
{
	if (!index)
		return;

	index->name = name;
}

QString Playlist::getPlaylistName(const pListIndex * const index) const
{
	if (!index)
		return QString::null;

	return index->name;
}

void Playlist::setPlaylistDescr(pListIndex * const index, const QString & descr)
{
	if (!index)
		return;

	index->description = descr;
}

QString Playlist::getPlaylistDescr(const pListIndex * const index) const
{
	if (!index)
		return QString::null;

	return index->description;
}

void Playlist::delPlaylist(pListIndex * const index)
{
	if (!index)
		return;

	if (index->previous)
		index->previous->next = index->next;

	if (index->next)
		index->next->previous = index->previous;

	if (index == mainGuard)
		mainGuard = index->previous;

	if (index == mainIndex)
		mainIndex = index->next;

	del(index, -1);

	delete index;

	playlistCount--;
}

pListIndex * Playlist::getPlaylist(const int & which) const
{
	pListIndex * tmp = mainIndex;

	if ((!tmp) || (which < 0) || (which >= playlistCount))
		return 0;

	if (which <= (playlistCount / 2))
	{
		for (int i = 0 ; i < which ; i++)
		{
			if (tmp->next)
				tmp = tmp->next;
			else
				return 0;
		}
	}
	else
	{
		tmp = mainGuard;

		for (int i = (playlistCount - 1) ; i > which ; i--)
		{
			if (tmp->previous)
				tmp = tmp->previous;
			else
				return 0;
		}
	}

	return tmp;
}

void Playlist::movePlaylist(const int & which, const int & where, int & current, int & dest)
{
	pListIndex * src = getPlaylist(which);
	pListIndex * dst = getPlaylist(where);

	if ((!src) || (!dst) || (src == dst))
		return;

	if (which < where)
	{
		if (src->previous)
			src->previous->next = src->next;

		if (src->next)
			src->next->previous = src->previous;

		if (mainIndex == src)
			mainIndex = src->next;

		if (mainGuard == dst)
			mainGuard = src;

		src->next = dst->next;
		src->previous = dst;

		if (dst->next)
			dst->next->previous =  src;

		dst->next = src;

		if ((current >= which) && (current <= where))
		{
			if (which == current)
				current = where;
			else
				current--;
		}

		if ((dest >= which) && (dest <= where))
		{
			if (which == dest)
				dest = where;
			else
				dest--;
		}
	}
	else
	{
		if (src->next)
			src->next->previous = src->previous;

		if (src->previous)
			src->previous->next = src->next;

		if (mainIndex == dst)
			mainIndex = src;

		if (mainGuard == src)
			mainGuard = src->previous;

		src->previous = dst->previous;
		src->next = dst;

		if (dst->previous)
			dst->previous->next = src;

		dst->previous = src;

		if ((current <= which) && (current >= where))
		{
			if (which == current)
				current = where;
			else
				current++;
		}

		if ((dest <= which) && (dest >= where))
		{
			if (which == dest)
				dest = where;
			else
				dest++;
		}
	}
}

unsigned int Playlist::getCount() const
{
	return count;
}

int Playlist::getPlaylistCount() const
{
	return playlistCount;
}

length Playlist::getTotalLength() const
{
	return totalLen;
}

pList * Playlist::add(pListIndex * const index, const QString & name, const QString & path,
			const tags * const ftags)
{
	if ((!index) || (index->count == INT_MAX))
		return 0;

	pList * newItem = new pList;

	if (!newItem)
		return NULL;

	newItem->trackName = name;
	if (ftags)
		newItem->fileTags = *ftags;
	newItem->filePath = path;
	
	if (!index->list)
	{
		index->list = newItem;
		index->last = newItem;
	}
	else
	{
		index->last->next = newItem;
		newItem->previous = index->last;
		index->last = newItem;
	}

	count++;
	index->count++;
	index->len += newItem->fileTags.tagLengthSeconds;	
	totalLen += newItem->fileTags.tagLengthSeconds;

	return newItem;
}

void Playlist::del(pListIndex * const index, const int & which)
{
	if (!index)
		return;

	pList * tmp = index->list;

	if (!tmp)
		return;

	if (which < 0)
	{
		pList * temp = 0;

		while (tmp)
		{
			temp = tmp->next;
			
			index->len -= tmp->fileTags.tagLengthSeconds;
			totalLen -= tmp->fileTags.tagLengthSeconds;

			delete tmp->myId;
			delete tmp;

			tmp = temp;
			
			count--;
		}

		index->count = 0;

		index->list = 0;
		index->last = 0;
	}
	else
	{
		if (!(tmp = get(index, which)))
			return;

		if (tmp->previous)
			tmp->previous->next = tmp->next;

		if (tmp->next)
			tmp->next->previous = tmp->previous;

		if ((!tmp->previous) && (!tmp->next))
		{
			index->list = 0;
			index->last = 0;
		}
		else
		{
			if (tmp == index->list)
				index->list = tmp->next;
			else
			{
				if (tmp == index->last)
					index->last = tmp->previous;
			}
		}

		index->len -= tmp->fileTags.tagLengthSeconds;
		totalLen -= tmp->fileTags.tagLengthSeconds;

		delete tmp->myId;
		delete tmp;

		count--;
		index->count--;
	}

	index->recent = 0;
}

void Playlist::up(pListIndex * const index, pList * item)
{
	if ((!index) || (!item))
		return;

	pList * temp = 0;

	if (item == index->last)
		index->last = item->previous;

	if (item->previous == index->list)
		index->list = item;

	if (item->next)
		item->next->previous = item->previous;

	if (item->previous)
	{
		temp = item->previous->previous;

		if (item->previous->previous)
			item->previous->previous->next = item;

		item->previous->next = item->next;
		item->previous->previous = item;
	}

	item->next = item->previous;
	item->previous = temp;

	index->recent = 0;
}

void Playlist::up(pListIndex * const index, const int & which)
{
	if ((!index) || (which < 1))
		return;

	pList * tmp = 0;
	
	if (!(tmp = get(index, which)))
		return;

	up(index, tmp);
}

void Playlist::down(pListIndex * const index, pList * item)
{
	if ((!index) || (!item))
		return;

	pList * temp = 0;

	if (item == index->list)
		index->list = item->next;

	if (item->next == index->last)
		index->last = item;

	if (item->previous)
		item->previous->next = item->next;

	if (item->next)
	{
		temp = item->next->next;

		if (item->next->next)
			item->next->next->previous = item;

		item->next->previous = item->previous;
		item->next->next = item;
	}

	item->previous = item->next;
	item->next = temp;

	index->recent = 0;
}

void Playlist::down(pListIndex * const index, const int & which)
{
	if ((!index) || (which >= index->count))
		return;

	pList * tmp = 0;
	
	if (!(tmp = get(index, which)))
		return;

	down(index, tmp);
}

void Playlist::swap(pListIndex * const index, const int & which, const int & where)
{
	pList * l = 0, * r = 0;
	int diff = (which - where);

	if (!diff)
		return;

	if (!(l = get(index, which)))
		return;

	if (!(r = get(index, where)))
		return;

	if ((diff == 1) || (diff == -1))
	{
		if (which > where)
			up(index, l);
		else
			down(index, l);
	}
	else
	{
		if (l == index->list)
			index->list = r;
		else
		{
			if (r == index->list)
				index->list = l;
		}

		if (l == index->last)
			index->last = r;
		else
		{
			if (r == index->last)
				index->last = l;
		}

		pList * l_previous = l->previous, * l_next = l->next;
	
		if (r->previous)
			r->previous->next = l;

		if (r->next)
			r->next->previous = l;

		if (l->previous)
			l->previous->next = r;

		if (l->next)
			l->next->previous = r;
	
		l->previous = r->previous;
		l->next = r->next;
		r->previous = l_previous;
		r->next = l_next;
	}

	index->recent = 0;
}

void Playlist::move(pListIndex * const index, const int & which, const int & where)
{
	pList * l = 0, * r = 0;
	int diff = (which - where);

	if (!diff)
		return;

	if (!(l = get(index, which)))
		return;

	if (!(r = get(index, where)))
		return;

	if ((diff == 1) || (diff == -1))
	{
		if (which > where)
			up(index, l);
		else
			down(index, l);
	}
	else
	{
		if (l == index->list)
			index->list = l->next;
		else
		{
			if (r == index->list)
				index->list = l;
		}

		if (l == index->last)
			index->last = l->previous;
		else
		{
			if (r == index->last)
				index->last = l;
		}

		if (l->previous)
			l->previous->next = l->next;

		if (l->next)
			l->next->previous = l->previous;

		if (diff > 0)
		{
			if (r->previous)
				r->previous->next = l;
	
			l->previous = r->previous;
			l->next = r;
			r->previous = l;
		}
		else
		{
			if (r->next)
				r->next->previous = l;
	
			l->previous = r;
			l->next = r->next;
			r->next = l;
		}
	}

	index->recent = 0;
}

pList * Playlist::get(const pListIndex * const index, const int & which) const
{
	if ((!index) || (which < 0) || (which >= index->count))
		return 0;

	int mid_i = (index->count / 2);
	int diff_mid = (mid_i - which);
	int diff_recent = (index->recent_i - which);

	bool begin = true;
	bool use_recent = false;

	pList * tmp = index->list;

	if ((index->recent) && (abs(diff_recent) < abs(diff_mid)))
	{
		use_recent = true;

		tmp = index->recent;

		if (diff_recent > 0)
			begin = false;

//		qDebug("recent");
	}
	else
	{
		if (diff_mid > 0)
		{
			tmp = index->last;

			begin = false;
		}

//		qDebug("first/last");
	}

	if (begin)
	{
//		qDebug("from begin");

		int i = 0;

		if (use_recent)
			i = index->recent_i;

		for (; i < which ; i++)
		{
			if (tmp->next)
				tmp = tmp->next;
			else
				return 0;
		}
	}
	else
	{
//		qDebug("from end");

		int i = (index->count - 1);

		if (use_recent)
			i = index->recent_i;

		for (; i > which ; i--)
		{
			if (tmp->previous)
				tmp = tmp->previous;
			else
				return 0;
		}
	}

	index->recent_i = which;
	index->recent = tmp;

//	qDebug(QString::number(int(tmp)));

	return tmp;
}

int Playlist::assignId(const pListIndex * const index)
{
	pList * item = 0;
	int i = 0;

	while ((item = get(index, i)))
	{
		delete item->myId;
		item->myId = new id(i);
		i++;
	}

	return i;
}

int Playlist::abs(const int & val) const
{
	if (val < 0)
		return (-val);
	else
		return val;
}

int Playlist::rmDup(pListIndex * const index, const int & pI)
{
	if (!index)
		return pI;

	int newPI = pI;

	pList * item = 0;
	int i = 0;

	if (!assignId(index))
		return pI;

	while ((item = get(index, i)))
	{
		pList * jtem = 0;
		int j = (i + 1);
		
		while ((jtem = get(index, j)))
		{
			if (item->filePath == jtem->filePath)
			{
				pList * tmp_item = 0;
				int tmp_k = j;

				if (newPI == j)
				{
					tmp_k = i;
					del(index, i--);
					item = jtem;
					newPI--;
				}
				else
				{
					del(index, j);

					if (newPI > j)
						newPI--;
				}

				while ((tmp_item = get(index, tmp_k)))
				{
					tmp_item->myId->current = tmp_k;
					tmp_k++;
				}
			}
			else
				j++;
		}

		i++;
	}

	return newPI;
}

int Playlist::rmDead(pListIndex * const index, const int & pI)
{
	if (!index)
		return pI;

	int newPI = pI;

	pList * item = 0;
	int i = 0;

	if (!assignId(index))
		return pI;

	while ((item = get(index, i)))
	{
		if ((item->filePath.left(6) != "cdda:/") &&
			(item->filePath.left(7) != "http://") &&
			(!QFile::exists(item->filePath)))
		{
			pList * tmp_item = 0;
			int tmp_k = i;

			del(index, i);

			if (newPI > i)
				newPI--;

			while ((tmp_item = get(index, tmp_k)))
			{
				tmp_item->myId->current = tmp_k;
				tmp_k++;
			}
		}
		else
			i++;
	}

	return newPI;
}

void Playlist::sort(pListIndex * const index, const unsigned short int & type, const bool & order)
{
	if (!index)
		return;

	assignId(index);

	qSort(index, 0, (index->count - 1), type, order);
}

void Playlist::qSort(pListIndex * const index, const int & first, const int & last, const unsigned short int & type, const bool & order)
{
	if (first >= last)
		return;

	int cur_i = first;
	int next_i = (first + 1);

	pList * cur = get(index, first);
	pList * next = 0;

	while (next_i <= last)
	{
		next = get(index, next_i);

		const QString * l = 0, * r = 0;
		int lV = 0, rV = 0;

		switch (type)
		{
			case SORT_TITLE:
				l = &cur->fileTags.tagTitle;
				r = &next->fileTags.tagTitle;

				if (l->isEmpty())
					l = &cur->trackName;

				if (r->isEmpty())
					r = &next->trackName;
			break;
			case SORT_ALBUM:
				l = &cur->fileTags.tagAlbum;
				r = &next->fileTags.tagAlbum;
			break;
			case SORT_ARTIST:
				l = &cur->fileTags.tagArtist;
				r = &next->fileTags.tagArtist;
			break;
			case SORT_BITRATE:
				lV = cur->fileTags.tagBitrate.toInt();
				rV = next->fileTags.tagBitrate.toInt();
			break;
			case SORT_LENGTH:
				lV = cur->fileTags.tagLengthSeconds;
				rV = next->fileTags.tagLengthSeconds;
			break;
		}

		bool _swap = false;

		if ((type != SORT_BITRATE) && (type != SORT_LENGTH))
		{
			if (order)
			{
				if ((*l) > (*r))
					_swap = true;
			}
			else
			{
				if ((*l) < (*r))
					_swap = true;
			}
		}
		else
		{
			if (order)
			{
				if (lV > rV)
					_swap = true;
			}
			else
			{
				if (lV < rV)
					_swap = true;
			}
		}

		if (_swap)
		{
			cur->myId->current++;
			next->myId->current = cur_i;

			for (int i = (cur_i + 1) ; i < next_i ; i++)
			{
				pList * tmp = get(index, i);

				if (!tmp)
					break;

				tmp->myId->current++;
			}

			move(index, next_i, cur_i);

			cur_i++;
		}
		else
			next_i++;
	}

	qSort(index, first, (cur_i - 1), type, order);
	qSort(index, (cur_i + 1), last, type, order);
}

void Playlist::shuffle(pListIndex * const index)
{
	if (!index)
		return;

	int cnt = 0;

	pList * item = 0;
	int i = 0;

	if ((cnt = assignId(index)) <= 1)
		return;

	while ((item = get(index, i++)))
	{
		int previous = item->myId->current;
		item->myId->current = (std::rand() % cnt);

		if (previous == item->myId->current)
			continue;

		get(index, item->myId->current)->myId->current = previous;
		
		swap(index, previous, item->myId->current);
	}
}

#endif
