#include "version.h"

#include <qcheckbox.h>
#include <qdatetime.h>
#include <qdir.h>
#include <qdragobject.h>
#include <qfileinfo.h>
#include <qheader.h>
#include <qmessagebox.h>
#include <qpixmap.h>
#include <qprocess.h>
#include <qstatusbar.h>
#include <qtextcodec.h>
#include <qtextedit.h>
#include <qtextstream.h>
#include <qtooltip.h>

#include <kaboutapplication.h>
#include <kcmdlineargs.h>
#include <kdirselectdialog.h>
#include <kdiskfreesp.h>
#include <kfiledialog.h>
#include <kmdcodec.h>
#include <kprogress.h>
#include <kiconeffect.h>
#include <kurl.h>

#include <fileref.h>
#include <tag.h>

#include <cstdlib>

#define CURRENT_PLAYLIST collection.getPlaylist(tabPlaylist->currentPageIndex())
#define CURRENT_TRACK lstPlaylist->selectedItem()
#define WAIT_CURSOR setCursor(Qt::WaitCursor);
#define ARROW_CURSOR setCursor(Qt::ArrowCursor);

void frmMain::init()
{
	QTime midnight(0, 0, 0);
	std::srand((unsigned int)midnight.secsTo(QTime::currentTime(Qt::LocalTime)));

	grpControl->setTitle(i18n("Control"));
	grpVolume->setTitle(i18n("Volume"));
	grpPlaylist->setTitle(i18n("Playlist"));
	grpTrack->setTitle(i18n("Track"));

	QToolTip::add(pshNewPlaylist, i18n("Add playlist"));
	QToolTip::add(pshUp, i18n("Move up"));
	QToolTip::add(pshDown, i18n("Move down"));
	QToolTip::add(pshGoToCurrent, i18n("Go to current"));
	QToolTip::add(pshRemove, i18n("Remove"));
	QToolTip::add(pshDeletePlaylist, i18n("Delete playlist"));
	QToolTip::add(pshClearList, i18n("Clear playlist"));

	lstPlaylist->addColumn(i18n("Title"), 255);
	lstPlaylist->addColumn(i18n("Album"), 222);
	lstPlaylist->addColumn(i18n("Artist"), 196);
	lstPlaylist->addColumn(i18n("Bitrate"), 106);
	lstPlaylist->addColumn(i18n("Length"), 76);

	sldPos->setLineStep(1);
	sldPos->setPageStep(1);

	objsToMoveHor.append(objWithPos(grpPlaylist, grpPlaylist->geometry().x(), grpPlaylist->geometry().y()));
	objsToMoveHor.append(objWithPos(grpTrack, grpTrack->geometry().x(), grpTrack->geometry().y()));
	objsToMoveHor.append(objWithPos(grpVolume, grpVolume->geometry().x(), grpVolume->geometry().y()));
	objsToMoveHor.append(objWithPos(lblTime, lblTime->geometry().x(), lblTime->geometry().y()));

	objsToMoveVer.append(objWithPos(tabPlaylist, tabPlaylist->geometry().x(), tabPlaylist->geometry().y()));

	QTimer::singleShot(0, this, SLOT(restore()));
}

void frmMain::restore()
{
	qDebug("Welcome to " + (QString)FALF_NAM + " " + (QString)FALF_VER +
		" (using libxine " + player.getXineVer() + ")!");

	KConfig * conf = KGlobal::config();

	conf->setGroup("Accels");
	pshAction->setAccel(conf->readEntry("tray_play", "Ctrl+G"));
	pshStop->setAccel(conf->readEntry("tray_stop", "Ctrl+H"));
	pshLeft->setAccel(conf->readEntry("tray_left", "Ctrl+P"));
	pshRight->setAccel(conf->readEntry("tray_right", "Ctrl+N"));
	pshNewPlaylist->setAccel(conf->readEntry("wnd_newplaylist", "Ctrl+T"));
	pshUp->setAccel(conf->readEntry("wnd_up", "Ctrl+Up"));
	pshDown->setAccel(conf->readEntry("wnd_down", "Ctrl+Down"));
	pshRemove->setAccel(conf->readEntry("wnd_remove", "Delete"));
	pshGoToCurrent->setAccel(conf->readEntry("wnd_current", "Ctrl+Right"));

	systemTray = new KMSystemTray(this, "TrayIcon");
	QPixmap icon;
	KIconLoader * loader = KGlobal::iconLoader();

	pixPlay = loader->loadIcon("player_play", KIcon::Toolbar);
	pixPause = loader->loadIcon("player_pause", KIcon::Toolbar);
	pixStop = loader->loadIcon("player_stop", KIcon::Toolbar);
	pixPlayLoop = loader->loadIcon("reload", KIcon::Toolbar);
	pixPlayOnce = loader->loadIcon("bottom", KIcon::Toolbar);
	pixTrackLoop = loader->loadIcon("lock", KIcon::Toolbar);
	pixRndOn = loader->loadIcon("wizard", KIcon::Toolbar);
	pixRndOff = loader->loadIcon("forward", KIcon::Toolbar);
	pixFollow = loader->loadIcon("next", KIcon::Toolbar);
	pixCopyOnPlaylist = loader->loadIcon("editcopy", KIcon::Toolbar);
	pixSendOnPlaylist = loader->loadIcon("goto", KIcon::Toolbar);
	pixAppendToPlaylist = loader->loadIcon("add", KIcon::Toolbar);

	popupMenu = systemTray->contextMenu();
	icon = loader->loadIcon("player_playlist", KIcon::Toolbar);
	popupMenu->insertItem(pixPlay, i18n("Play"), this, SLOT(onAction()), pshAction->accel(), TRAY_PLAY);
	popupMenu->insertItem("", this, SLOT(onStop()), pshStop->accel(), TRAY_STOP);
	popupMenu->insertItem("", this, SLOT(onLeft()), pshLeft->accel(), TRAY_LEFT);
	popupMenu->insertItem("", this, SLOT(onRight()), pshRight->accel(), TRAY_RIGHT);
	popupMenu->insertSeparator();
	icon = loader->loadIcon("messagebox_info", KIcon::Toolbar);
	popupMenu->insertItem(icon, i18n("About..."), this, SLOT(onAbout()));

	systemTray->setPixmap(pixStop);
	systemTray->setCaption((QString)FALF_NAM);
	systemTray->show();

	pshAction->setPixmap(pixPlay);
	pshLoop->setPixmap(pixPlayLoop);

	actionSelectAll = new QAction(this);
	actionSelectAll->setAccel(conf->readEntry("action_selectall", "Ctrl+A"));
	connect(actionSelectAll, SIGNAL(activated()), lstPlaylist, SLOT(selectAll()));
	connect(actionSelectAll, SIGNAL(activated()), this, SLOT(onDragging()));

	actionCopyOnDest = new QAction(this);
	actionCopyOnDest->setAccel(conf->readEntry("action_copyondest", "Shift+C"));
	connect(actionCopyOnDest, SIGNAL(activated()), this, SLOT(copyOnDest()));

	actionSendOnDest = new QAction(this);
	actionSendOnDest->setAccel(conf->readEntry("action_sendondest", "Shift+S"));
	connect(actionSendOnDest, SIGNAL(activated()), this, SLOT(sendOnDest()));

	actionAppendToDest = new QAction(this);
	actionAppendToDest->setAccel(conf->readEntry("action_appendtodest", "Shift+A"));
	connect(actionAppendToDest, SIGNAL(activated()), this, SLOT(appendToDest()));

	actionMakeVolHigher = new QAction(this);
	actionMakeVolHigher->setAccel(conf->readEntry("action_volhigher", "+"));
	connect(actionMakeVolHigher, SIGNAL(activated()), sldVol, SLOT(addStep()));

	actionMakeVolLower = new QAction(this);
	actionMakeVolLower->setAccel(conf->readEntry("action_vollower", "-"));
	connect(actionMakeVolLower, SIGNAL(activated()), sldVol, SLOT(subtractStep()));

	actionMakePrevCurrent = new QAction(this);
	actionMakePrevCurrent->setAccel(conf->readEntry("action_prevcurrent", "Left"));
	connect(actionMakePrevCurrent, SIGNAL(activated()), this, SLOT(onMakePrevCurrent()));

	actionMakeNextCurrent = new QAction(this);
	actionMakeNextCurrent->setAccel(conf->readEntry("action_nextcurrent", "Right"));
	connect(actionMakeNextCurrent, SIGNAL(activated()), this, SLOT(onMakeNextCurrent()));

	actionQuit = new QAction(this);
	actionQuit->setAccel(conf->readEntry("action_quit", "Ctrl+Q"));
	connect(actionQuit, SIGNAL(activated()), this, SLOT(onQuitSelected()));

	actionPopupCopyOnDest = new QAction(this);
	actionPopupCopyOnDest->setAccel(conf->readEntry("action_popupcopyondest", "Ctrl+,"));
	connect(actionPopupCopyOnDest, SIGNAL(activated()), this, SLOT(onPopupCopyOnDest()));

	actionPopupSendOnDest = new QAction(this);
	actionPopupSendOnDest->setAccel(conf->readEntry("action_popupsendondest", "Ctrl+."));
	connect(actionPopupSendOnDest, SIGNAL(activated()), this, SLOT(onPopupSendOnDest()));

	actionPopupAppendTo = new QAction(this);
	actionPopupAppendTo->setAccel(conf->readEntry("action_popupappendto", "Ctrl+/"));
	connect(actionPopupAppendTo, SIGNAL(activated()), this, SLOT(onPopupAppendTo()));

	contextMenu = new KPopupMenu(lstPlaylist);
	icon = loader->loadIcon("falf", KIcon::Toolbar);
	contextMenu->insertTitle(icon, (QString)FALF_NAM + " " + (QString)FALF_VER);
	icon = loader->loadIcon("sound", KIcon::Toolbar);
	contextMenu->insertItem(icon, i18n("Add file(s)") + "...", this, SLOT(addFiles()), conf->readEntry("context_addfiles", "Ctrl+O"), CONTEXT_ADDFILES);
	icon = loader->loadIcon("folder", KIcon::Toolbar);
	contextMenu->insertItem(icon, i18n("Add folder") + "...", this, SLOT(addFolders()), conf->readEntry("context_addfolder", "Ctrl+D"), CONTEXT_ADDFOLDER);
	icon = loader->loadIcon("cdaudio_unmount", KIcon::Toolbar);
	contextMenu->insertItem(icon, i18n("Add audio CD contents") + "...", this, SLOT(openAudioCD()), conf->readEntry("context_addcd", "Ctrl+C"), CONTEXT_ADDCD);
	contextMenu->insertSeparator();
	
	subMenuSettings = new QPopupMenu(contextMenu);
	icon = loader->loadIcon("edit_user", KIcon::Toolbar);
	subMenuSettings->insertItem(icon, "last.fm...", this, SLOT(editLfm()), conf->readEntry("context_lastfm", "Ctrl+U"), CONTEXT_LASTFM);
	dlgLfm.setIcon(icon);
	icon = loader->loadIcon("music_eightnote", KIcon::Toolbar);
	dlgEqualizer.setIcon(icon);
	subMenuSettings->insertItem(icon, i18n("Equalizer") + "...", this, SLOT(onEqualizer()), conf->readEntry("context_equalizer", "Ctrl+Z"), CONTEXT_EQUALIZER);
	icon = loader->loadIcon("kcmsound", KIcon::Toolbar);
	subMenuSettings->insertItem(icon, i18n("Sound system") + "...", this, SLOT(onSoundSystem()), conf->readEntry("context_sound", "Ctrl+X"), CONTEXT_SOUND);
	icon = loader->loadIcon("key_bindings", KIcon::Toolbar);
	dlgAccels.setIcon(icon);
	subMenuSettings->insertItem(icon, i18n("Accelerators") + "...", this, SLOT(onAccels()), conf->readEntry("context_accels", "Ctrl+K"), CONTEXT_ACCELS);
	dlgSoundSystem.setIcon(icon);
	icon = loader->loadIcon("configure", KIcon::Toolbar);
	dlgPlaylist.setIcon(icon);
	contextMenu->insertItem(icon, i18n("Settings"), subMenuSettings);
	contextMenu->insertSeparator();
	trackMenu = new QPopupMenu(contextMenu);
	playlistMenu = new QPopupMenu(contextMenu);
	subMenuSend = new QPopupMenu(trackMenu);
	subMenuCopy = new QPopupMenu(trackMenu);
	subMenuAppend = new QPopupMenu(playlistMenu);
	icon = loader->loadIcon("filesave", KIcon::Toolbar);
	playlistMenu->insertItem(icon, i18n("Save") + "...", this, SLOT(saveM3u()), conf->readEntry("context_save", "Ctrl+S"), CONTEXT_SAVE);
	subMenuRemove = new QPopupMenu(playlistMenu);
	subMenuRemove->insertItem(i18n("duplicates"), 0);
	subMenuRemove->insertItem(i18n("dead entries"), 1);
	subMenuRemove->insertSeparator();
	subMenuRemove->insertItem(i18n("both kinds"), CONTEXT_RMBOTH);
	subMenuRemove->setAccel(conf->readEntry("context_rmboth", "Ctrl+R"), CONTEXT_RMBOTH);
	icon = loader->loadIcon("editdelete", KIcon::Toolbar);
	playlistMenu->insertItem(icon, i18n("Remove"), subMenuRemove);
	connect(subMenuRemove, SIGNAL(activated(int)), this, SLOT(makeUpPlaylist(int)));
	icon = loader->loadIcon("roll", KIcon::Toolbar);
	playlistMenu->insertItem(icon, i18n("Shuffle"), this, SLOT(shufflePlaylist()), conf->readEntry("context_shuffle", "Ctrl+F"), CONTEXT_SHUFFLE);
	playlistMenu->insertItem(pixAppendToPlaylist, i18n("Append to playlist"), subMenuAppend);
	icon = loader->loadIcon("edit", KIcon::Toolbar);
	trackMenu->insertItem(icon, i18n("Edit tags") + "...", this, SLOT(editTags()), conf->readEntry("context_tags", "Ctrl+E"), CONTEXT_TAGS);
	dlgTags.setIcon(icon);
	icon = loader->loadIcon("contents", KIcon::Toolbar);
	trackMenu->insertItem(icon, i18n("Show lyrics") + "...", this, SLOT(onLyrics()), conf->readEntry("context_lyrics", "Ctrl+L"), CONTEXT_LYRICS);
	dlgLyrics.setIcon(icon);
	dlgLyrics.pshDelLyrics->setText(i18n("Download lyrics again"));
	connect(dlgLyrics.pshDelLyrics, SIGNAL(clicked()), this, SLOT(onDelLyrics()));
	connect(dlgLyrics.cmbCodec, SIGNAL(activated(int)), this, SLOT(onCodecChanged(int)));
	connect(&dlgEqualizer, SIGNAL(someEqValChanged(const int &, const int *)),
		this, SLOT(updateFreq(const int &, const int *)));
	trackMenu->insertItem(pixCopyOnPlaylist, i18n("Copy on playlist"), subMenuCopy);
	playlistMenu->insertItem(icon, i18n("Duplicate"), this, SLOT(copyPlaylist()), conf->readEntry("context_duplicate", "Ctrl+V"), CONTEXT_DUPLICATE);
	trackMenu->insertItem(pixSendOnPlaylist, i18n("Send on playlist"), subMenuSend);
	icon = loader->loadIcon("ipod_unmount", KIcon::Toolbar);
	dlgRemovableDevice.setIcon(icon);
	trackMenu->insertItem(icon, i18n("Copy to removable device") + "...", this, SLOT(onRemovableDevice()), conf->readEntry("context_removable", "Ctrl+B"), CONTEXT_REMOVABLE);
	icon = loader->loadIcon("view_tree", KIcon::Toolbar);
	contextMenu->insertItem(icon, i18n("Track"), trackMenu);
	icon = loader->loadIcon("player_playlist", KIcon::Toolbar);
	contextMenu->insertItem(icon, i18n("Playlist"), playlistMenu);
	connect(subMenuSend, SIGNAL(activated(int)), this, SLOT(sendOnPlaylist(int)));
	connect(subMenuCopy, SIGNAL(activated(int)), this, SLOT(copyOnPlaylist(int)));
	connect(subMenuAppend, SIGNAL(activated(int)), this, SLOT(appendPlaylist(int)));
	icon = loader->loadIcon("view_multicolumn", KIcon::Toolbar);
	playlistMenu->insertItem(icon, i18n("Columns") + "...", this, SLOT(onShowPlaylistHeaderMenu()), 0);
	stopMenu = new QPopupMenu(this);
	stopMenu->insertItem(i18n("now"), 0, 0);
	stopMenu->insertItem(i18n("after track"), 1, 1);
	connect(stopMenu, SIGNAL(activated(int)), this, SLOT(onStopMenu(int)));
	connect(pshStop, SIGNAL(clicked()), this, SLOT(onStop()));
	connect(pshStop, SIGNAL(pressed()), this, SLOT(onStopPressed()));

	connect(this, SIGNAL(doubleClicked()), this, SLOT(onNewPlaylist()));

	dlgAccels.addItem(i18n("Play"), pshAction->accel());
	dlgAccels.addItem(i18n("Stop") + "/" + i18n("Pause"), pshStop->accel());
	dlgAccels.addItem(i18n("Previous"), pshLeft->accel());
	dlgAccels.addItem(i18n("Next"), pshRight->accel());
	dlgAccels.addItem(i18n("Add playlist"), pshNewPlaylist->accel());
	dlgAccels.addItem(i18n("Move up"), pshUp->accel());
	dlgAccels.addItem(i18n("Move down"), pshDown->accel());
	dlgAccels.addItem(i18n("Remove"), pshRemove->accel());
	dlgAccels.addItem(i18n("Go to current"), pshGoToCurrent->accel());
	dlgAccels.addItem(i18n("Add file(s)"), contextMenu->accel(CONTEXT_ADDFILES));
	dlgAccels.addItem(i18n("Add folder"), contextMenu->accel(CONTEXT_ADDFOLDER));
	dlgAccels.addItem(i18n("Add audio CD contents"), contextMenu->accel(CONTEXT_ADDCD));
	dlgAccels.addItem("last.fm", subMenuSettings->accel(CONTEXT_LASTFM));
	dlgAccels.addItem(i18n("Equalizer"), subMenuSettings->accel(CONTEXT_EQUALIZER));
	dlgAccels.addItem(i18n("Sound system"), subMenuSettings->accel(CONTEXT_SOUND));
	dlgAccels.addItem(i18n("Accelerators"), subMenuSettings->accel(CONTEXT_ACCELS));
	dlgAccels.addItem(i18n("Save"), playlistMenu->accel(CONTEXT_SAVE));
	dlgAccels.addItem(i18n("Remove duplicates and dead entries"), subMenuRemove->accel(CONTEXT_RMBOTH));
	dlgAccels.addItem(i18n("Shuffle"), playlistMenu->accel(CONTEXT_SHUFFLE));
	dlgAccels.addItem(i18n("Edit tags"), trackMenu->accel(CONTEXT_TAGS));
	dlgAccels.addItem(i18n("Show lyrics"), trackMenu->accel(CONTEXT_LYRICS));
	dlgAccels.addItem(i18n("Duplicate"), playlistMenu->accel(CONTEXT_DUPLICATE));
	dlgAccels.addItem(i18n("Copy to removable device"), trackMenu->accel(CONTEXT_REMOVABLE));
	dlgAccels.addItem(i18n("Select all"), actionSelectAll->accel());
	dlgAccels.addItem(i18n("Copy on destination playlist"), actionCopyOnDest->accel());
	dlgAccels.addItem(i18n("Send on destination playlist"), actionSendOnDest->accel());
	dlgAccels.addItem(i18n("Append to destination playlist"), actionAppendToDest->accel());
	dlgAccels.addItem(i18n("Increase volume"), actionMakeVolHigher->accel());
	dlgAccels.addItem(i18n("Decrease volume"), actionMakeVolLower->accel());
	dlgAccels.addItem(i18n("Make previous playlist current"), actionMakePrevCurrent->accel());
	dlgAccels.addItem(i18n("Make next playlist current"), actionMakeNextCurrent->accel());
	dlgAccels.addItem(i18n("Copy on playlist"), actionPopupCopyOnDest->accel());
	dlgAccels.addItem(i18n("Send on playlist"), actionPopupSendOnDest->accel());
	dlgAccels.addItem(i18n("Append to playlist"), actionPopupAppendTo->accel());
	dlgAccels.addItem(i18n("Quit"), actionQuit->accel());

	lstPlaylist->restoreLayout(conf, "KMListView lstPlaylist");

	lstPlaylist->setSorting(-1);
	lstPlaylist->setSelectionModeExt(KListView::Extended);
	lstPlaylist->setResizeMode(QListView::NoColumn);
	lstPlaylist->setDropVisualizer(false);
	lstPlaylist->setAllColumnsShowFocus(true);

	headerMenu = new QPopupMenu(lstPlaylist->header());
	for (int col_i = 0 ; col_i != lstPlaylist->columns() ; col_i++)
	{
		QCheckBox * tmp = new QCheckBox(lstPlaylist->columnText(col_i), headerMenu);
		tmp->setChecked((bool)lstPlaylist->columnWidth(col_i));
		if (!tmp->isChecked())
			lstPlaylist->header()->setResizeEnabled(false, col_i);
		headerMenuItems.append(tmp);
		headerMenu->insertItem(tmp, col_i, col_i);
	}
	QPushButton * pshHideHeaderMenu = new QPushButton("OK", headerMenu);
	headerMenu->insertItem(pshHideHeaderMenu, lstPlaylist->columns(), lstPlaylist->columns());
	connect(headerMenu, SIGNAL(aboutToHide()), this, SLOT(onPlaylistHeaderMenu()));
	connect(pshHideHeaderMenu, SIGNAL(clicked()), headerMenu, SLOT(hide()));
	connect(lstPlaylist->header(), SIGNAL(clicked(int)), this, SLOT(sortPlaylist(int)));

	dlgCopyFiles.setCaption(i18n("Copying tracks to removable device") + "...");
	connect(&opCopyFiles, SIGNAL(startedNextCopy(const QPtrList<QNetworkOperation> &)),
		this, SLOT(removableDeviceStartedNextCopy(const QPtrList<QNetworkOperation> &)));

	homeDir = QDir::homeDirPath();
	QDir dir(homeDir + "/.FALF");
	confFile.setName(dir.path() + "/FALF.conf");
	currentTrackFile.setName(dir.path() + "/track.info");
	lockFile.setName(dir.path() + "/lock");
	tmpFiles.setName(dir.path() + "/tmp_files");
	lyrics.setPath(dir.path() + "/lyrics");
	browseDir = homeDir;
	filePattern = "*.mp3 *.ogg *.mpc *.wav *.mp4 *.m4a *.wma *.mp2 *.aif *.mpp *.aac *.mod *.flac *.aiff *.speex *.xm *.it *.s3m *.m3u";

	KCmdLineArgs * args = KCmdLineArgs::parsedArgs();
	QString files(QString::null);

	for (int argsCount = 0 ; argsCount != args->count() ; argsCount++)
		files += ("file://" + QString::fromLocal8Bit(args->arg(argsCount)));

	alreadyRunning = false;

	if (lockFile.exists())
	{
		if (files.isEmpty())
		{
			if (QMessageBox::information(this, i18n("Info"),
				(QString)FALF_NAM + " " + i18n("is already running") + "!\n" +
				i18n("Running multiple copies of FALF will damage configuration file.") + "\n" +
				i18n("Do You really wanna run next copy?"),
				QMessageBox::Yes, QMessageBox::No | QMessageBox::Default,
				QMessageBox::NoButton) != QMessageBox::Yes)
			{
				alreadyRunning = true;
				destroy();
				exit(1);
			}
		}
		else
		{
			tmpFiles.open(IO_WriteOnly | IO_Append);
			QTextStream tmp(&tmpFiles);
			tmp.setEncoding(QTextStream::Unicode);
				tmp << files;
			tmpFiles.close();

			alreadyRunning = true;
			destroy();
			exit(0);
		}
	}
	else
	{
		lockFile.open(IO_WriteOnly);
		QTextStream lock(&lockFile);
			lock << QString::null;
		lockFile.close();
	}

	if (!dir.exists())
		QDir().mkdir(dir.path());

	http = 0;
	lyricsErrStr = i18n("No lyrics found");
	frame = 0;
	sldVol->setValue(50);
	player.setVolume(50);
	statBar = statusBar();
	currentTrack = QString::null;
	action = STOP;
	lfmChallenge = 0;
	lfmBuffer = 0;
	lfmAddr = 0;
	lfmServer = 0;
	lfmSubmit = 0;
	stopAfter = false;
	allowRefresh = false;

	conf->setGroup("Player");
	desiredVolume = conf->readNumEntry("Volume", 50);
	loop = conf->readNumEntry("Loop", 0);
	rnd = conf->readNumEntry("Random", 0);
	destTab = conf->readNumEntry("DestTab", -1);
	playTab = conf->readNumEntry("Tab", 0);
	playIndex = conf->readNumEntry("Track", 0);
	playTimeDirection = conf->readBoolEntry("TimeDirection", true);
	audioCDDevice = conf->readEntry("AudioCDDevice", "/dev/");
	frame = conf->readNumEntry("Frame", sldPos->value());
	dlgSoundSystem.cmbSoundSystem->setCurrentItem(conf->readNumEntry("SoundSystem", 0));

	conf->setGroup("Geometry");
	int x = conf->readNumEntry("X", geometry().x());
	int y = conf->readNumEntry("Y", geometry().y());
	int w = conf->readNumEntry("W", geometry().width());
	int h = conf->readNumEntry("H", geometry().height());
	setGeometry(x, y, w, h);

	conf->setGroup("Last.fm");
	lfmOn = conf->readBoolEntry("Enabled", false);
	lfmUser = conf->readEntry("User", "");
	lfmPassword = conf->readEntry("Password", "");
	lfmPercent = conf->readNumEntry("Percent", 90);

	if ((lfmPercent < 50) || (lfmPercent > 100))
		lfmPercent = 90;

	conf->setGroup("Browser");
	browseDir = conf->readEntry("Directory", QDir::homeDirPath());

	conf->setGroup("Lyrics");
	dlgLyrics.cmbCodec->setCurrentItem(conf->readNumEntry("Codec", 10));
	x = conf->readNumEntry("X", dlgLyrics.geometry().x());
	y = conf->readNumEntry("Y", dlgLyrics.geometry().y());
	w = conf->readNumEntry("W", dlgLyrics.geometry().width());
	h = conf->readNumEntry("H", dlgLyrics.geometry().height());
	dlgLyrics.setGeometry(x, y, w, h);

	conf->setGroup("RemovableDevice");
	dlgRemovableDevice.txtDevice->setText(conf->readEntry("Device", "/dev/sda1"));
	dlgRemovableDevice.txtDir->setText(conf->readEntry("Destination", "mp3"));

	if (confFile.exists())
	{
		frmRestore dlgRestore;
		dlgRestore.prgRestore->setProgress(0);
		dlgRestore.show();

		if (confFile.open(IO_ReadOnly))
		{
			qDebug("Restoring...");

			QTextStream streamIn(&confFile);
			streamIn.setEncoding(QTextStream::Unicode);

			conf->setGroup("Collection");
			dlgRestore.prgRestore->setTotalSteps(conf->readNumEntry("Files", 0));

			QString buffer;

			while (!confFile.atEnd())
			{
				buffer = streamIn.readLine();

				if (buffer == "<equalizer>")
					break;

				int pListPos = buffer.find("</playlist>");
				QString plName = buffer.mid(10, pListPos - 10);
				
				buffer = streamIn.readLine();

				pListPos = buffer.find("</descr>");
				QString plDescr = buffer.mid(7, pListPos - 7);

				pListIndex * p = collection.newPlaylist(plName, plDescr);

				if (!p)
					continue;

				streamIn.readLine();

				while (((buffer = streamIn.readLine()) != "<end>") &&
					(!confFile.atEnd()) && (!buffer.isEmpty()))
				{
					int filePos = buffer.find("</file>");

					QString path = buffer.mid(7, filePos - 7);
					QString name;

					if (path.left(7).lower() == "http://")
						collection.add(p, path, path, 0);
					else
					{
						if (path.left(6).lower() == "cdda:/")
							name = (i18n("Track") + " " + path.right(path.length() - 6));
						else
							name = path.mid(path.findRev("/") + 1, path.length() - 4);

						exTags t = readTags(path);

						collection.add(p, name, path, &t.basic);
					}

					dlgRestore.prgRestore->advance(1);
				}
			}

			eqProfiles & ps = dlgEqualizer.getEqProfiles();

			while (!confFile.atEnd())
			{
				buffer = streamIn.readLine();

				if (buffer == "</equalizer>")
					break;

				int pPos = buffer.find("</profile>");
				eqProfile p(buffer.mid(10, pPos - 10));
				
				ps.push_back(p);
			}
			
			conf->setGroup("Equalizer");
			dlgEqualizer.refreshProfilesList(conf->readNumEntry("Profile", 0));
			dlgEqualizer.chkEqualizer->setChecked(conf->readBoolEntry("Enabled", false));

			player.setSoundSystem(dlgSoundSystem.cmbSoundSystem->currentText().lower());

			if (collection.getPlaylistCount())
			{
				refreshPlaylist(playTab);

				if (playTab == 0)
					refreshList();
			
				if (playIndex >= 0)
				{
					QListViewItem * i = lstPlaylist->itemAtIndex(playIndex);

					if (i)
					{
						lstPlaylist->clearSelection();
						lstPlaylist->setSelected(i, true);
					}
				}

				if ((frame > 0) && (files.isEmpty()))
				{
					player.setPos(frame);
					sldPos->setValue(frame);
					QTimer::singleShot(0, this, SLOT(onAction()));
				}
			}

			confFile.close();
		}
	}
	else
		onSoundSystem();

	if (!files.isEmpty())
	{
		collection.newPlaylist(
			QDate::currentDate(Qt::LocalTime).toString() + " " +
			QTime::currentTime(Qt::LocalTime).toString(),
			i18n("Double click to rename playlist"));
		
		frame = 0;
		playIndex = 0;
		playTab = (collection.getPlaylistCount() - 1);
		refreshPlaylist(playTab);
		parse(files);
	}

	loop--;
	onLoop();

	rnd--;
	onRnd();

	if (!collection.getPlaylistCount())
	{
		collection.newPlaylist(i18n("Unnamed Playlist"), i18n("Double click to rename playlist"));
		refreshPlaylist(0);
		refreshList();
		onOff();
	}

	if (!dlgEqualizer.getEqProfiles().count())
	{
		eqProfile p(i18n("Unnamed"));
		dlgEqualizer.getEqProfiles().push_back(p);
		dlgEqualizer.refreshProfilesList(0);
		dlgEqualizer.chkEqualizer->setChecked(false);
	}

	setAcceptDrops(true);

	sldVol->setOrientation(Qt::Horizontal);
	sldVol->setPageStep(5);
	sldVol->setValue(desiredVolume);

	connect(sldVol, SIGNAL(valueChanged(int)), this, SLOT(onVolume()));
	
	connect(lblTime, SIGNAL(leftMouseButtonClicked()), this, SLOT(onTimeClicked()));

	connect(lstPlaylist, SIGNAL(dropped(QDropEvent *, QListViewItem *)), this, SLOT(fileDropped(QDropEvent *, QListViewItem *)));
	connect(lstPlaylist, SIGNAL(doubleClicked(QListViewItem *)), this, SLOT(onPlaylist()));
	connect(lstPlaylist, SIGNAL(mouseButtonReleased(const QPoint &)), this, SLOT(onMouseButtonReleased(const QPoint &)));
	connect(lstPlaylist, SIGNAL(keyPressed(QKeyEvent *)), this, SLOT(onKeyPressed(QKeyEvent *)));
	connect(lstPlaylist, SIGNAL(rightMouseButtonClicked(const QPoint &)), this, SLOT(showContextMenu(const QPoint &)));
	connect(lstPlaylist, SIGNAL(dragging()), this, SLOT(onDragging()));
	
	if (player.getPos())
	{
		player.setVolume(0);
		connect(&volumeTimer, SIGNAL(timeout()), this, SLOT(restoreVolume()));
		volumeTimer.start(500);
	}
	else
		onVolume();

	connect(&playTimer, SIGNAL(timeout()), this, SLOT(onPlayTimer()));
	connect(&searchTimer, SIGNAL(timeout()), this, SLOT(onSearchTimer()));
	connect(&lfmTimer, SIGNAL(timeout()), this, SLOT(lastFmQueue()));
	connect(&tmpFilesTimer, SIGNAL(timeout()), this, SLOT(addTmpFiles()));

	tmpFilesTimer.start(3000);

	connect(systemTray, SIGNAL(quitSelected()), this, SLOT(onQuitSelected()));
	connect(systemTray, SIGNAL(wheelMoved(QWheelEvent *)), this, SLOT(scrollVol(QWheelEvent *)));

	icon = loader->loadIcon("view_remove", KIcon::Toolbar);
	pshClearList->setPixmap(icon);
	icon = loader->loadIcon("tab_remove", KIcon::Toolbar);
	pshDeletePlaylist->setPixmap(icon);
	icon = loader->loadIcon("player_start", KIcon::Toolbar);
	pshLeft->setPixmap(icon);
	popupMenu->changeItem(TRAY_LEFT, icon, i18n("Previous"));
	icon = loader->loadIcon("player_end", KIcon::Toolbar);
	pshRight->setPixmap(icon);
	popupMenu->changeItem(TRAY_RIGHT, icon, i18n("Next"));
	icon = loader->loadIcon("locationbar_erase", KIcon::Toolbar);
	dlgPlaylist.pshClearTxtPlaylist->setPixmap(icon);
	dlgPlaylist.pshClearTxtDescription->setPixmap(icon);
	dlgEqualizer.pshClearTxtName->setPixmap(icon);
	pshNewPlaylist->setPixmap(pixAppendToPlaylist);
	icon = loader->loadIcon("kmix", KIcon::Toolbar);
	pixVol->setPixmap(icon);
	icon = loader->loadIcon("up", KIcon::Toolbar);
	pshUp->setPixmap(icon);
	icon = loader->loadIcon("down", KIcon::Toolbar);
	pshDown->setPixmap(icon);
	icon = loader->loadIcon("redo", KIcon::Toolbar);
	pshGoToCurrent->setPixmap(icon);
	icon = loader->loadIcon("remove", KIcon::Toolbar);
	pshRemove->setPixmap(icon);
	pshStop->setPixmap(pixStop);
	popupMenu->changeItem(TRAY_STOP, pixStop, i18n("Stop"));
	popupMenu->setItemEnabled(TRAY_STOP, false);

	tabPlaylist->setTabReorderingEnabled(true);
	
	setTip();
	setPlayTimeMsg();

	txtSearch->hide();

	if (lfmOn)
		lastFmHandShake();

	onTabChanged();
}

void frmMain::destroy()
{
	player.termThread();
	player.wait();

	if (!alreadyRunning)
	{
		currentTrackFile.remove();
		lockFile.remove();

		qDebug("Storing...");
		store();

		qDebug("Graceful shutdown...");
	}
	else
		qDebug("Aborted!");
}

void frmMain::dragEnterEvent(QDragEnterEvent * event)
{
	event->accept(QTextDrag::canDecode(event));
}

void frmMain::dropEvent(QDropEvent * event)
{
	QString text;

	if (QTextDrag::decode(event, text))
		parse(text);
}

void frmMain::onNewPlaylist()
{
	dlgPlaylist.ok = false;
	dlgPlaylist.txtPlaylist->setText(i18n("Unnamed Playlist"));
	dlgPlaylist.txtDescription->setText("");
	dlgPlaylist.txtPlaylist->setFocus();
	dlgPlaylist.chkDest->setChecked(false);

	dlgPlaylist.exec();

	if (dlgPlaylist.ok)
	{
		const int c = tabPlaylist->count();

		if (!dlgPlaylist.txtPlaylist->text().isEmpty())
			collection.newPlaylist(dlgPlaylist.txtPlaylist->text(), dlgPlaylist.txtDescription->text());
		else
			collection.newPlaylist(i18n("Unnamed Playlist"), dlgPlaylist.txtDescription->text());

		if (dlgPlaylist.chkDest->isChecked())
			destTab = c;

		refreshPlaylist(c);
	}
}

void frmMain::refreshPlaylist(const int & index)
{
	QWidget * w = 0;

	tabPlaylist->disconnect();

	while (tabPlaylist->count())
	{
		w = tabPlaylist->currentPage();
		tabPlaylist->removePage(w);
		delete w;
	}

	for (int i = 0 ; i != collection.getPlaylistCount() ; i++)
	{
		w = new QWidget;
		
		const pListIndex * p = collection.getPlaylist(i);

		if (p)
			tabPlaylist->addTab(w, collection.getPlaylistName(p));
	}

	connect(tabPlaylist, SIGNAL(currentChanged(QWidget*)), this, SLOT(onTabChanged()));
	connect(tabPlaylist, SIGNAL(testCanDecode(const QDragMoveEvent *, bool &)),
		this, SLOT(onTestCanDecode(const QDragMoveEvent *, bool &)));
	connect(tabPlaylist, SIGNAL(receivedDropEvent(QWidget *, QDropEvent *)),
		this, SLOT(onReceivedDropEvent(QWidget *, QDropEvent *)));
	connect(tabPlaylist, SIGNAL(initiateDrag(QWidget *)), this, SLOT(onInitiateDrag(QWidget *)));
	connect(tabPlaylist, SIGNAL(mouseDoubleClick(QWidget *)), this, SLOT(onTabDblClicked(QWidget *)));
	
	if (index != tabPlaylist->currentPageIndex())
		tabPlaylist->setCurrentPage(index);

	setTabTip();
}

void frmMain::setTabTip()
{
	const pListIndex * p = 0;
	QWidget * w = 0;

	for (int i = 0 ; i != collection.getPlaylistCount() ; i++)
	{
		p = collection.getPlaylist(i);

		if (!p)
			return;

		w = tabPlaylist->page(i);

		QString name = collection.getPlaylistName(p);

		if (i == destTab)
			name = ("<u>" + name + "</u>");

		QString s;
		s += "<b>" + name + "</b>";
		QString descr = collection.getPlaylistDescr(p);
		if (!descr.isEmpty())
		{
			s += "<br>";
			s += "<i>" + descr + "</i>";
		}
		s += "<br><br>";
		QString len = QString("%1:%2:%3").arg(p->len.h, 2).arg(p->len.m, 2).arg(p->len.s, 2);
		len.replace(" ", "0");
		s += QString::number(p->count) + " " + i18n("track(s)") + " (" + len + ")";
	
		tabPlaylist->setTabToolTip(w, s);
	}
}

void frmMain::refreshList()
{
	const pListIndex * p = collection.getPlaylist(tabPlaylist->currentPageIndex());

	QListViewItem * item = lstPlaylist->selectedItem();

	int curSel = 0;

	if (item)
		curSel = lstPlaylist->itemIndex(item);

	lstPlaylist->clear();

	if ((p) && (p->list))
	{
		int i = 0;
		const pList * l = 0;

		while ((l = collection.get(p, i++)))
		{
			QListViewItem * newItem = NULL;

			QString title = l->fileTags.tagTitle;

			if (title.isEmpty())
				title = l->trackName;

			newItem = new KListViewItem(lstPlaylist, lstPlaylist->lastItem(),
				title,
				l->fileTags.tagAlbum,
				l->fileTags.tagArtist,
				l->fileTags.tagBitrate.isEmpty() ? "" : (l->fileTags.tagBitrate + " kbps"),
				l->fileTags.tagLength);

			newItem->setDropEnabled(true);
			newItem->setDragEnabled(true);
		}

		lstPlaylist->setSelected(lstPlaylist->itemAtIndex(curSel), true);
	}

	onOff();

	setStatBarMsg();
	setTabTip();
}

void frmMain::onDeletePlaylist()
{
	if (collection.getPlaylistCount() == 1)
		return;

	int index = tabPlaylist->currentPageIndex();

	QMessageBox msg(i18n("Question"), i18n("Do You really wanna delete playlist") + " \"" + tabPlaylist->label(index).remove("&") + "\"?", QMessageBox::Warning,
		QMessageBox::Yes, QMessageBox::No | QMessageBox::Default, QMessageBox::NoButton);

	if (msg.exec() != QMessageBox::Yes)
		return;

	QWidget * w = tabPlaylist->currentPage();
	pListIndex * p = collection.getPlaylist(tabPlaylist->indexOf(w));

	if (!p)
		return;

	clearStop(true);

	collection.delPlaylist(p);
	tabPlaylist->removePage(w);
	delete w;

	tabPlaylist->setCurrentPage(index - 1);

	setTip();
}

void frmMain::onTabChanged()
{
	if (!allowRefresh)
	{
		allowRefresh = true;
		return;
	}

	txtSearch->hide();

	subMenuSend->clear();
	subMenuCopy->clear();
	subMenuAppend->clear();

	for (int i = 0 ; i != collection.getPlaylistCount() ; i++)
	{
		const pListIndex * p = collection.getPlaylist(i);

		if (p)
		{
			if (i != tabPlaylist->currentPageIndex())
			{
				subMenuSend->insertItem(p->name, i, i);
				subMenuCopy->insertItem(p->name, i, i);
				subMenuAppend->insertItem(p->name, i, i);
			}
			else
			{
				refreshList();

				subMenuSend->insertSeparator();
				subMenuCopy->insertSeparator();
				subMenuAppend->insertSeparator();
			}
		}

	}

	lstPlaylist->clearSelection();

	if (tabPlaylist->currentPageIndex() == playTab)
		lstPlaylist->setSelected(lstPlaylist->itemAtIndex(playIndex), true);
	else
	{
		QListViewItem * i = lstPlaylist->firstChild();
		if (i)
			lstPlaylist->setSelected(i, true);
	}

	lstPlaylist->header()->setSortIndicator(-1, Qt::Descending);

	lstPlaylist->setFocus();
	ensureVisible();
}

void frmMain::onClearList()
{
	unsigned int index = tabPlaylist->currentPageIndex();

	QMessageBox msg(i18n("Question"), i18n("Do You really wanna delete all files from playlist") + " \"" + tabPlaylist->label(index).remove("&") + "\"?", QMessageBox::Warning,
		QMessageBox::Yes, QMessageBox::No | QMessageBox::Default, QMessageBox::NoButton);

	if (msg.exec() != QMessageBox::Yes)
		return;

	pListIndex * p = collection.getPlaylist(index);

	if (!p)
		return;

	lstPlaylist->clear();

	pshClearList->setEnabled(false);

	clearStop();

	collection.del(p, -1);

	refreshList();
	setTip();
}

void frmMain::onAction()
{
	const pListIndex * p = NULL;
	const pList * l = NULL;

	QListViewItem * item = lstPlaylist->selectedItem();

	if ((!item) && (action != PLAY))
		return;

	bool enable = true;

	lfmTimer.stop();

	lstPlaylist->clearSelection();

	markCurrent(playIndex);

	switch (action)
	{
		case PAUSE:
		case STOP:
			playTab = tabPlaylist->currentPageIndex();
			playIndex = lstPlaylist->itemIndex(item);

			lstPlaylist->setSelected(item, true);

			p = collection.getPlaylist(playTab);

			if (!p)
				return;

			l = collection.get(p, playIndex);

			if (!l)
				return;

			player.setTrack(l->filePath);
			player.wakeThread();

			if (!player.running())
				player.start(QThread::TimeCriticalPriority);

			playTimer.start(1000, false);
			sldPos->setEnabled(true);

			if (!frame)
				player.setPos(sldPos->value());
			else
			{
				player.setPos(frame);
				frame = 0;
			}

			setPlayTimeMsg(true);

			if ((action != PAUSE) && (rnd == 1) && (loop != 2))
				randList.addIndex(playIndex);

			if (l->filePath.left(7).lower() != "http://")
			{
				httpStream = false;

				if ((lfmOn) && (action != PAUSE))
				{
					if (l->fileTags.tagLengthSeconds >= 30)
						lfmTimer.start((l->fileTags.tagLengthSeconds * lfmPercent * 10), true);
				}
			}
			else
				httpStream = true;

			pshAction->setPixmap(pixPause);
			systemTray->setPixmap(pixPlay);
			popupMenu->changeItem(0, pixPause, i18n("Pause"));
			
			if (dlgLyrics.isVisible())
				onLyrics();

			action = PLAY;
		break;
		case PLAY:
			player.termThread();
			player.wait();
			playTimer.stop();

			if (tabPlaylist->currentPageIndex() != playTab)
				tabPlaylist->setCurrentPage(playTab);

			lstPlaylist->setSelected(lstPlaylist->itemAtIndex(playIndex), true);

			pshAction->setPixmap(pixPlay);
			systemTray->setPixmap(pixPause);
			popupMenu->changeItem(0, pixPlay, i18n("Play"));
			action = PAUSE;
			enable = false;
		break;
	}

	lstPlaylist->setEnabled(enable);
	tabPlaylist->setEnabled(enable);

	if ((tabPlaylist->count() != 1) && (action != PAUSE))
		pshDeletePlaylist->setEnabled(true);
	else
		pshDeletePlaylist->setEnabled(false);

	pshClearList->setEnabled(enable);
	pshNewPlaylist->setEnabled(enable);
	pshUp->setEnabled(enable);
	pshDown->setEnabled(enable);
	pshRemove->setEnabled(enable);
	pshStop->setEnabled(enable);
	popupMenu->setItemEnabled(1, enable);
	popupMenu->setItemEnabled(2, enable);
	popupMenu->setItemEnabled(3, enable);
	pshLeft->setEnabled(enable);
	pshRight->setEnabled(enable);
	sldPos->setEnabled(enable);
	lblTime->setEnabled(enable);

	setTip();
	markCurrent();
	ensureVisible();

	lstPlaylist->setFocus();
}

void frmMain::onVolume()
{
	volumeTimer.stop();
	volumeTimer.disconnect();
	player.setVolume(sldVol->value());
}

void frmMain::onPlayTimer()
{
	static bool eqAdj = false;
	
	sldPos->setMinValue(0);
	sldPos->setMaxValue(player.getLength()); // doesn't work without it ;[
	sldPos->setValue(player.getPos());

	if (!player.running())
	{
		eqAdj = false;
		playTimer.stop();

		if ((rnd != 2) && (tabPlaylist->currentPageIndex() != playTab))
			tabPlaylist->setCurrentPage(playTab);

		if (stopAfter)
		{
			onStop();
			return;
		}

		if (loop == 2)
		{
			action = STOP;
			lstPlaylist->setSelected(lstPlaylist->itemAtIndex(playIndex), true);
			onAction();
			return;
		}

		switch (rnd)
		{
			case 0:
			case 1:
				onRight();
			break;
			case 2:
				if ((tabPlaylist->currentPageIndex() == playTab)
					&& (lstPlaylist->itemIndex(lstPlaylist->selectedItem()) == playIndex))
					onRight();
				else
				{
					action = STOP;
					onAction();
				}
			break;
		}
	}
	else
	{
		if ((dlgEqualizer.chkEqualizer->isChecked()) && (!eqAdj))
		{
			dlgEqualizer.somethingChanged();
			eqAdj = true;
		}

		if (httpStream)
		{
			pList * l = collection.get(collection.getPlaylist(playTab), playIndex);

			if ((l) && (l->filePath.left(7).lower() == "http://"))
			{
				tags t = httpStreamMeta();
	
				if ((t.tagTitle != l->fileTags.tagTitle) ||
					(t.tagAlbum != l->fileTags.tagAlbum) ||
					(t.tagArtist != l->fileTags.tagArtist))
				{
					l->fileTags = t;

					if (tabPlaylist->currentPageIndex() == playTab)
						refreshList();
					setTip();
				}
			}
		}
		else
			setPlayTimeMsg();
	}
}

tags frmMain::httpStreamMeta()
{
	tags ft;

	xine_stream_t * tmp = (xine_stream_t *)player.getXineStreamHandle();
	
	if (!tmp)
		return ft;

	ft.tagTitle = QString::fromUtf8(xine_get_meta_info(tmp, XINE_META_INFO_TITLE));

	if (!ft.tagTitle.isEmpty())
	{
		ft.tagAlbum =
			QString::fromUtf8(xine_get_meta_info(tmp, XINE_META_INFO_ALBUM));
        	ft.tagArtist =
			QString::fromUtf8(xine_get_meta_info(tmp, XINE_META_INFO_ARTIST));
	}

	int brate = xine_get_stream_info(tmp, XINE_STREAM_INFO_AUDIO_BITRATE) / 1000;

	int length = 0;
	xine_get_pos_length(tmp, NULL, NULL, &length);

	length /= 1000;
	
	ft.tagBitrate = QString::number(brate);
	ft.tagLength = getTimeAsString(length);
	ft.tagLengthSeconds = length;

	return ft;
}

void frmMain::onSldPosPressed()
{
	currentTrack = player.getTrack();
	lfmTimer.stop();
	playTimer.stop();
	player.termThread();
	player.wait();
}

void frmMain::onSldPosReleased()
{
	if (player.getPos() == 0)
		setPlayTimeMsg();

	player.setTrack(currentTrack);
	player.wakeThread();
	player.start(QThread::TimeCriticalPriority);
	playTimer.start(1000, false);
}

void frmMain::onSldPos()
{
	const int & pos = sldPos->value();

	player.setPos(pos);

	setPlayTimeMsg();
}

void frmMain::onLeft()
{
	if ((rnd != 2) && (tabPlaylist->currentPageIndex() != playTab))
		tabPlaylist->setCurrentPage(playTab);

	QListViewItem * item = lstPlaylist->selectedItem();

	if (!item)
		return;

	int i = lstPlaylist->itemIndex(item);

	switch (action)
	{
		case PLAY:
			player.termThread();
			player.wait();
			playTimer.stop();
			sldPos->setValue(0);
			action = STOP;

			lstPlaylist->clearSelection();

			if (rnd == 1)
				random();
			else
			{
				if ((rnd == 2) && 
					((tabPlaylist->currentPageIndex() != playTab) ||
						(i != playIndex)))
				{
					lstPlaylist->setSelected(lstPlaylist->itemAtIndex(i), true);
					onAction();
					break;
				}

				if (playIndex > 0)
				{
					lstPlaylist->setSelected(lstPlaylist->itemAtIndex(playIndex - 1), true);
					onAction();
				}
				else
				{
					onStop();
					lstPlaylist->setSelected(lstPlaylist->itemAtIndex(playIndex), true);
				}
			}
		break;
		case STOP:
		break;
	}

	onOff();
	setStatBarMsg();
	ensureVisible();
}

void frmMain::onRight()
{
	if ((rnd != 2) && (tabPlaylist->currentPageIndex() != playTab))
		tabPlaylist->setCurrentPage(playTab);

	QListViewItem * item = lstPlaylist->selectedItem();

	if (!item)
		return;

	int i = lstPlaylist->itemIndex(item);

	switch (action)
	{
		case PLAY:
			player.termThread();
			player.wait();
			playTimer.stop();
			sldPos->setValue(0);
			action = STOP;
			
			lstPlaylist->clearSelection();

			if (rnd == 1)
				random();
			else
			{
				if ((rnd == 2) && 
					((tabPlaylist->currentPageIndex() != playTab) ||
						(i != playIndex)))
				{
					lstPlaylist->setSelected(lstPlaylist->itemAtIndex(i), true);
					onAction();
					break;
				}

				if (playIndex < (lstPlaylist->childCount() - 1))
				{
					lstPlaylist->setSelected(lstPlaylist->itemAtIndex(playIndex + 1), true);
					onAction();
				}
				else
				{
					if ((loop == 0) || (loop == 2))
					{
						onStop();
						lstPlaylist->setSelected(lstPlaylist->itemAtIndex(playIndex), true);
					}
					else
					{
						QListViewItem * first = lstPlaylist->firstChild();

						if (first)
						{
							lstPlaylist->setSelected(first, true);
							onAction();
						}
					}
				}
			}
		break;
		case STOP:
		break;
	}

	onOff();
	setStatBarMsg();
	ensureVisible();
}

void frmMain::onPlaylist()
{
	QListViewItem * i = lstPlaylist->selectedItem();

	if (!i)
		return;

	player.termThread();
	player.wait();
	playTimer.stop();
	sldPos->setValue(0);
	action = STOP;

	markCurrent(playIndex);

	playTab = tabPlaylist->currentPageIndex();
	playIndex = lstPlaylist->itemIndex(i);

	randList.remove();
	setStatBarMsg();

	onAction();
}

void frmMain::onUp()
{
	pListIndex * p = collection.getPlaylist(tabPlaylist->currentPageIndex());

	if (!p)
		return;

	QListViewItems items = lstPlaylist->selectedItems();
	QListViewItemsIterator it(items);
	QListViewItem * item = 0;

	ints indexes;

	while ((item = it.current()))
	{
		int i = lstPlaylist->itemIndex(item);

		if (i <= 0)
			return;

		indexes.append((i - 1));

		collection.up(p, i);
		randMove(i, (i - 1));

		++it;
	}

	refreshList();
	lstPlaylist->clearSelection();

	for (ints::iterator i = indexes.begin() ; i != indexes.end() ; i++)
		lstPlaylist->setSelected(lstPlaylist->itemAtIndex((*i)), true);

	ensureVisible();
}

void frmMain::onDown()
{
	pListIndex * p = collection.getPlaylist(tabPlaylist->currentPageIndex());

	if (!p)
		return;

	QListViewItems items = lstPlaylist->selectedItems();
	QListViewItem * item = items.last();

	ints indexes;

	for (;;)
	{
		if (!item)
			break;

		int i = lstPlaylist->itemIndex(item);

		if ((i + 1) >= lstPlaylist->childCount())
			return;

		indexes.append((i + 1));

		collection.down(p, i);
		randMove(i, (i + 1));

		item = items.prev();
	}

	refreshList();
	lstPlaylist->clearSelection();

	for (ints::iterator i = indexes.begin() ; i != indexes.end() ; i++)
		lstPlaylist->setSelected(lstPlaylist->itemAtIndex((*i)), true);

	ensureVisible();
}

void frmMain::onRemove(bool ask)
{
	pListIndex * p = collection.getPlaylist(tabPlaylist->currentPageIndex());

	if (!p)
		return;

	int i = 0;
	bool wasPlaying = false;

	QListViewItems items = lstPlaylist->selectedItems();
	//QListViewItemsIterator it(items);
	QListViewItem * item = items.first();

	if ((ask) && (items.count() > 1))
	{
		QMessageBox msg(i18n("Question"), i18n("Do You really wanna remove") + " " + QString::number(items.count()) + " " + i18n("tracks from current playlist?"), QMessageBox::Warning,
			QMessageBox::Yes, QMessageBox::No | QMessageBox::Default, QMessageBox::NoButton);

		if (msg.exec() != QMessageBox::Yes)
			return;
	}

	for (;;)
	{
		if (!item)
			break;

		i = lstPlaylist->itemIndex(item);

		if (tabPlaylist->currentPageIndex() == playTab)
		{
			if (i == playIndex)
			{
				if (player.running())
					wasPlaying = true;
				onStop();
			}
			else
			{
				if (playIndex > i)
					playIndex--;
			}

			if (!randList.isEmpty())
			{
				if (randList.searchForIndex(i))
					randList.deleteIndex(i);

				for (int c = (i + 1) ; c < lstPlaylist->childCount() ; c++)
				{
					if (randList.searchForIndex(c))
					{
						randList.deleteIndex(c);
						randList.addIndex(c - 1);
					}
				}
			}
		}
	
		delete item;
		
		collection.del(p, i);

		item = items.next();
	}

	setStatBarMsg();
	setTabTip();

	lstPlaylist->clearSelection();

	if (i + 1 <= lstPlaylist->childCount())
		lstPlaylist->setSelected(lstPlaylist->itemAtIndex(i), true);
	else
		lstPlaylist->setSelected(lstPlaylist->itemAtIndex(0), true);

	if (!player.running())
	{
		sldPos->setValue(0);

		if (lstPlaylist->childCount() == 0)
			sldPos->setEnabled(false);

		if (wasPlaying)
		{
			if (rnd == 1)
				random();
			else
				onAction();
		}
	}

	ensureVisible();
}

void frmMain::onGoToCurrent()
{
	tabPlaylist->setCurrentPage(playTab);
}

void frmMain::onQuitSelected()
{
	int x = this->x(), y = this->y();

	unsigned short int count = queue.count();
	if (count > 0)
	{
		QMessageBox msg(i18n("Question"), QString::number(count) + " " + 
			i18n("track(s) remain(s) in last.fm's queue.") + "\n" +
			i18n("Do You really wanna quit?"), QMessageBox::Warning,
			QMessageBox::Yes, QMessageBox::No | QMessageBox::Default, QMessageBox::NoButton);

		if (msg.exec() != QMessageBox::Yes)
			return;
	}

	opCopyFiles.stop();

	dlgCopyFiles.close();
	dlgLyrics.close();
	dlgEqualizer.close();
	systemTray->close();

	move(x, y);

	close();
}

void frmMain::onAbout()
{
	KAboutApplication aboutApp;
	aboutApp.exec();
}

void frmMain::onLoop()
{
	if (loop == 2)
		loop = 0;
	else
		loop++;

	switch (loop)
	{
		case 0:
			pshLoop->setPixmap(pixPlayOnce);
			QToolTip::add(pshLoop, i18n("Repeat playlist"));
		break;
		case 1:
			pshLoop->setPixmap(pixPlayLoop);
			QToolTip::add(pshLoop, i18n("Repeat track"));
		break;
		case 2:
			pshLoop->setPixmap(pixTrackLoop);
			QToolTip::add(pshLoop, i18n("Don't repeat playlist"));
		break;
	}
}

QString frmMain::getTimeAsString(const int & time)
{
	length len;
	len += time;

	QString s;
	s = QString("%1:%2:%3").arg(len.h, 2).arg(len.m, 2).arg(len.s, 2);
	s.replace(" ", "0");

	if (!len.h)
		s = s.mid(3);

	return s;
}

exTags frmMain::readTags(const QString & file)
{
	exTags ft;

	if (file.left(6).lower() != "cdda:/")
	{
		QString crap("! WWW.POLSKIE-MP3.TK !");

		TagLib::FileRef f((const char *)file.local8Bit());

		if (!f.isNull())
		{
			if (TagLib::Tag * tag = f.tag())
			{
				ft.basic.tagTitle = TStringToQString(tag->title());
				ft.basic.tagAlbum = TStringToQString(tag->album());
				ft.basic.tagArtist = TStringToQString(tag->artist());

				ft.basic.tagTitle.remove(crap);
				ft.basic.tagTitle.remove(crap.lower());
				ft.basic.tagAlbum.remove(crap);
				ft.basic.tagAlbum.remove(crap.lower());
				ft.basic.tagArtist.remove(crap);
				ft.basic.tagArtist.remove(crap.lower());

				ft.basic.tagTitle = ft.basic.tagTitle.simplifyWhiteSpace();
				ft.basic.tagAlbum = ft.basic.tagAlbum.simplifyWhiteSpace();
				ft.basic.tagArtist = ft.basic.tagArtist.simplifyWhiteSpace();

				ft.tagYear = QString::number(tag->year());
				ft.tagComment = TStringToQString(tag->comment());
				ft.tagTrack = QString::number(tag->track());
				ft.tagGenre = TStringToQString(tag->genre());
			}

			if (TagLib::AudioProperties * properties = f.audioProperties())
			{
				int brate = properties->bitrate();

				ft.basic.tagBitrate = (QString::number(brate));
				ft.basic.tagLength = getTimeAsString(properties->length());
				ft.basic.tagLengthSeconds = properties->length();
			}
		}
	}
	else
	{
		static xine_t * xine = (xine_t *)player.getXineHandle();
		
		if (!xine)
			return ft;

		xine_stream_t * tmp = xine_stream_new(xine, NULL, NULL);
		
		if (xine_open(tmp, QFile::encodeName(file)))
		{
			ft.basic.tagTitle = QString::fromUtf8(xine_get_meta_info(tmp, XINE_META_INFO_TITLE));

			if (!ft.basic.tagTitle.isEmpty())
			{
				ft.basic.tagAlbum =
					QString::fromUtf8(xine_get_meta_info(tmp, XINE_META_INFO_ALBUM));
                		ft.basic.tagArtist =
					QString::fromUtf8(xine_get_meta_info(tmp, XINE_META_INFO_ARTIST));
			}

			int sampleRate = xine_get_stream_info(tmp, XINE_STREAM_INFO_AUDIO_SAMPLERATE);
			int bitsPerSample = xine_get_stream_info(tmp, XINE_STREAM_INFO_AUDIO_BITS);
			int numberOfChannels = xine_get_stream_info(tmp, XINE_STREAM_INFO_AUDIO_CHANNELS);
			int brate = (sampleRate * bitsPerSample * numberOfChannels) / 1000;

			int length = 0;
			xine_get_pos_length(tmp, NULL, NULL, &length);

			length /= 1000;
			
			ft.basic.tagBitrate = QString::number(brate);
			ft.basic.tagLength = getTimeAsString(length);
			ft.basic.tagLengthSeconds = length;

			xine_close(tmp);
        	}

		xine_dispose(tmp);
	}

	return ft;
}

void frmMain::editTags()
{
	if (lstPlaylist->childCount() == 0)
	{
		QMessageBox::information(this, i18n("Info"), i18n("First select some file!"));
		return;
	}

	pListIndex * p = collection.getPlaylist(tabPlaylist->currentPageIndex());

	if (!p)
		return;

	QListViewItems items = lstPlaylist->selectedItems();
	QListViewItemsIterator it(items);
	QListViewItem * item = 0;

	ints indexes;

	while ((item = it.current()))
	{
		if (!item)
			break;

		int i = lstPlaylist->itemIndex(item);
		indexes.append(i);

		pList * l = collection.get(p, i);

		if (!l)
		{
			++it;
			continue;
		}

		dlgTags.txtLocation->setText(l->filePath);

		dlgTags.txtTitle->setText(l->fileTags.tagTitle);
		dlgTags.txtAlbum->setText(l->fileTags.tagAlbum);
		dlgTags.txtArtist->setText(l->fileTags.tagArtist);

		exTags eT = readTags(l->filePath);

		dlgTags.txtYear->setText(eT.tagYear);
		dlgTags.txtComment->setText(eT.tagComment);
		dlgTags.txtTrack->setText(eT.tagTrack);
		dlgTags.txtGenre->setText(eT.tagGenre);

		dlgTags.txtTitle->setFocus();

		dlgTags.ok = false;
		
		dlgTags.exec();

		if (dlgTags.ok)
		{
			writeTags(l->filePath,
				dlgTags.txtTitle->text(),
				dlgTags.txtAlbum->text(),
				dlgTags.txtArtist->text(),
				dlgTags.txtYear->text(),
				dlgTags.txtComment->text(),
				dlgTags.txtTrack->text(),
				dlgTags.txtGenre->text());

			eT = readTags(l->filePath);
		
			l->fileTags.tagTitle = eT.basic.tagTitle;
			l->fileTags.tagAlbum = eT.basic.tagAlbum;
			l->fileTags.tagArtist = eT.basic.tagArtist;
		}

		++it;
	}

	refreshList();
	ensureVisible();
	setTip();

	for (ints::iterator i = indexes.begin() ; i != indexes.end() ; i++)
		lstPlaylist->setSelected(lstPlaylist->itemAtIndex((*i)), true);
}

void frmMain::writeTags(const QString & file,
	const QString & title,
	const QString & album,
	const QString & artist,
	const QString & year,
	const QString & comment,
	const QString & track,
	const QString & genre)
{
	tags ft;

	TagLib::FileRef f((const char *)file.local8Bit());

	if (!f.isNull())
	{
		TagLib::Tag * tag = f.tag();
		
		tag->setTitle(QStringToTString(title));
		tag->setAlbum(QStringToTString(album));
		tag->setArtist(QStringToTString(artist));

		tag->setYear(year.toUInt());
		tag->setComment(QStringToTString(comment));
		tag->setTrack(track.toUInt());
		tag->setGenre(QStringToTString(genre));

		f.save();
	}
}

void frmMain::setTip()
{
	currentTrackFile.open(IO_WriteOnly);
	QTextStream streamOut(&currentTrackFile);
	streamOut.setEncoding(QTextStream::Unicode);

	QString title(QString::null),
		album(QString::null),
		artist(QString::null),
		bitrate(QString::null),
		length(QString::null),
		text(QString::null),
		pureText(QString::null);

	if (player.running())
	{
		const pListIndex * p = collection.getPlaylist(playTab);

		if (!p)
			return;

		const pList * l = collection.get(p, playIndex);

		if (!l)
			return;

		if (l->fileTags.tagTitle.isEmpty())
			title = l->trackName;
		else
			title = l->fileTags.tagTitle;
		
		text += ("<center><b><i>" + (QString)FALF_NAM + "</i></b><br></center>");
		text += ("<left><b>" + i18n("Title") + ":</b> " + title + "<br>");
		pureText += title;

		if (!l->fileTags.tagAlbum.isEmpty())
		{
			album = l->fileTags.tagAlbum;
			text += ("<left><b>" + i18n("Album") + ":</b> " + album + "<br>");
			pureText += (" - " + album);
		}

		if (!l->fileTags.tagArtist.isEmpty())
		{
			artist = l->fileTags.tagArtist;
			text += ("<left><b>" + i18n("Artist") + ":</b> " + artist + "<br>");
			pureText += (" - " + artist);
		}

		if (!l->fileTags.tagBitrate.isEmpty())
		{
			bitrate = l->fileTags.tagBitrate;
			text += ("<left><b>" + i18n("Bitrate") + ":</b> " + bitrate + " kbps");
			pureText += (" - " + bitrate + " kbps");
		}
	}
	else
		text = (QString)FALF_NAM;

	if (text != (QString)FALF_NAM)
	{
		if (pureText.isEmpty())
			pureText = text;

		setCaption((QString)FALF_NAM + " - [" + pureText + "]");

		streamOut << "TITLE: " << title
			<< "\nALBUM: "<< album
			<< "\nARTIST: " << artist
			<< "\nBITRATE: " << bitrate
			<< "\n" + (QString)FALF_NAM + " " + (QString)FALF_VER
			<< "\n";
	}
	else
	{
		setCaption((QString)FALF_NAM);
		streamOut << QString::null;
	}

	QToolTip::add(systemTray, text);

	currentTrackFile.close();
}

void frmMain::fileDropped(QDropEvent * event, QListViewItem * after)
{
	Q_UNUSED(after);

	dropEvent(event);
}

void frmMain::scrollVol(QWheelEvent * e)
{
	if (e->delta() > 0)
		sldVol->addStep();
	else
		sldVol->subtractStep();
}

void frmMain::onMouseButtonReleased(const QPoint & point)
{
	pListIndex * p = collection.getPlaylist(tabPlaylist->currentPageIndex());

	if (!p)
		return;

	QListViewItem * i = lstPlaylist->itemAt(lstPlaylist->contentsToViewport(point));
	const int itemIndex = lstPlaylist->itemIndex(i);

	QListViewItems items = lstPlaylist->selectedItems();
	QListViewItemsIterator it(items);
	QListViewItem * item = 0;

	int where = itemIndex, sub = 0;

	while ((item = it.current()))
	{
		if (!item)
			break;

		int currentItemIndex = lstPlaylist->itemIndex(item);
		
		collection.move(p, (currentItemIndex - sub), where);
		randMove((currentItemIndex - sub), where);
		
		++it;

		if (where < currentItemIndex)
			where++;
		else
			sub++;
	}

	refreshList();
	lstPlaylist->clearSelection();

	if (sub)
		sub--;

	for (int c = 0 ; c != (int)items.count() ; c++)
		lstPlaylist->setSelected(lstPlaylist->itemAtIndex(itemIndex + c - sub), true);

	ensureVisible();
}

void frmMain::ensureVisible()
{
	QListViewItem * c = lstPlaylist->selectedItem();

	if (!c)
		return;

	lstPlaylist->ensureItemVisible(c);
	lstPlaylist->center(0, c->itemPos());
}

void frmMain::onStop()
{
	if (tabPlaylist->currentPageIndex() != playTab)
		tabPlaylist->setCurrentPage(playTab);
	else
		ensureVisible();

	player.termThread();
	player.wait();
	playTimer.stop();
	lfmTimer.stop();
	sldPos->setValue(0);
	sldPos->setEnabled(false);
	pshAction->setPixmap(pixPlay);
	systemTray->setPixmap(pixStop);
	popupMenu->changeItem(0, pixPlay, i18n("Play"));
	action = STOP;
	stopAfter = false;

	randList.remove();
	
	onOff();
	markCurrent(playIndex);
	setStatBarMsg();

	setPlayTimeMsg();

	setTip();
}

QString frmMain::browse(const QString & dir)
{
	QFileInfo info;
	QDir directory;
	QStringList files, dirs;
	QString results(QString::null);

	directory.setPath(dir);
	directory.setFilter(QDir::Files);
	directory.setNameFilter((filePattern + " " + filePattern.upper()));

	files = directory.entryList();

	QStringList::const_iterator i;

	for (i = files.begin() ; i != files.end() ; i++)
		results += ("file://" + dir + "/" + (*i));

	directory.setFilter(QDir::Dirs);
	directory.setNameFilter("*");

	dirs = directory.entryList();
	dirs.remove(".");
	dirs.remove("..");

	for (i = dirs.begin() ; i != dirs.end() ; i++)
		results += browse((dir + "/" + (*i)));

	return results;
}

void frmMain::store()
{
	const pListIndex * p = 0;
	const pList * l = 0;

	if (confFile.open(IO_WriteOnly))
	{
		QTextStream streamOut(&confFile);
		streamOut.setEncoding(QTextStream::Unicode);

		int emptyPlay = 0, emptyDest = 0;
		for (int i = 0 ; i < tabPlaylist->count() ; i++)
		{
			p = collection.getPlaylist(i);

			if (!p)
				continue;

			l = p->list;

			if (!l)
			{
				if (i < playTab)
					emptyPlay++;
				if (i < destTab)
					emptyDest++;
				continue;
			}

			streamOut << "<playlist>" << collection.getPlaylistName(p) << "</playlist>\n"
				<< "<descr>" << collection.getPlaylistDescr(p) << "</descr>\n" << "<begin>\n";

			while (l)
			{
				streamOut << "\t<file>" << l->filePath << "</file>\n";
				l = l->next;
			}

			streamOut << "<end>\n";
		}

		const eqProfiles & p = dlgEqualizer.getEqProfiles();

		streamOut << "<equalizer>\n";

		for (eqProfiles::const_iterator i = p.begin() ; i != p.end() ; i++)
			streamOut << "\t<profile>" << (*i).getProfileAsString() << "</profile>\n";

		streamOut << "</equalizer>\n";

		KConfig * conf = KGlobal::config();

		lstPlaylist->saveLayout(conf, "KMListView lstPlaylist");

		conf->setGroup("Collection");
		conf->writeEntry("Files", collection.getCount());

		conf->setGroup("Player");
		conf->writeEntry("Volume", sldVol->value());
		conf->writeEntry("Loop", loop);
		conf->writeEntry("Random", rnd);
		conf->writeEntry("DestTab", (destTab - emptyDest));
		conf->writeEntry("Tab", (playTab - emptyPlay));
		conf->writeEntry("Track", playIndex);
		conf->writeEntry("TimeDirection", playTimeDirection);
		conf->writeEntry("AudioCDDevice", audioCDDevice);
		conf->writeEntry("Frame", sldPos->value());
		conf->writeEntry("SoundSystem", dlgSoundSystem.cmbSoundSystem->currentItem());

		conf->setGroup("Geometry");
		conf->writeEntry("X", geometry().x());
		conf->writeEntry("Y", geometry().y());
		conf->writeEntry("W", geometry().width());
		conf->writeEntry("H", geometry().height());
	
		conf->setGroup("Last.fm");
		conf->writeEntry("Enabled", lfmOn);
		conf->writeEntry("User", lfmUser);
		conf->writeEntry("Password", lfmPassword);
		conf->writeEntry("Percent", lfmPercent);

		conf->setGroup("Equalizer");
		conf->writeEntry("Enabled", dlgEqualizer.chkEqualizer->isChecked());
		conf->writeEntry("Profile", dlgEqualizer.cmbProfiles->currentItem());
	
		conf->setGroup("Browser");
		conf->writeEntry("Directory", browseDir);
	
		conf->setGroup("Lyrics");
		conf->writeEntry("Codec", dlgLyrics.cmbCodec->currentItem());
		conf->writeEntry("X", dlgLyrics.geometry().x());
		conf->writeEntry("Y", dlgLyrics.geometry().y());
		conf->writeEntry("W", dlgLyrics.geometry().width());
		conf->writeEntry("H", dlgLyrics.geometry().height());

		conf->setGroup("RemovableDevice");
		conf->writeEntry("Device", dlgRemovableDevice.txtDevice->text());
		conf->writeEntry("Destination", dlgRemovableDevice.txtDir->text());

		conf->setGroup("Accels");
		conf->writeEntry("tray_play", (QString)pshAction->accel());
		conf->writeEntry("tray_stop", (QString)pshStop->accel());
		conf->writeEntry("tray_left", (QString)pshLeft->accel());
		conf->writeEntry("tray_right", (QString)pshRight->accel());
		conf->writeEntry("wnd_newplaylist", (QString)pshNewPlaylist->accel());
		conf->writeEntry("wnd_up", (QString)pshUp->accel());
		conf->writeEntry("wnd_down", (QString)pshDown->accel());
		conf->writeEntry("wnd_remove", (QString)pshRemove->accel());
		conf->writeEntry("wnd_current", (QString)pshGoToCurrent->accel());
		conf->writeEntry("context_addfiles", (QString)contextMenu->accel(CONTEXT_ADDFILES));
		conf->writeEntry("context_addfolder", (QString)contextMenu->accel(CONTEXT_ADDFOLDER));
		conf->writeEntry("context_addcd", (QString)contextMenu->accel(CONTEXT_ADDCD));
		conf->writeEntry("context_lastfm", (QString)subMenuSettings->accel(CONTEXT_LASTFM));
		conf->writeEntry("context_equalizer", (QString)subMenuSettings->accel(CONTEXT_EQUALIZER));
		conf->writeEntry("context_sound", (QString)subMenuSettings->accel(CONTEXT_SOUND));
		conf->writeEntry("context_accels", (QString)subMenuSettings->accel(CONTEXT_ACCELS));
		conf->writeEntry("context_save", (QString)playlistMenu->accel(CONTEXT_SAVE));
		conf->writeEntry("context_rmboth", (QString)subMenuRemove->accel(CONTEXT_RMBOTH));
		conf->writeEntry("context_shuffle", (QString)playlistMenu->accel(CONTEXT_SHUFFLE));
		conf->writeEntry("context_tags", (QString)trackMenu->accel(CONTEXT_TAGS));
		conf->writeEntry("context_lyrics", (QString)trackMenu->accel(CONTEXT_LYRICS));
		conf->writeEntry("context_duplicate", (QString)playlistMenu->accel(CONTEXT_DUPLICATE));
		conf->writeEntry("context_removable", (QString)trackMenu->accel(CONTEXT_REMOVABLE));
		conf->writeEntry("action_selectall", (QString)actionSelectAll->accel());
		conf->writeEntry("action_copyondest", (QString)actionCopyOnDest->accel());
		conf->writeEntry("action_sendondest", (QString)actionSendOnDest->accel());
		conf->writeEntry("action_appendtodest", (QString)actionAppendToDest->accel());
		conf->writeEntry("action_volhigher", (QString)actionMakeVolHigher->accel());
		conf->writeEntry("action_vollower", (QString)actionMakeVolLower->accel());
		conf->writeEntry("action_prevcurrent", (QString)actionMakePrevCurrent->accel());
		conf->writeEntry("action_nextcurrent", (QString)actionMakeNextCurrent->accel());
		conf->writeEntry("action_popupcopyondest", (QString)actionPopupCopyOnDest->accel());
		conf->writeEntry("action_popupsendondest", (QString)actionPopupSendOnDest->accel());
		conf->writeEntry("action_popupappendto", (QString)actionPopupAppendTo->accel());
		conf->writeEntry("action_quit", (QString)actionQuit->accel());
	}

	confFile.close();
}

void frmMain::addFiles()
{
	QStringList files = KFileDialog::getOpenFileNames(browseDir,
		(filePattern + "|" + i18n("Audio files")), this, i18n("Add file(s)"));
	QStringList::const_iterator i = files.begin();

	if (i == files.end())
		return;

	QFileInfo fileInfo((*i));
	browseDir = fileInfo.dirPath(true);

	QString results(QString::null);

	for (; i != files.end() ; i++)
		results += ("file://" + (*i));

	if (!results.isEmpty())
		parse(results);
}

void frmMain::addFolders()
{
	KDirSelectDialog dlg(browseDir, true, this, 0, true);
	dlg.setCaption(i18n("Add folder"));
	
	if (dlg.exec() == QDialog::Rejected)
		return;

	QString dirName = dlg.url().path();
	QDir dir(dirName);

	if (!dirName.isEmpty())
		parse(dirName, true);

	if (!dir.cdUp())
		return;

	browseDir = dir.absPath();
}

void frmMain::saveM3u()
{
	if (!lstPlaylist->childCount())
		return;

	QString file = KFileDialog::getSaveFileName(browseDir, "*.m3u", this, i18n("Save playlist"));
	
	if (file.isEmpty())
		return;

	QFileInfo fileInfo(file);
	browseDir = fileInfo.dirPath(true);

	QFile fout(file);
	fout.open(IO_WriteOnly);

	QTextStream out(&fout);

	const pListIndex * p = collection.getPlaylist(tabPlaylist->currentPageIndex());

	if (!p)
		return;

	const pList * l = 0;
	unsigned int i = 0;

	while ((l = collection.get(p, i++)))
		out << l->filePath << "\n";

	fout.close();
}

void frmMain::openM3u(const QString & file)
{
	QFile fin(file);
	fin.open(IO_ReadOnly);

	QTextStream in(&fin);

	QString files;
	QString streams;
	QString dir = file.left(file.findRev("/") + 1);

	while (!fin.atEnd())
	{
		QString tmp = in.readLine();
		int len = 0;

		for (unsigned int i = 0 ; i != tmp.length() ; i++)
		{
			if (tmp.at(i) != '#')
				len++;
			else
			{
				while ((i-- != 0))
				{
					if (tmp.at(i) == ' ')
						len--;
					else
						break;
				}

				break;
			}
		}

		if (len != 0)
		{
			if (tmp.left(7).lower() == "http://")
				streams += tmp;
			else
			{
				if ((tmp = tmp.left(len)).at(0) != '/')
					tmp = (dir + tmp);
			
				files += ("file://" + tmp);
			}
		}
	}

	fin.close();

	parse(files, true);
	parse(streams, true);
}

void frmMain::parse(const QString & data, bool ignoreM3u)
{
	int index = tabPlaylist->currentPageIndex();

	if (index < 0)
		return;

	pListIndex * p = collection.getPlaylist(index);

	if (!p)
		return;

	WAIT_CURSOR
	
	QStringList files;
	bool stream = false;

	if (data.find("http://", 0, false) < 0)
		files = QStringList::split("file://", data);
	else
	{
		stream = true;
		files = QStringList::split("http://", data.lower());
	}

	frmRestore dlgParse;
	dlgParse.setCaption(i18n("Adding files") + "...");
	dlgParse.prgRestore->setProgress(0);
	dlgParse.prgRestore->setTotalSteps(files.count());

	if (files.count() > 5)
		dlgParse.show();

	for (QStringList::const_iterator i = files.begin() ; i != files.end() ; i++)
	{
		QString file = (*i);
		file = file.stripWhiteSpace();
		file = KURL::decode_string(file);

		QString name = file.right(file.length() - file.findRev("/") - 1);

		QFileInfo info(file);
		
		if (stream)
		{
			file = "http://" + file;
			collection.add(p, file, file, 0);
		}
		else
		{
			if (info.isFile())
			{
				QString ext3 = file.right(4);
			
				if (filePattern.find(ext3, 0, false) != -1)
				{
					if (ext3.lower() != ".m3u")
					{
						exTags t = readTags(file);
						collection.add(p, name, file, &t.basic);
					}
					else
					{
						if (!ignoreM3u)
							openM3u(file);
					}
				}
			}
			else
			{
				if (info.isDir())
				{
					QString res = browse(file);
					parse(res, true);
				}
			}
		}

		dlgParse.prgRestore->advance(1);
	}

	ARROW_CURSOR

	onTabChanged();
	setTabTip();
}

void frmMain::onSldPosValChanged()
{
	if ((!player.running()) || (!sldPos->maxValue()))
		return;

	QImage tmpImage = pixPlay.convertToImage();
	QPixmap tmpPixmap, dstPixmap = pixPlay;

	KIconEffect::toGamma(tmpImage, 1.0);
	tmpPixmap.convertFromImage(tmpImage);

	float
		percent = ((sldPos->value() * 100.0) / sldPos->maxValue()),
		toCpy = (dstPixmap.height() - (dstPixmap.height() * (percent / 100.0)));

	copyBlt(&tmpPixmap, 0, (int)toCpy, &dstPixmap, 0, (int)toCpy);

	systemTray->setPixmap(tmpPixmap);
}

int frmMain::getRandom()
{
	int index, c = lstPlaylist->childCount(), add = 0, sub = 0;
	
	if (randList.getCount() == c)
		randList.remove();
	else
	{
		int i;

		for (i = 0 ; i != (c - 1) ; i++)
		{
			if (randList.searchForIndex(i))
				add++;
			else
				break;
		}

		for (i = (c - 1) ; i != 0 ; i--)
		{
			if (randList.searchForIndex(i))
				sub++;
			else
				break;
		}
	}

	do
		index = ((std::rand() % (c - sub - add)) + add);
	while (randList.searchForIndex(index));

	return index;
}

void frmMain::random()
{
	action = STOP;

	if ((loop == 0) && (randList.getCount() == lstPlaylist->childCount()))
	{
		onStop();
		lstPlaylist->setSelected(lstPlaylist->itemAtIndex(playIndex), true);

	}
	else
	{
		if (loop != 2)
			lstPlaylist->setSelected(lstPlaylist->itemAtIndex(getRandom()), true);
		else
			lstPlaylist->setSelected(lstPlaylist->currentItem(), true);
		
		onAction();
	}

	setStatBarMsg();
}

void frmMain::onRnd()
{
	if (rnd == 2)
		rnd = 0;
	else
		rnd++;

	switch (rnd)
	{
		case 0:
			pshRnd->setPixmap(pixRndOff);
			QToolTip::add(pshRnd, i18n("Play randomly"));
			randList.remove();
			setStatBarMsg();
		break;
		case 1:
			pshRnd->setPixmap(pixRndOn);
			QToolTip::add(pshRnd, i18n("Follow selected track"));

			if ((player.running()) && (!randList.searchForIndex(playIndex)))
				randList.addIndex(playIndex);
		break;
		case 2:
			pshRnd->setPixmap(pixFollow);
			QToolTip::add(pshRnd, i18n("Play normally"));
			randList.remove();
			setStatBarMsg();
		break;
	}
}

void frmMain::randMove(const int & current, const int & newCurrent)
{
	int add;

	if (current < newCurrent)
		add = 1;
	else
		add = -1;

	if (tabPlaylist->currentPageIndex() == playTab)
	{
		if (playIndex == newCurrent)
			playIndex = (newCurrent - add);
		else
		{
			if (playIndex == current)
				playIndex = newCurrent;
			else
			{
				if (((playIndex > current) && (playIndex < newCurrent)) ||
					((playIndex < current) && (playIndex > newCurrent)))
					playIndex -= add;
			}
		}
		
	}
	else
		return;

	if (randList.isEmpty())
		return;

	bool currPresent = randList.searchForIndex(current);

	if (currPresent)
		randList.deleteIndex(current);

	for (int c = (current + add) ; c != newCurrent ; c += add)
	{
		if (randList.searchForIndex(c))
		{
			randList.deleteIndex(c);
			randList.addIndex(c - add);
		}
	}

	if (!randList.searchForIndex(newCurrent))
	{
		if (currPresent)
		{
			if (randList.searchForIndex(newCurrent - add))
				randList.deleteIndex(newCurrent - add);

			randList.addIndex(newCurrent);
		}
	}
	else
	{
		if (!randList.searchForIndex(newCurrent - add))
			randList.addIndex(newCurrent - add);

		if (!currPresent)
			randList.deleteIndex(newCurrent);
	}
}

void frmMain::onOff()
{
	for (int i = 1 ; i != 4 ; i++)
		popupMenu->setItemEnabled(i, false);

	pshAction->setEnabled(true);
	popupMenu->setItemEnabled(0, true);
	pshStop->setEnabled(false);
	pshLeft->setEnabled(false);
	pshRight->setEnabled(false);
	lblTime->setEnabled(false);

	if (player.running())
	{
		pshStop->setEnabled(true);
		pshLeft->setEnabled(true);
		pshRight->setEnabled(true);
		lblTime->setEnabled(true);
		popupMenu->setItemEnabled(1, true);
		popupMenu->setItemEnabled(2, true);
		popupMenu->setItemEnabled(3, true);
		pshAction->setPixmap(pixPause);
		popupMenu->changeItem(0, pixPause, i18n("Pause"));
	}
	else
	{
		pshAction->setPixmap(pixPlay);
		popupMenu->changeItem(0, pixPlay, i18n("Play"));
		
		if (action == STOP)
			systemTray->setPixmap(pixStop);
	}
	
	bool enable = true;
	if (!lstPlaylist->childCount())
		enable = false;

	if ((!enable) && (action != PLAY))
	{
		pshAction->setEnabled(false);
		popupMenu->setItemEnabled(0, false);
	}
	
	if (action == PAUSE)
		return;

	pshUp->setEnabled(enable);
	pshDown->setEnabled(enable);
	pshRemove->setEnabled(enable);
	pshClearList->setEnabled(enable);

	if (tabPlaylist->count() != 1)
		pshDeletePlaylist->setEnabled(true);
	else
		pshDeletePlaylist->setEnabled(false);

	markCurrent();
}

void frmMain::lyricsDownloaded(bool e)
{
	QString text;

	if (!e)
	{
		if ((text = lyrics.getFromBuffer(buffer)).isEmpty())
			text = lyricsErrStr;
	}
	else
	{
		text = lyricsErrStr;
		delete buffer;
		buffer = 0;
	}
	
	QTextCodec * codec = QTextCodec::codecForName(dlgLyrics.cmbCodec->currentText());

	if (!codec)
		return;

	QString unicodeString = codec->toUnicode(text.latin1());

	dlgLyrics.txtLyrics->setText(unicodeString);
	dlgLyrics.txtLyrics->setContentsPos(0, 0);
}

void frmMain::onDelLyrics()
{
	onLyrics(true);
}

void frmMain::onCodecChanged(int i)
{
	Q_UNUSED(i);

	onLyrics();
}

void frmMain::onLyrics(bool force)
{
	const pListIndex * p = collection.getPlaylist(playTab);

	if (!p)
		return;

	const pList * item = collection.get(p, playIndex);

	if (!item)
		return;
	
	if (http)
	{
		http->disconnect();
		http->abort();
	}
	else
		http = new QHttp("www.autolyrics.com");
	
	QString title = item->fileTags.tagTitle;
	QString artist = item->fileTags.tagArtist;

	title.remove('/');
	artist.remove('/');

	if ((!title.isEmpty()) && (!artist.isEmpty()))
		dlgLyrics.setCaption(title + " - " + artist);
	else
		dlgLyrics.setCaption(QString::null);

	dlgLyrics.txtLyrics->setText(i18n("Looking for lyrics") + "...");

	if (!dlgLyrics.isVisible())
		dlgLyrics.show();

	if (title.isEmpty() || artist.isEmpty())
	{
		dlgLyrics.txtLyrics->setText(lyricsErrStr);
		return;
	}

	title.replace(" ", "_");
	artist.replace(" ", "_");

	lyrics.setTitle(title);
	lyrics.setArtist(artist);
	
	QString text;

	if ((!force) && !(text = lyrics.getFromLocal()).isEmpty())
	{
		QTextCodec * codec = QTextCodec::codecForName(dlgLyrics.cmbCodec->currentText());

		if (!codec)
			return;

		QString unicodeString = codec->toUnicode(text.latin1());

		dlgLyrics.txtLyrics->setText(unicodeString);
		dlgLyrics.txtLyrics->setContentsPos(0, 0);
		return;
	}

	buffer = new QBuffer;
	
	if (!buffer)
		return;

	connect(http, SIGNAL(done(bool)), this, SLOT(lyricsDownloaded(bool)));
	
	QString url = "/lyrics/for/" + title + "/by/" + artist;
	url = KURL::encode_string(url);

	http->get(url, buffer);
}

void frmMain::onKeyPressed(QKeyEvent * e)
{
	QListViewItem * item = lstPlaylist->selectedItem();

	int i = -1;

	if (item)
		i = lstPlaylist->itemIndex(item);

	switch (e->key())
	{
		case Qt::Key_Return:
		case Qt::Key_Enter:
			onPlaylist();
			return;
		case Qt::Key_Up:
			if (i > 0)
			{
				lstPlaylist->clearSelection();
				lstPlaylist->setSelected(lstPlaylist->itemAtIndex(i - 1), true);
				ensureVisible();
			}
			return;
		case Qt::Key_Down:
			if ((i != -1) && ((i + 1) < lstPlaylist->childCount()))
			{
				lstPlaylist->clearSelection();
				lstPlaylist->setSelected(lstPlaylist->itemAtIndex(i + 1), true);
				ensureVisible();
			}
			return;
		case Qt::Key_PageUp:
			lstPlaylist->clearSelection();
			if (i > 15)
				lstPlaylist->setSelected(lstPlaylist->itemAtIndex(i - 16), true);
			else
			{
				QListViewItem * i = lstPlaylist->firstChild();
				if (i)
					lstPlaylist->setSelected(i, true);
			}
			ensureVisible();
			return;
		case Qt::Key_PageDown:
			lstPlaylist->clearSelection();
			if ((i != -1) && ((i + 16) < lstPlaylist->childCount()))
				lstPlaylist->setSelected(lstPlaylist->itemAtIndex(i + 16), true);
			else
			{
				QListViewItem * i = lstPlaylist->lastItem();
				if (i)
					lstPlaylist->setSelected(i, true);
			}
			ensureVisible();
			return;
		case Qt::Key_Escape:
			onSearch(false);
			return;
		default:
			break;
	}

	if (e->text().isEmpty() || !lstPlaylist->childCount() ||
		(e->text().length() > 1) || (!QChar(e->text().at(0).isLetterOrNumber())))
		return;

	txtSearch->setFocus();
	txtSearch->setText("");
	txtSearch->setText(e->text());
	txtSearch->show();

	searchTimer.stop();
	searchTimer.start(1500);
}

void frmMain::onSearch(bool begin)
{
	static bool hidden = false;

	QListViewItemIterator it(lstPlaylist);
	QListViewItem * item = 0;

	if (begin)
	{
		if (txtSearch->text().isEmpty())
			return;

		const QString text = txtSearch->text();

		while ((item = it.current()))
		{
			if ((item->text(0).find(text, 0, false) == -1) &&
				(item->text(1).find(text, 0, false) == -1) &&
					(item->text(2).find(text, 0, false) == -1))
				item->setVisible(false);

			++it;
		}

		hidden = true;
	}
	else
	{
		if (!hidden)
			return;

		while ((item = it.current()))
		{
			item->setVisible(true);
			
			++it;
		}

		hidden = false;

		ensureVisible();
		setStatBarMsg();
	}
}

void frmMain::onSearchTimer()
{
	txtSearch->hide();
	searchTimer.stop();

	statBar->message(i18n("press Esc to cancel filter"));
}

void frmMain::onSearchTextChanged()
{
	searchTimer.stop();
	searchTimer.start(1500);

	onSearch(false);
	onSearch(true);
}

void frmMain::showContextMenu(const QPoint & p)
{
	contextMenu->exec(p);
}

void frmMain::openAudioCD()
{
	int index = tabPlaylist->currentPageIndex();

	if (index < 0)
		return;

	pListIndex * p = collection.getPlaylist(index);

	if (!p)
		return;

	if (audioCDDevice.isEmpty())
	{
		audioCDDevice = "/dev/";
		dlgAudioCDDevice.txtDevice->setFocus();
	}
	else
		dlgAudioCDDevice.pshCancel->setFocus();

	dlgAudioCDDevice.txtDevice->setText(audioCDDevice);
	dlgAudioCDDevice.exec();

	if (!dlgAudioCDDevice.ok)
		return;

	audioCDDevice = dlgAudioCDDevice.txtDevice->text();

	QStringList contents = player.getAudioCDContents(audioCDDevice);

	WAIT_CURSOR

	frmRestore dlgParseCD;
	dlgParseCD.setCaption(i18n("Adding audio CD contents") + "...");
	dlgParseCD.prgRestore->setProgress(0);
	dlgParseCD.prgRestore->setTotalSteps(contents.count());

	if (contents.count() > 5)
		dlgParseCD.show();

	int counter = 1;
	for (QStringList::const_iterator i = contents.begin() ; i != contents.end() ; i++, counter++)
	{
		exTags tags = readTags((*i));
		collection.add(p, i18n("Track") + " " + QString::number(counter), (*i), &tags.basic);
		dlgParseCD.prgRestore->advance(1);
	}

	ARROW_CURSOR

	onTabChanged();
}

void frmMain::clearStop(bool remove)
{
	if (!player.running())
		pshStop->setEnabled(false);
	
	const int c = tabPlaylist->currentPageIndex();

	if (playTab == c)
	{
		onStop();

		playIndex = 0;

		QListViewItem * i = lstPlaylist->firstChild();

		if (i)
			lstPlaylist->setSelected(i, true);
	}
	else
	{
		if (remove)
		{
			if (c < playTab)
				playTab--;

			if (c < destTab)
				destTab--;
			else
			{
				if (c == destTab)
					destTab = -1;
			}
		}
	}
}

void frmMain::copyOnPlaylist(int i)
{
	pListIndex * p = collection.getPlaylist(tabPlaylist->currentPageIndex());

	if (!p)
		return;

	static QTime lastCalled;

	if (lastCalled.restart() < 1000)
	{
		QMessageBox msg(i18n("Question"), i18n("Do You really wanna duplicate tracks?"), QMessageBox::Warning,
			QMessageBox::Yes, QMessageBox::No | QMessageBox::Default, QMessageBox::NoButton);

		if (msg.exec() != QMessageBox::Yes)
			return;
	}

	QListViewItems items = lstPlaylist->selectedItems();
	QListViewItemsIterator it(items);
	QListViewItem * item = 0;

	pListIndex * dest = collection.getPlaylist(i);

	if (!dest)
		return;

	while ((item = it.current()))
	{
		if (!item)
			break;

		int curSel = lstPlaylist->itemIndex(item);

		pList * l = collection.get(p, curSel);

		if (l)
			collection.add(dest, l->trackName, l->filePath, &l->fileTags);

		++it;
	}

	setTabTip();
	setStatBarMsg();
}

void frmMain::sendOnPlaylist(int i)
{
	copyOnPlaylist(i);
	onRemove(false);
}

void frmMain::copyPlaylist()
{
	pListIndex * p = CURRENT_PLAYLIST;
	pListIndex * np = 0;

	if ((!p) || (!p->count))
		return;

	static QTime lastCalled;

	if (lastCalled.restart() < 1000)
	{
		QMessageBox msg(i18n("Question"), i18n("Do You really wanna duplicate playlist?"), QMessageBox::Warning,
			QMessageBox::Yes, QMessageBox::No | QMessageBox::Default, QMessageBox::NoButton);

		if (msg.exec() != QMessageBox::Yes)
			return;
	}

	np = collection.newPlaylist(p->name + " [" + i18n("copy").upper() + "]", p->description);

	pList * l = 0;
	int i = 0;

	while ((l = collection.get(p, i++)))
		collection.add(np, l->trackName, l->filePath, &l->fileTags);

	refreshPlaylist(tabPlaylist->currentPageIndex());
}

void frmMain::appendPlaylist(int i)
{
	pListIndex * p = CURRENT_PLAYLIST;
	pListIndex * np = collection.getPlaylist(i);

	if ((!p) || (!np) || (!p->count))
		return;

	static QTime lastCalled;

	if (lastCalled.restart() < 1000)
	{
		QMessageBox msg(i18n("Question"), i18n("Do You really wanna append playlist?"), QMessageBox::Warning,
			QMessageBox::Yes, QMessageBox::No | QMessageBox::Default, QMessageBox::NoButton);

		if (msg.exec() != QMessageBox::Yes)
			return;
	}

	pList * l = 0;
	int j = 0;
	const int count = p->count;

	while ((j != count) && (l = collection.get(p, j++)))
		collection.add(np, l->trackName, l->filePath, &l->fileTags);

	setTabTip();
	setStatBarMsg();
}

void frmMain::restoreVolume()
{
	const int & vol = player.getVolume();

	if (vol < desiredVolume)
	{
		if ((vol + 5) > desiredVolume)
			sldVol->setValue(desiredVolume);
		else
			player.setVolume(vol + 5);
	}
	else
		volumeTimer.stop();
}

void frmMain::makeUpPlaylist(int id)
{
	const int & page = tabPlaylist->currentPageIndex();

	pListIndex * p = collection.getPlaylist(page);

	if (!p)
		return;

	WAIT_CURSOR

	int i = -1;

	if (page == playTab)
		i = playIndex;

	switch (id)
	{
		case 0:
			i = collection.rmDup(p, i);
		break;
		case 1:
			i = collection.rmDead(p, i);
		break;
		case CONTEXT_RMBOTH:
			makeUpPlaylist(1);
			makeUpPlaylist(0);
			return;
		break;
	}

	RandTree tmp_rand(randList);

	if (page == playTab)
	{
		playIndex = i;
		randList.remove();
	}
	else
		tmp_rand.remove();

	pList * next = 0;
	i = 0;

	while ((next = collection.get(p, i++)))
	{
		if ((!tmp_rand.isEmpty()) && (tmp_rand.searchForIndex(next->myId->previous)))
			randList.addIndex(next->myId->current);

		delete next->myId;
		next->myId = 0;
	}

	ARROW_CURSOR

	onTabChanged();
	setTabTip();
}

void frmMain::lastFmHandShake()
{
	qDebug("lastFmHandShake()");

	if (!lfmServer)
		lfmServer = new QHttp("post.audioscrobbler.com");
	else
	{
		lfmServer->disconnect();
		lfmServer->abort();
	}

	connect(lfmServer, SIGNAL(done(bool)), this, SLOT(lastFmResolved(bool)));

	delete lfmBuffer;
	lfmBuffer = new QBuffer;
		
	lfmServer->get(
		QString("/?hs=true&p=1.1&c=%1&v=%2&u=%3")
		.arg("fap")
		.arg("0.1")
		.arg(lfmUser),
		lfmBuffer);
}

void frmMain::lastFmResolved(bool e)
{
	Q_UNUSED(e);

	qDebug("lastFmResolved()");
	
	QString result((QString)lfmBuffer->buffer());
	
	delete lfmBuffer;
	lfmBuffer = 0;

	if (!result.startsWith("UPTODATE"))
		return;

	delete lfmAddr;
	delete lfmChallenge;
	
	lfmChallenge = new QString(result.section('\n', 1, 1));
	QString addr(result.section('\n', 2, 2));
	lfmAddr = new QString(addr.mid(7, addr.findRev(":") - 7));
}

void frmMain::lastFmQueue()
{
	qDebug("lastFmQueue()");

	if (!lfmOn)
		return;

	const pListIndex * p = collection.getPlaylist(playTab);

	if (!p)
		return;

	const pList * i = collection.get(p, playIndex);

	if (!i)
		return;

	const tags & itemTags = i->fileTags;

	if (itemTags.tagArtist.isEmpty() || itemTags.tagTitle.isEmpty() || itemTags.tagLength.isEmpty())
		return;

	lfmItem item(itemTags.tagArtist, itemTags.tagTitle, itemTags.tagAlbum,
		QString::number(itemTags.tagLengthSeconds),
		QDateTime::currentDateTime(Qt::UTC).toString("yyyy-MM-dd hh:mm:ss"));

	item.artist = KURL::encode_string(item.artist);
	item.title = KURL::encode_string(item.title);
	item.album = KURL::encode_string(item.album);

	if (queue.count() == 10)
		queue.remove(queue.begin());

	queue.append(item);
	
	if ((lfmServer) && (lfmServer->currentId()))
		return;

	if ((!lfmAddr) || (!lfmChallenge))
	{
		lastFmHandShake();
		return;
	}

	QString addr("/protocol_1.1"), request(QString::null);
	request = ("u=" + lfmUser + "&s=");
	request += KMD5(KMD5(lfmPassword).hexDigest() + lfmChallenge->utf8()).hexDigest();

	unsigned short int num = 0;
	for (lfmItemList::iterator it = queue.begin() ; it != queue.end() ; it++, num++)
	{
		QString n = QString::number(num);
		request += "&a[" + n + "]=";
		request += (*it).artist;
		request += "&t[" + n + "]=";
		request += (*it).title;
		request += "&b[" + n + "]=";
		request += (*it).album;
		request += "&m[" + n + "]=";
		request += "&l[" + n + "]=";
		request += (*it).length;
		request += "&i[" + n + "]=";
		request += (*it).date;
	}
	
	if (!lfmSubmit)
		lfmSubmit = new QHttp((*lfmAddr));
	else
	{
		lfmSubmit->disconnect();
		lfmSubmit->abort();
		lfmSubmit->setHost((*lfmAddr));
	}

	QHttpRequestHeader header("POST", addr);
	header.setValue("Host", (*lfmAddr));
	header.setContentType("application/x-www-form-urlencoded");

	connect(lfmSubmit, SIGNAL(done(bool)), this, SLOT(lastFmSubmited(bool)));
	
	delete lfmBuffer;
	lfmBuffer = new QBuffer;

	lfmSubmit->request(header, request.utf8(), lfmBuffer);

	qDebug("sizeof(lastFmQueue): " + QString::number(queue.count()));
}

void frmMain::lastFmSubmited(bool e)
{
	qDebug("lastFmSubmited()");

	QString data((QString)lfmBuffer->buffer());

	delete lfmBuffer;
	lfmBuffer = 0;

	if ((!e) && (data.startsWith("OK")))
	{
		statBar->message("last.fm: " + i18n("information submitted successfully"));
		queue.clear();
	}
	else
	{
		statBar->message("last.fm: " + i18n("information submission failed"));
		lastFmHandShake();
	}
}

void frmMain::editLfm()
{
	dlgLfm.ok = false;
	dlgLfm.txtUser->setText(lfmUser);
	dlgLfm.txtPassword->setText(lfmPassword);
	dlgLfm.chkLfm->setChecked(lfmOn);
	dlgLfm.sldPercent->setValue(lfmPercent);
	dlgLfm.txtUser->setFocus();
	dlgLfm.exec();
	
	if (dlgLfm.ok)
	{
		QString prevUser, prevPassword;
		int prevPercent;

		prevUser = lfmUser;
		prevPassword = lfmPassword;
		prevPercent = lfmPercent;

		lfmUser = dlgLfm.txtUser->text();
		lfmPassword = dlgLfm.txtPassword->text();
		lfmPercent = dlgLfm.sldPercent->value();
		
		if (!dlgLfm.chkLfm->isChecked())
		{
			lfmTimer.stop();
			queue.clear();

			if (lfmServer)
			{
				lfmServer->disconnect();
				lfmServer->abort();
				delete lfmServer;
				lfmServer = 0;
			}

			if (lfmSubmit)
			{
				lfmSubmit->disconnect();
				lfmSubmit->abort();
				delete lfmSubmit;
				lfmSubmit = 0;
			}
		}
		else
		{
			if ((lfmOn != dlgLfm.chkLfm->isChecked()) ||
				(prevUser != lfmUser) || (prevPassword != lfmPassword) || (prevPercent != lfmPercent))
				lastFmHandShake();
		}

		lfmOn = dlgLfm.chkLfm->isChecked();
	}
}

void frmMain::markCurrent(int old)
{
	if (tabPlaylist->currentPageIndex() != playTab)
		return;

	QListViewItem * item = 0;

	if (old < 0)
	{
		item = lstPlaylist->itemAtIndex(playIndex);

		if ((item) && (player.running()))
			item->setPixmap(0, pixPlay);
	}
	else
	{
		item = lstPlaylist->itemAtIndex(old);

		if (item)
			item->setPixmap(0, 0);
	}
}

void frmMain::sortPlaylist(int type)
{
	const int page = tabPlaylist->currentPageIndex();
	pListIndex * p = collection.getPlaylist(page);

	if (!p)
		return;

	QHeader * h = lstPlaylist->header();

	static int section = type;
	bool order = true;

	if (type != section)
		h->setSortIndicator(type, Qt::Descending);

	if (h->sortIndicatorOrder() == Qt::Ascending)
	{
		h->setSortIndicator(type, Qt::Descending);

		order = false;
	}
	else
		h->setSortIndicator(type, Qt::Ascending);

	section = type;

	WAIT_CURSOR

	collection.sort(p, type, order);

	pList * l = 0;
	int k = 0, item = 0;
	bool found = false;

	RandTree tmp_rand(randList);

	if (playTab == page)
		randList.remove();
	else
		tmp_rand.remove();

	while ((l = collection.get(p, k++)))
	{
		if (!l->myId)
			continue;

		if (playTab == page)
		{
			if ((!found) && (l->myId->previous == playIndex))
			{
				found = true;
				playIndex = l->myId->current;
				item = playIndex;
			}

			if ((!tmp_rand.isEmpty()) && (tmp_rand.searchForIndex(l->myId->previous)))
				randList.addIndex(l->myId->current);
		}

		delete l->myId;
		l->myId = 0;
	}

	ARROW_CURSOR

	refreshList();
	lstPlaylist->clearSelection();
	
	lstPlaylist->setSelected(lstPlaylist->firstChild(), true);
}

void frmMain::shufflePlaylist()
{
	const int page = tabPlaylist->currentPageIndex();
	pListIndex * p = collection.getPlaylist(page);

	if (!p)
		return;

	WAIT_CURSOR

	collection.shuffle(p);

	pList * l = 0;
	int k = 0, item = 0;
	bool found = false;

	RandTree tmp_rand(randList);

	if (playTab == page)
		randList.remove();
	else
		tmp_rand.remove();

	while ((l = collection.get(p, k++)))
	{
		if (!l->myId)
			continue;

		if (playTab == page)
		{
			if ((!found) && (l->myId->previous == playIndex))
			{
				found = true;
				playIndex = l->myId->current;
				item = playIndex;
			}

			if ((!tmp_rand.isEmpty()) && (tmp_rand.searchForIndex(l->myId->previous)))
				randList.addIndex(l->myId->current);
		}

		delete l->myId;
		l->myId = 0;
	}

	ARROW_CURSOR

	lstPlaylist->header()->setSortIndicator(-1, Qt::Descending);

	refreshList();
	lstPlaylist->clearSelection();
	
	lstPlaylist->setSelected(lstPlaylist->firstChild(), true);
}

void frmMain::onStopPressed()
{
	QTimer::singleShot(500, this, SLOT(onShowStopMenu()));
}

void frmMain::onShowStopMenu()
{
	if (pshStop->isDown())
	{
		stopMenu->exec(QCursor::pos());
		pshStop->setDown(false);
	}
}

void frmMain::onStopMenu(int i)
{
	if (i == 0)
		onStop();
	else
		stopAfter = true;
}

void frmMain::onShowPlaylistHeaderMenu()
{
	headerMenu->exec(QCursor::pos());
}

void frmMain::onPlaylistHeaderMenu()
{
	ints checked, unchecked;

	int i = 0;
	for (chks::const_iterator it = headerMenuItems.begin() ; it != headerMenuItems.end() ; it++, i++)
	{
		if ((*it)->isChecked())
			checked.append(i);
		else
			unchecked.append(i);
	}

	if (checked.count() == 0)
	{
		i = 0;
		for (chks::const_iterator it = headerMenuItems.begin() ; it != headerMenuItems.end() ; it++, i++)
			(*it)->setChecked((bool)lstPlaylist->columnWidth(i));
		return;
	}

	for (ints::const_iterator it = checked.begin() ; it != checked.end() ; it++)
	{
		if (lstPlaylist->columnWidth((*it)) == 0)
		{
			lstPlaylist->setColumnWidth((*it), 125);
			lstPlaylist->header()->setResizeEnabled(true, (*it));
		}
	}

	for (ints::const_iterator it = unchecked.begin() ; it != unchecked.end() ; it++)
	{
		if (lstPlaylist->columnWidth((*it)) > 0)
		{
			lstPlaylist->hideColumn((*it));
			lstPlaylist->header()->setResizeEnabled(false, (*it));
		}
	}
}

void frmMain::setStatBarMsg()
{
	const pListIndex * p = collection.getPlaylist(tabPlaylist->currentPageIndex());
	const length & total = collection.getTotalLength();

	QString len, totalLen;

	if (p)
	{
		len = QString("(%1:%2:%3)").arg(p->len.h, 2).arg(p->len.m, 2).arg(p->len.s, 2);
		len.replace(" ", "0");
		len = (" " + len);

		totalLen = QString("(%1:%2:%3)").arg(total.h, 2).arg(total.m, 2).arg(total.s, 2);
		totalLen.replace(" ", "0");
		totalLen = (" " + totalLen);
	}

	QString msg = (QString::number(lstPlaylist->childCount()) + " " +
		i18n("track(s) on currently selected playlist") + len);

	if ((randList.getCount() > 1) && (playTab == tabPlaylist->currentPageIndex()))
		msg += (" (" + QString::number((randList.getCount() - 1)) + " " + i18n("played randomly") + ")");

	msg += (", " + i18n("in total") + " " +
		QString::number(collection.getCount()) + " " + i18n("track(s) on") + " " +
		QString::number(tabPlaylist->count()) + " " + i18n("playlist(s)") + totalLen);

	statBar->message(msg);
}

void frmMain::addTmpFiles()
{
	if (!tmpFiles.exists())
		return;

	tmpFilesTimer.stop();

	tmpFiles.open(IO_ReadOnly);
	QTextStream tmp(&tmpFiles);
	tmp.setEncoding(QTextStream::Unicode);
		QString files = tmp.readLine();
	tmpFiles.close();
	tmpFiles.remove();

	if (files.isEmpty())
		return;

	QMessageBox msg(i18n("Question"), QString::number(files.contains("file://")) + " " + 
		i18n("file(s) to add") + ".\n" + i18n("Do You wanna create new playlist?"), QMessageBox::Warning,
		QMessageBox::Yes | QMessageBox::Default, QMessageBox::No, QMessageBox::NoButton);

	show();

	if (msg.exec() != QMessageBox::No)
	{
		collection.newPlaylist(
			QDate::currentDate(Qt::LocalTime).toString() + " " +
			QTime::currentTime(Qt::LocalTime).toString(),
			i18n("Double click to rename playlist"));
	
		refreshPlaylist(collection.getPlaylistCount() - 1);
	}
	
	parse(files);

	tmpFilesTimer.start(3000);
}

void frmMain::onSoundSystem()
{
	int index = dlgSoundSystem.cmbSoundSystem->currentItem();
	dlgSoundSystem.ok = false;
	dlgSoundSystem.cmbSoundSystem->setFocus();
	dlgSoundSystem.exec();

	if (dlgSoundSystem.ok)
	{
		if (index == dlgSoundSystem.cmbSoundSystem->currentItem())
			return;

		bool wasRunning = false;

		if (player.running())
		{
			onAction();
			wasRunning = true;
		}

		player.setSoundSystem(dlgSoundSystem.cmbSoundSystem->currentText().lower());

		if (wasRunning)
			onAction();

		lstPlaylist->setFocus();
	}
	else
		dlgSoundSystem.cmbSoundSystem->setCurrentItem(index);
}

void frmMain::onTestCanDecode(const QDragMoveEvent * e, bool & accept)
{
	accept = QTextDrag::canDecode(e);
}

void frmMain::onReceivedDropEvent(QWidget * w, QDropEvent * e)
{
	QString txt;
	QTextDrag::decode(e, txt);
	
	if ((!txt.isEmpty()) || (!e->source()))
		return;

	const int oldIndex = tabPlaylist->indexOf(e->source()),
		newIndex = tabPlaylist->indexOf(w);

	collection.movePlaylist(oldIndex, newIndex, playTab, destTab);

	refreshPlaylist(newIndex);
}

void frmMain::onInitiateDrag(QWidget * w)
{
	Q_UNUSED(w);

	QTextDrag * d = new QTextDrag(QString::null, tabPlaylist->currentPage());

	d->drag();
}

void frmMain::onTabDblClicked(QWidget * w)
{
	const int i = tabPlaylist->indexOf(w);

	pListIndex * p = collection.getPlaylist(i);

	if (!p)
		return;

	dlgPlaylist.ok = false;
	dlgPlaylist.txtPlaylist->setText(collection.getPlaylistName(p));
	dlgPlaylist.txtDescription->setText(collection.getPlaylistDescr(p));
	dlgPlaylist.txtPlaylist->setFocus();

	if (i == destTab)
		dlgPlaylist.chkDest->setChecked(true);
	else
		dlgPlaylist.chkDest->setChecked(false);

	dlgPlaylist.exec();

	if (dlgPlaylist.ok)
	{
		if (!dlgPlaylist.txtPlaylist->text().isEmpty())
			collection.setPlaylistName(p, dlgPlaylist.txtPlaylist->text());
		else
			collection.setPlaylistName(p, i18n("Unnamed Playlist"));

		collection.setPlaylistDescr(p, dlgPlaylist.txtDescription->text());

		if (dlgPlaylist.chkDest->isChecked())
			destTab = i;
		else
		{
			if (destTab == i)
				destTab = -1;
		}

		refreshPlaylist(tabPlaylist->currentPageIndex());
	}
}

void frmMain::onEqualizer()
{
	dlgEqualizer.show();
}

void frmMain::updateFreq(const int & amp, const int * freq)
{
	player.setEqualizer(amp, freq);
}

void frmMain::resizeEvent(QResizeEvent * e)
{
	int diffX = width() - baseSize().width();
	int diffY = height() - baseSize().height();

	if (diffX <= 0)
	{
		for (objsWithPos::iterator i = objsToMoveHor.begin() ; i != objsToMoveHor.end() ; i++)
		{
			QWidget * w = (QWidget *)(*i).getObject();
			w->move((*i).getX(), w->geometry().y());
		}

		sldPos->setGeometry(sldPos->x(), sldPos->y(),
			sldPos->baseSize().width(), sldPos->geometry().height());
		grpControl->setGeometry(grpControl->x(), grpControl->y(),
			grpControl->baseSize().width(), grpControl->geometry().height());
		lstPlaylist->setGeometry(lstPlaylist->x(), lstPlaylist->y(),
			lstPlaylist->baseSize().width(), lstPlaylist->geometry().height());
		tabPlaylist->setGeometry(tabPlaylist->x(), tabPlaylist->y(),
			tabPlaylist->baseSize().width(), tabPlaylist->geometry().height());
	}
	else
	{
		for (objsWithPos::iterator i = objsToMoveHor.begin() ; i != objsToMoveHor.end() ; i++)
		{
			QWidget * w = (QWidget *)(*i).getObject();
			w->move((*i).getX() + diffX, w->geometry().y());
		}

		sldPos->setGeometry(sldPos->geometry().x(), sldPos->geometry().y(),
			sldPos->baseSize().width() + diffX, sldPos->geometry().height());
		grpControl->setGeometry(grpControl->geometry().x(), grpControl->geometry().y(),
			grpControl->baseSize().width() + diffX, grpControl->geometry().height());
		lstPlaylist->setGeometry(lstPlaylist->geometry().x(), lstPlaylist->geometry().y(),
			lstPlaylist->baseSize().width() + diffX, lstPlaylist->geometry().height());
		tabPlaylist->setGeometry(tabPlaylist->geometry().x(), tabPlaylist->geometry().y(),
			tabPlaylist->baseSize().width() + diffX, tabPlaylist->geometry().height());
	}

	if (diffY <= 0)
	{
		for (objsWithPos::iterator i = objsToMoveVer.begin() ; i != objsToMoveVer.end() ; i++)
		{
			QWidget * w = (QWidget *)(*i).getObject();
			w->move(w->geometry().x(), (*i).getY());
		}

		lstPlaylist->setGeometry(lstPlaylist->geometry().x(), lstPlaylist->geometry().y(),
			lstPlaylist->geometry().width(), lstPlaylist->baseSize().height());
	}
	else
	{
		for (objsWithPos::iterator i = objsToMoveVer.begin() ; i != objsToMoveVer.end() ; i++)
		{
			QWidget * w = (QWidget *)(*i).getObject();
			w->move(w->geometry().x(), (*i).getY() + diffY);
		}

		lstPlaylist->setGeometry(lstPlaylist->geometry().x(), lstPlaylist->geometry().y(),
			lstPlaylist->geometry().width(), lstPlaylist->baseSize().height() + diffY);
	}

	QMainWindow::resizeEvent(e);
}

void frmMain::onRemovableDevice()
{
	if (!lstPlaylist->selectedItems().count())
		return;

	dlgRemovableDevice.txtDir->setFocus();
	dlgRemovableDevice.exec();

	if (!dlgRemovableDevice.ok)
		return;

	opCopyFiles.stop();

	dlgCopyFiles.prgRestore->setProgress(0);
	dlgCopyFiles.show();

	QProcess unmount_first;
	unmount_first.addArgument("umount");
	unmount_first.addArgument(dlgRemovableDevice.txtDevice->text());

	qDebug("Unmounting " + dlgRemovableDevice.txtDevice->text() + " (just in case)...");

	unmount_first.start();

	while (unmount_first.isRunning())
		usleep(100);

	QProcess mount;
	mount.addArgument("mount");
	mount.addArgument(dlgRemovableDevice.txtDevice->text());

	qDebug("Mounting " + dlgRemovableDevice.txtDevice->text() + "...");

	mount.start();

	while (mount.isRunning())
		usleep(100);

	if (!mount.readStderr().isEmpty())
	{
		dlgCopyFiles.hide();

		QMessageBox::warning(this, i18n("Mount error"),
			i18n("Could not mount removable device.") + "\n" +
			i18n("Device is not connected?") + "\n" +
			i18n("Check fstab settings."),
			QMessageBox::Ok, QMessageBox::NoButton, QMessageBox::NoButton);

		return;
	}

	QProcess ls_dev;
	ls_dev.addArgument("mount");

	ls_dev.start();

	while (ls_dev.isRunning());

	QString std_out = ls_dev.readStdout();

	int on_pos = std_out.find(" on ", (std_out.find(dlgRemovableDevice.txtDevice->text()) + 1));
	int start_pos = (on_pos + 4);
	int space_pos = std_out.find(' ', start_pos);

	QString mount_point = std_out.mid(start_pos, (space_pos - start_pos));
	QString dest_dir = mount_point + "/" + dlgRemovableDevice.txtDir->text();

	int slash_pos = 0;

	qDebug("Creating destination directory...");

	while ((slash_pos = dlgRemovableDevice.txtDir->text().find('/', slash_pos)) != -1)
		QDir().mkdir((mount_point + "/" + dlgRemovableDevice.txtDir->text().left(slash_pos++)));

	QDir().mkdir(dest_dir);

	if (!QDir(dest_dir).exists())
	{
		dlgCopyFiles.hide();

		QMessageBox::warning(this, i18n("Directory error"),
			i18n("Could not create destination directory.") + "\n" +
			i18n("Device is mounted read only?") + "\n" +
			i18n("Check fstab settings."),
			QMessageBox::Ok, QMessageBox::NoButton, QMessageBox::NoButton);
	}

	KDiskFreeSp * free_space = new KDiskFreeSp;

	connect(
		free_space,
		SIGNAL(foundMountPoint(const unsigned long &, const unsigned long &, const unsigned long &, const QString &)),
		this,
		SLOT(removableDeviceFoundMountPoint(const unsigned long &, const unsigned long &, const unsigned long &, const QString &)));
	connect(free_space, SIGNAL(done()), free_space, SLOT(deleteLater()));

	qDebug("Mount point is " + mount_point + ".");

	if (free_space->readDF(mount_point) != 1)
	{
		dlgCopyFiles.hide();

		QMessageBox::warning(this, i18n("Disk space error"),
			i18n("Could not obtain free space information."),
			QMessageBox::Ok, QMessageBox::NoButton, QMessageBox::NoButton);
	}
}

void frmMain::removableDeviceFoundMountPoint(
	const unsigned long & kBSize, const unsigned long & kBUsed, const unsigned long & kBAvail, const QString & pt)
{
	Q_UNUSED(kBSize);
	Q_UNUSED(kBUsed);

	QListViewItems items = lstPlaylist->selectedItems();
	QListViewItemsIterator it(items);
	QListViewItem * item = 0;

	const pListIndex * p = collection.getPlaylist(tabPlaylist->currentPageIndex());

	if (!p)
		return;

	QStringList files;
	unsigned long total_size = 0;

	qDebug("Adding files to copy...");

	while ((item = it.current()))
	{
		int i = lstPlaylist->itemIndex(item);

		const pList * track = collection.get(p, i);

		if (track)
		{
			QString lower = track->filePath.lower();

			if ((lower.left(7) != "http://") && (lower.left(6) != "cdda:/"))
			{
				files.append(track->filePath);
				total_size += (QFileInfo(track->filePath).size() / 1024);
			}
		}

		++it;
	}

	if (total_size > kBAvail)
	{
		dlgCopyFiles.hide();

		if (QMessageBox::information(this, i18n("Disk space"),
			i18n("Required disk space:") + " " + KIO::convertSizeFromKB(total_size) + "\n" +
			i18n("Available disk space:") + " " + KIO::convertSizeFromKB(kBAvail) + "\n" +
			i18n("Missing disk space:") + " " + KIO::convertSizeFromKB((total_size - kBAvail)) + "\n\n" +
			i18n("Not all files will be copied."),
			QMessageBox::Ok, QMessageBox::Cancel | QMessageBox::Default, QMessageBox::NoButton) !=
			QMessageBox::Ok)
			files.clear();
		else
			dlgCopyFiles.show();
	}

	files.append("");

	dlgCopyFiles.prgRestore->setTotalSteps((files.count() - 1));

	qDebug("Copying started...");

	WAIT_CURSOR

	opCopyFiles.copy(files, (pt + "/" + dlgRemovableDevice.txtDir->text()));
}

void frmMain::removableDeviceStartedNextCopy(const QPtrList<QNetworkOperation> & lst)
{
	Q_UNUSED(lst);

	static int done = 0;

	qDebug(QString::number(done) + " files from " + QString::number(dlgCopyFiles.prgRestore->totalSteps()) + " copied.");

	dlgCopyFiles.prgRestore->setProgress(done);

	if (done == dlgCopyFiles.prgRestore->totalSteps())
	{
		opCopyFiles.stop();
		
		QProcess unmount_end;
		unmount_end.addArgument("umount");
		unmount_end.addArgument(dlgRemovableDevice.txtDevice->text());

		qDebug("Unmounting " + dlgRemovableDevice.txtDevice->text() + "...");

		unmount_end.start();

		while (unmount_end.isRunning())
			usleep(100);

		dlgCopyFiles.hide();

		ARROW_CURSOR

		if (!unmount_end.readStderr().isEmpty())
		{
			QMessageBox::warning(this, i18n("Unmount error"),
				i18n("Could not unmount removable device.") + "\n" +
				i18n("Device is not connected?"),
				QMessageBox::Ok, QMessageBox::NoButton, QMessageBox::NoButton);
		}

		qDebug("Files have been copied.");

		done = 0;
	}
	else
		done++;
}

void frmMain::hideEvent(QHideEvent * e)
{
	allowRefresh = false;

	QMainWindow::hideEvent(e);
}

void frmMain::setPlayTimeMsg(bool reload)
{
	static int len = -1;

	QString time("00:00:00");

	length playTime;
	playTime += (player.getPos() / 1000);

	if ((reload) || (len < 0))
	{
		const pListIndex * p = collection.getPlaylist(playTab);

		if (!p)
			goto set_time;

		const pList * l = collection.get(p, playIndex);

		if (!l)
			goto set_time;

		len = l->fileTags.tagLengthSeconds;
	}

	if ((playTimeDirection) || ((!player.running()) && (action == STOP)))
		time = QString("%1:%2:%3").arg(playTime.h, 2).arg(playTime.m, 2).arg(playTime.s, 2).replace(" ", "0");
	else
	{
		length diff;

		diff = len;
		diff -= playTime;

		time = QString("-%1:%2:%3").arg(diff.h, 2).arg(diff.m, 2).arg(diff.s, 2).replace(" ", "0");
	}

set_time:
	lblTime->setText(time);
}

void frmMain::onDragging()
{
	onSearch(false);
}

void frmMain::onTimeClicked()
{
	playTimeDirection = (!playTimeDirection);

	setPlayTimeMsg();
}

void frmMain::copyOnDest()
{
	if (destTab >= 0)
	{
		copyOnPlaylist(destTab);

		if (tabPlaylist->currentPageIndex() == destTab)
			refreshList();
	}
}

void frmMain::sendOnDest()
{
	if ((destTab >= 0) && (tabPlaylist->currentPageIndex() != destTab))
		sendOnPlaylist(destTab);
}

void frmMain::appendToDest()
{
	if (destTab >= 0)
	{
		appendPlaylist(destTab);

		if (tabPlaylist->currentPageIndex() == destTab)
			refreshList();
	}
}

void frmMain::onAccels()
{
	dlgAccels.ok = false;

	dlgAccels.item(0)->setAccel(pshAction->accel());
	dlgAccels.item(1)->setAccel(pshStop->accel());
	dlgAccels.item(2)->setAccel(pshLeft->accel());
	dlgAccels.item(3)->setAccel(pshRight->accel());
	dlgAccels.item(4)->setAccel(pshNewPlaylist->accel());
	dlgAccels.item(5)->setAccel(pshUp->accel());
	dlgAccels.item(6)->setAccel(pshDown->accel());
	dlgAccels.item(7)->setAccel(pshRemove->accel());
	dlgAccels.item(8)->setAccel(pshGoToCurrent->accel());
	dlgAccels.item(9)->setAccel(contextMenu->accel(CONTEXT_ADDFILES));
	dlgAccels.item(10)->setAccel(contextMenu->accel(CONTEXT_ADDFOLDER));
	dlgAccels.item(11)->setAccel(contextMenu->accel(CONTEXT_ADDCD));
	dlgAccels.item(12)->setAccel(subMenuSettings->accel(CONTEXT_LASTFM));
	dlgAccels.item(13)->setAccel(subMenuSettings->accel(CONTEXT_EQUALIZER));
	dlgAccels.item(14)->setAccel(subMenuSettings->accel(CONTEXT_SOUND));
	dlgAccels.item(15)->setAccel(subMenuSettings->accel(CONTEXT_ACCELS));
	dlgAccels.item(16)->setAccel(playlistMenu->accel(CONTEXT_SAVE));
	dlgAccels.item(17)->setAccel(subMenuRemove->accel(CONTEXT_RMBOTH));
	dlgAccels.item(18)->setAccel(playlistMenu->accel(CONTEXT_SHUFFLE));
	dlgAccels.item(19)->setAccel(trackMenu->accel(CONTEXT_TAGS));
	dlgAccels.item(20)->setAccel(trackMenu->accel(CONTEXT_LYRICS));
	dlgAccels.item(21)->setAccel(playlistMenu->accel(CONTEXT_DUPLICATE));
	dlgAccels.item(22)->setAccel(trackMenu->accel(CONTEXT_REMOVABLE));
	dlgAccels.item(23)->setAccel(actionSelectAll->accel());
	dlgAccels.item(24)->setAccel(actionCopyOnDest->accel());
	dlgAccels.item(25)->setAccel(actionSendOnDest->accel());
	dlgAccels.item(26)->setAccel(actionAppendToDest->accel());
	dlgAccels.item(27)->setAccel(actionMakeVolHigher->accel());
	dlgAccels.item(28)->setAccel(actionMakeVolLower->accel());
	dlgAccels.item(29)->setAccel(actionMakePrevCurrent->accel());
	dlgAccels.item(30)->setAccel(actionMakeNextCurrent->accel());
	dlgAccels.item(31)->setAccel(actionPopupCopyOnDest->accel());
	dlgAccels.item(32)->setAccel(actionPopupSendOnDest->accel());
	dlgAccels.item(33)->setAccel(actionPopupAppendTo->accel());
	dlgAccels.item(34)->setAccel(actionQuit->accel());

	dlgAccels.accelView()->clearSelection();
	dlgAccels.item(0)->setSelected(true);
	dlgAccels.accelView()->setCurrentItem(dlgAccels.item(0));
	dlgAccels.accelView()->setFocus();

	dlgAccels.exec();

	if (dlgAccels.ok)
	{
		pshAction->setAccel(dlgAccels.item(0)->accel());
		pshStop->setAccel(dlgAccels.item(1)->accel());
		pshLeft->setAccel(dlgAccels.item(2)->accel());
		pshRight->setAccel(dlgAccels.item(3)->accel());
		pshNewPlaylist->setAccel(dlgAccels.item(4)->accel());
		pshUp->setAccel(dlgAccels.item(5)->accel());
		pshDown->setAccel(dlgAccels.item(6)->accel());
		pshRemove->setAccel(dlgAccels.item(7)->accel());
		pshGoToCurrent->setAccel(dlgAccels.item(8)->accel());
		popupMenu->setAccel(dlgAccels.item(0)->accel(), TRAY_PLAY);
		popupMenu->setAccel(dlgAccels.item(1)->accel(), TRAY_STOP);
		popupMenu->setAccel(dlgAccels.item(2)->accel(), TRAY_LEFT);
		popupMenu->setAccel(dlgAccels.item(3)->accel(), TRAY_RIGHT);
		contextMenu->setAccel(dlgAccels.item(9)->accel(), CONTEXT_ADDFILES);
		contextMenu->setAccel(dlgAccels.item(10)->accel(), CONTEXT_ADDFOLDER);
		contextMenu->setAccel(dlgAccels.item(11)->accel(), CONTEXT_ADDCD);
		subMenuSettings->setAccel(dlgAccels.item(12)->accel(), CONTEXT_LASTFM);
		subMenuSettings->setAccel(dlgAccels.item(13)->accel(), CONTEXT_EQUALIZER);
		subMenuSettings->setAccel(dlgAccels.item(14)->accel(), CONTEXT_SOUND);
		subMenuSettings->setAccel(dlgAccels.item(15)->accel(), CONTEXT_ACCELS);
		playlistMenu->setAccel(dlgAccels.item(16)->accel(), CONTEXT_SAVE);
		subMenuRemove->setAccel(dlgAccels.item(17)->accel(), CONTEXT_RMBOTH);
		playlistMenu->setAccel(dlgAccels.item(18)->accel(), CONTEXT_SHUFFLE);
		trackMenu->setAccel(dlgAccels.item(19)->accel(), CONTEXT_TAGS);
		trackMenu->setAccel(dlgAccels.item(20)->accel(), CONTEXT_LYRICS);
		playlistMenu->setAccel(dlgAccels.item(21)->accel(), CONTEXT_DUPLICATE);
		trackMenu->setAccel(dlgAccels.item(22)->accel(), CONTEXT_REMOVABLE);
		actionSelectAll->setAccel(dlgAccels.item(23)->accel());
		actionCopyOnDest->setAccel(dlgAccels.item(24)->accel());
		actionSendOnDest->setAccel(dlgAccels.item(25)->accel());
		actionAppendToDest->setAccel(dlgAccels.item(26)->accel());
		actionMakeVolHigher->setAccel(dlgAccels.item(27)->accel());
		actionMakeVolLower->setAccel(dlgAccels.item(28)->accel());
		actionMakePrevCurrent->setAccel(dlgAccels.item(29)->accel());
		actionMakeNextCurrent->setAccel(dlgAccels.item(30)->accel());
		actionPopupCopyOnDest->setAccel(dlgAccels.item(31)->accel());
		actionPopupSendOnDest->setAccel(dlgAccels.item(32)->accel());
		actionPopupAppendTo->setAccel(dlgAccels.item(33)->accel());
		actionQuit->setAccel(dlgAccels.item(34)->accel());
	}
}

void frmMain::onMakePrevCurrent()
{
	tabPlaylist->setCurrentPage(tabPlaylist->currentPageIndex() - 1);
}

void frmMain::onMakeNextCurrent()
{
	tabPlaylist->setCurrentPage(tabPlaylist->currentPageIndex() + 1);
}

void frmMain::onPopupCopyOnDest()
{
	QPopupMenu popup;

	popup.insertItem(pixCopyOnPlaylist,
		i18n("Copy on playlist") + " " + 
			QString::number(lstPlaylist->selectedItems().count()) + " " + i18n("track(s)"), subMenuCopy);
	popup.exec(QCursor::pos());
}

void frmMain::onPopupSendOnDest()
{
	QPopupMenu popup;

	popup.insertItem(pixSendOnPlaylist,
		i18n("Send on playlist") + " " + 
			QString::number(lstPlaylist->selectedItems().count()) + " " + i18n("track(s)"), subMenuSend);
	popup.exec(QCursor::pos());
}

void frmMain::onPopupAppendTo()
{
	QPopupMenu popup;

	popup.insertItem(pixAppendToPlaylist, i18n("Append to playlist"), subMenuAppend);
	popup.exec(QCursor::pos());
}
