/***************************************************************************
                          pivottable.h
                             -------------------
    begin                : Sat May 22 2004
    copyright            : (C) 2004-2005 by Ace Jones
    email                : <ace.j@hotpop.com>
                           Thomas Baumgart <ipwizard@users.sourceforge.net>
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#ifndef PIVOTTABLE_H
#define PIVOTTABLE_H

// ----------------------------------------------------------------------------
// QT Includes
#include <qmap.h>
#include <qvaluelist.h>

// ----------------------------------------------------------------------------
// KDE Includes

// ----------------------------------------------------------------------------
// Project Includes
#include "../mymoney/mymoneyfile.h"
#include "../mymoney/mymoneyreport.h"
#include "reportaccount.h"

namespace reports {

class KReportChartView;

/**
  * Calculates a 'pivot table' of information about the transaction database.
  * Based on pivot tables in MS Excel, and implemented as 'Data Pilot' in
  * OpenOffice.Org Calc.
  *
  *              | Month,etc
  * -------------+------------
  * Expense Type | Sum(Value)
  *   Category   |
  *
  * This is a middle-layer class, between the UI and the engine.  The 
  * MyMoneyReport class holds only the CONFIGURATION parameters.  This 
  * class actually does the work of retrieving the data from the engine
  * and formatting it for the user.
  *
  * @author Ace Jones
  *
  * @short
**/
class PivotTable
{
public:
  /**
    * Create a Pivot table style report
    *
    * @param _config_f The configuration parameters for this report
    */
    PivotTable( const MyMoneyReport& _config_f );

  /**
    * Render the report to an HTML stream.
    *
    * @return QString HTML string representing the report
    */
    QString renderHTML( void ) const;
  /**
    * Render the report to a comma-separated-values stream.
    *
    * @return QString CSV string representing the report
    */
    QString renderCSV( void ) const;
    
  /**
    * Render the report to a graphical chart
    *
    * Currently, this is not implemented, but it's here as a stub for an
    * ambitious chart-writer.
    *
    * @param view The KReportChartView into which to draw the chart.
    */
    void drawChart( KReportChartView& view ) const;
    
  /**
    * Dump the report's HTML to a file
    *
    * @param file The filename to dump into
    */
    void dump( const QString& file ) const;

private:
  /**
    * The fundamental data construct of this class is a 'grid'.  It is organized as follows:
    *
    * A 'Row' is a row of money values, each column is a month.  The first month corresponds to
    * m_beginDate
    *
    * An 'Inner Group' contains a rows for each subordinate account within a single top-level
    * account.  It also contains a mapping from the account descriptor for the subordinate account
    * to its row data.  So if we have an Expense account called "Computers", with sub-accounts called
    * "Hardware", "Software", and "Peripherals", there will be one Inner Group for "Computers"
    * which contains three Rows.
    *
    * An 'Outer Group' contains Inner Row Groups for all the top-level accounts in a given
    * account class.  Account classes are Expense, Income, Asset, Liability.  In the case above,
    * the "Computers" Inner Group is contained within the "Expense" Outer Group.
    *
    * A 'Grid' is the set of all Outer Groups contained in this report.
    *
    */
    class TGridRow: public QValueList<MyMoneyMoney> { public: MyMoneyMoney m_total; };
    class TInnerGroup: public QMap<ReportAccount,TGridRow> { public: TGridRow m_total; };
    class TOuterGroup: public QMap<QString,TInnerGroup> { public: TGridRow m_total; };
    class TGrid: public QMap<QString,TOuterGroup> { public: TGridRow m_total; };

    TGrid m_grid;

    QStringList m_columnHeadings;
    unsigned m_numColumns;
    QDate m_beginDate;
    QDate m_endDate;
    
    MyMoneyReport m_config_f;

protected:
  /**
    * Creates a row in the grid if it doesn't already exist
    *
    * Downsteam assignment functions will assume that this row already
    * exists, so this function creates a row of the needed length populated
    * with zeros.
    *
    * @param outergroup The outer row group
    * @param row The row itself
    * @param recursive Whether to also recursively create rows for our parent accounts
    */
    void createRow( const QString& outergroup, const ReportAccount& row, bool recursive );

  /**
    * Assigns a value into the grid
    *
    * Adds the given value to the value which already exists at the specified grid position
    *
    * @param outergroup The outer row group
    * @param row The row itself
    * @param column The column
    * @param value The value to be added in
    */
    inline void assignCell( const QString& outergroup, const ReportAccount& row, unsigned column, MyMoneyMoney value );

  /**
    * Record the opening balances of all qualifying accounts into the grid.
    *
    * For accounts opened before the report period, places the balance into the '0' column.
    * For those opened during the report period, places the balance into the appropriate column
    * for the month when it was opened.
    */
    void calculateOpeningBalances( void );

  /**
    * Calculate the running sums.
    *
    * After calling this method, each cell of the report will contain the running sum of all
    * the cells in its row in this and earlier columns.
    *
    * For example, consider a row with these values:
    *   01 02 03 04 05 06 07 08 09 10
    *
    * After calling this function, the row will look like this:
    *   01 03 06 10 15 21 28 36 45 55
    */
    void calculateRunningSums( void );

  /**
    * Calculate the row and column totals
    *
    * This function will set the m_total members of all the TGrid objects.  Be sure the values are
    * all converted to the base currency first!!
    *
    */
    void calculateTotals( void );

  /**
    * Convert each value in the grid to the base currency
    *
    */
    void convertToBaseCurrency( void );

  /**
    * Convert each value in the grid to the account/category's deep currency
    *
    * See AccountDescriptor::deepCurrencyPrice() for a description of 'deep' currency
    *
    */
    void convertToDeepCurrency( void );

  /**
    * Turn month-long columns into larger time periods if needed
    *
    * For example, consider a row with these values:
    *   01 02 03 04 05 06 07 08 09 10
    *
    * If the column pitch is 3 (i.e. quarterly), after calling this function,
    * the row will look like this:
    *   06 15 26 10
    */
    void collapseColumns(void);
    
  /**
    * Determine the proper column headings based on the time periods covered by each column
    *
    */
    void calculateColumnHeadings(void);
    
  /**
    * Helper methods for collapseColumns
    *
    */
    void accumulateColumn(unsigned destcolumn, unsigned sourcecolumn);
    void clearColumn(unsigned column);  
};

}
#endif
// PIVOTTABLE_H
