/*
 * Copyright 2004-2006 Luc Verhaegen.
 * Copyright 2004      The Unichrome Project  [unichrome.sf.net]
 * Copyright 1998-2003 VIA Technologies, Inc. All Rights Reserved.
 * Copyright 2001-2003 S3 Graphics, Inc. All Rights Reserved.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sub license,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice (including the
 * next paragraph) shall be included in all copies or substantial portions
 * of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NON-INFRINGEMENT. IN NO EVENT SHALL
 * THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
 * DEALINGS IN THE SOFTWARE.
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include "via_driver.h"

#ifndef _XF86_ANSIC_H
#include <string.h>
#endif

#include "via_vgahw.h"
#include "via_id.h"
#include "via_mode.h"

#include "via_vt162x.h"

/*
 *
 */
enum VT162xDevices {
    VT1621,
    VT1622,
    VT1623
};

/*
 * Output->Private
 */
struct VT162xOutputPrivate {
    int Device;

#define VT162X_REGSSIZE 0x68
    CARD8 *Regs; /* I2C Registers */
    int  RegsSize; /* Number of registers */

    /* Options */
    int   Output;
    int   Standard;
    Bool  DotCrawl;
    int   Deflicker;
};

/*
 *
 */
static void
VT162xPrivateDestroy(struct ViaOutput *Output)
{
    struct VT162xOutputPrivate *Private = Output->Private;

    xfree(Private->Regs);
    xfree(Private);

    Output->PrivateDestroy = NULL;
}

/*
 *
 */
static struct VT162xOutputPrivate *
VT162xPrivateCreate(struct ViaOutput *Output)
{
    struct VT162xOutputPrivate *Private;

    VIAFUNC(Output->scrnIndex);

    Output->PrivSize = sizeof(struct VT162xOutputPrivate);
    Output->Private =  xnfcalloc(1, Output->PrivSize);
    Private = Output->Private;

    Private->RegsSize = VT162X_REGSSIZE;
    Private->Regs = xnfcalloc(VT162X_REGSSIZE, sizeof(CARD32));

    Output->PrivateDestroy = VT162xPrivateDestroy;

    return Output->Private;
}

/*
 *
 * Option handling.
 *
 */

/* For Output->Output */
#define TVOUTPUT_NONE       0x00
#define TVOUTPUT_COMPOSITE  0x01
#define TVOUTPUT_SVIDEO     0x02
#define TVOUTPUT_RGB        0x04
#define TVOUTPUT_YCBCR      0x08
#define TVOUTPUT_SC         0x16

enum VT162xOpts {
    OPTION_TVDEFLICKER,
    OPTION_TVDOTCRAWL,
    OPTION_TVOUTPUT,
    OPTION_TVSTANDARD
};

static OptionInfoRec VT162xOptions[] =
{
    {OPTION_TVDEFLICKER, "TVDeflicker", OPTV_INTEGER, {0}, FALSE},
    {OPTION_TVDOTCRAWL,  "TVDotCrawl",  OPTV_BOOLEAN, {0}, FALSE},
    {OPTION_TVOUTPUT,    "TVOutput",    OPTV_ANYSTR,  {0}, FALSE},
    {OPTION_TVSTANDARD,  "TVStandard",  OPTV_ANYSTR,  {0}, FALSE},
    {-1,                 NULL,          OPTV_NONE,    {0}, FALSE}
};

/*
 *
 */
static OptionInfoPtr
VT162xGetOptions(ScrnInfoPtr pScrn, struct VT162xOutputPrivate *Private)
{
    OptionInfoPtr  Options;
    char  *s = NULL;

    Options = xnfalloc(sizeof(VT162xOptions));
    memcpy(Options, VT162xOptions, sizeof(VT162xOptions));

    xf86ProcessOptions(pScrn->scrnIndex, pScrn->options, Options);

    /* TV Deflicker */
    Private->Deflicker = 0;
    if (xf86GetOptValInteger(Options, OPTION_TVDEFLICKER, &Private->Deflicker))
        xf86DrvMsg(pScrn->scrnIndex, X_CONFIG, "Option: TVDeflicker %d\n",
                   Private->Deflicker );

    /* TV DotCrawl Enable Option */
    if (xf86ReturnOptValBool(Options, OPTION_TVDOTCRAWL, FALSE)) {
        Private->DotCrawl = TRUE;
        xf86DrvMsg(pScrn->scrnIndex, X_CONFIG, "DotCrawl is Enabled\n");
    } else
        Private->DotCrawl = FALSE;

    /* TV output combination */
    Private->Output = TVOUTPUT_NONE;
    if ((s = xf86GetOptValString(Options, OPTION_TVOUTPUT))) {
        if (!xf86NameCmp(s, "S-Video")) {
            Private->Output = TVOUTPUT_SVIDEO;
            xf86DrvMsg(pScrn->scrnIndex, X_CONFIG, "TV Output Signal is S-Video\n");
        } else if(!xf86NameCmp(s, "Composite")) {
            Private->Output = TVOUTPUT_COMPOSITE;
            xf86DrvMsg(pScrn->scrnIndex, X_CONFIG, "TV Output Signal is Composite\n");
        } else if(!xf86NameCmp(s, "SC")) {
           Private->Output = TVOUTPUT_SC;
            xf86DrvMsg(pScrn->scrnIndex, X_CONFIG, "TV Output Signal is SC\n");
        } else if(!xf86NameCmp(s, "RGB")) {
            Private->Output = TVOUTPUT_RGB;
            xf86DrvMsg(pScrn->scrnIndex, X_CONFIG, "TV Output Signal is RGB\n");
        } else if(!xf86NameCmp(s, "YCbCr")) {
            Private->Output = TVOUTPUT_YCBCR;
            xf86DrvMsg(pScrn->scrnIndex, X_CONFIG, "TV Output Signal is YCbCr\n");
        }
    }

    /* TV standard */
    Private->Standard = VIAPTR(pScrn)->Scratch->TVStandard;
    if ((s = xf86GetOptValString(Options, OPTION_TVSTANDARD))) {
        if (!xf86NameCmp(s, "NTSC")) {
            Private->Standard = TVSTANDARD_NTSC;
            xf86DrvMsg(pScrn->scrnIndex, X_CONFIG, "TV Standard is NTSC\n");
        } else if(!xf86NameCmp(s, "PAL")) {
            Private->Standard = TVSTANDARD_PAL;
            xf86DrvMsg(pScrn->scrnIndex, X_CONFIG, "TV Standard is PAL\n");
        }
    }

    return Options;
}

/*
 *
 */
static void
VT162xPrintRegs(struct ViaOutput *Output, const char *function)
{
    CARD8 i, buf;
    
    ViaDebug(Output->scrnIndex, "%s: Printing registers for %s\n",
             function, Output->I2CDev->DevName);

    for (i = 0; i < 0x68; i++) {
	xf86I2CReadByte(Output->I2CDev, i, &buf);
	ViaDebug(Output->scrnIndex, "TV%02X: 0x%02X\n", i, buf);
    }

    ViaDebug(Output->scrnIndex, "End of TV registers.\n");
}

/*
 *
 */
static void
VT162xSave(struct ViaOutput *Output)
{
    struct VT162xOutputPrivate *Private = Output->Private;
    CARD8 buf = 0x00;

    VIAFUNC(Output->scrnIndex);

    xf86I2CWriteRead(Output->I2CDev, &buf,1, Private->Regs, Private->RegsSize);
}

/*
 *
 */
static void
VT162xRestore(struct ViaOutput *Output)
{
    struct VT162xOutputPrivate *Private = Output->Private;
    int i;

    VIAFUNC(Output->scrnIndex);

    for (i = 0; i < Private->RegsSize; i++)
	xf86I2CWriteByte(Output->I2CDev, i, Private->Regs[i]);
}

/*
 * the same for VT1621 as for VT1622/VT1622A/VT1623, result is different though
 * still needs testing on vt1621 of course.
 */
static CARD8
VT162xDACSenseI2C(I2CDevPtr pDev)
{
    CARD8  save, sense;

    xf86I2CReadByte(pDev, 0x0E, &save);
    xf86I2CWriteByte(pDev, 0x0E, 0x00);
    xf86I2CWriteByte(pDev, 0x0E, 0x80);
    xf86I2CWriteByte(pDev, 0x0E, 0x00);
    xf86I2CReadByte(pDev, 0x0F, &sense);
    xf86I2CWriteByte(pDev, 0x0E, save);
    
    return (sense & 0x0F);
}

/*
 * VT1621 only knows composite and s-video
 */
static Bool
VT1621DACSense(struct ViaOutput *Output)
{
    struct VT162xOutputPrivate *Private = Output->Private;
    CARD8  sense;
    
    VIAFUNC(Output->scrnIndex);
    
    /* Config already set this */
    if (Private->Output != TVOUTPUT_NONE)
        return TRUE;

    sense = VT162xDACSenseI2C(Output->I2CDev);
    switch (sense) {
    case 0x00:
	Private->Output = TVOUTPUT_SC;
	xf86DrvMsg(Output->scrnIndex, X_PROBED, "VT1621: S-Video & Composite connected.\n");
	return TRUE;
    case 0x01:
	Private->Output = TVOUTPUT_COMPOSITE;
	xf86DrvMsg(Output->scrnIndex, X_PROBED, "VT1621: Composite connected.\n");
	return TRUE;
    case 0x02:
	Private->Output = TVOUTPUT_SVIDEO;
	xf86DrvMsg(Output->scrnIndex, X_PROBED, "VT1621: S-Video connected.\n");
	return TRUE;
    case 0x03:
	Private->Output = TVOUTPUT_NONE;
	xf86DrvMsg(Output->scrnIndex, X_PROBED, "VT1621: Nothing connected.\n");
	return FALSE;
    default:
	Private->Output = TVOUTPUT_NONE;
	xf86DrvMsg(Output->scrnIndex, X_WARNING, "VT1621: Unknown cable combination: 0x0%2X.\n",
		   sense);
	return FALSE;
    }
}


/*
 * VT1622, VT1622A and VT1623 know composite, s-video, RGB and YCBCR
 */
static Bool
VT1622DACSense(struct ViaOutput *Output)
{
    struct VT162xOutputPrivate *Private = Output->Private;
    CARD8  sense;

    VIAFUNC(Output->scrnIndex);

    /* Config already set this */
    if (Private->Output != TVOUTPUT_NONE)
        return TRUE;

    sense = VT162xDACSenseI2C(Output->I2CDev);
    switch (sense) {
    case 0x00: /* DAC A,B,C,D */
	Private->Output = TVOUTPUT_RGB;
	xf86DrvMsg(Output->scrnIndex, X_PROBED, "VT162x: RGB connected.\n");
	return TRUE;
    case 0x01: /* DAC A,B,C */
	Private->Output = TVOUTPUT_SC;
	xf86DrvMsg(Output->scrnIndex, X_PROBED, "VT162x: S-Video & Composite connected.\n");
	return TRUE;
    case 0x07: /* DAC A */
	Private->Output = TVOUTPUT_COMPOSITE;
	xf86DrvMsg(Output->scrnIndex, X_PROBED, "VT162x: Composite connected.\n");
	return TRUE;
    case 0x08: /* DAC B,C,D */
	Private->Output = TVOUTPUT_YCBCR;
	xf86DrvMsg(Output->scrnIndex, X_PROBED, "VT162x: YcBcR connected.\n");
	return TRUE;
    case 0x09: /* DAC B,C */
	Private->Output = TVOUTPUT_SVIDEO;
	xf86DrvMsg(Output->scrnIndex, X_PROBED, "VT162x: S-Video connected.\n");
	return TRUE;
    case 0x0F:
	Private->Output = TVOUTPUT_NONE;
	xf86DrvMsg(Output->scrnIndex, X_PROBED, "VT162x: Nothing connected.\n");
	return FALSE;
    default:
	Private->Output = TVOUTPUT_NONE;
	xf86DrvMsg(Output->scrnIndex, X_WARNING, "VT162x: Unknown cable combination: 0x0%2X.\n",
		   sense);
	return FALSE;
    }
}

/*
 * Depends on Output Name, Modes and Private being initialised already.
 */
static MonPtr
VT1621Monitor(struct ViaOutput *Output)
{
    struct VT162xOutputPrivate *Private = Output->Private;
    MonPtr Monitor;

    VIAFUNC(Output->scrnIndex);

    Monitor = xnfcalloc(1, sizeof(MonRec));
    memset(Monitor, 0, sizeof(MonRec));

    Monitor->vendor = xnfstrdup("VIA");
    Monitor->model = xnfstrdup(Output->Name);

    if (Private->Standard == TVSTANDARD_NTSC) {
        Monitor->id = xnfstrdup("TV (NTSC)");

        Monitor->nHsync = 1;
        Monitor->hsync[0].lo = 31.4;
        Monitor->hsync[0].hi = 45.0;

        Monitor->nVrefresh = 1;
        Monitor->vrefresh[0].lo = 59.94;
        Monitor->vrefresh[0].hi = 59.94;

        ViaMonitorAddModetable(Monitor, VT1621ModesNTSC);
    } else {
        Monitor->id = xnfstrdup("TV (PAL)");

        Monitor->nHsync = 1;
        Monitor->hsync[0].lo = 25.0;
        Monitor->hsync[0].hi = 37.5;

        Monitor->nVrefresh = 1;
        Monitor->vrefresh[0].lo = 50.00;
        Monitor->vrefresh[0].hi = 50.00;

        ViaMonitorAddModetable(Monitor, VT1621ModesPAL);
    }

    Output->ModesExclusive = TRUE;

#ifdef MONREC_HAS_REDUCED
    /* Make sure that we are not allowing reduced blanking here. */
    Monitor->reducedblanking = FALSE;
#endif

    return Monitor;
}

/*
 *
 */
static CARD8
VT1621ModeIndex(struct ViaOutput *Output, DisplayModePtr mode)
{
    struct VT162xOutputPrivate *Private = Output->Private;
    int i;

    VIAFUNC(Output->scrnIndex);

    for (i = 0; VT1621Table[i].Width; i++) {
	if ((VT1621Table[i].Width == mode->CrtcHDisplay) &&
	    (VT1621Table[i].Height == mode->CrtcVDisplay) &&
	    (VT1621Table[i].Standard == Private->Standard) &&
	    !(strcmp(VT1621Table[i].name, mode->name)))
	    return i;
    }
    xf86DrvMsg(Output->scrnIndex, X_WARNING, "%s: Mode \"%s\" not found in "
               "Table\n", __func__, mode->name);
    return 0xFF;
}

/*
 *
 */
static ModeStatus
VT1621ModeValid(struct ViaOutput *Output, DisplayModePtr mode)
{
    struct VT162xOutputPrivate *Private = Output->Private;

    VIAFUNC(Output->scrnIndex);

    if ((mode->PrivSize != sizeof(struct VT162xModePrivate)) ||
	((mode->Private != (void *) &VT162xModePrivateNTSC) &&
	 (mode->Private != (void *) &VT162xModePrivatePAL))) {
	xf86DrvMsg(Output->scrnIndex, X_INFO, "Not a mode defined by the TV Encoder.\n");
	return MODE_BAD;
    }

    if ((Private->Standard == TVSTANDARD_NTSC) &&
	(mode->Private != (void *) &VT162xModePrivateNTSC)) {
	xf86DrvMsg(Output->scrnIndex, X_INFO, "TV standard is NTSC. This is a PAL mode.\n");
	return MODE_BAD;
    } else if ((Private->Standard == TVSTANDARD_PAL) &&
	       (mode->Private != (void *) &VT162xModePrivatePAL)) {
	xf86DrvMsg(Output->scrnIndex, X_INFO, "TV standard is PAL. This is a NTSC mode.\n");
	return MODE_BAD;
    }

    if (VT1621ModeIndex(Output, mode) != 0xFF)
	return MODE_OK;
    return MODE_BAD;
}

/*
 * Depends on Output Name, Modes and Private being initialised already.
 */
static MonPtr
VT1622Monitor(struct ViaOutput *Output)
{
    struct VT162xOutputPrivate *Private = Output->Private;
    MonPtr Monitor;

    VIAFUNC(Output->scrnIndex);

    Monitor = xnfcalloc(1, sizeof(MonRec));
    memset(Monitor, 0, sizeof(MonRec));

    Monitor->vendor = xnfstrdup("VIA");
    Monitor->model = xnfstrdup(Output->Name);

    if (Private->Standard == TVSTANDARD_NTSC) {
        Monitor->id = xnfstrdup("TV (NTSC)");

        Monitor->nHsync = 1;
        Monitor->hsync[0].lo = 27.5;
        Monitor->hsync[0].hi = 56.7;

        Monitor->nVrefresh = 1;
        Monitor->vrefresh[0].lo = 59.94;
        Monitor->vrefresh[0].hi = 59.94;

        ViaMonitorAddModetable(Monitor, VT1622ModesNTSC);
    } else {
        Monitor->id = xnfstrdup("TV (PAL)");

        Monitor->nHsync = 1;
        Monitor->hsync[0].lo = 25.0;
        Monitor->hsync[0].hi = 47.5;

        Monitor->nVrefresh = 1;
        Monitor->vrefresh[0].lo = 50.00;
        Monitor->vrefresh[0].hi = 50.00;

        ViaMonitorAddModetable(Monitor, VT1622ModesPAL);
    }

    Output->ModesExclusive = TRUE;

#ifdef MONREC_HAS_REDUCED
    /* Make sure that we are not allowing reduced blanking here. */
    Monitor->reducedblanking = FALSE;
#endif

    return Monitor;
}

/*
 * Depends on Output Name, Modes and Private being initialised already.
 */
static MonPtr
VT1623Monitor(struct ViaOutput *Output)
{
    struct VT162xOutputPrivate *Private = Output->Private;
    MonPtr Monitor;

    VIAFUNC(Output->scrnIndex);

    Monitor = xnfcalloc(1, sizeof(MonRec));
    memset(Monitor, 0, sizeof(MonRec));

    Monitor->vendor = xnfstrdup("VIA");
    Monitor->model = xnfstrdup(Output->Name);

    if (Private->Standard == TVSTANDARD_NTSC) {
        Monitor->id = xnfstrdup("TV (NTSC)");

        Monitor->nHsync = 1;
        Monitor->hsync[0].lo = 31.4;
        Monitor->hsync[0].hi = 56.7;

        Monitor->nVrefresh = 1;
        Monitor->vrefresh[0].lo = 59.94;
        Monitor->vrefresh[0].hi = 59.94;

        ViaMonitorAddModetable(Monitor, VT1623ModesNTSC);
    } else {
        Monitor->id = xnfstrdup("TV (PAL)");

        Monitor->nHsync = 1;
        Monitor->hsync[0].lo = 25.0;
        Monitor->hsync[0].hi = 47.5;

        Monitor->nVrefresh = 1;
        Monitor->vrefresh[0].lo = 50.00;
        Monitor->vrefresh[0].hi = 50.00;

        ViaMonitorAddModetable(Monitor, VT1623ModesPAL);
    }

    Output->ModesExclusive = TRUE;

#ifdef MONREC_HAS_REDUCED
    /* Make sure that we are not allowing reduced blanking here. */
    Monitor->reducedblanking = FALSE;
#endif

    return Monitor;
}

/*
 *
 */
static CARD8
VT1622ModeIndex(struct ViaOutput *Output, DisplayModePtr mode)
{
    struct VT162xOutputPrivate *Private = Output->Private;
    struct VT162XTableRec *Table;
    int i;

    VIAFUNC(Output->scrnIndex);

    if (Private->Device == VT1622)
	Table = VT1622Table;
    else
	Table = VT1623Table;

    for (i = 0; Table[i].Width; i++) {
	if ((Table[i].Width == mode->CrtcHDisplay) &&
	    (Table[i].Height == mode->CrtcVDisplay) &&
	    (Table[i].Standard == Private->Standard) &&
	    !strcmp(Table[i].name, mode->name))
	    return i;
    }
    xf86DrvMsg(Output->scrnIndex, X_WARNING, "%s: Mode \"%s\" not found in "
               "Table\n", __func__, mode->name);
    return 0xFF;
}

/*
 *
 */
static ModeStatus
VT1622ModeValid(struct ViaOutput *Output, DisplayModePtr mode)
{
    struct VT162xOutputPrivate *Private = Output->Private;

    VIAFUNC(Output->scrnIndex);

    if ((mode->PrivSize != sizeof(struct VT162xModePrivate)) ||
	((mode->Private != (void *) &VT162xModePrivateNTSC) &&
	 (mode->Private != (void *) &VT162xModePrivatePAL))) {
	xf86DrvMsg(Output->scrnIndex, X_INFO, "Not a mode defined by the TV Encoder.\n");
	return MODE_BAD;
    }

    if ((Private->Standard == TVSTANDARD_NTSC) &&
	(mode->Private != (void *) &VT162xModePrivateNTSC)) {
	xf86DrvMsg(Output->scrnIndex, X_INFO, "TV standard is NTSC. This is a PAL mode.\n");
	return MODE_BAD;
    } else if ((Private->Standard == TVSTANDARD_PAL) &&
	       (mode->Private != (void *) &VT162xModePrivatePAL)) {
	xf86DrvMsg(Output->scrnIndex, X_INFO, "TV standard is PAL. This is a NTSC mode.\n");
	return MODE_BAD;
    }

    if (VT1622ModeIndex(Output, mode) != 0xFF)
	return MODE_OK;
    return MODE_BAD;
}

/*
 *
 */
static void
VT162xSetSubCarrier(I2CDevPtr pDev, CARD32 SubCarrier)
{
    xf86I2CWriteByte(pDev, 0x16, SubCarrier & 0xFF);
    xf86I2CWriteByte(pDev, 0x17, (SubCarrier >> 8) & 0xFF);
    xf86I2CWriteByte(pDev, 0x18, (SubCarrier >> 16) & 0xFF);
    xf86I2CWriteByte(pDev, 0x19, (SubCarrier >> 24) & 0xFF);
}

/*
 *
 */
static void
VT1621Mode(struct ViaOutput *Output, DisplayModePtr mode)
{
    struct VT162xOutputPrivate *Private = Output->Private;
    struct VT1621TableRec Table = VT1621Table[VT1621ModeIndex(Output, mode)];
    CARD8 i;

    VIAFUNC(Output->scrnIndex);

    for (i = 0; i < 0x16; i++)
	xf86I2CWriteByte(Output->I2CDev, i, Table.TV[i]);

    VT162xSetSubCarrier(Output->I2CDev, Table.SubCarrier);

    /* skip reserved (1A) and version id (1B). */

    xf86I2CWriteByte(Output->I2CDev, 0x1C, Table.TV[0x1C]);

    /* skip software reset (1D) */

    for (i = 0x1E; i < 0x24; i++)
	xf86I2CWriteByte(Output->I2CDev, i, Table.TV[i]);

    /* write some zeroes? */
    xf86I2CWriteByte(Output->I2CDev, 0x24, 0x00);
    for (i = 0; i < 0x08; i++)
	xf86I2CWriteByte(Output->I2CDev, 0x4A + i, 0x00);

    if (Private->Output == TVOUTPUT_COMPOSITE)
	for (i = 0; i < 0x10; i++)
	    xf86I2CWriteByte(Output->I2CDev, 0x52 + i, Table.TVC[i]);
    else
	for (i = 0; i < 0x10; i++)
	    xf86I2CWriteByte(Output->I2CDev, 0x52 + i, Table.TVS[i]);

    if (Private->DotCrawl) {
	if (Table.DotCrawlSubCarrier) {
	    xf86I2CReadByte(Output->I2CDev, 0x11, &i);
	    xf86I2CWriteByte(Output->I2CDev, 0x11, i | 0x08);
	    
	    VT162xSetSubCarrier(Output->I2CDev, Table.DotCrawlSubCarrier);
	} else
	    xf86DrvMsg(Output->scrnIndex, X_INFO, "This mode does not currently "
		       "support DotCrawl suppression.\n");  
    }
}

/*
 * also suited for VT1622A, VT1623
 */
static void
VT1622Mode(struct ViaOutput *Output, DisplayModePtr mode)
{
    struct VT162xOutputPrivate *Private = Output->Private;
    struct VT162XTableRec Table;
    CARD8 save, i;

    VIAFUNC(Output->scrnIndex);

    if (Private->Device == VT1622)
	Table = VT1622Table[VT1622ModeIndex(Output, mode)];
    else /* VT1622A/VT1623 */
	Table = VT1623Table[VT1622ModeIndex(Output, mode)];

    /* TV Reset */
    xf86I2CWriteByte(Output->I2CDev, 0x1D, 0x00);
    xf86I2CWriteByte(Output->I2CDev, 0x1D, 0x80);

    for (i = 0; i < 0x16; i++)
	xf86I2CWriteByte(Output->I2CDev, i, Table.TV1[i]);

    VT162xSetSubCarrier(Output->I2CDev, Table.SubCarrier);

    xf86I2CWriteByte(Output->I2CDev, 0x1A, Table.TV1[0x1A]);
    
    /* skip version id */

    xf86I2CWriteByte(Output->I2CDev, 0x1C, Table.TV1[0x1C]);

    /* skip software reset */

    for (i = 0x1E; i < 0x30; i++)
	xf86I2CWriteByte(Output->I2CDev, i, Table.TV1[i]);

    for (i = 0; i < 0x1B; i++)
	xf86I2CWriteByte(Output->I2CDev, 0x4A + i, Table.TV2[i]);

    if (Private->DotCrawl) {
	if (Table.DotCrawlSubCarrier) {
	    xf86I2CReadByte(Output->I2CDev, 0x11, &save);
	    xf86I2CWriteByte(Output->I2CDev, 0x11, save | 0x08);
	    
	    VT162xSetSubCarrier(Output->I2CDev, Table.DotCrawlSubCarrier);
	} else
	    xf86DrvMsg(Output->scrnIndex, X_INFO, "This mode does not currently "
		       "support DotCrawl suppression.\n");  
    }

    if (Private->Output == TVOUTPUT_RGB) {
	xf86I2CWriteByte(Output->I2CDev, 0x02, 0x2A);
	xf86I2CWriteByte(Output->I2CDev, 0x65, Table.RGB[0]);
	xf86I2CWriteByte(Output->I2CDev, 0x66, Table.RGB[1]);
	xf86I2CWriteByte(Output->I2CDev, 0x67, Table.RGB[2]);
	if (Table.RGB[3])
	    xf86I2CWriteByte(Output->I2CDev, 0x27, Table.RGB[3]);
	if (Table.RGB[4])
	    xf86I2CWriteByte(Output->I2CDev, 0x2B, Table.RGB[4]);
	if (Table.RGB[5])
	    xf86I2CWriteByte(Output->I2CDev, 0x2C, Table.RGB[5]);
    } else if (Private->Output == TVOUTPUT_YCBCR) {
	xf86I2CWriteByte(Output->I2CDev, 0x02, 0x03);
	xf86I2CWriteByte(Output->I2CDev, 0x65, Table.YCbCr[0]);
	xf86I2CWriteByte(Output->I2CDev, 0x66, Table.YCbCr[1]);
	xf86I2CWriteByte(Output->I2CDev, 0x67, Table.YCbCr[2]);
    }

    /* Configure flicker filter */
    xf86I2CReadByte(Output->I2CDev, 0x03, &save);
    save &= 0xFC;
    if (Private->Deflicker == 1)
	save |= 0x01;
    else if (Private->Deflicker == 2)
        save |= 0x02;
    xf86I2CWriteByte(Output->I2CDev, 0x03, save);
}

/*
 *
 */
static void
VT1621Power(struct ViaOutput *Output, Bool On)
{
    VIAFUNC(Output->scrnIndex);
    
    if (On)
	xf86I2CWriteByte(Output->I2CDev, 0x0E, 0x00);
    else
	xf86I2CWriteByte(Output->I2CDev, 0x0E, 0x03);
}

/*
 *
 */
static void
VT1622Power(struct ViaOutput *Output, Bool On)
{
    VIAFUNC(Output->scrnIndex);
    
    if (On)
	xf86I2CWriteByte(Output->I2CDev, 0x0E, 0x00);
    else
	xf86I2CWriteByte(Output->I2CDev, 0x0E, 0x0F);
}

/*
 *
 */
struct ViaOutput *
ViaVT162xInit(ScrnInfoPtr pScrn, I2CDevPtr pDev)
{
    struct ViaOutput *Output;
    struct VT162xOutputPrivate *Private;
    CARD8 buf;

    VIAFUNC(pScrn->scrnIndex);

    if (!xf86I2CReadByte(pDev, 0x1B, &buf)) {
        xf86DrvMsg(pScrn->scrnIndex, X_ERROR, "Unable to read from %s Slave %d.\n",
                   pDev->pI2CBus->BusName, pDev->SlaveAddr);
        return NULL;
    }

    Output = xnfcalloc(1, sizeof(struct ViaOutput));

    Output->Prev = NULL;
    Output->Next = NULL;
    Output->scrnIndex = pScrn->scrnIndex;
    Output->I2CDev = pDev;
    Output->Type = OUTPUT_TV;

    Private = VT162xPrivateCreate(Output);

    Output->Options = VT162xGetOptions(pScrn, Private);

    switch (buf) {
    case 0x02:
        xf86DrvMsg(pScrn->scrnIndex, X_PROBED, 
                   "Detected VIA Technologies VT1621 TV Encoder\n");
        pDev->DevName = "VT1621";
        Output->Name = "VT1621";
        Private->Device = VT1621;
        
        Output->ClockMaster = TRUE;
        Output->Monitor = VT1621Monitor(Output);

        Output->Save = VT162xSave;
        Output->Restore = VT162xRestore;
        Output->Sense = VT1621DACSense;
        Output->ModeValid = VT1621ModeValid;
        Output->Mode = VT1621Mode;
        Output->Power = VT1621Power;
        Output->PrintRegs = VT162xPrintRegs;

        return Output;
    case 0x03:
        xf86DrvMsg(pScrn->scrnIndex, X_PROBED, 
                   "Detected VIA Technologies VT1622 TV Encoder\n");
        pDev->DevName = "VT1622";
        Output->Name = "VT1622";
        Private->Device = VT1622;

        Output->ClockMaster = TRUE;
        Output->Monitor = VT1622Monitor(Output);

        Output->Save = VT162xSave;
        Output->Restore = VT162xRestore;
        Output->Sense = VT1622DACSense;
        Output->ModeValid = VT1622ModeValid;
        Output->Mode = VT1622Mode;
        Output->Power = VT1622Power;
        Output->PrintRegs = VT162xPrintRegs;

        return Output;
    case 0x10:
        xf86DrvMsg(pScrn->scrnIndex, X_PROBED, 
		   "Detected VIA Technologies VT1622A/VT1623 TV Encoder\n");
        pDev->DevName = "VT1623";
        Output->Name = "VT1623";
        Private->Device = VT1623;

        Output->ClockMaster = TRUE;
        Output->Monitor = VT1623Monitor(Output);

        Output->Save = VT162xSave;
        Output->Restore = VT162xRestore;
        Output->Sense = VT1622DACSense;
        Output->ModeValid = VT1622ModeValid;
        Output->Mode = VT1622Mode;
        Output->Power = VT1622Power;
        Output->PrintRegs = VT162xPrintRegs;

        return Output;
    case 0x50:
        xf86DrvMsg(pScrn->scrnIndex, X_PROBED, 
                   "Detected VIA Technologies VT1625 TV Encoder\n");
#if 0
        pDev->DevName = "VT1625";

        ...

        return Output;
#else 
        xf86DrvMsg(pScrn->scrnIndex, X_WARNING, "VT1625 is not supported yet.\n");
        break;
#endif
    default:
        xf86DrvMsg(pScrn->scrnIndex, X_WARNING, "Unknown TV Encoder found at %s %X.\n",
                   pDev->pI2CBus->BusName, pDev->SlaveAddr);
        break;
    }

    Output->PrivateDestroy(Output);

    xfree(Output->Options);
    xfree(Output);
    return NULL;
}
