/*
 * Copyright 2004-2006 Luc Verhaegen.
 * Copyright 2004-2005 The Unichrome Project  [unichrome.sf.net]
 * Copyright 1998-2003 VIA Technologies, Inc. All Rights Reserved.
 * Copyright 2001-2003 S3 Graphics, Inc. All Rights Reserved.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sub license,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice (including the
 * next paragraph) shall be included in all copies or substantial portions
 * of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NON-INFRINGEMENT. IN NO EVENT SHALL
 * THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
 * DEALINGS IN THE SOFTWARE.
 */

/*************************************************************************
 *
 *  File:       via_driver.c
 *  Content:    XFree86 4.0 for VIA/S3G UniChrome
 *
 ************************************************************************/

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include "xf86RAC.h"
#include "shadowfb.h"

#include "globals.h"
#define DPMS_SERVER
#include "X11/extensions/dpms.h"

#include "via_driver.h"
#include "../git_version.h"

#ifndef _XF86_ANSIC_H
#include <string.h>
#endif

#ifdef XF86DRI
#include "dri.h"
#endif
#include "via_vgahw.h"
#include "via_id.h"
#include "via_mode.h"

/*
 * prototypes
 */

static void VIAIdentify(int flags);
static Bool VIAProbe(DriverPtr drv, int flags);
static Bool VIAPreInit(ScrnInfoPtr pScrn, int flags);
static Bool VIAEnterVT(int scrnIndex, int flags);
static void VIALeaveVT(int scrnIndex, int flags);
static void VIASave(ScrnInfoPtr pScrn);
static void VIARestore(ScrnInfoPtr pScrn);
static void VIAWriteMode(ScrnInfoPtr pScrn, DisplayModePtr mode);
static Bool VIACloseScreen(int scrnIndex, ScreenPtr pScreen);
static Bool VIAScreenInit(int scrnIndex, ScreenPtr pScreen, int argc,
                          char **argv);
static int VIAInternalScreenInit(ScrnInfoPtr pScrn, ScreenPtr pScreen);
static void VIAFreeScreen(int scrnIndex, int flags);
static Bool VIASwitchMode(int scrnIndex, DisplayModePtr mode, int flags);
static void VIAAdjustFrame(int scrnIndex, int y, int x, int flags);
static void VIADPMS(ScrnInfoPtr pScrn, int mode, int flags);
static const OptionInfoRec * VIAAvailableOptions(int chipid, int busid);

static Bool VIAMapMMIO(ScrnInfoPtr pScrn);
static void VIAUnmapMem(ScrnInfoPtr pScrn);
static void ViaFBEnable(vgaHWPtr hwp);

#ifndef _X_EXPORT
#define _X_EXPORT
#endif

_X_EXPORT DriverRec VIA =
{
    VIA_VERSION,
    DRIVER_NAME,
    VIAIdentify,
    VIAProbe,
    VIAAvailableOptions,
    NULL,
    0
};


/* Supported chipsets */

static SymTabRec VIAChipsets[] = {
    {VT3122,  "VT3122"},
    {VT7205,  "VT7205"},
    {VT3108,  "VT3108"},
    {VT3118,  "VT3118"},
    {VT3344,  "VT3344"},
    {VT3157,  "VT3157"},
    {VT3230,  "VT3230"},
    {VT3343,  "VT3343"},
    {-1,      NULL }
};

/* This table maps a PCI device ID to a chipset family identifier. */
static PciChipsets VIAPciChipsets[] = {
    {VT3122,  PCI_CHIP_VT3122,  RES_SHARED_VGA},
    {VT7205,  PCI_CHIP_VT7205,  RES_SHARED_VGA},
    {VT3108,  PCI_CHIP_VT3108,  RES_SHARED_VGA},
    {VT3118,  PCI_CHIP_VT3118,  RES_SHARED_VGA},
    {VT3344,  PCI_CHIP_VT3344,  RES_SHARED_VGA},
    {VT3157,  PCI_CHIP_VT3157,  RES_SHARED_VGA},
    {VT3230,  PCI_CHIP_VT3230,  RES_SHARED_VGA},
    {VT3343,  PCI_CHIP_VT3343,  RES_SHARED_VGA},
    {-1,      -1,               RES_UNDEFINED}
};

int gVIAEntityIndex = -1;

typedef enum {
    OPTION_PRINTVGAREGS,
    OPTION_PRINTTVREGS,
    OPTION_PRINTSWOVREGS,
    OPTION_I2CSCAN,
    OPTION_PCI_BURST,
    OPTION_PCI_RETRY,
    OPTION_NOACCEL,
    OPTION_SWCURSOR,
    OPTION_HWCURSOR,
    OPTION_SHADOW_FB,
    OPTION_ROTATE,
    OPTION_VIDEORAM,
    OPTION_FORCEMEMCLOCK,
    OPTION_ACTIVEDEVICE,
    OPTION_DISABLEVQ,
    OPTION_DRIXINERAMA,
    OPTION_DISABLEIRQ,
    OPTION_INSECUREDRI,
    OPTION_AGP_DMA,
    OPTION_2D_DMA,
    OPTION_USEDMACOPY
} VIAOpts;


static OptionInfoRec VIAOptions[] =
{
    /* Debug only. Don't document these */
    {OPTION_PRINTVGAREGS, "PrintVGARegs", OPTV_BOOLEAN, {0}, FALSE},
    {OPTION_PRINTTVREGS, "PrintTVRegs",  OPTV_BOOLEAN, {0}, FALSE},
    {OPTION_PRINTSWOVREGS, "PrintSwovRegs",  OPTV_BOOLEAN, {0}, FALSE},
    {OPTION_I2CSCAN, "I2CScan", OPTV_BOOLEAN, {0}, FALSE},
    /* Normal options */
    {OPTION_NOACCEL,    "NoAccel",      OPTV_BOOLEAN, {0}, FALSE},
    {OPTION_HWCURSOR,   "HWCursor",     OPTV_BOOLEAN, {0}, FALSE},
    {OPTION_SWCURSOR,   "SWCursor",     OPTV_BOOLEAN, {0}, FALSE},
    {OPTION_SHADOW_FB,  "ShadowFB",     OPTV_BOOLEAN, {0}, FALSE},
    {OPTION_ROTATE,     "Rotate",       OPTV_ANYSTR,  {0}, FALSE},
    {OPTION_VIDEORAM,   "VideoRAM",     OPTV_INTEGER, {0}, FALSE},
    {OPTION_FORCEMEMCLOCK, "ForceMemClock", OPTV_INTEGER, {0}, FALSE},
    {OPTION_ACTIVEDEVICE,     "ActiveDevice",       OPTV_ANYSTR,  {0}, FALSE},
    {OPTION_DISABLEVQ,  "DisableVQ",    OPTV_BOOLEAN, {0}, FALSE},
    {OPTION_DRIXINERAMA,  "DRIXINERAMA",    OPTV_BOOLEAN, {0}, FALSE},
    {OPTION_DISABLEIRQ, "DisableIRQ", OPTV_BOOLEAN, {0}, FALSE},
    {OPTION_AGP_DMA, "EnableAGPDMA", OPTV_BOOLEAN, {0}, FALSE},
    {OPTION_2D_DMA, "NoAGPFor2D", OPTV_BOOLEAN, {0}, FALSE},
    {OPTION_USEDMACOPY, "UseDMACopy", OPTV_BOOLEAN, {0}, FALSE},
    {-1,                NULL,           OPTV_NONE,    {0}, FALSE}
};


static const char *vgaHWSymbols[] = {
    "vgaHWGetHWRec",
    "vgaHWSetMmioFuncs",
    "vgaHWGetIOBase",
    "vgaHWSave",
    "vgaHWProtect",
    "vgaHWRestore",
    "vgaHWSaveScreen",
    "vgaHWLock",
    "vgaHWUnlock",
    "vgaHWFreeHWRec",
    NULL
};


static const char *ramdacSymbols[] = {
    "xf86InitCursor",
    "xf86CreateCursorInfoRec",
    "xf86DestroyCursorInfoRec",
    NULL
};

static const char *ddcSymbols[] = {
    "xf86PrintEDID",
    "xf86DoEDID_DDC2",
    "xf86SetDDCproperties",
    NULL
};


static const char *i2cSymbols[] = {
    "xf86CreateI2CBusRec",
    "xf86I2CBusInit",
    "xf86CreateI2CDevRec",
    "xf86I2CDevInit",
    "xf86I2CWriteRead",
    "xf86I2CProbeAddress",
    "xf86DestroyI2CDevRec",
    "xf86I2CReadByte",
    "xf86I2CWriteByte",
    NULL
};

static const char *xaaSymbols[] = {
#ifdef X_HAVE_XAAGETROP
    "XAAGetCopyROP",
    "XAAGetCopyROP_PM",
    "XAAGetPatternROP",
#else
    "XAACopyROP",
    "XAACopyROP_PM",
    "XAAPatternROP",
#endif
    "XAACreateInfoRec",
    "XAADestroyInfoRec",
    "XAAInit",
    NULL
};

static const char *shadowSymbols[] = {
    "ShadowFBInit",
    NULL
};

#ifdef USE_FB
static const char *fbSymbols[] = {
    "fbScreenInit",
    "fbPictureInit",
    NULL
};
#else
static const char *cfbSymbols[] = {
    "cfbScreenInit",
    "cfb16ScreenInit",
    "cfb32ScreenInit",
    NULL
};
#endif

#ifdef XFree86LOADER
#ifdef XF86DRI
static const char *drmSymbols[] = {
    "drmAddMap",
    "drmAgpAcquire",
    "drmAgpAlloc",
    "drmAgpBase",
    "drmAgpBind",
    "drmAgpEnable",
    "drmAgpFree",
    "drmAgpGetMode",
    "drmAgpRelease",
    "drmCtlInstHandler",
    "drmCtlUninstHandler",
    "drmCommandWrite",
    "drmCommandWriteRead",
    "drmGetInterruptFromBusID",
    "drmGetVersion",
    "drmMap",
    "drmUnmap",
    "drmAgpUnbind",
    "drmRmMap",
    NULL
};

static const char *driSymbols[] = {
    "DRICloseScreen",
    "DRICreateInfoRec",
    "DRIDestroyInfoRec",
    "DRIFinishScreenInit",
    "DRIGetSAREAPrivate",
    "DRILock",
    "DRIQueryVersion",
    "DRIScreenInit",
    "DRIUnlock",
    "GlxSetVisualConfigs",
    NULL
};
#endif

static MODULESETUPPROTO(VIASetup);

static XF86ModuleVersionInfo VIAVersRec = {
    "via",
    "http://unichrome.sf.net/",
    MODINFOSTRING1,
    MODINFOSTRING2,
#ifdef XORG_VERSION_CURRENT
    XORG_VERSION_CURRENT,
#else
    XF86_VERSION_CURRENT,
#endif
    VERSION_MAJOR, VERSION_MINOR, PATCHLEVEL,
    ABI_CLASS_VIDEODRV,
    ABI_VIDEODRV_VERSION,
    MOD_CLASS_VIDEODRV,
    {0, 0, 0, 0}
};

XF86ModuleData viaModuleData = {&VIAVersRec, VIASetup, NULL};

static pointer VIASetup(
#ifdef HaveModuleDescPtr /* XFree86 4.6.0 */
                        ModuleDescPtr module,
#else
                        pointer module,
#endif
                        pointer opts, int *errmaj, int *errmin)
{
    static Bool setupDone = FALSE;

    if (!setupDone) {
        setupDone = TRUE;
        xf86AddDriver(&VIA, module, 0);
        LoaderRefSymLists(vgaHWSymbols,
#ifdef USE_FB
                          fbSymbols,
#else
                          cfbSymbols,
#endif
                          ramdacSymbols,
                          xaaSymbols,
                          shadowSymbols,
                          i2cSymbols,
                          ddcSymbols,
#ifdef XF86DRI
			  drmSymbols,
			  driSymbols,
#endif
                          NULL);

        return (pointer) 1;
    }
    else {
        if (errmaj)
            *errmaj = LDR_ONCEONLY;

        return NULL;
    }
} /* VIASetup */

#endif /* XFree86LOADER */

static Bool 
VIAGetRec(ScrnInfoPtr pScrn)
{
    VIAPtr pVia;

    VIAFUNC(pScrn->scrnIndex);

    if (pScrn->driverPrivate)
        return TRUE;

    pScrn->driverPrivate = xnfcalloc(sizeof(VIARec), 1);
    pVia = VIAPTR(pScrn);
    pVia->scrnIndex = pScrn->scrnIndex;

    pVia->Scratch = NULL;
    pVia->Outputs = NULL;
    pVia->Monitor = NULL;

    pVia->Cursor = NULL;
    pVia->CursorImage = NULL;

    return TRUE;
}


static void 
VIAFreeRec(ScrnInfoPtr pScrn)
{
    VIAPtr pVia;

    VIAFUNC(pScrn->scrnIndex);

    if (!pScrn->driverPrivate)
        return;

    pVia = VIAPTR(pScrn);

    ViaOutputsDestroy(pScrn);

    if (pVia->Scratch)
        xfree(pVia->Scratch);

    if (pVia->Monitor) {
        xfree(pVia->Monitor->id);
        xfree(pVia->Monitor->vendor);
        xfree(pVia->Monitor->model);
        ViaModesDestroy(pVia->Monitor->Modes);
        xfree(pVia->Monitor);
    }

    VIAUnmapMem(pScrn);

    xfree(pScrn->driverPrivate);
    pScrn->driverPrivate = NULL;
}


static const OptionInfoRec *
VIAAvailableOptions(int chipid, int busid)
{

    return VIAOptions;

}


/*
 * This implementation is far more flexible and informative than
 * xf86PrintChipSets. It allows for some semi-subversive VIA bashing too.
 */
static void 
VIAIdentify(int flags)
{
    xf86Msg(X_INFO, "VIA: driver for VIA Unichrome integrated graphics.\n");
    xf86Msg(X_INFO, "    VT3122: CLE266 (CastleRock).\n");
    xf86Msg(X_INFO, "    VT7205: KM400, KM400A, KN400, P4M800 (Unichrome).\n");
    xf86Msg(X_INFO, "    VT3108: K8M800, K8N800, K8N800A (Unichrome Pro, aka Pro B).\n");
    xf86Msg(X_INFO, "    VT3118: CN400, PM800, PM880, PN800 (Unichrome Pro A).\n");
    xf86Msg(X_INFO, "    VT3344: VN800, P4M800Pro, CN700 (Unichrome Pro).\n");
    xf86Msg(X_INFO, "    VT3157: CX700 (Unichrome Pro - not yet released).\n");
    xf86Msg(X_INFO, "    VT3230: K8M890 (Chrome9 IGP).\n");
    xf86Msg(X_INFO, "    VT3343: P4M890 (Unichrome Pro - not yet released).\n");
}


static Bool VIAProbe(DriverPtr drv, int flags)
{
    GDevPtr *devSections;
    int     *usedChips;
    int     numDevSections;
    int     numUsed;
    Bool    foundScreen = FALSE;
    int     i;

    /* sanity checks */
    if ((numDevSections = xf86MatchDevice(DRIVER_NAME, &devSections)) <= 0)
        return FALSE;

    if (xf86GetPciVideoInfo() == NULL)
        return FALSE;

    numUsed = xf86MatchPciInstances(DRIVER_NAME,
                                    PCI_VENDOR_VIA,
                                    VIAChipsets,
                                    VIAPciChipsets,
                                    devSections,
                                    numDevSections,
                                    drv,
                                    &usedChips);
    xfree(devSections);

    if (numUsed <= 0)
        return FALSE;

    xf86Msg(X_NOTICE, "VIA Technologies does not support or endorse this driver in any way.\n");
    xf86Msg(X_NOTICE, "For support, please refer to http://unichrome.sf.net/ or your X vendor.\n");

#if PATCHLEVEL == 0
    xf86Msg(X_INFO, "Using development version of the unichrome driver.\n");
# ifndef GIT_USED
    xf86Msg(X_INFO, "No git version information present.\n");
# else
#  ifdef GIT_UNCOMMITTED  
    xf86Msg(X_INFO, "git SHA-ID: %s (on %s) with uncommitted changes.\n", GIT_SHAID, GIT_BRANCH);
#  else
    xf86Msg(X_INFO, "git SHA-ID: %s (on %s).\n", GIT_SHAID, GIT_BRANCH);
#  endif /* GIT_UNCOMMITTED */
# endif /* GIT_USED */
#endif /* PATCHLEVEL */

    if (flags & PROBE_DETECT) {
        foundScreen = TRUE;
    }
    else {
        for (i = 0; i < numUsed; i++) {
            ScrnInfoPtr pScrn = xf86AllocateScreen(drv, 0);
            EntityInfoPtr pEnt;
            if ((pScrn = xf86ConfigPciEntity(pScrn, 0, usedChips[i],
                 VIAPciChipsets, 0, 0, 0, 0, 0)))
            {
                pScrn->driverVersion = VIA_VERSION;
                pScrn->driverName = DRIVER_NAME;
                pScrn->name = "VIA";
                pScrn->Probe = VIAProbe;
                pScrn->PreInit = VIAPreInit;
                pScrn->ScreenInit = VIAScreenInit;
                pScrn->SwitchMode = VIASwitchMode;
                pScrn->AdjustFrame = VIAAdjustFrame;
                pScrn->EnterVT = VIAEnterVT;
                pScrn->LeaveVT = VIALeaveVT;
                pScrn->FreeScreen = VIAFreeScreen;
                pScrn->ValidMode = ViaValidMode;
                foundScreen = TRUE;
            }
            /*
            xf86ConfigActivePciEntity(pScrn,
                                      usedChips[i],
                                      VIAPciChipsets,
                                      NULL,
                                      NULL,
                                      NULL,
                                      NULL,
                                      NULL);
	    */
            pEnt = xf86GetEntityInfo(usedChips[i]);

            /* Secondary needs a lot of work, this is just the tip of the iceberg.
             * And... erm... sharable? -- Luc */
            /* CLE266 card support Dual-Head, mark the entity as sharable*/
            if (pEnt->chipset == VT3122 || pEnt->chipset == VT7205) {
                static int instance = 0;
                DevUnion* pPriv;

                xf86SetEntitySharable(usedChips[i]);
                xf86SetEntityInstanceForScreen(pScrn,
                    pScrn->entityList[0], instance);

                if(gVIAEntityIndex < 0)
                {
                    gVIAEntityIndex = xf86AllocateEntityPrivateIndex();
                    pPriv = xf86GetEntityPrivate(pScrn->entityList[0],
                            gVIAEntityIndex);

                    if (!pPriv->ptr)
                    {
                        VIAEntPtr pVIAEnt;
                        pPriv->ptr = xnfcalloc(sizeof(VIAEntRec), 1);
                        pVIAEnt = pPriv->ptr;
                        pVIAEnt->IsDRIEnabled = FALSE;
                        pVIAEnt->BypassSecondary = FALSE;
                        pVIAEnt->HasSecondary = FALSE;
                        pVIAEnt->IsSecondaryRestored = FALSE;
                    }
                }
                instance++;
            }
            xfree(pEnt);
        }
    }

    xfree(usedChips);

    return foundScreen;

} /* VIAProbe */


static int LookupChipSet(PciChipsets *pset, int chipSet)
{
  while (pset->numChipset >= 0) {
    if (pset->numChipset == chipSet) return pset->PCIid;
    pset++;
  }
  return -1;
}

/*
 * Concentrate all VIA scratch area stuff in one struct.
 * Or, keep all nastyness close together and far, far away.
 *
 * Known VIA Scratch Registers:
 *   SR32, SR33, SR34, SR39, SR3A
 *   CR3B, CR3C, CR3D, CR3E, CR3F,
 *   CR49, CR4A, CR4B, CR4C, CR4D, CR4E, CR4F.
 *
 */
static struct ViaScratch *
ViaScratchGet(ScrnInfoPtr pScrn)
{
    vgaHWPtr hwp = VGAHWPTR(pScrn);
    VIAPtr pVia = VIAPTR(pScrn);
    struct ViaScratch *Scratch = xnfcalloc(sizeof(struct ViaScratch), 1);
    int tmp;

    VIAFUNC(pScrn->scrnIndex);

    if (!xf86IsScreenPrimary(pScrn->scrnIndex))
        xf86DrvMsg(pScrn->scrnIndex, X_INFO, "%s called when not the primary"
                   "VGA device. Some scratch areas might be uninitialised.\n",
                   __func__);

    /*
     * Get Memory size first. Should really use the RAM controller pci config.
     */
    if (pVia->Chipset == VT3122) {
        tmp = hwp->readSeq(hwp, 0x34);
    
        if (!tmp) {
            xf86DrvMsg(pScrn->scrnIndex, X_WARNING, "%s: VideoRam Scratch area"
                       " uninitialised. Trying CR39.\n", __func__);
            tmp = hwp->readSeq(hwp, 0x39);
        }
    } else
        tmp = hwp->readSeq(hwp, 0x39);
    
    if ((tmp > 16) && (tmp <= 128))
        Scratch->VideoRam = (tmp + 1) << 9;
    else if ((tmp > 0) && (tmp < 31))
        Scratch->VideoRam = tmp << 12;
    else {
        xf86DrvMsg(pScrn->scrnIndex, X_WARNING, "%s: VideoRam Scratch area"
                   " uninitialised.\n", __func__);
        Scratch->VideoRam = 16 << 10;
    }

    ViaDebug(pScrn->scrnIndex, "%s: VideoRam: %d\n", __func__,
             Scratch->VideoRam);


    /*
     * Get the memory clock.
     */
    tmp = hwp->readCrtc(hwp, 0x3D) >> 4;
    
    /* Actual values:
     *  0: SDR  66Mhz
     *  1: SDR 100Mhz
     *  2: SDR 133Mhz
     *  3: DDR 100Mhz (PC1600 or DDR200)
     *  4: DDR 133Mhz (PC2100 or DDR266)
     *  5: DDR 166Mhz (PC2700 or DDR333)
     *  6: DDR 200Mhz (PC3200 or DDR400)
     *  7: fairytale.
     */
    /* SDR won't happen, assume badly set up register. */
    if ((tmp < 3) || (tmp > 6)) {
        xf86DrvMsg(pScrn->scrnIndex, X_WARNING, "%s: Unknown Memory Clock (%d)."
                   " Defaulting to DDR200\n", __func__, tmp);
        Scratch->MemClk = VIA_MEM_DDR200;
    } else
        Scratch->MemClk = tmp - 3;
    ViaDebug(pScrn->scrnIndex, "%s: MemClk: %d\n", __func__, Scratch->MemClk);

    /*
     * Get output device status.
     */
    Scratch->ActiveDevice = hwp->readCrtc(hwp, 0x3E) >> 4;
    ViaDebug(pScrn->scrnIndex, "%s: ActiveDevice: %d\n", __func__,
             Scratch->ActiveDevice);

    /*
     * Get Panel size.
     */
    Scratch->PanelSize = hwp->readCrtc(hwp, 0x3F) >> 4;
    ViaDebug(pScrn->scrnIndex, "%s: PanelSize: %d\n",
             __func__, Scratch->PanelSize);
    if (!Scratch->PanelSize)
        xf86DrvMsg(pScrn->scrnIndex, X_WARNING, "%s: It is possible that"
                   " PanelSize wasn't initialised properly\n", __func__);

    /*
     * Get TV Standard.
     */
    if (hwp->readCrtc(hwp, 0x3B) & 0x02)
        Scratch->TVStandard = TVSTANDARD_PAL;
    else
        Scratch->TVStandard = TVSTANDARD_NTSC;
    ViaDebug(pScrn->scrnIndex, "%s: TVStandard: %d\n", __func__,
             Scratch->TVStandard);

    return Scratch;
}

/*
 *
 */
static CARD8
ViaGetMemoryClock(ScrnInfoPtr pScrn)
{
    int temp;
    CARD8 MemClk = VIA_MEM_NONE;

    VIAFUNC(pScrn->scrnIndex);

    if (xf86GetOptValInteger(VIAOptions, OPTION_FORCEMEMCLOCK, &temp)) {
        switch (temp) {
        case 200:
        case 1600:
            MemClk = VIA_MEM_DDR200;
            break;
        case 266:
        case 2100:
            MemClk = VIA_MEM_DDR266;
            break;
        case 333:
        case 2700:
            MemClk = VIA_MEM_DDR333;
            break;
        case 400:
        case 3200:
            MemClk = VIA_MEM_DDR400;
            break;
        default:
            xf86DrvMsg(pScrn->scrnIndex, X_WARNING, "Unsupported ForceMemClock:"
                       " %d.\n", MemClk);
            break;
        }
    }

    if (MemClk == VIA_MEM_NONE)
        MemClk = VIAPTR(pScrn)->Scratch->MemClk;

    ViaDebug(pScrn->scrnIndex, "Memory clock: %d.\n", MemClk);
    return MemClk;
}

/*
 * Good VIA idea: poke the ramcontroller directly for VideoRam.
 * But of course we do this properly, and we also poke the right
 * register, since we test code :)
 */
static void
ViaVideoRamGetFromRamController(ScrnInfoPtr pScrn)
{
    static struct {
        int    Chipset;
        CARD16 RamController;
        CARD8  Register;
    } ViaRamController[] = {
        { VT3122, 0x3123, 0xE1},
        { VT7205, 0x3205, 0xE1}, /* KM400 */
        { VT7205, 0x3296, 0xA1}, /* P4M800 */
        { VT3108, 0x3204, 0xA1},
        { VT3118, 0x3259, 0xA1},
        { VT3344, 0x3314, 0xA1},
        { VT3157, 0x3324, 0xA1},
        { VT3230, 0x3336, 0xA1},
        { VT3343, 0x3327, 0xA1},
        { 0xFFFF, 0x0000, 0x00}
    };
    PCITAG Tag;
    CARD8 tmp;
    int i;

    VIAFUNC(pScrn->scrnIndex);

    for (i = 0; ViaRamController[i].RamController; i++) {
        if (ViaRamController[i].Chipset != VIAPTR(pScrn)->Chipset)
            continue;
        
        Tag = pciFindFirst((ViaRamController[i].RamController << 16) |
                           PCI_VENDOR_VIA, 0xFFFFFFFF);
        if (Tag == PCI_NOT_FOUND)
            continue;

        tmp = pciReadByte(Tag, ViaRamController[i].Register);
        pScrn->videoRam = (1 << ((tmp & 0x70) >> 4)) * 1024;
        return;
    }

    xf86DrvMsg(pScrn->scrnIndex, X_WARNING, 
               "%s: Unable to determine VideoRam Size.\n", __func__);
    xf86DrvMsg(pScrn->scrnIndex, X_WARNING, 
               "%s: Please contact libv@skynet.be ASAP.\n", __func__);
}

/*
 * Bandwidth
 *
 */
/* used for impossible entries: allow a minimum bandwidth in case this does happen */
#define VIA_BW_MIN 74000000 /* > 640x480@60Hz@32bpp */

/* index to table */
#define VIA_BW_VT3122A  0
#define VIA_BW_VT3122C  1
#define VIA_BW_VT7205   2
#define VIA_BW_VT7205A  3
#define VIA_BW_VT3108   4
#define VIA_BW_VT3118   5
#define VIA_BW_VT3344   6
#define VIA_BW_VT3157   7
#define VIA_BW_ALL      8

/*
 * 393216000 is for SDR133 in via_refresh.h
 * 460800000 is for DDR266
 */
static struct {
    CARD8 Device; /* equal to index */
    CARD32 Bandwidth[VIA_MEM_END];
} ViaBandwidthTable[VIA_BW_ALL] = {
    { VIA_BW_VT3122A, { 394000000,  461000000, VIA_BW_MIN, VIA_BW_MIN } },
    { VIA_BW_VT3122C, { 394000000,  461000000, VIA_BW_MIN, VIA_BW_MIN } },
    { VIA_BW_VT7205,  { 394000000,  461000000,  461000000, VIA_BW_MIN } },
    { VIA_BW_VT7205A, { 394000000,  461000000,  461000000,  461000000 } },
    { VIA_BW_VT3108,  { 394000000,  461000000,  461000000,  461000000 } },
    { VIA_BW_VT3118,  { 394000000,  461000000,  461000000,  461000000 } },
    { VIA_BW_VT3344,  { 394000000,  461000000,  461000000,  461000000 } },
    { VIA_BW_VT3157,  { 394000000,  461000000,  461000000,  461000000 } },
};

/*
 *
 */
static CARD32
ViaGetMemoryBandwidth(ScrnInfoPtr pScrn)
{
    VIAPtr pVia = VIAPTR(pScrn);

    VIAFUNC(pScrn->scrnIndex);
   
    switch (pVia->Chipset) {
    case VT3122:
	if (VT3122_REV_IS_AX(pVia->ChipRev))
	    return ViaBandwidthTable[VIA_BW_VT3122A].Bandwidth[pVia->MemClk];
	else
	    return ViaBandwidthTable[VIA_BW_VT3122C].Bandwidth[pVia->MemClk];
    case VT7205:
        {
            /* is this a KM400 or a P4M800 ? */
            Bool KM400 = FALSE;
            
            /* check host bridge pci device id. */
            if (pciReadWord(0x00000000, 0x02) == 0x3205)
                KM400 = TRUE;
            
            /* 0x84 is earliest known KM400A device, 0x80 is more likely though */
            if (KM400 && (pVia->ChipRev < 0x84))
                return ViaBandwidthTable[VIA_BW_VT7205].Bandwidth[pVia->MemClk];
            else
                return ViaBandwidthTable[VIA_BW_VT7205A].Bandwidth[pVia->MemClk];
        }
    case VT3108:
	return ViaBandwidthTable[VIA_BW_VT3108].Bandwidth[pVia->MemClk];
    case VT3118:
	return ViaBandwidthTable[VIA_BW_VT3118].Bandwidth[pVia->MemClk];
    case VT3344:
        return ViaBandwidthTable[VIA_BW_VT3344].Bandwidth[pVia->MemClk];
    case VT3157:
        return ViaBandwidthTable[VIA_BW_VT3157].Bandwidth[pVia->MemClk];
    default:
	xf86DrvMsg(pScrn->scrnIndex, X_ERROR, "%s: Unknown Chipset.\n", __func__);
	return VIA_BW_MIN;
    }
}

static int LookupChipID(PciChipsets* pset, int ChipID)
{
    /* Is there a function to do this for me? */
    while (pset->numChipset >= 0)
    {
        if (pset->PCIid == ChipID)
            return pset->numChipset;

        pset++;
    }

    return -1;

}

/*
 *
 */
static Bool 
VIAPreInit(ScrnInfoPtr pScrn, int flags)
{
    EntityInfoPtr   pEnt;
    VIAPtr          pVia;
    MessageType     from = X_DEFAULT;
    ClockRangePtr   clockRanges;
    char            *s = NULL;
#ifndef USE_FB
    char            *mod = NULL;
    const char      *reqSym = NULL;
#endif
    vgaHWPtr        hwp;
    int             i;
    int  MaxHVirtual, MaxVVirtual;

    VIAFUNC(pScrn->scrnIndex);

    if (flags & PROBE_DETECT) {
        xf86DrvMsg(pScrn->scrnIndex, X_ERROR, "libv doesn't believe in "
                   "PROBE_DETECT calling PreInit.\n");
        return FALSE;
    }

    if (pScrn->numEntities > 1)
        return FALSE;

    if (!xf86LoadSubModule(pScrn, "vgahw"))
        return FALSE;

    xf86LoaderReqSymLists(vgaHWSymbols, NULL);
    if (!vgaHWGetHWRec(pScrn))
        return FALSE;

#if 0
    /* Here we can alter the number of registers saved and restored by the
     * standard vgaHWSave and Restore routines.
     */
    vgaHWSetRegCounts(pScrn, VGA_NUM_CRTC, VGA_NUM_SEQ, VGA_NUM_GFX, VGA_NUM_ATTR);
#endif

    if (!VIAGetRec(pScrn)) {
        return FALSE;
    }

    pVia = VIAPTR(pScrn);

    pVia->IsSecondary = FALSE;
    pEnt = xf86GetEntityInfo(pScrn->entityList[0]);
    if (pEnt->resources) {
        xfree(pEnt);
        VIAFreeRec(pScrn);
        return FALSE;
    }

    pVia->EntityIndex = pEnt->index;

    if(xf86IsEntityShared(pScrn->entityList[0]))
    {
        if(xf86IsPrimInitDone(pScrn->entityList[0]))
        {
            DevUnion* pPriv;
            VIAEntPtr pVIAEnt;
            VIAPtr pVia1;

            pVia->IsSecondary = TRUE;
            pPriv = xf86GetEntityPrivate(pScrn->entityList[0],
                    gVIAEntityIndex);
            pVIAEnt = pPriv->ptr;
            if (pVIAEnt->BypassSecondary) {
		xfree(pEnt);
		VIAFreeRec(pScrn);
		return FALSE;
	    }
            pVIAEnt->pSecondaryScrn = pScrn;
            pVIAEnt->HasSecondary = TRUE;
            pVia1 = VIAPTR(pVIAEnt->pPrimaryScrn);
            pVia1->HasSecondary = TRUE;
	    pVia->sharedData = pVia1->sharedData;
        }
        else
        {
            DevUnion* pPriv;
            VIAEntPtr pVIAEnt;

            xf86SetPrimInitDone(pScrn->entityList[0]);
            pPriv = xf86GetEntityPrivate(pScrn->entityList[0],
                    gVIAEntityIndex);
	    pVia->sharedData = xnfcalloc(sizeof(ViaSharedRec),1);
            pVIAEnt = pPriv->ptr;
            pVIAEnt->pPrimaryScrn = pScrn;
            pVIAEnt->IsDRIEnabled = FALSE;
            pVIAEnt->BypassSecondary = FALSE;
            pVIAEnt->HasSecondary = FALSE;
            pVIAEnt->RestorePrimary = FALSE;
            pVIAEnt->IsSecondaryRestored = FALSE;
        }
    } else {
	pVia->sharedData = xnfcalloc(sizeof(ViaSharedRec),1);
    }

    /*
     * We support depths of 8, 16 and 24.
     * We support bpp of 8, 16, and 32.
     */

    if (!xf86SetDepthBpp(pScrn, 0, 0, 0, Support32bppFb)) {
        xfree(pEnt);
        VIAFreeRec(pScrn);
        return FALSE;
    }
    else {
        switch (pScrn->depth) {
        case 8:
        case 16:
        case 24:
        case 32:
            /* OK */
            break;
        default:
            xf86DrvMsg(pScrn->scrnIndex, X_ERROR,
                       "Given depth (%d) is not supported by this driver\n",
                       pScrn->depth);
            xfree(pEnt);
            VIAFreeRec(pScrn);
            return FALSE;
        }
    }

    xf86PrintDepthBpp(pScrn);

    if (pScrn->depth == 32) {
        pScrn->depth = 24;
    }

    if (pScrn->depth > 8) {
        rgb zeros = {0, 0, 0};

        if (!xf86SetWeight(pScrn, zeros, zeros)) {
	    xfree(pEnt);
	    VIAFreeRec(pScrn);
            return FALSE;
	} else {
            /* TODO check weight returned is supported */
            ;
        }
    }

    if (!xf86SetDefaultVisual(pScrn, -1)) {
        return FALSE;
    }
    else {
        /* We don't currently support DirectColor at > 8bpp */
        if (pScrn->depth > 8 && pScrn->defaultVisual != TrueColor) {
            xf86DrvMsg(pScrn->scrnIndex, X_ERROR, "Given default visual"
                       " (%s) is not supported at depth %d\n",
                       xf86GetVisualName(pScrn->defaultVisual), pScrn->depth);
            xfree(pEnt);
            VIAFreeRec(pScrn);
            return FALSE;
        }
    }

    /* We use a programmable clock */
    pScrn->progClock = TRUE;

    /* xf86CollectOptions depends on pScrn->monitor instead of
       pScrn->confScreen->monitor */
    pScrn->monitor = pScrn->confScreen->monitor;
    xf86CollectOptions(pScrn, NULL);
    pScrn->monitor = NULL; /* we might end up using a different MonRec anyway */

    /* Set the bits per RGB for 8bpp mode */
    if (pScrn->depth == 8)
        pScrn->rgbBits = 6;

    xf86ProcessOptions(pScrn->scrnIndex, pScrn->options, VIAOptions);

#ifdef XF86DRI
    pVia->drixinerama = FALSE;
    if (xf86IsOptionSet(VIAOptions, OPTION_DRIXINERAMA))
        pVia->drixinerama = TRUE;
#else
    if (xf86IsOptionSet(VIAOptions, OPTION_DRIXINERAMA))
    	xf86DrvMsg(pScrn->scrnIndex, X_CONFIG,
    		"Option: drixinerama ignored, no DRI support compiled into driver.\n");
#endif
    if (xf86ReturnOptValBool(VIAOptions, OPTION_PCI_BURST, FALSE)) {
        pVia->pci_burst = TRUE;
        xf86DrvMsg(pScrn->scrnIndex, X_CONFIG,
                   "Option: pci_burst - PCI burst read enabled\n");
    }
    else {
        pVia->pci_burst = FALSE;
    }

    pVia->NoPCIRetry = 1;       /* default */
    if (xf86ReturnOptValBool(VIAOptions, OPTION_PCI_RETRY, FALSE)) {
        if (xf86ReturnOptValBool(VIAOptions, OPTION_PCI_BURST, FALSE)) {
            pVia->NoPCIRetry = 0;
            xf86DrvMsg(pScrn->scrnIndex, X_CONFIG, "Option: pci_retry\n");
        }
        else {
            xf86DrvMsg(pScrn->scrnIndex, X_CONFIG,
                       "\"pci_retry\" option requires \"pci_burst\"\n");
        }
    }

    if (xf86IsOptionSet(VIAOptions, OPTION_SHADOW_FB)) {
        pVia->shadowFB = TRUE;
        xf86DrvMsg(pScrn->scrnIndex, X_CONFIG, "Option: ShadowFB %s.\n",
                   pVia->shadowFB ? "enabled" : "disabled");
    }
    else {
        pVia->shadowFB = FALSE;
    }

    if ((s = xf86GetOptValString(VIAOptions, OPTION_ROTATE))) {
        if (!xf86NameCmp(s, "CW")) {
            /* accel is disabled below for shadowFB */
            pVia->shadowFB = TRUE;
            pVia->rotate = 1;
            pVia->hwcursor = FALSE;
            xf86DrvMsg(pScrn->scrnIndex, X_CONFIG,
                       "Rotating screen clockwise - acceleration disabled\n");
        }
        else if(!xf86NameCmp(s, "CCW")) {
            pVia->shadowFB = TRUE;
            pVia->rotate = -1;
            pVia->hwcursor = FALSE;
            xf86DrvMsg(pScrn->scrnIndex, X_CONFIG,  "Rotating screen"
                       "counter clockwise - acceleration disabled\n");
        }
        else {
            xf86DrvMsg(pScrn->scrnIndex, X_CONFIG, "\"%s\" is not a valid"
                       "value for Option \"Rotate\"\n", s);
            xf86DrvMsg(pScrn->scrnIndex, X_INFO,
                       "Valid options are \"CW\" or \"CCW\"\n");
        }
    }

    if (xf86ReturnOptValBool(VIAOptions, OPTION_NOACCEL, FALSE)) {
        pVia->NoAccel = TRUE;
        xf86DrvMsg(pScrn->scrnIndex, X_CONFIG,
                   "Option: NoAccel -Acceleration Disabled\n");
    }
    else {
        pVia->NoAccel = FALSE;
    }

    if (pVia->shadowFB && !pVia->NoAccel) {
        xf86DrvMsg(pScrn->scrnIndex, X_WARNING,
                   "HW acceleration not supported with \"shadowFB\".\n");
        pVia->NoAccel = TRUE;
    }

    /*
     * The SWCursor setting takes priority over HWCursor.  The default
     * if neither is specified is HW.
     */

    from = X_DEFAULT;
    pVia->hwcursor = pVia->shadowFB ? FALSE : TRUE;
    if (xf86GetOptValBool(VIAOptions, OPTION_HWCURSOR, &pVia->hwcursor))
        from = X_CONFIG;

    if (xf86ReturnOptValBool(VIAOptions, OPTION_SWCURSOR, FALSE)) {
        pVia->hwcursor = FALSE;
        from = X_CONFIG;
    }

    if (pVia->IsSecondary) pVia->hwcursor = FALSE;

    xf86DrvMsg(pScrn->scrnIndex, from, "Using %s cursor\n",
               pVia->hwcursor ? "HW" : "SW");

    from = X_DEFAULT;

    pScrn->videoRam = 0;
    if(xf86GetOptValInteger(VIAOptions, OPTION_VIDEORAM, &pScrn->videoRam)) {
    	xf86DrvMsg( pScrn->scrnIndex, X_CONFIG,
    	            "Option: VideoRAM %dkB\n", pScrn->videoRam );
    }

    if (xf86ReturnOptValBool(VIAOptions, OPTION_DISABLEVQ, FALSE)) {
        pVia->VQEnable = FALSE;
        xf86DrvMsg(pScrn->scrnIndex, X_CONFIG,
                   "Option: DisableVQ -VQ Disabled\n");
    }
    else {
        pVia->VQEnable = TRUE;
    }

    if (xf86ReturnOptValBool(VIAOptions, OPTION_DISABLEIRQ, FALSE)) {
        pVia->DRIIrqEnable = FALSE;
        xf86DrvMsg(pScrn->scrnIndex, X_CONFIG,
                   "Option: DisableIRQ - DRI IRQ Disabled\n");
    }
    else {
        pVia->DRIIrqEnable = TRUE;
    }


    if (xf86ReturnOptValBool(VIAOptions, OPTION_AGP_DMA, FALSE)) {
        pVia->agpEnable = TRUE;
        xf86DrvMsg(pScrn->scrnIndex, X_CONFIG,
                   "Option: EnableAGPDMA - Enabling AGP DMA\n");
    } else {
	pVia->agpEnable = FALSE;
    }

    if (xf86ReturnOptValBool(VIAOptions, OPTION_2D_DMA, FALSE)) {
        pVia->dma2d = FALSE;
	if (pVia->agpEnable) {
	    xf86DrvMsg(pScrn->scrnIndex, X_CONFIG,
		       "Option: NoAGPFor2D - AGP DMA is not used for 2D "
		       "acceleration\n");
	}
    } else {
        pVia->dma2d = TRUE;
    }

#ifdef XF86DRI
    if (xf86ReturnOptValBool(VIAOptions, OPTION_USEDMACOPY, TRUE))
        pVia->UseDMACopy = TRUE; /* properly checked from DRIScreenInit */
    else
#endif
        pVia->UseDMACopy = FALSE;

    /* ActiveDevice Option for device selection */
    pVia->ActiveDevice = 0x00;
    if ((s = xf86GetOptValString(VIAOptions, OPTION_ACTIVEDEVICE))) {
	if (strstr(s, "CRT"))
	    pVia->ActiveDevice |= VIA_DEVICE_CRT;
	if (strstr(s, "DFP")) /* just treat this the same as LCD */
	    pVia->ActiveDevice |= VIA_DEVICE_LCD;
	if (strstr(s, "TV"))
	    pVia->ActiveDevice |= VIA_DEVICE_TV;
    }

    pVia->PciInfo = xf86GetPciInfoForEntity(pEnt->index);
    xf86RegisterResources(pEnt->index, NULL, ResNone);
    /*
    xf86SetOperatingState(RES_SHARED_VGA, pEnt->index, ResUnusedOpr);
    xf86SetOperatingState(resVgaMemShared, pEnt->index, ResDisableOpr);
    */

    if (pEnt->device->chipset && *pEnt->device->chipset) {
        pScrn->chipset = pEnt->device->chipset;
        pVia->Chipset = xf86StringToToken(VIAChipsets, pScrn->chipset);
        pVia->ChipId = pEnt->device->chipID = LookupChipSet(VIAPciChipsets, pVia->Chipset);
        from = X_CONFIG;
    } else if (pEnt->device->chipID >= 0) {
        pVia->ChipId = pEnt->device->chipID;
        pVia->Chipset = LookupChipID(VIAPciChipsets, pVia->ChipId);
        pScrn->chipset = (char *)xf86TokenToString(VIAChipsets,
                                                   pVia->Chipset);
        from = X_CONFIG;
        xf86DrvMsg(pScrn->scrnIndex, X_CONFIG, "ChipID override: 0x%04X\n",
                   pEnt->device->chipID);
    } else {
        from = X_PROBED;
        pVia->ChipId = pVia->PciInfo->chipType;
        pVia->Chipset = LookupChipID(VIAPciChipsets, pVia->ChipId);
        pScrn->chipset = (char *)xf86TokenToString(VIAChipsets,
                                                   pVia->Chipset);
    }

    if (pEnt->device->chipRev >= 0) {
        pVia->ChipRev = pEnt->device->chipRev;
        xf86DrvMsg(pScrn->scrnIndex, X_CONFIG, "ChipRev override: %d\n",
                   pVia->ChipRev);
    }
    else {
        /*pVia->ChipRev = pVia->PciInfo->chipRev;*/
        /* Read PCI bus 0, dev 0, function 0, index 0xF6 to get chip rev. */
        pVia->ChipRev = pciReadByte(pciTag(0, 0, 0), 0xF6);
    }

    if (pEnt->device->videoRam != 0) {
        if (!pScrn->videoRam)
    	    pScrn->videoRam = pEnt->device->videoRam;
    	else {
            xf86DrvMsg(pScrn->scrnIndex, X_WARNING,
                       "Video Memory Size in Option is %d KB, Detect is %d KB!",
                       pScrn->videoRam, pEnt->device->videoRam);
    	}
    }

    xfree(pEnt);

    /* maybe throw in some more sanity checks here */
    xf86DrvMsg(pScrn->scrnIndex, from, "Chipset: \"%s\"\n", pScrn->chipset);

    /* No dotclock == No workable mode == Completely useless */
    switch (pVia->Chipset) {
    case VT3122:
    case VT7205:
    case VT3108:
        break;
    default:
	xf86DrvMsg(pScrn->scrnIndex, X_ERROR, "%s is not supported yet at this"
                   " time.\n", pScrn->chipset);
        VIAFreeRec(pScrn);
        return FALSE;
    }

    pVia->PciTag = pciTag(pVia->PciInfo->bus, pVia->PciInfo->device,
                          pVia->PciInfo->func);

    if (!VIAMapMMIO(pScrn)) {
	VIAFreeRec(pScrn);
        return FALSE;
    }
    hwp = VGAHWPTR(pScrn);

    if (xf86ReturnOptValBool(VIAOptions, OPTION_PRINTVGAREGS, FALSE)) {
        pVia->PrintVGARegs = TRUE;
	ViaVgaPrintRegs(pScrn, __func__); /* Do this as early as possible */
    } else
        pVia->PrintVGARegs = FALSE;

    if (xf86ReturnOptValBool(VIAOptions, OPTION_PRINTTVREGS, FALSE))
        pVia->PrintTVRegs = TRUE;
    else
        pVia->PrintTVRegs = FALSE;

    if (xf86ReturnOptValBool(VIAOptions, OPTION_PRINTSWOVREGS, FALSE))
        pVia->PrintSwovRegs = TRUE;
    else
        pVia->PrintSwovRegs = FALSE;

    if (xf86ReturnOptValBool(VIAOptions, OPTION_I2CSCAN, FALSE))
        pVia->I2CScan = TRUE;
    else
        pVia->I2CScan = FALSE;

    xf86DrvMsg(pScrn->scrnIndex, from, "Chipset Rev.: %d\n", pVia->ChipRev);

    ViaCheckCardId(pScrn);   

    pVia->Scratch = ViaScratchGet(pScrn);

    pVia->MemClk = ViaGetMemoryClock(pScrn);
    pVia->Bandwidth = ViaGetMemoryBandwidth(pScrn);

    if (!pScrn->videoRam)
        ViaVideoRamGetFromRamController(pScrn);
    if (!pScrn->videoRam)
        pScrn->videoRam = pVia->Scratch->VideoRam;

    /* Split FB for SAMM */
    /* FIXME: For now, split FB into two equal sections. This should
     *        be able to be adjusted by user with a config option. */
    if (pVia->IsSecondary) {
        DevUnion* pPriv;
        VIAEntPtr pVIAEnt;
        VIAPtr    pVia1;

        pPriv = xf86GetEntityPrivate(pScrn->entityList[0],
              gVIAEntityIndex);
        pVIAEnt = pPriv->ptr;
        pScrn->videoRam = pScrn->videoRam >> 1;
        pVIAEnt->pPrimaryScrn->videoRam = pScrn->videoRam;
        pVia1 = VIAPTR(pVIAEnt->pPrimaryScrn);
        pVia1->videoRambytes = pScrn->videoRam << 10;
        pVia->FrameBufferBase += (pScrn->videoRam << 10);
    }

    pVia->videoRambytes = pScrn->videoRam << 10;
    xf86DrvMsg(pScrn->scrnIndex, X_PROBED,"videoram =  %dk\n", pScrn->videoRam);

    pScrn->monitor = pScrn->confScreen->monitor;

    if (!xf86LoadSubModule(pScrn, "i2c")) {
        VIAFreeRec(pScrn);
        return FALSE;
    }
    else {
        xf86LoaderReqSymLists(i2cSymbols,NULL);
        ViaI2CInit(pScrn);
    }

    if (!xf86LoadSubModule(pScrn, "ddc")) {
	VIAFreeRec(pScrn);
	return FALSE;
    } else
	xf86LoaderReqSymLists(ddcSymbols, NULL);

    ViaOutputsDetect(pScrn);
    if (!ViaOutputsSelect(pScrn)) {
	xf86DrvMsg(pScrn->scrnIndex, X_ERROR, "No outputs possible.\n");
	VIAFreeRec(pScrn);
	return FALSE;
    }

    ViaMonitorInitialise(pScrn);

    /*
     * Setup the ClockRanges, which describe what clock ranges are available,
     * and what sort of modes they can be used for.
     */
    
    clockRanges = xnfalloc(sizeof(ClockRange));
    clockRanges->next = NULL;
    clockRanges->minClock = 20000;
    clockRanges->maxClock = 230000;
    
    clockRanges->clockIndex = -1;
    clockRanges->interlaceAllowed = FALSE;
    clockRanges->doubleScanAllowed = FALSE;
    
    /* Virtual mode limitations:
     * 1) Offset: or FB line stride.
     *    Primary: max 16kB - 8 (32bit: 4088 pixels per line)
     *    Secondary: max 8kB - 8 (32bit: 2040 pixels per line)
     *
     * 2) StartAddress:
     *    Primary: Here we run into separate limits.
     *      VT3122A: 32MB - 2 or roughly 2048 lines.
     *      VT7205A: 128MB - 2 or twice the FB size :), so use 4096.
     *    Secondary: 128MB - 8, but this needs verification.
     *
     * xf86ValidateModes forces you to hand it pixel counts for both H and V.
     * Max pitch and Max startAddress would've been far more effective and
     * far more dynamic.
     *
     * We should be able to limit the memory available for a mode to 32MB,
     * yet xf86ValidateModes (or miScanLineWidth) fails to catch this properly
     * (apertureSize).
     */
    MaxHVirtual = 4096 - 8;
    if (pVia->Chipset == VT3122)
        MaxVVirtual = 2048;
    else
        MaxVVirtual = 4096;

    /* Select valid modes from those available */
    i = xf86ValidateModes(pScrn,
                          pScrn->monitor->Modes,    /* availModes */
                          pScrn->display->modes,    /* modeNames */
                          clockRanges,              /* list of clock ranges */
                          NULL,                     /* list of line pitches */
                          256,                      /* mini line pitch */
                          MaxHVirtual,              /* max line pitch */
                          8 * pScrn->bitsPerPixel,  /* pitch inc (bits) */
                          128,                      /* min height */
                          MaxVVirtual,              /* max height */
                          pScrn->display->virtualX, /* virtual width */
                          pScrn->display->virtualY, /* virtual height */
                          pVia->videoRambytes,      /* apertureSize */
                          LOOKUP_BEST_REFRESH);     /* lookup mode flags */
    
    
    if (i == -1) {
        xf86DrvMsg(pScrn->scrnIndex, X_ERROR, "xf86ValidateModes failure\n");
        VIAFreeRec(pScrn);
        return FALSE;
    }
    
    xf86PruneDriverModes(pScrn);
    
    if (i == 0 || pScrn->modes == NULL) {
        xf86DrvMsg(pScrn->scrnIndex, X_ERROR, "No valid modes found\n");
        VIAFreeRec(pScrn);
        return FALSE;
    }
    
    /* Set up screen parameters. */
    pVia->Bpp = pScrn->bitsPerPixel >> 3;
    pVia->Bpl = pScrn->displayWidth * pVia->Bpp;

    pScrn->currentMode = pScrn->modes;
    xf86PrintModes(pScrn);
    xf86SetDpi(pScrn, 0, 0);

#ifdef USE_FB
    if (xf86LoadSubModule(pScrn, "fb") == NULL) {
        VIAFreeRec(pScrn);
        return FALSE;
    }

    xf86LoaderReqSymLists(fbSymbols, NULL);

#else
    /* load bpp-specific modules */
    switch (pScrn->bitsPerPixel) {
        case 8:
            mod = "cfb";
            reqSym = "cfbScreenInit";
            break;
        case 16:
            mod = "cfb16";
            reqSym = "cfb16ScreenInit";
            break;
        case 32:
            mod = "cfb32";
            reqSym = "cfb32ScreenInit";
            break;
    }

    if (mod && xf86LoadSubModule(pScrn, mod) == NULL) {
        VIAFreeRec(pScrn);
        return FALSE;
    }

    xf86LoaderReqSymbols(reqSym, NULL);
#endif

    if (!pVia->NoAccel) {
        if(!xf86LoadSubModule(pScrn, "xaa")) {
            VIAFreeRec(pScrn);
            return FALSE;
        }
        xf86LoaderReqSymLists(xaaSymbols, NULL);
    }

    if (pVia->hwcursor) {
        if (!xf86LoadSubModule(pScrn, "ramdac")) {
            VIAFreeRec(pScrn);
            return FALSE;
        }
        xf86LoaderReqSymLists(ramdacSymbols, NULL);
    }

    if (pVia->shadowFB) {
        if (!xf86LoadSubModule(pScrn, "shadowfb")) {
            VIAFreeRec(pScrn);
            return FALSE;
        }
        xf86LoaderReqSymLists(shadowSymbols, NULL);
    }

    /* this needs to have pScrn->monitor initialised */
    {
        Gamma zeros = {0.0, 0.0, 0.0};

        if (!xf86SetGamma(pScrn, zeros)) {
	    VIAFreeRec(pScrn);
            return FALSE;
        }
    }

    VIAUnmapMem(pScrn);

    return TRUE;
}


static Bool
VIAEnterVT(int scrnIndex, int flags)
{
    ScrnInfoPtr pScrn = xf86Screens[scrnIndex];
    VIAPtr      pVia = VIAPTR(pScrn);
    vgaHWPtr hwp = VGAHWPTR(pScrn);

    /* FIXME: Rebind AGP memory here */
    VIAFUNC(scrnIndex);
    
    VIASave(pScrn);

    ViaFBEnable(hwp);

    ViaOutputsPower(pScrn, FALSE);

    VIAWriteMode(pScrn, pScrn->currentMode);

    vgaHWUnlock(hwp);

    /* Patch for APM suspend resume, HWCursor has garbage */
    ViaCursorRestore(pScrn); 

#ifdef XF86DRI
    if (pVia->directRenderingEnabled) {
	VIADRIRingBufferInit(pScrn);
	DRIUnlock(screenInfo.screens[scrnIndex]);
    }
#endif

    /* How does this interact with DRI? */
    if (pVia->VQEnable) 
        ViaVQEnable(pScrn);

    ViaOutputsPower(pScrn, TRUE);

    return TRUE;
}


static void VIALeaveVT(int scrnIndex, int flags)
{
    ScrnInfoPtr pScrn = xf86Screens[scrnIndex];
    vgaHWPtr    hwp = VGAHWPTR(pScrn);
    VIAPtr      pVia = VIAPTR(pScrn);

    VIAFUNC(scrnIndex);

#ifdef XF86DRI
    if (pVia->directRenderingEnabled)
	DRILock(pScrn->pScreen, 0);
#endif

    VIAAccelSync(pScrn);


#ifdef XF86DRI
    if (pVia->directRenderingEnabled) {

	/*
	 * Next line apparently helps fix 3D hang on VT switch.
	 * No idea why. Taken from VIA's binary drivers.
	 */

        hwp->writeSeq(hwp, 0x1A, pVia->SavedReg.SR1A | 0x40);

	VIADRIRingBufferCleanup(pScrn); 
    }
#endif

    if (pVia->VQEnable) 
	ViaVQDisable(pScrn);

    ViaCursorStore(pScrn);

    VIARestore(pScrn);

    vgaHWLock(hwp);

}


static void
VIASave(ScrnInfoPtr pScrn)
{
    vgaHWPtr  hwp = VGAHWPTR(pScrn);
    VIAPtr  pVia = VIAPTR(pScrn);
    VIARegPtr  Regs = &pVia->SavedReg;
    int i;

    VIAFUNC(pScrn->scrnIndex);

    if(pVia->IsSecondary)
    {
        DevUnion* pPriv;
        VIAEntPtr pVIAEnt;
        VIAPtr   pVia1;
        vgaHWPtr hwp1;
        pPriv = xf86GetEntityPrivate(pScrn->entityList[0],
              gVIAEntityIndex);
        pVIAEnt = pPriv->ptr;
        hwp1 = VGAHWPTR(pVIAEnt->pPrimaryScrn);
        pVia1 = VIAPTR(pVIAEnt->pPrimaryScrn);
        hwp->SavedReg = hwp1->SavedReg;
        pVia->SavedReg = pVia1->SavedReg;
    }
    else {
        vgaHWProtect(pScrn, TRUE);

        if (xf86IsPrimaryPci(pVia->PciInfo))
            vgaHWSave(pScrn, &hwp->SavedReg, VGA_SR_ALL);
        else
            vgaHWSave(pScrn, &hwp->SavedReg, VGA_SR_MODE);

        /* Unlock Extended Regs */
	hwp->writeSeq(hwp, 0x10, 0x01);
	Regs->SR14 = hwp->readSeq(hwp, 0x14);
	Regs->SR15 = hwp->readSeq(hwp, 0x15);
	Regs->SR16 = hwp->readSeq(hwp, 0x16);
	Regs->SR17 = hwp->readSeq(hwp, 0x17);
	Regs->SR18 = hwp->readSeq(hwp, 0x18);
	Regs->SR19 = hwp->readSeq(hwp, 0x19);
	Regs->SR1A = hwp->readSeq(hwp, 0x1A);
	Regs->SR1B = hwp->readSeq(hwp, 0x1B);
	Regs->SR1C = hwp->readSeq(hwp, 0x1C);
	Regs->SR1D = hwp->readSeq(hwp, 0x1D);
	Regs->SR1E = hwp->readSeq(hwp, 0x1E);
	Regs->SR1F = hwp->readSeq(hwp, 0x1F);

	Regs->SR22 = hwp->readSeq(hwp, 0x22);
	Regs->SR23 = hwp->readSeq(hwp, 0x23);
	Regs->SR24 = hwp->readSeq(hwp, 0x24);
	Regs->SR25 = hwp->readSeq(hwp, 0x25);
	Regs->SR26 = hwp->readSeq(hwp, 0x26);
	Regs->SR27 = hwp->readSeq(hwp, 0x27);
	Regs->SR28 = hwp->readSeq(hwp, 0x28);
	Regs->SR29 = hwp->readSeq(hwp, 0x29);
	Regs->SR2A = hwp->readSeq(hwp, 0x2A);
	Regs->SR2B = hwp->readSeq(hwp, 0x2B);

	Regs->SR2E = hwp->readSeq(hwp, 0x2E);	

	Regs->SR44 = hwp->readSeq(hwp, 0x44);
	Regs->SR45 = hwp->readSeq(hwp, 0x45);
	Regs->SR46 = hwp->readSeq(hwp, 0x46);
	Regs->SR47 = hwp->readSeq(hwp, 0x47);

	Regs->CR13 = hwp->readCrtc(hwp, 0x13);

	Regs->CR32 = hwp->readCrtc(hwp, 0x32);
	Regs->CR33 = hwp->readCrtc(hwp, 0x33);
	Regs->CR34 = hwp->readCrtc(hwp, 0x34);
	Regs->CR35 = hwp->readCrtc(hwp, 0x35);
	Regs->CR36 = hwp->readCrtc(hwp, 0x36);

        ViaOutputsSave(pScrn);

        /* Save LCD control regs */
        for (i = 0; i < 68; i++)
	    Regs->CRTCRegs[i] = hwp->readCrtc(hwp, i + 0x50);

        if (pVia->Chipset == VT3108) {
            Regs->CR97 = hwp->readCrtc(hwp, 0x97);
            Regs->CR99 = hwp->readCrtc(hwp, 0x99);
        }

        vgaHWProtect(pScrn, FALSE);

        /* If we're using secondary for textmode, then, upon restoration,
         * the standard VGA clock set through Misc will ruin the mode.
         */
        if (((Regs->CRTCRegs[0x1A] & 0xC0) == 0x40) && /* CRTC2 active */
            (Regs->CRTCRegs[0x1B] & 0x08) && /* Simultaneous */
            ((hwp->SavedReg.MiscOutReg & 0x0C) != 0x0C)) { /* VGA clock set */
            xf86DrvMsg(pScrn->scrnIndex, X_WARNING, "Console uses both CRTCs"
                       " while depending on VGA clock selection.\n");
            xf86DrvMsg(pScrn->scrnIndex, X_WARNING, "Upon restoration, console "
                       "will not be useful without a workaround.\n");
            xf86DrvMsg(pScrn->scrnIndex, X_WARNING, "This workaround means that"
                       " future VGA mode changes might break.\n");
            hwp->SavedReg.MiscOutReg |= 0x0C;
        }
    }
}

static void 
VIARestore(ScrnInfoPtr pScrn)
{
    vgaHWPtr  hwp = VGAHWPTR(pScrn);
    VIAPtr  pVia = VIAPTR(pScrn);
    VIARegPtr  Regs = &pVia->SavedReg;
    int  i;
    CARD8  tmp;

    VIAFUNC(pScrn->scrnIndex);

    /* Secondary? */

    vgaHWProtect(pScrn, TRUE);
    /* Unlock Extended Regs */
    hwp->writeSeq(hwp, 0x10, 0x01);
    ViaCrtcMask(hwp, 0x47, 0x00, 0x01);

    hwp->writeCrtc(hwp, 0x6A, 0x00);
    hwp->writeCrtc(hwp, 0x6B, 0x00);
    hwp->writeCrtc(hwp, 0x6C, 0x00);
 
    ViaOutputsRestore(pScrn);

    /* restore the standard vga regs */
    if (xf86IsPrimaryPci(pVia->PciInfo))
        vgaHWRestore(pScrn, &hwp->SavedReg, VGA_SR_ALL);
    else
        vgaHWRestore(pScrn, &hwp->SavedReg, VGA_SR_MODE);

    /* restore extended regs */
    hwp->writeSeq(hwp, 0x14, Regs->SR14);
    hwp->writeSeq(hwp, 0x15, Regs->SR15);
    hwp->writeSeq(hwp, 0x16, Regs->SR16);
    hwp->writeSeq(hwp, 0x17, Regs->SR17);
    hwp->writeSeq(hwp, 0x18, Regs->SR18);
    hwp->writeSeq(hwp, 0x19, Regs->SR19);
    hwp->writeSeq(hwp, 0x1A, Regs->SR1A);
    hwp->writeSeq(hwp, 0x1B, Regs->SR1B);
    hwp->writeSeq(hwp, 0x1C, Regs->SR1C);
    hwp->writeSeq(hwp, 0x1D, Regs->SR1D);
    hwp->writeSeq(hwp, 0x1E, Regs->SR1E);
    hwp->writeSeq(hwp, 0x1F, Regs->SR1F);

    hwp->writeSeq(hwp, 0x22, Regs->SR22);
    hwp->writeSeq(hwp, 0x23, Regs->SR23);
    hwp->writeSeq(hwp, 0x24, Regs->SR24);
    hwp->writeSeq(hwp, 0x25, Regs->SR25);
    hwp->writeSeq(hwp, 0x26, Regs->SR26);
    hwp->writeSeq(hwp, 0x27, Regs->SR27);
    hwp->writeSeq(hwp, 0x28, Regs->SR28);
    hwp->writeSeq(hwp, 0x29, Regs->SR29);
    hwp->writeSeq(hwp, 0x2A, Regs->SR2A);
    hwp->writeSeq(hwp, 0x2B, Regs->SR2B);

    hwp->writeSeq(hwp, 0x2E, Regs->SR2E);
    
    hwp->writeSeq(hwp, 0x44, Regs->SR44);
    hwp->writeSeq(hwp, 0x45, Regs->SR45);
    hwp->writeSeq(hwp, 0x46, Regs->SR46);
    hwp->writeSeq(hwp, 0x47, Regs->SR47);

    /* reset dotclocks */
    ViaSeqMask(hwp, 0x40, 0x06, 0x06);
    ViaSeqMask(hwp, 0x40, 0x00, 0x06);
	
    hwp->writeCrtc(hwp, 0x13, Regs->CR13);
    hwp->writeCrtc(hwp, 0x32, Regs->CR32);
    hwp->writeCrtc(hwp, 0x33, Regs->CR33);
    hwp->writeCrtc(hwp, 0x34, Regs->CR34);
    hwp->writeCrtc(hwp, 0x35, Regs->CR35);
    hwp->writeCrtc(hwp, 0x36, Regs->CR36);

    /* Restore LCD control regs */
    for (i = 0; i < 68; i++)
        hwp->writeCrtc(hwp, i + 0x50, Regs->CRTCRegs[i]);

    if (pVia->Chipset == VT3108) {
        hwp->writeCrtc(hwp, 0x97, Regs->CR97);
        hwp->writeCrtc(hwp, 0x99, Regs->CR99);
    }

    /* Re-enable Panel before returning */
    {
        struct ViaOutput *Output = pVia->Outputs;
        
        while (Output) {
            if ((Output->Type == OUTPUT_PANEL) && Output->Power)
                Output->Power(Output, TRUE);

            Output = Output->Next;
        }
    }

    ViaDisablePrimaryFIFO(pScrn);

    /* Reset clock */
    tmp = hwp->readMiscOut(hwp);
    hwp->writeMiscOut(hwp, tmp);

    vgaHWProtect(pScrn, FALSE);
}

/*
 * FIXME: This is a hack to get rid of offending wrongly sized
 * MTRR regions set up by the VIA BIOS. Should be taken care of
 * in the OS support layer.
 */
static void
ViaBrokenMTRRHack(ScrnInfoPtr pScrn)
{
    VIAPtr pVia = VIAPTR(pScrn);
    CARD8 *tmp;

    VIAFUNC(pScrn->scrnIndex);

    tmp = xf86MapPciMem(pScrn->scrnIndex, VIDMEM_MMIO, pVia->PciTag,
                        pVia->FrameBufferBase, pVia->videoRambytes);
    xf86UnMapVidMem(pScrn->scrnIndex, (pointer)tmp, pVia->videoRambytes);

    /* And, as if this wasn't enough, 2.6 series kernels doesn't remove MTRR
     * regions on the first attempt. Try again. */
    tmp = xf86MapPciMem(pScrn->scrnIndex, VIDMEM_MMIO, pVia->PciTag,
                        pVia->FrameBufferBase, pVia->videoRambytes);
    xf86UnMapVidMem(pScrn->scrnIndex, (pointer)tmp, pVia->videoRambytes);
}

static Bool
VIAMapMMIO(ScrnInfoPtr pScrn)
{
    VIAPtr pVia = VIAPTR(pScrn);

    VIAFUNC(pScrn->scrnIndex);

    pVia->FrameBufferBase = pVia->PciInfo->memBase[0];
    pVia->MmioBase = pVia->PciInfo->memBase[1];

    xf86DrvMsg(pScrn->scrnIndex, X_PROBED,
               "mapping MMIO @ 0x%lx with size 0x%x\n",
               pVia->MmioBase, VIA_MMIO_REGSIZE);

    pVia->MapBase = xf86MapPciMem(pScrn->scrnIndex, VIDMEM_MMIO, pVia->PciTag,
                                  pVia->MmioBase, VIA_MMIO_REGSIZE);

    xf86DrvMsg(pScrn->scrnIndex, X_PROBED,
               "mapping BitBlt MMIO @ 0x%lx with size 0x%x\n",
               pVia->MmioBase + VIA_MMIO_BLTBASE, VIA_MMIO_BLTSIZE);

    pVia->BltBase = xf86MapPciMem(pScrn->scrnIndex, VIDMEM_MMIO, pVia->PciTag,
                                  pVia->MmioBase + VIA_MMIO_BLTBASE,
                                  VIA_MMIO_BLTSIZE);

    if (!pVia->MapBase || !pVia->BltBase) {
        xf86DrvMsg(pScrn->scrnIndex, X_ERROR,
                   "Internal error: cound not map registers\n");
        return FALSE;
    }

    /* Set up MMIO vgaHW */
    {
	vgaHWPtr hwp = VGAHWPTR(pScrn);
	CARD8 val;

	vgaHWSetMmioFuncs(hwp, pVia->MapBase, 0x8000);
	
	val = hwp->readEnable(hwp);
	hwp->writeEnable(hwp, val | 0x01);
	
	val = hwp->readMiscOut(hwp);
	hwp->writeMiscOut(hwp, val | 0x01);

        /* Unlock Extended IO Space */
        hwp->writeSeq(hwp, 0x10, 0x01);

        /* Enable MMIO -- probably very VT3122/VT7205 specific  */
        if (pVia->IsSecondary)
            ViaSeqMask(hwp, 0x1A, 0x30, 0x30);
        else
            ViaSeqMask(hwp, 0x1A, 0x60, 0x60);

	vgaHWGetIOBase(hwp);
    }

    return TRUE;
}

/*
 * Gets called at MapFB (ScreenInit) and EnterVTs
 */
static void
ViaFBEnable(vgaHWPtr hwp)
{
    /* Enable writing to all VGA memory planes */
    hwp->writeSeq(hwp, 0x02, 0x0F);

    /* Enable Extended VGA memory */
    hwp->writeSeq(hwp, 0x04, 0x0E);

    /* Enable Extended Memory access */
    ViaSeqMask(hwp, 0x1A, 0x08, 0x08);  
}

/*
 *
 */
static Bool 
ViaMapFB(ScrnInfoPtr pScrn)
{
    VIAPtr pVia = VIAPTR(pScrn);

    VIAFUNC(pScrn->scrnIndex);

    xf86DrvMsg(pScrn->scrnIndex, X_PROBED,
               "mapping framebuffer @ 0x%lx with size 0x%lx\n",
               pVia->FrameBufferBase, pVia->videoRambytes);

    pVia->FBBase = xf86MapPciMem(pScrn->scrnIndex, VIDMEM_FRAMEBUFFER,
                                 pVia->PciTag, pVia->FrameBufferBase,
                                 pVia->videoRambytes);
    if (!pVia->FBBase) {
        xf86DrvMsg(pScrn->scrnIndex, X_ERROR,
                   "Internal error: could not map framebuffer\n");
        return FALSE;
    }

    pVia->FBFreeStart = 
        (pScrn->displayWidth * pScrn->bitsPerPixel >> 3) * pScrn->virtualY;
    pVia->FBFreeEnd = pVia->videoRambytes;

    xf86DrvMsg(pScrn->scrnIndex, X_PROBED,
               "Frame buffer start: %p, free start: 0x%x end: 0x%x\n",
               pVia->FBBase, pVia->FBFreeStart, pVia->FBFreeEnd);

    pScrn->memPhysBase = pVia->PciInfo->memBase[0];

    if (pVia->IsSecondary)
        pScrn->fbOffset = pScrn->videoRam << 10;
    else
        pScrn->fbOffset = 0;

    ViaFBEnable(VGAHWPTR(pScrn));

    /* Stop previous content from disturbing users. */
    memset(VIAPTR(pScrn)->FBBase, 0x00,
           pScrn->virtualX * pScrn->virtualY * (pScrn->bitsPerPixel >> 3));

    return TRUE;
}


static void
VIAUnmapMem(ScrnInfoPtr pScrn)
{
    VIAPtr pVia = VIAPTR(pScrn);

    VIAFUNC(pScrn->scrnIndex);

    /* Disable MMIO */
    ViaSeqMask(VGAHWPTR(pScrn), 0x1A, 0x00, 0x60);

    if (pVia->MapBase)
        xf86UnMapVidMem(pScrn->scrnIndex, (pointer)pVia->MapBase, VIA_MMIO_REGSIZE);

    if (pVia->BltBase)
        xf86UnMapVidMem(pScrn->scrnIndex, (pointer)pVia->BltBase, VIA_MMIO_BLTSIZE);

    if (pVia->FBBase)
        xf86UnMapVidMem(pScrn->scrnIndex, (pointer)pVia->FBBase, pVia->videoRambytes);
}

/*
 * Doesn't do secondary/merged currently.
 */
static void
ViaPaletteLoad(ScrnInfoPtr pScrn, int numColors, int *indices,
               LOCO *colors, VisualPtr pVisual)
{
    vgaHWPtr hwp = VGAHWPTR(pScrn);
    int i, j, index;

    VIAFUNC(pScrn->scrnIndex);

    hwp->enablePalette(hwp);
    hwp->writeDacMask(hwp, 0xFF);

    /* We need the same palette contents for 16 as 24bit, but X doesn't
     * play. X cmap handling is hopelessly intertwined with just about any X
     * subsystem you can care to name. So we just space out RGB values over the
     * 256*3 lut. This is quite unlike me, but even i draw a line somewhere.
     * Digging out the DIX is somewhere. */
    switch (pScrn->bitsPerPixel) {
    case 16:
        for (i = 0; i < numColors; i++) {
            index = indices[i];
            hwp->writeDacWriteAddr(hwp, index * 4);

            for (j = 0; j < 4; j++) {
                hwp->writeDacData(hwp, colors[index/2].red);
                hwp->writeDacData(hwp, colors[index].green);
                hwp->writeDacData(hwp, colors[index/2].blue);
            }
        }
        break;
    case 8:
    case 24:
    case 32:
        for (i = 0; i < numColors; i++) {
            index = indices[i];
            hwp->writeDacWriteAddr(hwp, index);

            hwp->writeDacData(hwp, colors[index].red);
            hwp->writeDacData(hwp, colors[index].green);
            hwp->writeDacData(hwp, colors[index].blue);
        }
        break;
    default:
        xf86DrvMsg(pScrn->scrnIndex, X_ERROR, "%s: Unsupported bitdepth: %d\n",
                   __func__, pScrn->bitsPerPixel);
        break;
    }

    hwp->disablePalette(hwp);
}

/*
 * standard VGA Overscan register.
 */
static void
ViaOverScan(ScrnInfoPtr pScrn, int overscan)
{
    vgaHWPtr hwp = VGAHWPTR(pScrn);

    VIAFUNC(pScrn->scrnIndex);

    switch (pScrn->bitsPerPixel) {
    case 16:
        hwp->writeAttr(hwp, 0x11, overscan * 4);
        break;
    case 8:
    case 24:
    case 32:
        hwp->writeAttr(hwp, 0x11, overscan);
        break;
    default:
        xf86DrvMsg(pScrn->scrnIndex, X_ERROR, "%s: Unsupported bitdepth: %d\n",
                   __func__, pScrn->bitsPerPixel);
        break;
    }
}

/*
 *
 */
static Bool
ViaWrapvgaHWSaveScreen(ScreenPtr pScreen, int mode)
{
    return vgaHWSaveScreen(pScreen, mode);
}

/*
 *
 */
static Bool 
VIAScreenInit(int scrnIndex, ScreenPtr pScreen, int argc, char **argv)
{
    ScrnInfoPtr pScrn = xf86Screens[scrnIndex];
    vgaHWPtr hwp = VGAHWPTR(pScrn);
    VIAPtr pVia = VIAPTR(pScrn);

    VIAFUNC(scrnIndex);

    ViaBrokenMTRRHack(pScrn);
    if (!VIAMapMMIO(pScrn))
        return FALSE;

    VIASave(pScrn);

    if (!ViaMapFB(pScrn))
        return FALSE;

    vgaHWUnlock(hwp);

    vgaHWSaveScreen(pScreen, SCREEN_SAVER_ON);

    ViaOutputsPower(pScrn, FALSE);

    VIAWriteMode(pScrn, pScrn->currentMode);

    pScrn->AdjustFrame(scrnIndex, pScrn->frameX0, pScrn->frameY0, 0);

    ViaDebug(scrnIndex, "- Blanked\n");

    miClearVisualTypes();

    if (pScrn->bitsPerPixel > 8 && !pVia->IsSecondary) {
        if (!miSetVisualTypes(pScrn->depth, TrueColorMask,
                              pScrn->rgbBits, pScrn->defaultVisual))
            return FALSE;
        if (!miSetPixmapDepths())
            return FALSE;
    } else {
        if (!miSetVisualTypes(pScrn->depth, miGetDefaultVisualMask(pScrn->depth),
                              pScrn->rgbBits, pScrn->defaultVisual))
            return FALSE;
        if (!miSetPixmapDepths())
            return FALSE;
    }

#ifdef XF86DRI
    pVia->directRenderingEnabled = VIADRIScreenInit(pScrn, pScreen);
    if (!pVia->directRenderingEnabled) {
        xf86DrvMsg(pScrn->scrnIndex, X_INFO, "[drm] Not using DMA for "
                   "copying to FB (no DRI).\n");
        pVia->UseDMACopy = FALSE;
    }
#endif

    ViaDebug(scrnIndex, "- Visuals set up\n");

    if (!VIAInternalScreenInit(pScrn, pScreen))
	return FALSE;

    xf86SetBlackWhitePixels(pScreen);
    ViaDebug(scrnIndex, "- B & W\n");

    if (pScrn->bitsPerPixel > 8) {
        VisualPtr visual;

        visual = pScreen->visuals + pScreen->numVisuals;
        while (--visual >= pScreen->visuals) {
            if ((visual->class | DynamicClass) == DirectColor) {
                visual->offsetRed = pScrn->offset.red;
                visual->offsetGreen = pScrn->offset.green;
                visual->offsetBlue = pScrn->offset.blue;
                visual->redMask = pScrn->mask.red;
                visual->greenMask = pScrn->mask.green;
                visual->blueMask = pScrn->mask.blue;
            }
        }
    }

#ifdef USE_FB
    /* must be after RGB ordering fixed */
    fbPictureInit(pScreen, 0, 0);
#endif

    if (!pVia->NoAccel) {
        VIAInitAccel(pScrn, pScreen);
    } 
#ifdef X_USE_LINEARFB
    else {
	/*
	 * This is needed because xf86InitFBManagerLinear in VIAInitLinear
	 * needs xf86InitFBManager to have been initialized, and 
	 * xf86InitFBManager needs at least one line of free memory to
	 * work. This is only for Xv in Noaccel part, and since Xv is in some
	 * sense accelerated, it might be a better idea to disable it
	 * altogether.
	 */ 
        BoxRec AvailFBArea;

        AvailFBArea.x1 = 0;
        AvailFBArea.y1 = 0;
        AvailFBArea.x2 = pScrn->displayWidth;
        AvailFBArea.y2 = pScrn->virtualY + 1;
	/* 
	 * Update FBFreeStart also for other memory managers, since 
	 * we steal one line to make xf86InitFBManager work.
	 */
	pVia->FBFreeStart = (AvailFBArea.y2 + 1) * pVia->Bpl;
	xf86InitFBManager(pScreen, &AvailFBArea);	
    }
#endif /* X_USE_LINEARFB */

    miInitializeBackingStore(pScreen);
    xf86SetBackingStore(pScreen);
    /*xf86SetSilkenMouse(pScreen);*/
    ViaDebug(scrnIndex, "- Backing store set up\n");

    if(!pVia->shadowFB)         /* hardware cursor needs to wrap this layer */
        VIADGAInit(pScrn, pScreen);

    miDCInitialize(pScreen, xf86GetPointerScreenFuncs());
    ViaDebug(scrnIndex, "- SW cursor set up\n");

    if (pVia->hwcursor && !ViaCursorInit(pScrn, pScreen))
        xf86DrvMsg(pScrn->scrnIndex, X_ERROR,
                   "Hardware cursor initialization failed\n");

    if (pVia->shadowFB)
	ViaShadowFBInit(pScrn, pScreen);

    if (!miCreateDefColormap(pScreen))
        return FALSE;

    if (!xf86HandleColormaps(pScreen, 256, 8, ViaPaletteLoad, ViaOverScan,
                             CMAP_RELOAD_ON_MODE_SWITCH
                             | CMAP_PALETTED_TRUECOLOR))
        return FALSE;

    vgaHWBlankScreen(pScrn, TRUE);

    pVia->CloseScreen = pScreen->CloseScreen;
    pScreen->SaveScreen = ViaWrapvgaHWSaveScreen;
    pScreen->CloseScreen = VIACloseScreen;

    xf86DPMSInit(pScreen, VIADPMS, 0);
    ViaDebug(scrnIndex, "- DPMS set up\n");

    ViaDebug(scrnIndex, "- Color maps etc. set up\n");

#ifdef XF86DRI
    if (pVia->directRenderingEnabled)
	pVia->directRenderingEnabled = VIADRIFinishScreenInit(pScrn, pScreen);

    if (pVia->directRenderingEnabled)
        xf86DrvMsg(scrnIndex, X_INFO, "direct rendering enabled\n");
    else {
        xf86DrvMsg(scrnIndex, X_INFO, "direct rendering disabled\n");
	ViaMemInit(pScrn, pScreen);
    }
#else    
    ViaMemInit(pScrn, pScreen);
#endif

#ifdef XvExtension
    ViaVideoInit(pScrn, pScreen);
#endif

    ViaOutputsPower(pScrn, TRUE);

    if (serverGeneration == 1)
        xf86ShowUnusedOptions(pScrn->scrnIndex, pScrn->options);

    if (pVia->PrintVGARegs)
	ViaVgaPrintRegs(pScrn, __func__);
    
    if (pVia->PrintTVRegs)
	ViaOutputsPrintRegs(pScrn, __func__);

    ViaDebug(scrnIndex, "- Done\n");
    return TRUE;
}


static int 
VIAInternalScreenInit(ScrnInfoPtr pScrn, ScreenPtr pScreen)
{
    VIAPtr          pVia = VIAPTR(pScrn);
    int             width, height, displayWidth;
    unsigned char   *FBStart;

    VIAFUNC(pScrn->scrnIndex);

    displayWidth = pScrn->displayWidth;

    if (pVia->rotate) {
        height = pScrn->virtualX;
        width = pScrn->virtualY;
    } else {
        width = pScrn->virtualX;
        height = pScrn->virtualY;
    }

    if (pVia->shadowFB) {
        pVia->ShadowPitch = BitmapBytePad(pScrn->bitsPerPixel * width);
        pVia->ShadowPtr = xalloc(pVia->ShadowPitch * height);
        displayWidth = pVia->ShadowPitch / (pScrn->bitsPerPixel >> 3);
        FBStart = pVia->ShadowPtr;
    }
    else {
        pVia->ShadowPtr = NULL;
        FBStart = pVia->FBBase;
    }

#ifdef USE_FB
    return fbScreenInit(pScreen, FBStart, width, height,
			pScrn->xDpi, pScrn->yDpi, displayWidth,
			pScrn->bitsPerPixel);
#else
    switch (pScrn->bitsPerPixel) {
    case 8:
        return cfbScreenInit(pScreen, FBStart, width, height, pScrn->xDpi,
			     pScrn->yDpi, displayWidth);
    case 16:
        return cfb16ScreenInit(Screen, FBStart, width, height, pScrn->xDpi,
			       pScrn->yDpi, displayWidth);
    case 32:
        return cfb32ScreenInit(pScreen, FBStart, width, height, pScrn->xDpi,
			       pScrn->yDpi, displayWidth);
    default:
        xf86DrvMsg(pScrn->scrnIndex, X_ERROR, "Internal error: invalid bpp (%d) in "
		   "VIAInternalScreenInit\n", pScrn->bitsPerPixel);
        return FALSE;
    }
#endif
}

static void
VIAWriteMode(ScrnInfoPtr pScrn, DisplayModePtr mode)
{
    VIAPtr pVia = VIAPTR(pScrn);

    VIAFUNC(pScrn->scrnIndex);

    /* this is waaaay too simplistic and even plain wrong. */
#ifdef _VIA_VIDEO_H_
    pVia->Swov->Allowed = FALSE;
#endif

    pScrn->vtSema = TRUE;

    /* FIXME - need DRI lock for this bit - see i810 */
    if (!pVia->IsSecondary)
	ViaModePrimary(pScrn, mode);
    else
	ViaModeSecondary(pScrn, mode);

    /* Enable the graphics engine. */
    if (!pVia->NoAccel)
	VIAInitialize2DEngine(pScrn);
    
#ifdef XF86DRI
    VIAInitialize3DEngine(pScrn);
#endif 

    VIAAdjustFrame(pScrn->scrnIndex, pScrn->frameX0, pScrn->frameY0, 0);
}


static Bool VIACloseScreen(int scrnIndex, ScreenPtr pScreen)
{
    ScrnInfoPtr pScrn = xf86Screens[scrnIndex];
    vgaHWPtr    hwp = VGAHWPTR(pScrn);
    VIAPtr      pVia = VIAPTR(pScrn);

    VIAFUNC(scrnIndex);

    /* Is the display currently visible ? */
    if(pScrn->vtSema)
    {

#ifdef XF86DRI
        if (pVia->directRenderingEnabled)
            DRILock(pScreen, 0);
#endif
        /* Wait Hardware Engine idle to exit graphical mode */
        VIAAccelSync(pScrn);
 

#ifdef XF86DRI
	/* Fix 3D Hang after X restart */

	if (pVia->directRenderingEnabled)
	    hwp->writeSeq(hwp, 0x1A, pVia->SavedReg.SR1A | 0x40);
#endif 

        if (pVia->Cursor)
            pVia->Cursor->HideCursor(pScrn);

        if (pVia->VQEnable)
	    ViaVQDisable(pScrn);
    }
#ifdef XF86DRI
    if (pVia->directRenderingEnabled)
	VIADRICloseScreen(pScrn, pScreen);
#endif
    if (pVia->AccelInfoRec) {
        XAADestroyInfoRec(pVia->AccelInfoRec);
        pVia->AccelInfoRec = NULL;
    }
    if (pVia->Cursor) {
        xf86DestroyCursorInfoRec(pVia->Cursor);
        pVia->Cursor = NULL;
    }
    if (pVia->ShadowPtr) {
        xfree(pVia->ShadowPtr);
        pVia->ShadowPtr = NULL;
    }
    if (pVia->DGAModes) {
        xfree(pVia->DGAModes);
        pVia->DGAModes = NULL;
    }

    if (pScrn->vtSema) {
        VIARestore(pScrn);
	vgaHWLock(hwp);
        VIAUnmapMem(pScrn);
    }
    pScrn->vtSema = FALSE;
    pScreen->CloseScreen = pVia->CloseScreen;
    return (*pScreen->CloseScreen)(scrnIndex, pScreen);
}

/*
 * This only gets called when a screen is being deleted.  It does not
 * get called routinely at the end of a server generation.
 */
static void VIAFreeScreen(int scrnIndex, int flags)
{
    VIAFUNC(scrnIndex);

    VIAFreeRec(xf86Screens[scrnIndex]);

    if (xf86LoaderCheckSymbol("vgaHWFreeHWRec"))
        vgaHWFreeHWRec(xf86Screens[scrnIndex]);
}

static void
VIAAdjustFrame(int scrnIndex, int x, int y, int flags)
{
    ScrnInfoPtr pScrn = xf86Screens[scrnIndex];
    vgaHWPtr    hwp = VGAHWPTR(pScrn);
    VIAPtr      pVia = VIAPTR(pScrn);
    CARD32      Base;

    /* VIAFUNC(scrnIndex); */ /* noisy */

    Base = (y * pScrn->displayWidth + x) * (pScrn->bitsPerPixel / 8);

    /* now program the start address registers */
    if (pVia->IsSecondary) {
	Base = (Base + pScrn->fbOffset) >> 3;
	ViaCrtcMask(hwp, 0x62, (Base & 0x7F) << 1 , 0xFE);
	hwp->writeCrtc(hwp, 0x63, (Base & 0x7F80) >>  7);
	hwp->writeCrtc(hwp, 0x64, (Base & 0x7F8000) >>  15);
    }
    else {
        Base = Base >> 1;
        hwp->writeCrtc(hwp, 0x0C, (Base & 0xFF00) >> 8);
	hwp->writeCrtc(hwp, 0x0D, Base & 0xFF);
	hwp->writeCrtc(hwp, 0x34, (Base & 0xFF0000) >> 16);
	/* VT3122A doesn't implement this, but luckily it simply doesn't care.
         * Value is properly limited in PreInit anyway. */
	ViaCrtcMask(hwp, 0x48, Base >> 24, 0x03);
    }
}


static Bool
VIASwitchMode(int scrnIndex, DisplayModePtr mode, int flags)
{
    ScrnInfoPtr pScrn = xf86Screens[scrnIndex];
    VIAPtr      pVia = VIAPTR(pScrn);
    
    VIAFUNC(scrnIndex);

    ViaOutputsPower(pScrn, FALSE);

#ifdef XF86DRI
    if (pVia->directRenderingEnabled)
	DRILock(pScrn->pScreen, 0);
#endif
    
    VIAAccelSync(pScrn);
    
#ifdef XF86DRI
    if (pVia->directRenderingEnabled)
	VIADRIRingBufferCleanup(pScrn); 
#endif
    
    if (pVia->VQEnable)
	ViaVQDisable(pScrn);
    
    VIAWriteMode(pScrn, mode);

    if (pVia->VQEnable)
	ViaVQEnable(pScrn);

#ifdef XF86DRI
    if (pVia->directRenderingEnabled) {
	VIADRIRingBufferInit(pScrn);
	DRIUnlock(screenInfo.screens[scrnIndex]);
    }
#endif

    ViaOutputsPower(pScrn, TRUE);

    return TRUE;
}

/*
 *
 */
static void VIADPMS(ScrnInfoPtr pScrn, int mode, int flags)
{
    VIAFUNC(pScrn->scrnIndex);

    switch (mode) {
    case DPMSModeOn:
        ViaOutputsPower(pScrn, TRUE);
        break;
    case DPMSModeStandby:
    case DPMSModeSuspend:
    case DPMSModeOff:
        ViaOutputsPower(pScrn, FALSE);
        break;
    default:
        xf86DrvMsg(pScrn->scrnIndex, X_ERROR, "Invalid DPMS mode %d\n", mode);
        break;
    }
    return;
}

#ifdef XF86DRI
void
VIAInitialize3DEngine(ScrnInfoPtr pScrn)
{
    VIAPtr  pVia = VIAPTR(pScrn);
    int i;

    VIAFUNC(pScrn->scrnIndex);

    /*
     * FIXME: Reinitializing the 3D engine after VT switch seems to be the way
     * to go to avoid hangs and rendering errors. But what happens with secondary?
     * Test this for a while, but keep the old code in case...
     */

    if (1 /*!pVia->sharedData->b3DRegsInitialized */)
    {

        VIASETREG(0x43C, 0x00010000);

        for (i = 0; i <= 0x7D; i++)
        {
            VIASETREG(0x440, (CARD32) i << 24);
        }

        VIASETREG(0x43C, 0x00020000);

        for (i = 0; i <= 0x94; i++)
        {
            VIASETREG(0x440, (CARD32) i << 24);
        }

        VIASETREG(0x440, 0x82400000);

        VIASETREG(0x43C, 0x01020000);


        for (i = 0; i <= 0x94; i++)
        {
            VIASETREG(0x440, (CARD32) i << 24);
        }

        VIASETREG(0x440, 0x82400000);
        VIASETREG(0x43C, 0xfe020000);

        for (i = 0; i <= 0x03; i++)
        {
            VIASETREG(0x440, (CARD32) i << 24);
        }

        VIASETREG(0x43C, 0x00030000);

        for (i = 0; i <= 0xff; i++)
        {
            VIASETREG(0x440, 0);
        }
        VIASETREG(0x43C, 0x00100000);
        VIASETREG(0x440, 0x00333004);
        VIASETREG(0x440, 0x10000002);
        VIASETREG(0x440, 0x60000000);
        VIASETREG(0x440, 0x61000000);
        VIASETREG(0x440, 0x62000000);
        VIASETREG(0x440, 0x63000000);
        VIASETREG(0x440, 0x64000000);

        VIASETREG(0x43C, VIA_REG_VIRTUAL_QUEUE);

        if (pVia->ChipRev >= 3 )
            VIASETREG(0x440,0x40008c0f);
        else
            VIASETREG(0x440,0x4000800f);

        VIASETREG(0x440,0x44000000);
        VIASETREG(0x440,0x45080C04);
        VIASETREG(0x440,0x46800408);
        VIASETREG(0x440,0x50000000); /* WTF!!! */
        VIASETREG(0x440,0x51000000); /* Why are we killing VQ sizing? */
        VIASETREG(0x440,0x52000000);
        VIASETREG(0x440,0x53000000);

        pVia->sharedData->b3DRegsInitialized = 1;
        xf86DrvMsg(pScrn->scrnIndex, X_INFO,
            "3D Engine has been initialized.\n");
    }
    
    VIASETREG(0x43C,VIA_REG_VIRTUAL_QUEUE);
    VIASETREG(0x440,0x08000001);
    VIASETREG(0x440,0x0A000183);
    VIASETREG(0x440,0x0B00019F);
    VIASETREG(0x440,0x0C00018B);
    VIASETREG(0x440,0x0D00019B);
    VIASETREG(0x440,0x0E000000);
    VIASETREG(0x440,0x0F000000);
    VIASETREG(0x440,0x10000000);
    VIASETREG(0x440,0x11000000);
    VIASETREG(0x440,0x20000000);
}
#endif


/*
 * Since X lacks anything of the sort.
 * Used by VIAFUNC macro.
 */
void
ViaDebug(int scrnIndex, const char *format, ...)
{
    va_list ap;

    va_start(ap, format);
    xf86VDrvMsgVerb(scrnIndex, X_INFO, LOG_DEBUG, format, ap);
    va_end(ap);

}
