/*
 * SCIM Bridge
 *
 * Copyright (c) 2006 Ryo Dairiki <ryo-dairiki@users.sourceforge.net>
 *
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.*
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.See the
 * GNU Lesser General Public License for more details.*
 * You should have received a copy of the GNU Lesser General Public
 * License along with this program; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place, Suite 330,
 * Boston, MA  02111-1307  USA
 */

#include <assert.h>
#include <errno.h>
#include <signal.h>
#include <string.h>
#include <unistd.h>

#include <sys/select.h>
#include <sys/time.h>
#include <sys/types.h>

#include <list>
#include <vector>

#define Uses_SCIM_BACKEND
#define Uses_SCIM_CONFIG
#define Uses_SCIM_CONFIG_MODULE
#define Uses_SCIM_CONFIG_PATH
#define Uses_SCIM_EVENT
#define Uses_SCIM_FRONTEND
#define Uses_SCIM_PANEL_CLIENT
#define Uses_SCIM_HOTKEY
#define Uses_SCIM_IMENGINE
#define Uses_SCIM_IMENGINE_MODULE

#include <scim.h>

#include "scim-bridge-agent.h"
#include "scim-bridge-agent-accept-listener.h"
#include "scim-bridge-agent-client-listener.h"
#include "scim-bridge-agent-imcontext.h"
#include "scim-bridge-agent-interruption-listener.h"
#include "scim-bridge-agent-panel-listener.h"
#include "scim-bridge-agent-protected.h"
#include "scim-bridge-agent-signal-listener.h"
#include "scim-bridge-agent-socket-client.h"

#include "scim-bridge-debug.h"
#include "scim-bridge-display.h"
#include "scim-bridge-output.h"
#include "scim-bridge-path.h"

using std::endl;
using std::ifstream;
using std::list;
using std::ofstream;
using std::vector;

using namespace scim;

/* Macros */
#ifndef SCIM_KEYBOARD_ICON_FILE
#define SCIM_KEYBOARD_ICON_FILE (SCIM_ICONDIR "/keyboard.png")
#endif


/* Static Constants */
static const int LOCKFILE_UPDATE_INTERVAL = 60 * 60;


/* Class definition */
class ScimBridgeAgentImpl: public ScimBridgeAgent, public ScimBridgeAgentProtected
{

    public:

        ScimBridgeAgentImpl ();
        ~ScimBridgeAgentImpl ();

        retval_t launch ();
        void set_noexit_enabled (bool enabled);
        void set_standalone_enabled (bool enabled);

        /* Semi public funcitons */
        retval_t initialize ();
        retval_t finalize ();

        ScimBridgeAgentPanelListener *get_panel_listener (const ScimBridgeDisplay *new_display);

        void interrupt ();

        void quit ();
        void load_config ();
        void save_config ();

        void add_client_listener (ScimBridgeAgentClientListener *client_listener);
        void remove_client_listener (ScimBridgeAgentClientListener *client_listener);

        bool filter_hotkeys (scim_bridge_imcontext_id_t imcontext_id, const KeyEvent &key_event);
        virtual bool filter_key_event (scim_bridge_imcontext_id_t imcontext_id, const KeyEvent &key_event);

        void request_factory_menu ();

        scim_bridge_imcontext_id_t alloc_imcontext (ScimBridgeAgentClientListener *client_listener);

        void free_imcontext (scim_bridge_imcontext_id_t imcontext_id, const ScimBridgeAgentClientListener *client_listener);

        void reset_imcontext (scim_bridge_imcontext_id_t imcontext_id);

        void set_cursor_location (scim_bridge_imcontext_id_t imcontext_id, int cursor_x, int cursor_y);

        void set_preedit_mode (scim_bridge_imcontext_id_t imcontext_id, scim_bridge_preedit_mode_t preedit_mode);

        void change_focus (scim_bridge_imcontext_id_t imcontext_id, bool focus_in);

    private:

        bool noexit_enabled;
        bool standalone_enabled;

        bool active;

        list<ScimBridgeAgentSocketClient*> clients;
        size_t client_app_count;

        String scim_language;

        ConfigModule *scim_config_module;
        ConfigPointer scim_config;
        BackEndPointer scim_backend;

        KeyboardLayout scim_keyboard_layout;

        FrontEndHotkeyMatcher scim_frontend_hotkey_matcher;
        IMEngineHotkeyMatcher scim_imengine_hotkey_matcher;

        ScimBridgeAgentAcceptListener *accept_listener;
        ScimBridgeAgentInterruptionListener *interruption_listener;
        ScimBridgeAgentPanelListener *panel_listener;
        ScimBridgeAgentSignalListener *signal_listener;

        ScimBridgeDisplay *display;

        retval_t initialize_scim ();
        retval_t finalize_scim ();

        retval_t run_event_loop ();

        retval_t check_lockfile ();
        retval_t update_lockfile (pid_t pid);
        retval_t destroy_lockfile ();

        void slot_reload_config (const ConfigPointer &config);

};

/* Helper functions */
static bool check_socket_frontend ()
{
    SocketAddress address;
    SocketClient client;

    uint32 magic;

    address.set_address (scim_get_default_socket_frontend_address ());

    if (!client.connect (address))
        return false;

    if (!scim_socket_open_connection (magic, "ConnectionTester", "SocketFrontEnd", client, 1000)) {
        return false;
    } else {
        return true;
    }
}


/* Implimentations */
ScimBridgeAgent *ScimBridgeAgent::alloc ()
{
    ScimBridgeAgentImpl *agent = new ScimBridgeAgentImpl ();
    if (agent->initialize ()) {
        delete agent;
        return NULL;
    } else {
        return agent;
    }
}


ScimBridgeAgentImpl::ScimBridgeAgentImpl ():
active (true), noexit_enabled (false), standalone_enabled (false), client_app_count (0),
accept_listener (NULL), interruption_listener (NULL), panel_listener (NULL), signal_listener (NULL), display (NULL)
{
}


ScimBridgeAgentImpl::~ScimBridgeAgentImpl ()
{
    finalize ();
}


void ScimBridgeAgentImpl::set_noexit_enabled (bool enabled)
{
    noexit_enabled = enabled;
}


void ScimBridgeAgentImpl::set_standalone_enabled (bool enabled)
{
    standalone_enabled = enabled;
}


retval_t ScimBridgeAgentImpl::run_event_loop ()
{
    scim_bridge_pdebugln (5, "run_event_loop ()");

    timeval lockfile_timestamp;
    lockfile_timestamp.tv_sec = 0;
    lockfile_timestamp.tv_usec = 0;

    while (active) {

        int max_fd = -1;

        fd_set read_set;
        fd_set write_set;
        fd_set error_set;
        FD_ZERO (&read_set);
        FD_ZERO (&write_set);
        FD_ZERO (&error_set);
        for (list<ScimBridgeAgentSocketClient*>::iterator i = clients.begin (); i != clients.end ();) {
            ScimBridgeAgentSocketClient *client = *i;

            const scim_bridge_agent_event_type_t triggers = client->get_trigger_events ();
            const int socket_fd = client->get_socket_fd ();
            if (socket_fd < 0) {
                if (triggers & SCIM_BRIDGE_AGENT_EVENT_ERROR) {
                    if (!client->handle_event (SCIM_BRIDGE_AGENT_EVENT_ERROR)) {                        
                        i = clients.erase (i);
                        delete client;
                    }
                }
            } else {
                if (socket_fd > max_fd) max_fd = socket_fd;
                if (triggers & SCIM_BRIDGE_AGENT_EVENT_READ) {
                    scim_bridge_pdebugln (1, "FD (%d) is registred as a reading socket", socket_fd);
                    FD_SET (socket_fd, &read_set);
                }
                if (triggers & SCIM_BRIDGE_AGENT_EVENT_WRITE) {
                    FD_SET (socket_fd, &write_set);
                    scim_bridge_pdebugln (1, "FD (%d) is registred as a writing socket", socket_fd);
                }
                if (triggers & SCIM_BRIDGE_AGENT_EVENT_ERROR) {
                    FD_SET (socket_fd, &error_set);
                    scim_bridge_pdebugln (1, "FD (%d) is registred as a error socket", socket_fd);
                }
                ++i;
            }
        }

        timeval current_time;
        gettimeofday (&current_time, NULL);

        int past_sec = (current_time.tv_sec - lockfile_timestamp.tv_sec) + (current_time.tv_usec - lockfile_timestamp.tv_usec) / 1000000;
        if (past_sec > LOCKFILE_UPDATE_INTERVAL) {
            update_lockfile (getpid ());
            lockfile_timestamp = current_time;
            past_sec = 0;
        }

        scim_bridge_pdebugln (2, "Waiting for an event...");
        timeval remain_time;
        remain_time.tv_sec = LOCKFILE_UPDATE_INTERVAL - past_sec;
        remain_time.tv_usec = 0;
        const int retval = select (max_fd + 1, &read_set, &write_set, &error_set, &remain_time);
        if (retval < 0 && errno != EINTR) {
            scim_bridge_perrorln ("An exception occurred at selecting the sockets: %s", strerror (errno));
            abort ();
        }

        const bool interrupted = interruption_listener->is_interrupted ();
        interruption_listener->clear_interruption ();
        if (interrupted) scim_bridge_pdebugln (3, "Caught an interruption");

        for (list<ScimBridgeAgentSocketClient*>::iterator i = clients.begin (); i != clients.end ();) {
            ScimBridgeAgentSocketClient *client = *i;

            const int socket_fd = client->get_socket_fd ();
            const scim_bridge_agent_event_type_t triggers = client->get_trigger_events ();

            scim_bridge_agent_event_type_t events = SCIM_BRIDGE_AGENT_EVENT_NONE;

            if (FD_ISSET (socket_fd, &read_set) && (triggers & SCIM_BRIDGE_AGENT_EVENT_READ)) {
                if (events == SCIM_BRIDGE_AGENT_EVENT_NONE) {
                    scim_bridge_pdebug (2, "Invoked triggers: READ");
                } else {
                    scim_bridge_pdebug (2, ", READ");
                }
                events |= SCIM_BRIDGE_AGENT_EVENT_READ;
            }
            if (FD_ISSET (socket_fd, &write_set) && (triggers & SCIM_BRIDGE_AGENT_EVENT_WRITE)) {
                if (events == SCIM_BRIDGE_AGENT_EVENT_NONE) {
                    scim_bridge_pdebug (2, "Invoked triggers: WRITE");
                } else {
                    scim_bridge_pdebug (2, ", WRITE");
                }
                events |= SCIM_BRIDGE_AGENT_EVENT_WRITE;
            }
            if (FD_ISSET (socket_fd, &error_set) && (triggers & SCIM_BRIDGE_AGENT_EVENT_ERROR)) {
                if (events == SCIM_BRIDGE_AGENT_EVENT_NONE) {
                    scim_bridge_pdebug (2, "Invoked triggers: ERROR");
                } else {
                    scim_bridge_pdebug (2, ", ERROR");
                }
                events |= SCIM_BRIDGE_AGENT_EVENT_ERROR;
            }
            if (interrupted && (triggers & SCIM_BRIDGE_AGENT_EVENT_INTERRUPT)) {
                if (events == SCIM_BRIDGE_AGENT_EVENT_NONE) {
                    scim_bridge_pdebug (2, "Invoked triggers: INTERRUPT");
                } else {
                    scim_bridge_pdebug (2, ", INTERRUPT");
                }
                events |= SCIM_BRIDGE_AGENT_EVENT_INTERRUPT;
            }

            if (events != SCIM_BRIDGE_AGENT_EVENT_NONE) {
                scim_bridge_pdebug (2, "\n");
                if (!client->handle_event (events)) {
                    i = clients.erase (i);
                    delete client;
                    continue;
                }
            }
            ++i;
        }
    }

    return RETVAL_SUCCEEDED;
}


retval_t ScimBridgeAgentImpl::launch ()
{
    scim_bridge_pdebugln (5, "launch ()");

    if (!standalone_enabled) {
        scim_bridge_pdebugln (5, "Daemonizing...");
        const pid_t pid = fork ();

        if (pid < 0) {
            scim_bridge_perrorln ("Cannot fork myself: %s", strerror (errno));
            return RETVAL_FAILED;
        } else if (pid > 0) {
            // This is the parent process.
            if (update_lockfile (pid)) {
                scim_bridge_perrorln ("The cannot create the lockfile...");
                return RETVAL_FAILED;
            }

            _exit (0);
        } else {
            // This is the child process.
            if (chdir ("/")) {
                scim_bridge_perrorln ("Cannot change the working directory: %s", strerror (errno));
                return RETVAL_FAILED;
            }
            close (STDIN_FILENO);
            close (STDOUT_FILENO);
            close (STDERR_FILENO);
        }

        scim_bridge_pdebugln (5, "Daemonize done");

    } else {

        if (update_lockfile (getpid ())) {
            scim_bridge_perrorln ("Cannot create the lockfile...");
            return RETVAL_FAILED;
        }

    }

    run_event_loop ();
    destroy_lockfile ();
    return RETVAL_SUCCEEDED;
}


retval_t ScimBridgeAgentImpl::check_lockfile ()
{
    scim_bridge_pdebugln (8, "Checking the lockfile...");
    const char *lockfile_path = scim_bridge_path_get_lockfile ();
    ifstream input_stream (lockfile_path);

    pid_t pid = -1;
    input_stream >> pid;

    input_stream.close ();

    if (input_stream.good () && pid > 0) {
        scim_bridge_pdebugln (5, "Pinging for the process: pid = %d", pid);
        if (kill (pid, 0) == 0) {
            scim_bridge_println ("Another agent is running...");
            return RETVAL_FAILED;
        }
    }

    return RETVAL_SUCCEEDED;
}


retval_t ScimBridgeAgentImpl::update_lockfile (pid_t pid)
{
    scim_bridge_pdebugln (2, "Updating the lockfile...");

    const char *lockfile_path = scim_bridge_path_get_lockfile ();
    ofstream output_stream (lockfile_path);

    output_stream << pid << endl;
    output_stream.close ();

    if (!output_stream.good ()) {
        scim_bridge_perrorln ("Cannot update the lockfile");
        return RETVAL_FAILED;
    } else {
        scim_bridge_pdebugln (7, "The lockfile is updated.");
        return RETVAL_SUCCEEDED;
    }
}


retval_t ScimBridgeAgentImpl::destroy_lockfile ()
{
    scim_bridge_pdebugln (7, "Destroying the lockfile...");

    const char *lockfile_path = scim_bridge_path_get_lockfile ();

    if (unlink (lockfile_path)) {
        scim_bridge_perrorln ("Cannot destroy the lockfile");
        return RETVAL_FAILED;
    } else {
        scim_bridge_perrorln ("The lockfile is destroied");
        return RETVAL_SUCCEEDED;
    }
}


retval_t ScimBridgeAgentImpl::initialize ()
{
    display = scim_bridge_alloc_display ();
    if (scim_bridge_display_fetch_current (display)) {
        scim_bridge_perrorln ("Failed to allocate the current display");
        return RETVAL_FAILED;
    }

    if (check_lockfile ()) return RETVAL_FAILED;

    if (initialize_scim ()) {
        scim_bridge_perrorln ("Failed to initialize scim");
        return RETVAL_FAILED;
    }

    scim_bridge_pdebugln (4, "Loading configurations...");
    slot_reload_config (scim_config);
    scim_config->signal_connect_reload (slot (this, &ScimBridgeAgentImpl::slot_reload_config));

    interruption_listener = ScimBridgeAgentInterruptionListener::alloc ();
    if (interruption_listener == NULL) return RETVAL_FAILED;
    clients.push_back (interruption_listener);

    panel_listener = ScimBridgeAgentPanelListener::alloc (scim_config->get_name (), display, this);
    if (panel_listener == NULL) return RETVAL_FAILED;
    clients.push_back (panel_listener);

    accept_listener = ScimBridgeAgentAcceptListener::alloc (this);
    if (accept_listener == NULL) return RETVAL_FAILED;
    clients.push_back (accept_listener);

    signal_listener = ScimBridgeAgentSignalListener::alloc (this);
    if (signal_listener == NULL) return RETVAL_FAILED;
    clients.push_back (signal_listener);

    ScimBridgeAgentIMContext::static_initialize (panel_listener, scim_language, scim_backend);

    scim_bridge_pdebugln (4, "ScimBridgeAgent is now ready");
    return RETVAL_SUCCEEDED;
}


retval_t ScimBridgeAgentImpl::finalize ()
{
    for (list<ScimBridgeAgentSocketClient*>::iterator i = clients.begin (); i != clients.end (); ++i) {
        ScimBridgeAgentSocketClient *client = *i;
        delete client;
    }

    clients.clear ();

    ScimBridgeAgentIMContext::static_finalize ();

    finalize_scim ();

    if (display != NULL) {
        scim_bridge_free_display (display);
        display = NULL;
    }

    scim_bridge_pdebugln (4, "Finalize, done");
    return RETVAL_SUCCEEDED;
}


retval_t ScimBridgeAgentImpl::initialize_scim ()
{
    scim_bridge_pdebugln (6, "Initializing scim...");

    // Get system language.
    scim_language = scim_get_locale_language (scim_get_current_locale ());

    // Get modules list
    vector<String> imengine_vector;
    vector<String> config_vector;
    scim_get_imengine_module_list (imengine_vector);
    scim_get_config_module_list (config_vector);

    // Use socket if avaiable.
    bool use_socket = true;
    if (find (imengine_vector.begin (), imengine_vector.end (), "socket") == imengine_vector.end () ||
        find (config_vector.begin (), config_vector.end (), "socket") == config_vector.end ())
        use_socket = false;

    // Use simple config module as default if available.
    String config_module_name;
    if (config_vector.size () > 0) {
        config_module_name = scim_global_config_read (SCIM_GLOBAL_CONFIG_DEFAULT_CONFIG_MODULE, String ("simple"));
        if (find (config_vector.begin (), config_vector.end (), config_module_name) == config_vector.end ())
            config_module_name = config_vector [0];
    } else {
        config_module_name = "dummy";
    }

    // Use user defined immodules if available
    bool use_user_imegine = false;
    vector <String> user_imengine_vector;
    const char *imengine_list_str = getenv ("SCIM_BRIDGE_IMENGINE_MODULES");
    if (imengine_list_str != NULL) {
        vector <String> spec_imengine_vector;
        scim_split_string_list (spec_imengine_vector, imengine_list_str, ',');

        user_imengine_vector.clear ();

        for (size_t i = 0; i < spec_imengine_vector.size (); ++i) {
            if (find (imengine_vector.begin (), imengine_vector.end (), imengine_vector [i]) != imengine_vector.end ()) {
                user_imengine_vector.push_back (spec_imengine_vector [i]);
                use_user_imegine = true;
            }
        }
    }

    // If you try to use the socket feature manually,
    // then let you do it by yourself.
    if (config_module_name == "socket" ||
        find (user_imengine_vector.begin (), user_imengine_vector.end (), "socket") != user_imengine_vector.end ())
        use_socket = false;

    // Try to start a SCIM SocketFrontEnd process if necessary.
    if (scim_get_default_socket_frontend_address () != scim_get_default_socket_imengine_address () &&
        scim_get_default_socket_frontend_address () != scim_get_default_socket_config_address ())
        use_socket = false;

    if (use_socket) {
        // If no Socket FrontEnd is running, then launch one.
        // And set manual to false.
        if (!check_socket_frontend ()) {
            scim_bridge_perrorln ("Launching a SCIM daemon with Socket FrontEnd...");
            char *new_argv [] = { "--no-stay", 0 };
            scim_launch (true,
                config_module_name,
                (!user_imengine_vector.empty () ? scim_combine_string_list (user_imengine_vector, ',') : "all"),
                "socket",
                new_argv);
            use_user_imegine = false;
        }

        // If there is one Socket FrontEnd running and it's not manual mode,
        // then just use this Socket Frontend.
        if (!use_user_imegine) {
            for (int i = 0; i < 100; ++i) {
                if (check_socket_frontend ()) {
                    config_module_name = "socket";
                    user_imengine_vector.clear ();
                    user_imengine_vector.push_back ("socket");
                    break;
                }
                scim_usleep (100000);
            }
        }
    }

    if (config_module_name != "dummy") {
        //load config module
        scim_bridge_pdebugln (2, "Loading Config module...: %s", config_module_name.c_str ());
        scim_config_module = new ConfigModule (config_module_name);

        //create config instance
        if (scim_config_module != NULL && scim_config_module->valid ())
            scim_config = scim_config_module->create_config ();
    }

    if (scim_config.null ()) {
        scim_bridge_pdebugln (2, "Config module cannot be loaded, using dummy Config.");

        if (scim_config_module) delete scim_config_module;
        scim_config_module = NULL;

        scim_config = new DummyConfig ();
        config_module_name = "dummy";
    }

    // create backend
    scim_backend = new CommonBackEnd (scim_config, !user_imengine_vector.empty () ? user_imengine_vector : imengine_vector);

    if (scim_backend.null ()) {
        scim_bridge_perrorln ("Cannot create BackEnd Object!");
        return RETVAL_FAILED;
    }

    scim_bridge_pdebugln (4, "Initialize scim, done!");
    return RETVAL_SUCCEEDED;
}


retval_t ScimBridgeAgentImpl::finalize_scim ()
{
    scim_bridge_pdebugln (6, "Finalizing scim...");

    return RETVAL_SUCCEEDED;
}


void ScimBridgeAgentImpl::slot_reload_config (const ConfigPointer &config)
{
    scim_bridge_pdebugln (8, "slot_reload_config ()");

    scim_frontend_hotkey_matcher.load_hotkeys (scim_config);
    scim_imengine_hotkey_matcher.load_hotkeys (scim_config);

    ScimBridgeAgentIMContext::set_enabled_by_default (scim_config->read (String (SCIM_CONFIG_FRONTEND_IM_OPENED_BY_DEFAULT), ScimBridgeAgentIMContext::is_enabled_by_default ()));
    ScimBridgeAgentIMContext::set_imengine_shared (scim_config->read (String (SCIM_CONFIG_FRONTEND_SHARED_INPUT_METHOD), ScimBridgeAgentIMContext::is_imengine_shared ()));
    ScimBridgeAgentIMContext::set_on_the_spot_enabled (scim_config->read (String (SCIM_CONFIG_FRONTEND_ON_THE_SPOT), ScimBridgeAgentIMContext::is_on_the_spot_enabled ()));

    // Get keyboard layout setting
    // Flush the global config first, in order to load the new configs from disk.
    scim_global_config_flush ();

    scim_keyboard_layout = scim_get_default_keyboard_layout ();
}


ScimBridgeAgentPanelListener *ScimBridgeAgentImpl::get_panel_listener (const ScimBridgeDisplay *new_display)
{
    if (scim_bridge_display_equals (display, new_display)) {
        return panel_listener;
    } else {
        return NULL;
    }
}


void ScimBridgeAgentImpl::quit ()
{
    scim_bridge_pdebugln (5, "quit ()");

    active = false;
}


void ScimBridgeAgentImpl::interrupt ()
{
    scim_bridge_pdebugln (2, "interrupt ()");

    interruption_listener->interrupt ();
}


void ScimBridgeAgentImpl::add_client_listener (ScimBridgeAgentClientListener *client_listener)
{
    scim_bridge_pdebugln (8, "add_client_listener ()");

    clients.push_back (client_listener);
    ++client_app_count;
}


void ScimBridgeAgentImpl::remove_client_listener (ScimBridgeAgentClientListener *client_listener)
{
    scim_bridge_pdebugln (8, "remove_client_listener ()");

    --client_app_count;
    if (!noexit_enabled && client_app_count <= 0) quit ();

    ScimBridgeAgentIMContext::free_by_client (client_listener);
}


scim_bridge_imcontext_id_t ScimBridgeAgentImpl::alloc_imcontext (ScimBridgeAgentClientListener *client_listener)
{
    scim_bridge_pdebugln (7, "alloc_imcontext ()");
    ScimBridgeAgentIMContext *imcontext = ScimBridgeAgentIMContext::alloc (client_listener);

    return imcontext->get_id ();
}


void ScimBridgeAgentImpl::free_imcontext (scim_bridge_imcontext_id_t imcontext_id, const ScimBridgeAgentClientListener *client_listener)
{
    scim_bridge_pdebugln (7, "free_imcontext ()");

    ScimBridgeAgentIMContext *imcontext = ScimBridgeAgentIMContext::find (imcontext_id);
    if (imcontext == NULL) {
        scim_bridge_perrorln ("No such IMContext");
    } else if (imcontext->get_client_listener () == client_listener) {
        panel_listener->prepare (imcontext_id);
        delete imcontext;
        panel_listener->send ();
    }
}


void ScimBridgeAgentImpl::change_focus (scim_bridge_imcontext_id_t imcontext_id, bool focus_in)
{
    scim_bridge_pdebugln (6, "change_focus ()");

    ScimBridgeAgentIMContext *imcontext = ScimBridgeAgentIMContext::find (imcontext_id);
    if (imcontext == NULL) {
        scim_bridge_perrorln ("No such IMContext");
    } else {
        panel_listener->prepare (imcontext_id);
        if (focus_in) {
            imcontext->focus_in ();
        } else {
            imcontext->focus_out ();
        }
        panel_listener->send ();
    }
}


bool ScimBridgeAgentImpl::filter_hotkeys (scim_bridge_imcontext_id_t imcontext_id, const KeyEvent &key_event)
{
    scim_bridge_pdebugln (6, "filter_hotkeys ()");

    ScimBridgeAgentIMContext *imcontext = ScimBridgeAgentIMContext::find (imcontext_id);
    if (imcontext == NULL) {
        scim_bridge_perrorln ("No such IMContext");
        return false;
    }

    scim_frontend_hotkey_matcher.push_key_event (key_event);

    const FrontEndHotkeyAction hotkey_action = scim_frontend_hotkey_matcher.get_match_result ();

    if (hotkey_action == SCIM_FRONTEND_HOTKEY_TRIGGER) {
        const bool new_status = !imcontext->is_enabled ();
        imcontext->set_enabled (new_status);
        save_config ();
        return true;
    } else if (hotkey_action == SCIM_FRONTEND_HOTKEY_ON) {
        const bool new_status = true;
        imcontext->set_enabled (new_status);
        save_config ();
        return true;
    } else if (hotkey_action == SCIM_FRONTEND_HOTKEY_OFF) {
        const bool new_status = false;
        imcontext->set_enabled (new_status);
        save_config ();
        return true;
    } else if (hotkey_action == SCIM_FRONTEND_HOTKEY_NEXT_FACTORY) {
        imcontext->open_next_imengine ();
        return true;
    } else if (hotkey_action == SCIM_FRONTEND_HOTKEY_PREVIOUS_FACTORY) {
        imcontext->open_previous_imengine ();
        return true;
    } else if (hotkey_action == SCIM_FRONTEND_HOTKEY_SHOW_FACTORY_MENU) {
        request_factory_menu ();
        return true;
    } else {
        scim_imengine_hotkey_matcher.push_key_event (key_event);
        if (scim_imengine_hotkey_matcher.is_matched ()) {
            const String factory_uuid = scim_imengine_hotkey_matcher.get_match_result ();
            imcontext->open_imengine_by_uuid (factory_uuid);
            return true;
        }
    }

    return false;
}


bool ScimBridgeAgentImpl::filter_key_event (scim_bridge_imcontext_id_t imcontext_id, const KeyEvent &key_event)
{
    scim_bridge_pdebugln (5, "filter_key_event ()");

    ScimBridgeAgentIMContext *imcontext = ScimBridgeAgentIMContext::find (imcontext_id);
    if (imcontext == NULL) {
        scim_bridge_perrorln ("No such IMContext");
        return false;
    } else {
        bool consumed = false;
        panel_listener->prepare (imcontext_id);
        if (filter_hotkeys (imcontext_id, key_event)) {
            consumed = true;
        } else if (imcontext->filter_key_event (key_event)) {
            consumed = true;
        } else {
            consumed = false;
        }
        panel_listener->send ();

        return consumed;
    }
}


void ScimBridgeAgentImpl::set_cursor_location (scim_bridge_imcontext_id_t imcontext_id, int cursor_x, int cursor_y)
{
    scim_bridge_pdebugln (7, "set_cursor_location ()");

    ScimBridgeAgentIMContext *imcontext = ScimBridgeAgentIMContext::find (imcontext_id);
    if (imcontext == NULL) {
        scim_bridge_perrorln ("No such IMContext");
    } else {
        panel_listener->prepare (imcontext_id);
        imcontext->set_cursor_location (cursor_x, cursor_y);
        panel_listener->send ();
    }
}


void ScimBridgeAgentImpl::set_preedit_mode (scim_bridge_imcontext_id_t imcontext_id, scim_bridge_preedit_mode_t preedit_mode)
{
    scim_bridge_pdebugln (6, "set_preedit_mode ()");

    ScimBridgeAgentIMContext *imcontext = ScimBridgeAgentIMContext::find (imcontext_id);
    if (imcontext == NULL) {
        scim_bridge_perrorln ("No such IMContext");
    } else {
        panel_listener->prepare (imcontext_id);
        imcontext->set_preedit_mode (preedit_mode);
        panel_listener->send ();
    }
}


void ScimBridgeAgentImpl::reset_imcontext (scim_bridge_imcontext_id_t imcontext_id)
{
    scim_bridge_pdebugln (6, "reset_imcontext ()");

    ScimBridgeAgentIMContext *imcontext = ScimBridgeAgentIMContext::find (imcontext_id);
    if (imcontext == NULL) {
        scim_bridge_perrorln ("No such IMContext");
    } else {
        panel_listener->prepare (imcontext_id);
        imcontext->reset ();
        panel_listener->send ();
    }
}


void ScimBridgeAgentImpl::request_factory_menu ()
{
    scim_bridge_pdebugln (6, "request_factory_menu ()");

    vector<IMEngineFactoryPointer> factories;
    vector<PanelFactoryInfo> menu;

    scim_backend->get_factories_for_encoding (factories, "UTF-8");

    for (size_t i = 0; i < factories.size (); ++i) {
        menu.push_back (PanelFactoryInfo (factories [i]->get_uuid (), utf8_wcstombs (factories [i]->get_name ()), factories [i]->get_language (), factories [i]->get_icon_file ()));
    }
    if (!menu.empty ()) panel_listener->show_factory_menu (menu);
}


void ScimBridgeAgentImpl::load_config ()
{
    scim_bridge_pdebugln (6, "load_config ()");

    scim_config->reload ();
}


void ScimBridgeAgentImpl::save_config ()
{
    scim_bridge_pdebugln (6, "save_config ()");

    scim_config->write (String (SCIM_CONFIG_FRONTEND_IM_OPENED_BY_DEFAULT),
        ScimBridgeAgentIMContext::is_enabled_by_default ());
}
