/*************************************************************************
 *
 *  OpenOffice.org - a multi-platform office productivity suite
 *
 *  $RCSfile: shapeimporter.cxx,v $
 *
 *  $Revision: 1.10 $
 *
 *  last change: $Author: kz $ $Date: 2005/11/02 14:03:41 $
 *
 *  The Contents of this file are made available subject to
 *  the terms of GNU Lesser General Public License Version 2.1.
 *
 *
 *    GNU Lesser General Public License Version 2.1
 *    =============================================
 *    Copyright 2005 by Sun Microsystems, Inc.
 *    901 San Antonio Road, Palo Alto, CA 94303, USA
 *
 *    This library is free software; you can redistribute it and/or
 *    modify it under the terms of the GNU Lesser General Public
 *    License version 2.1, as published by the Free Software Foundation.
 *
 *    This library is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *    Lesser General Public License for more details.
 *
 *    You should have received a copy of the GNU Lesser General Public
 *    License along with this library; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *    MA  02111-1307  USA
 *
 ************************************************************************/

// must be first
#include "canvas/debug.hxx"
#include "vcl/cvtgrf.hxx"
#include "tools/urlobj.hxx"
#include "tools/stream.hxx"
#include "goodies/grfmgr.hxx"
#include "unotools/ucbstreamhelper.hxx"
#include "unotools/streamwrap.hxx"
#include "drawshape.hxx"
#include "mediashape.hxx"
#include "shapeimporter.hxx"
#include "slideshowexceptions.hxx"
#include "gdimtftools.hxx"
#include "tools.hxx"
#include "com/sun/star/awt/Rectangle.hpp"
#include "com/sun/star/drawing/ColorMode.hpp"
#include "com/sun/star/text/GraphicCrop.hpp"
#include "boost/shared_ptr.hpp"
#include "boost/scoped_ptr.hpp"

using namespace com::sun::star;

namespace presentation {
namespace internal {

namespace {

bool importShapeGraphic(
    GraphicObject & o_rGraphic,
    uno::Reference<beans::XPropertySet> const& xPropSet )
{
    rtl::OUString aURL;
    if (! (xPropSet->getPropertyValue( OUSTR("GraphicURL") ) >>= aURL) ||
        aURL.getLength() == 0) {
        // no or empty property - cannot import shape graphic
        return false;
    }
    
    rtl::OUString const aVndUrl( 
        RTL_CONSTASCII_USTRINGPARAM( "vnd.sun.star.GraphicObject:" ) );
    sal_Int32 nIndex( aURL.indexOf( aVndUrl ) );
    
    if (nIndex != -1) {
        // skip past the end of the "vnd..." prefix
        nIndex += aVndUrl.getLength();
        
        if (nIndex >= aURL.getLength()) {
            OSL_ENSURE( false, "ShapeImporter::importShape(): "
                        "embedded graphic has no graphic ID" );
            return false;
        }
        
        // unique ID string found in URL, extract
        // to separate string
        rtl::OUString const aUniqueId(
            aURL.copy( nIndex, aURL.getLength() - nIndex ) );
        
        // TODO(T2): Creating a GraphicObject is not
        // thread safe (internally calls VCL, and has
        // unguarded internal singleton mpGlobalMgr)
        
        // fetch already loaded graphic from graphic manager.
        ByteString const aOldString( static_cast<String>(aUniqueId),
                                     RTL_TEXTENCODING_UTF8 );
        o_rGraphic = GraphicObject( aOldString );
    }
    else {
        // no special string found, graphic must be
        // external. Load via GraphicIm porter
        INetURLObject aTmp( aURL );
        boost::scoped_ptr<SvStream> pGraphicStream(
            utl::UcbStreamHelper::CreateStream(
                aTmp.GetMainURL( INetURLObject::NO_DECODE ),
                STREAM_READ ) );
        if (! pGraphicStream) {
            OSL_ENSURE( false, "ShapeImporter::importShape(): "
                        "cannot create input stream for graphic" );
            return false;
        }
        
        Graphic aTmpGraphic;
        if (GraphicConverter::Import(
                *pGraphicStream, aTmpGraphic ) != ERRCODE_NONE) {
            OSL_ENSURE( false, "ShapeImporter::importShape(): "
                        "Failed to import shape graphic from given URL" );
            return false;
        }        
        o_rGraphic = GraphicObject( aTmpGraphic );
    }
    return true;
}

/** This shape implementation just acts as a dummy for the layermanager.
    Its sole role is for hit test detection of group shapes.
*/
class ShapeOfGroup : public Shape, private boost::noncopyable
{
public:
    ShapeOfGroup( ShapeSharedPtr const& pGroupShape,
                  uno::Reference<drawing::XShape> const& xShape,
                  uno::Reference<beans::XPropertySet> const& xPropSet,
                  double nPrio );
    
    // Shape:
    virtual uno::Reference<drawing::XShape> getXShape() const;
    virtual void addViewLayer( ViewLayerSharedPtr const& pNewLayer,
                               bool bRedrawLayer );
    virtual bool removeViewLayer( ViewLayerSharedPtr const& pNewLayer );
    virtual bool update() const;
    virtual bool render() const;
    virtual bool isUpdateNecessary() const;
    virtual basegfx::B2DRectangle getPosSize() const;
    virtual basegfx::B2DRectangle getDOMBounds() const;
    virtual basegfx::B2DRectangle getUpdateArea() const;
    virtual bool isVisible() const;
    virtual double getPriority() const;
    virtual bool isBackgroundDetached() const;
    virtual bool hasIntrinsicAnimation() const;
    virtual bool hasHyperlinks() const;
    virtual std::vector<HyperLinkRegion> getHyperlinkRegions() const;
    
private:
    ShapeSharedPtr const mpGroupShape;
    uno::Reference<drawing::XShape> const mxShape;
    double const mnPrio;
    basegfx::B2DPoint maPosOffset;
    double mnWidth;
    double mnHeight;
};

ShapeOfGroup::ShapeOfGroup(
    ShapeSharedPtr const& pGroupShape,
    uno::Reference<drawing::XShape> const& xShape,
    uno::Reference<beans::XPropertySet> const& xPropSet,
    double nPrio )
    : mpGroupShape(pGroupShape), mxShape(xShape), mnPrio(nPrio)
{
    // read bound rect
    uno::Any const aTmpRect_( xPropSet->getPropertyValue( OUSTR("BoundRect") ));
    awt::Rectangle const aTmpRect( aTmpRect_.get<awt::Rectangle>() );
    basegfx::B2DRectangle const groupPosSize( pGroupShape->getPosSize() );
    maPosOffset = basegfx::B2DPoint( aTmpRect.X - groupPosSize.getMinX(),
                                     aTmpRect.Y - groupPosSize.getMinY() );
    mnWidth = aTmpRect.Width;
    mnHeight = aTmpRect.Height;
}

uno::Reference<drawing::XShape> ShapeOfGroup::getXShape() const
{
    return mxShape;
}

void ShapeOfGroup::addViewLayer( ViewLayerSharedPtr const& pNewLayer,
                                 bool bRedrawLayer )
{
}

bool ShapeOfGroup::removeViewLayer( ViewLayerSharedPtr const& pNewLayer )
{
    return true;
}

bool ShapeOfGroup::update() const
{
    return true;
}

bool ShapeOfGroup::render() const
{
    return true;
}

bool ShapeOfGroup::isUpdateNecessary() const
{
    return false;
}

basegfx::B2DRectangle ShapeOfGroup::getPosSize() const
{
    basegfx::B2DRectangle const groupPosSize( mpGroupShape->getPosSize() );
    double const posX = (groupPosSize.getMinX() + maPosOffset.getX());
    double const posY = (groupPosSize.getMinY() + maPosOffset.getY());
    return basegfx::B2DRectangle( posX, posY, posX + mnWidth, posY + mnHeight );
}

basegfx::B2DRectangle ShapeOfGroup::getDOMBounds() const
{
    return getPosSize();
}

basegfx::B2DRectangle ShapeOfGroup::getUpdateArea() const
{
    return getPosSize();
}

bool ShapeOfGroup::isVisible() const
{
    return mpGroupShape->isVisible();
}

double ShapeOfGroup::getPriority() const
{
    return mnPrio;
}

bool ShapeOfGroup::isBackgroundDetached() const
{
    return false;
}

bool ShapeOfGroup::hasIntrinsicAnimation() const
{
    return false;
}

bool ShapeOfGroup::hasHyperlinks() const
{
    return false;
}

std::vector<Shape::HyperLinkRegion> ShapeOfGroup::getHyperlinkRegions() const
{
    return std::vector<HyperLinkRegion>();
}

} // anon namespace

ShapeSharedPtr ShapeImporter::createShape(
    uno::Reference<drawing::XShape> const& xCurrShape,
    uno::Reference<beans::XPropertySet> const& xPropSet,
    rtl::OUString const& shapeType ) const
{
    if (shapeType.equalsAsciiL( 
            RTL_CONSTASCII_STRINGPARAM("com.sun.star.drawing.MediaShape") )) {
        // Media shape (video etc.). This is a special object
        return ShapeSharedPtr( new MediaShape( xCurrShape, 
                                               mxPage,
                                               mnAscendingPrio ) );
    }
    else if (shapeType.equalsAsciiL(
                 RTL_CONSTASCII_STRINGPARAM(
                     "com.sun.star.drawing.GraphicObjectShape") )) {
        
        GraphicObject aGraphicObject;
        
        // to get hold of GIF animations, inspect Graphic
        // objects more thoroughly (the plain-jane shape
        // metafile of course would only contain the first
        // animation frame)
        if (!importShapeGraphic( aGraphicObject, xPropSet ) ||
            !aGraphicObject.IsAnimated()) {
            // import failed or no animation - fallback to
            // plain draw shape import
            
            // import shape as bitmap - either its a bitmap
            // anyway, or its a metafile, which currently the
            // metafile renderer might not display correctly.
            return ShapeSharedPtr( new DrawShape( xCurrShape, 
                                                  mxPage,
                                                  mnAscendingPrio,
                                                  true ) );
        }
        
        
        // now extract relevant shape attributes via API
        // ---------------------------------------------
        
        drawing::ColorMode eColorMode( drawing::ColorMode_STANDARD );
        sal_Int16 nLuminance(0);
        sal_Int16 nContrast(0);
        sal_Int16 nRed(0);
        sal_Int16 nGreen(0);
        sal_Int16 nBlue(0);
        double    nGamma(1.0);
        sal_Int16 nTransparency(0);
        sal_Int32 nRotation(0);
        
        getPropertyValue( eColorMode, xPropSet, OUSTR("GraphicColorMode") );
        getPropertyValue( nLuminance, xPropSet, OUSTR("AdjustLuminance") );
        getPropertyValue( nContrast, xPropSet, OUSTR("AdjustContrast") );
        getPropertyValue( nRed, xPropSet, OUSTR("AdjustRed") );
        getPropertyValue( nGreen, xPropSet, OUSTR("AdjustGreen") );
        getPropertyValue( nBlue, xPropSet, OUSTR("AdjustBlue") );
        getPropertyValue( nGamma, xPropSet, OUSTR("Gamma") );
        getPropertyValue( nTransparency, xPropSet, OUSTR("Transparency") );
        getPropertyValue( nRotation, xPropSet, OUSTR("RotateAngle") );
        
        GraphicAttr aGraphAttrs;
        aGraphAttrs.SetDrawMode( (GraphicDrawMode)eColorMode );
        aGraphAttrs.SetLuminance( nLuminance );
        aGraphAttrs.SetContrast( nContrast );
        aGraphAttrs.SetChannelR( nRed );
        aGraphAttrs.SetChannelG( nGreen );
        aGraphAttrs.SetChannelB( nBlue );
        aGraphAttrs.SetGamma( nGamma );
        aGraphAttrs.SetTransparency( static_cast<BYTE>(nTransparency) );
        aGraphAttrs.SetRotation( static_cast<USHORT>(nRotation*10) );
        
        text::GraphicCrop aGraphCrop;
        if (getPropertyValue( aGraphCrop, xPropSet, OUSTR("GraphicCrop") )) {
            aGraphAttrs.SetCrop( aGraphCrop.Left,
                                 aGraphCrop.Top,
                                 aGraphCrop.Right,
                                 aGraphCrop.Bottom );
        }
        
        // fetch readily transformed and color-modified
        // graphic
        // ---------------------------------------------
        
        Graphic aGraphic( 
            aGraphicObject.GetTransformedGraphic(
                aGraphicObject.GetPrefSize(),
                aGraphicObject.GetPrefMapMode(),
                aGraphAttrs ) );
        
        return ShapeSharedPtr( new DrawShape( xCurrShape, 
                                              mxPage,
                                              mnAscendingPrio,
                                              aGraphic ) );
    }
    else if (shapeType.equalsAsciiL(
                 RTL_CONSTASCII_STRINGPARAM("com.sun.star.drawing.OLE2Shape") ))
    {
        // #i46224# Import OLE shapes as a single bitmap;
        // currently, the metafile-to-canvas converted has a
        // few deficiencies (most important: no raster ops)
        return ShapeSharedPtr( new DrawShape( xCurrShape, 
                                              mxPage,
                                              mnAscendingPrio,
                                              true ) );
    }
    else {
        return ShapeSharedPtr( new DrawShape( xCurrShape, 
                                              mxPage,
                                              mnAscendingPrio,
                                        false ) );
    }
}

bool ShapeImporter::isSkip(
    uno::Reference<drawing::XShape> const& xCurrShape,
    uno::Reference<beans::XPropertySet> const& xPropSet,
    rtl::OUString const& shapeType ) const
{
    // skip empty presentation objects:
    bool bEmpty = false;
    if ((xPropSet->getPropertyValue( OUSTR("IsEmptyPresentationObject") )
         >>= bEmpty) && bEmpty)
        return true;
    
    // don't export presentation placeholders on masterpage
    // they can be non empty when user edits the default texts
    if (mbConvertingMasterPage) {
        if (shapeType.equalsAsciiL(
                RTL_CONSTASCII_STRINGPARAM("com.sun.star.presentation."
                                           "TitleTextShape") ) ||
            shapeType.equalsAsciiL(
                RTL_CONSTASCII_STRINGPARAM("com.sun.star.presentation."
                                           "OutlinerShape") )) {
            return true;
        }
    }
    return false;
}

ShapeSharedPtr ShapeImporter::importShape() // throw (ImportFailedException)
{
    ShapeSharedPtr pRet;
    bool bIsGroupShape = false;
    
    while (!maShapesStack.empty() && !pRet)
    {
        XShapesEntry & rTop = maShapesStack.top();
        if (rTop.mnPos < rTop.mnCount) {
            
            uno::Reference<drawing::XShape> const xCurrShape(
                rTop.mxShapes->getByIndex( rTop.mnPos ), uno::UNO_QUERY );
            ++rTop.mnPos;
            uno::Reference<beans::XPropertySet> xPropSet(
                xCurrShape, uno::UNO_QUERY );
            if (! xPropSet.is()) {
                // we definitely need the properties of 
                // the shape here. This will also fail,
                // if getByIndex did not return a valid
                // shape
                throw ImportFailedException();
            }

            rtl::OUString const shapeType( xCurrShape->getShapeType() );
            
            // is this shape presentation-invisible?
            if (! isSkip( xCurrShape, xPropSet, shapeType ))
            {
                bIsGroupShape = shapeType.equalsAsciiL(
                    RTL_CONSTASCII_STRINGPARAM(
                        "com.sun.star.drawing.GroupShape") );
                
                if (rTop.mpGroupShape) { // in group particle mode?
                    pRet.reset( new ShapeOfGroup(
                                    rTop.mpGroupShape /* container shape */,
                                    xCurrShape, xPropSet,
                                    mnAscendingPrio ) );
                }
                else {
                    pRet = createShape( xCurrShape, xPropSet, shapeType );
                }
                mnAscendingPrio += 1.0;
            }
        }
        if (rTop.mnPos >= rTop.mnCount) {
            // group or top-level shapes finished:
            maShapesStack.pop();
        }
        if (bIsGroupShape && pRet) {
            // push new group on the stack: group traversal
            maShapesStack.push( XShapesEntry( pRet ) );
        }
    }
    
    return pRet;
}

bool ShapeImporter::isImportDone() const
{
    return maShapesStack.empty();
}

ShapeImporter::ShapeImporter(
    uno::Reference<drawing::XDrawPage> const& xPage, 
    uno::Reference<drawing::XDrawPage> const& xActualPage,
    sal_Int32 nOrdNumStart,
    bool bConvertingMasterPage )
    : mxPage( xActualPage ),
      mnAscendingPrio( nOrdNumStart ),
      mbConvertingMasterPage( bConvertingMasterPage )
{
    uno::Reference<drawing::XShapes> const xShapes(
        xPage, uno::UNO_QUERY_THROW );
    maShapesStack.push( XShapesEntry(xShapes) );
}

} // namespace internal
} // namespace presentation

