/*
 *  Module for execution of addin commands.
 *
 * Copyright INOUE Seiichiro <inoue@ainet.or.jp>, licensed under the GPL.
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif
#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>
#if defined(HAVE_STRING_H)
#include <string.h>
#elif defined(HAVE_STRINGS_H)
#include <strings.h>
#endif
#include <fcntl.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/wait.h>
#include <signal.h>
#include <sys/time.h>
#include <unistd.h>

/*#include "xyaku.h"*/
typedef unsigned char	uchar;

/* Constant numbers */
#define MAX_WORD	64

/* Private variables */
static int child_die = 0;

/* Private function prototypes */
static void child_die_handler(int signal);
static void close_allfd_except_std(void);

/**
 * exec_addin_cmd:
 * Execute addin command, then pass the target word to its standard input.
 * Then, get an output from its pipe (from the command's standard output).
 * Input:
 * const char *addin_cmd; addin command path.
 * char **argv; arguments
 * const uchar *target; target string.
 * uchar *retbuf; return buffer. (caller should allocate)
 * int size_buf; (byte)size of the return buffer.
 * Output:
 * char *retbuf; return string.
 * return value; (byte)length of return string.
 **/
int
exec_addin_cmd(const char *addin_cmd, char **argv, const uchar *target, uchar *retbuf, int size_buf)
{
	int ret_lenb = 0;
	int left_lenb = size_buf;
	uchar *ptr = retbuf;
	int pid;
	int pipefds1[2];/* xyaku(parent) ==> addin-command(child) */
	int pipefds2[2];/* addin-command(child) ==> xyaku(parent) */
	int lenb;
	int status;
	int ret;
	fd_set fds;
	struct timeval tv;

	if (pipe(pipefds1) < 0) {
		perror("pipe");
		exit(1);
	}
	if (pipe(pipefds2) < 0) {
		perror("pipe");
		exit(1);
	}
	if ((pid = fork()) < 0) {
		perror("fork");
		exit(1);
	}
	
	if (pid == 0) {
		/* Child process */
		/*
		 * pipefds1[0]: for read (stdin)
		 * pipefds2[1]: for write (stdout)
		 */
#ifdef DEBUG
		fprintf(stderr, "child %s %s\n", addin_cmd, target);
#endif
		dup2(pipefds1[0], 0);
		dup2(pipefds2[1], 1);
#ifdef DEBUG
		dup2(pipefds2[1], 2);/* want a child process's error output. */
#else
		close(2);
#endif
		close_allfd_except_std();

		execv(addin_cmd, argv);
		perror("execl");
		exit(1);
	}
	
	/* PARENT process */
	/*
	 *  pipefds1[1]: for write
	 *  pipefds2[0]: for read
	 */
	/* Use select() and SIGCHLD signal handler.
	   In order to deal with the case the child puts no output to stdout.
	   In that case, it doesn't come back from read(). */
	child_die = 0;
	signal(SIGCHLD, child_die_handler);
	close(pipefds1[0]);
	close(pipefds2[1]);

	/* Write target(word) into the add-in command(child process) */
	ret = write(pipefds1[1], target, strlen(target));
#ifdef DEBUG		
	fprintf(stderr, "to write: %d, written: %d\n", strlen(target), ret);
#endif
	close(pipefds1[1]);

	/* Read the result from the add-in command(child process) */
	FD_ZERO(&fds);
	/* Wait up to five seconds. */
	tv.tv_sec = 5;
	tv.tv_usec = 0;
	while (1) {
		static char nullbuf[256];

		FD_SET(pipefds2[0], &fds);
		ret = select(pipefds2[0]+1, &fds, NULL, NULL, &tv);
#ifdef DEBUG		
		fprintf(stderr, "select %d\n", ret);
#endif
		if (ret > 0) {
			if (left_lenb) {
				lenb = read(pipefds2[0], ptr, left_lenb);
#ifdef DEBUG
				fprintf(stderr, "read %d\n", lenb);
#endif
				ptr += lenb;
				left_lenb -= lenb;
				ret_lenb += lenb;
			} else {
				/* Read it up all.
				   Because if the parent closes the pipe before reading it up,
				   it causes SIGPIPE for a child process. */
				lenb = read(pipefds2[0], nullbuf, sizeof(nullbuf));
			}
			if (lenb <= 0)
				break;
		}
		if (child_die || ret == -1)
			break;
	}
	close(pipefds2[0]);
	
	while (wait(&status) != pid)
		;

#ifdef DEBUG
	fprintf(stderr, "child done %d\n", WEXITSTATUS(status));
#endif
	
	return ret_lenb;
}

/**
 * child_die_handler:
 * handler for SIGCHLD.
 **/
static void
child_die_handler(int signal)
{
	child_die = 1;
}

/**
 * close_allfd_except_std:
 * close all file descriptors except standard fds(stdin, stdout, stderr).
 * Usually, called by a forked child process.
 * Its main purpose is to close fd for the socket connected to X server.
 **/
static void
close_allfd_except_std(void)
{
	int fd_max;
	int fd;

	fd_max = sysconf(_SC_OPEN_MAX);
	for (fd = STDERR_FILENO+1; fd < fd_max; fd++) {
		close(fd);
	}
}
