/*
 * Module for add-ins
 *
 * Example of the config file format (basically, it follows fvwm2rc)
 * See xyakurc file about details.
 *
 *  ModulePath /usr/local/libexec/xyaku
 *  Key F1 C use_cache max_nlines max_bufsiz max_width edict.sh
 *  AutoKey F1 C use_cache max_nlines max_bufsiz max_width interval edict.sh
 *
 * I ignore memory leaks if parse fails, because it can make code much simpler.
 *
 * Copyright INOUE Seiichiro <inoue@ainet.or.jp>, licensed under the GPL.
 */
#ifdef HAVE_CONFIG_H
#include "config.h"
#endif
#include <stdio.h>
#include <stdlib.h>
#if defined(HAVE_STRING_H)
#include <string.h>
#elif defined(HAVE_STRINGS_H)
#include <strings.h>
#endif
#include <ctype.h>
#include <unistd.h>
#include <limits.h>
#include <sys/stat.h>

#include <X11/Xlib.h>
#include <X11/Intrinsic.h>

#include "xyaku.h"
#include "mem.h"
#include "addin.h"


/* Forward declarations of types. */
typedef struct _AddinInfo AddinInfo;
typedef struct _AddinTable AddinTable;


/* Constant numbers and strings */
/* config file name */
#ifndef SYSTEM_XYAKURC	/* Normally, defined by the configure script */
#define SYSTEM_XYAKURC	"/usr/local/libexec/xyaku/xyakurc"
#endif

#define USER_XYAKURC	".xyakurc" /* $HOME/.xyakurc */


/* Identifiers (case-insensitive) */
typedef enum {
	N_IDENT_UNKNOWN,
	N_IDENT_VERSION,
	N_IDENT_MODULE_PATH,
	N_IDENT_KEY,
	N_IDENT_AUTOKEY
} IdentSymbol;

/* keywords definition */
#define IDENT_VERSION			"version"
#define IDENT_MODULE_PATH		"modulepath"
#define IDENT_KEY				"key"
#define IDENT_AUTOKEY			"autokey"


/* Version number. This is independent from xyaku's version.
   This is config file (xyakurc)'s version. */
typedef enum {
	VERSION_ONE = 1,
	VERSION_TWO = 2
} SupportVersion;

/* IDENT_VERSION line's column
   An example,
   Version 2 */
typedef enum {
	COL_VERSION_ID = 0,
	COL_VERSION_NUM = 1
} ColVersion;

/* IDENT_MODULE_PATH line's column
   An example,
   ModulePath /usr/local/libexec/xyaku */
typedef enum {
	COL_MODPATH_ID = 0,
	COL_MODPATH_PATH = 1
} ColModulePath;

/* IDENT_KEY and IDENT_AUTOKEY line's column
   An example,
   Key F1 C true 16 1024 640 foo.sh arg1 arg2
   AutoKey F1 C true 16 1024 640 1500 foo.sh arg1 arg2 */
typedef enum {
	COL_KEY_ID = 0,
	COL_KEY_KEYSYM = 1,
	COL_KEY_MODMASKS = 2,
	COL_KEY_USECACHE = 3,
	COL_KEY_NLINES = 4,
	COL_KEY_BUFSIZ = 5,
	COL_KEY_WIDTH = 6,
	COL_KEY_COMMAND = 7,
	COL_AUTOKEY_INTERVAL = 7,
	COL_AUTOKEY_COMMAND = 8,
	COL_KEY_ARGUMENTS
} ColKey2;/* version two */

/* IDENT_KEY and IDENT_AUTOKEY line's column (Version one)
   This is provided for backward compatibility */
typedef enum {
	COL_KEY1_ID = 0,
	COL_KEY1_KEYSYM = 1,
	COL_KEY1_MODMASKS = 2,
	COL_KEY1_COMMAND = 3,
	COL_KEY1_ARGUMENTS
} ColKey1;


/* Data structure */
/* Addin info on memory
   Currently, we have only one static instance for this. */
struct _AddinInfo {
	SupportVersion version;
	int num_path;		/* number of module_path */
	char **module_path;	/* array of char *module_path */
	AddinTable *atable;	/* Singly linked list of AddinTable */
};

/* Addin table
   A node of singly linked list */
struct _AddinTable {
	AddinTable *next;
	KeySym keysym;
	uint modmasks;	/* OR'd modifier keys */

	CmdInfo cmd;
	EtcInfo etc;
};


/* Private variables */	
static AddinInfo *addins;
static char *config_path = NULL;

/* Private function prototypes */
static void new_addin_info(void);
static void delete_addin_info(void);
static AddinTable* new_addin_table(void);
static void delete_addin_table(AddinTable *at);
static AddinTable* prepend_addin_table(AddinTable *head, AddinTable *node);
static void read_addin_info(void);

static int parse_version(char *buf);
static int parse_module_path(char *buf);
static int parse_key(char *buf, int f_automode);
static int parse_key2(char *buf, int f_automode);
static int parse_key1(char *buf);

static void set_grab_keys(void);
static void unset_grab_keys(void);
static uint str2modmasks(const char *str);
static char* get_cmd_path(const char *cmd);

static IdentSymbol determine_id(const char *word);
static int determine_use_cache(const char *word);

static void show_parse_result(void);


/**
 * new_config_path;
 * Look for the config file as following order,
 * 1. in specified path by an argument(-config)
 * 2. $HOME/.xyakurc
 * 3. $prefix/libexec/xyaku/xyakurc
 * Then initalize configuration data.
 * A static variable, config_path, is allocated here.
 * Input:
 * const char *confpath; specified path by an argument.
 **/
void
new_config_path(const char *confpath)
{
	const char *homedir;
	char *path;

	if (confpath && (access(confpath, R_OK) == 0)) {
		config_path = strdup(confpath);
	} else {
		homedir = getenv("HOME");
		if (homedir) {
			path = (char*)xmalloc(strlen(homedir) + 1 + strlen(USER_XYAKURC) + 1);
			sprintf(path, "%s/%s", homedir, USER_XYAKURC);
			if (access(path, R_OK) == 0) {
				config_path = path;
			} else {
				xfree(path);
				path = (char*)xmalloc(strlen(SYSTEM_XYAKURC) + 1);
				strcpy(path, SYSTEM_XYAKURC);
				if (access(path, R_OK) == 0) {
					config_path = path;
				} else {
					fprintf(stderr, "No config file \n");
					fprintf(stderr, "You can specify it as: xyaku -config path\n");
					xfree(path);
					exit(1);
				}
			}
		}
	}
#ifdef DEBUG
	fprintf(stderr, "config file=%s\n", config_path);
#endif
}

/**
 * delete_config_path:
 * Free the memory for config_path.
 **/
void
delete_config_path(void)
{
	xfree(config_path);
}


/**
 * init_addin_info:
 * Initialize AddinInfo. I.e. load config file.
 **/
void
init_addin_info(int parse_msg_flag)
{
	new_addin_info();
	if (parse_msg_flag)
		show_parse_result();
	set_grab_keys();
}

/**
 * finalize_addin_info:
 * Finalize AddinInfo.
 **/
void
finalize_addin_info(void)
{
	unset_grab_keys();
	delete_addin_info();
}

/**
 * refresh_addin_info:
 * Currently, called from a SIGHUP handler.
 * Refresh AddinInfo, i.e. reload the config file.
 **/
void
refresh_addin_info(void)
{
	finalize_addin_info();
	init_addin_info(0);
}

/**
 * find_addin_cmd:
 * With specified KeySym and Modified-key, find an addin command, and return it.
 * NOTE:
 * The contents of CmdInfo *cmd are set here,
 * but they point to static data.
 * So they should not be modified by caller. 
 * Input:
 * CmdInfo *cmd; Memory is allocated by caller.
 * KeySym ksym;
 * uint modkey;
 * EtcInfo *etc; Memory is allocated by caller.
 * Output:
 * CmdInfo *cmd;
 * EtcInfo *etc;
 * Return value; if found return 1, else return 0.
 **/
int
find_addin_cmd(CmdInfo *cmd, KeySym ksym, uint modkey, EtcInfo *etc)
{
	AddinTable *at;

	for (at = addins->atable; at; at = at->next) {
		if (at->keysym == ksym
			&& (at->modmasks == 0 || at->modmasks & modkey)) {
			*cmd = at->cmd;
			*etc = at->etc;
			return 1;
		}
	}
	return 0;
}



/* The followings are private functions. */
/**
 * new_addin_info:
 * AddinInfo, a static variable, is allocated.
 * Then, call read_addin_info() to load config file.
 **/
static void
new_addin_info(void)
{
	addins = (AddinInfo*)xmalloc(sizeof(AddinInfo));
	addins->version = VERSION_ONE;/* default */
	addins->num_path = 0;
	addins->module_path = NULL;
	addins->atable = NULL;

	read_addin_info();
}

/**
 * delete_addin_info:
 * Free the memory related to AddinInfo, a static variable.
 * Free its own memory.
 **/
static void
delete_addin_info(void)
{
	int i;
	AddinTable *at;

	for (i = 0; i < addins->num_path; i++) {
		xfree(addins->module_path[i]);
	}
	if (addins->num_path > 0)
		xfree(addins->module_path);

	for (at = addins->atable; at; ) {
		AddinTable *next = at->next;
		delete_addin_table(at);
		at = next;
	}
	if (addins)
		xfree(addins);
}


/**
 * new_addin_table:
 * Allocate AddinTable, initialize and return it.
 **/
static AddinTable*
new_addin_table(void)
{
	AddinTable *at;

	at = (AddinTable*)xcalloc(sizeof(AddinTable), 1);
	at->next = NULL;
	at->keysym = NoSymbol;
	at->modmasks = 0;

	at->cmd.cmd_path = NULL;
	at->cmd.argv = NULL;
	
	return at;
}

/**
 * delete_addin_table:
 * Delete the memory related to addin table.
 * Free its own memory.
 **/
static void
delete_addin_table(AddinTable *at)
{
	int i;

	xfree(at->cmd.cmd_path);

	for (i = 0; i < at->cmd.argc; i++) {
		xfree(at->cmd.argv[i]);
	}
	xfree(at->cmd.argv);

	xfree(at);
}

/**
 * prepend_addin_table:
 * Prepend the specified node to singly linked list of AddinTable.
 **/
static AddinTable*
prepend_addin_table(AddinTable *head, AddinTable *node)
{
	node->next = head;
	return node;
}


/**
 * read_addin_info:
 * Read from the config file and make addin info on memory.
 * XXX: I don't take care of the case that a line is longer than BUFSIZ.
 **/
static void
read_addin_info(void)
{
	FILE *fp;
	char buf[BUFSIZ];

	if ((fp = fopen(config_path, "r")) == NULL) {
		return;
	}
	while (fgets(buf, sizeof(buf), fp)) {
		IdentSymbol id = N_IDENT_UNKNOWN;
		int ret;
		char *p_lf;/* Used to remove the last '\n' */
		
		if (buf[0] == '\0' || buf[0] == '\n' || buf[0] == '#') /* comment */
			continue;
		if ((p_lf = strchr(buf, '\n')))/* remove the last '\n' */
			*p_lf = '\0';
		else
			fprintf(stderr, "A line is longer than %d.\n", BUFSIZ);

		id = determine_id(buf);
		if (id == N_IDENT_UNKNOWN) {
			fprintf(stderr, "unknown identifier: %s\n", buf);
			continue;/* next line */
		}

		switch (id) {
		case N_IDENT_VERSION:
			ret = parse_version(buf);
			break;
		case N_IDENT_MODULE_PATH:
			ret = parse_module_path(buf);
			break;
		case N_IDENT_KEY:
			ret = parse_key(buf, 0);
			break;
		case N_IDENT_AUTOKEY:
			ret = parse_key(buf, 1);
			break;
		default:
			ret = 0;
			break;
		}
		/* No error check */
	}
	fclose(fp);
}


/* The following parser functions return 1 if success. If fails, return 0. */

static int
parse_version(char *buf)
{
	int ret = 0;
	int col = 0; /* column */
	char *ptr;
	
	for (ptr = strtok(buf, " \t"); ptr; ptr = strtok(NULL, " \t"), col++) {
		switch (col) {
		case COL_VERSION_ID:
			break;
		case COL_VERSION_NUM:
			if (ptr[0] == '\0') {
				goto done;
			}
			addins->version = atoi(ptr);
			ret = 1;
			goto done;
		default:
			goto done;
		}
	}

 done:
	return ret;
}

/**
 * parse_module_path:
 * Parse module-path line in config file.
 * Allocate the memory for char **module_path as follows,
 * sizeof(char*) which is a pointer to string(char module_path[]),
 * and char module_path[] which is a string itself. 
 **/
static int
parse_module_path(char *buf)
{
	int ret = 0;
	int col = 0; /* column */
	char *ptr;
	
	for (ptr = strtok(buf, " \t"); ptr; ptr = strtok(NULL, " \t"), col++) {
		char *module_path;
		int lenb;
		
		switch (col) {
		case COL_MODPATH_ID:
			break;
		case COL_MODPATH_PATH:
			if (ptr[0] == '\0') {
				goto done;
			}
			module_path = strdup(ptr);
			/* Remove trailing slash. But multiple slashes still have a problem. */
			lenb = strlen(module_path);
			if (module_path[lenb - 1] == '/')
				module_path[lenb - 1] = '\0';
			addins->module_path = xrealloc(addins->module_path,
										   sizeof(char*) * (addins->num_path + 1));
			addins->module_path[addins->num_path] = module_path;
			addins->num_path++;
			ret = 1;
			goto done;
		default:
			goto done;
		}
	}

 done:
	return ret;
}

/**
 * parse_key:
 * Keep backward compatibility.
 **/
static int
parse_key(char *buf, int f_automode)
{
	if (addins->version == VERSION_ONE)
		return parse_key1(buf);
	else if (addins->version == VERSION_TWO)
		return parse_key2(buf, f_automode);
	else
		return 0;
}

/**
 * parse_key2:
 * Parse a line in config file, and set values to AddinTable.
 * For version two.
 **/
static int
parse_key2(char *buf, int f_automode)
{
	int ret = 0;
	AddinTable *at;
	int col = 0; /* column */
	char *ptr;
	
	at = new_addin_table();
	at->etc.f_automode = f_automode;
	
	for (ptr = strtok(buf, " \t"); ptr; ptr = strtok(NULL, " \t"), col++) {
		KeySym keysym;
		uint modmasks;
		char *cmd_path;
	
		switch (col) {
		case COL_KEY_ID:
			break;
		case COL_KEY_KEYSYM:
			keysym = XStringToKeysym(ptr);
			if (keysym != NoSymbol) {
				at->keysym = keysym;
			} else {
				goto done;
			}
			break;
		case COL_KEY_MODMASKS:
			modmasks = str2modmasks(ptr);
			if (modmasks != ~0) {
				at->modmasks = modmasks;
			} else {
				goto done;
			}
			break;
		case COL_KEY_USECACHE:
			at->etc.f_use_cache = determine_use_cache(ptr);
			break;
		case COL_KEY_NLINES:
			at->etc.max_nlines = atoi(ptr);
			break;
		case COL_KEY_BUFSIZ:
			at->etc.max_bufsiz = atoi(ptr);
			break;
		case COL_KEY_WIDTH:
			at->etc.max_width = atoi(ptr);
			break;

		case COL_KEY_COMMAND:	/*case COL_AUTOKEY_INTERVAL */
			if (f_automode) {	/* COL_AUTOKEY_INTERVAL */
				at->etc.interval = atoi(ptr);
			} else {			/* COL_KEY_COMMAND */
				cmd_path = get_cmd_path(ptr);
				if (cmd_path) {
					at->cmd.cmd_path = cmd_path;
					at->cmd.argv = xrealloc(at->cmd.argv, sizeof(char*) * (at->cmd.argc + 1));
					at->cmd.argv[at->cmd.argc] = strdup(ptr);/* the first argument is command name itself. */
					at->cmd.argc++;
					ret = 1;
				} else {
					goto done;
				}
			}
			break;
		case COL_KEY_ARGUMENTS:	/* COL_AUTOKEY_COMMAND */
		default:
			if (f_automode && col == COL_AUTOKEY_COMMAND) {
				cmd_path = get_cmd_path(ptr);
				if (cmd_path) {
					at->cmd.cmd_path = cmd_path;
					at->cmd.argv = xrealloc(at->cmd.argv, sizeof(char*) * (at->cmd.argc + 1));
					at->cmd.argv[at->cmd.argc] = strdup(ptr);/* the first argument is command name itself. */
					at->cmd.argc++;
					ret = 1;
				} else {
					goto done;
				}
			} else {
				at->cmd.argv = xrealloc(at->cmd.argv, sizeof(char*) * (at->cmd.argc + 1));
				at->cmd.argv[at->cmd.argc] = strdup(ptr);
				at->cmd.argc++;
				break;
			}
		}
	}
	
 done:
	if (ret == 1) {
		at->cmd.argv = xrealloc(at->cmd.argv, sizeof(char*) * (at->cmd.argc + 1));
		at->cmd.argv[at->cmd.argc] = NULL;
		at->cmd.argc++;
		addins->atable = prepend_addin_table(addins->atable, at);
	} else 
		xfree(at);/* still memory leak */

	return ret;
}

static int
parse_key1(char *buf)
{
	int ret = 0;
	AddinTable *at;
	int col = 0; /* column */
	char *ptr;
	
	at = new_addin_table();
	
	for (ptr = strtok(buf, " \t"); ptr; ptr = strtok(NULL, " \t"), col++) {
		KeySym keysym;
		uint modmasks;
		char *cmd_path;
	
		switch (col) {
		case COL_KEY1_ID:
			break;
		case COL_KEY1_KEYSYM:
			keysym = XStringToKeysym(ptr);
			if (keysym != NoSymbol) {
				at->keysym = keysym;
			} else {
				goto done;
			}
			break;
		case COL_KEY1_MODMASKS:
			modmasks = str2modmasks(ptr);
			if (modmasks != ~0) {
				at->modmasks = modmasks;
			} else {
				goto done;
			}
			break;
		case COL_KEY1_COMMAND:
			cmd_path = get_cmd_path(ptr);
			if (cmd_path) {
				at->cmd.cmd_path = cmd_path;
				at->cmd.argv = xrealloc(at->cmd.argv, sizeof(char*) * (at->cmd.argc + 1));
				at->cmd.argv[at->cmd.argc] = strdup(ptr);/* the first argument is command name itself. */
				at->cmd.argc++;
				ret = 1;
			} else {
				goto done;
			}
			break;
		case COL_KEY1_ARGUMENTS:
		default:
			at->cmd.argv = xrealloc(at->cmd.argv, sizeof(char*) * (at->cmd.argc + 1));
			at->cmd.argv[at->cmd.argc] = strdup(ptr);
			at->cmd.argc++;
			break;
		}
	}
	
 done:
	if (ret == 1) {
		at->cmd.argv = xrealloc(at->cmd.argv, sizeof(char*) * (at->cmd.argc + 1));
		at->cmd.argv[at->cmd.argc] = NULL;
		at->cmd.argc++;
		addins->atable = prepend_addin_table(addins->atable, at);
	} else 
		xfree(at);/* still memory leak */
	
	return ret;
}


/**
 * set_grab_keys:
 **/
static void
set_grab_keys(void)
{
	Window	rootwin;
	KeyCode	kcode;
	AddinTable *at;
#ifdef DEBUG
	int i;
#endif
	
	rootwin = DefaultRootWindow(dpy);
	for (at = addins->atable; at; at = at->next) {
		kcode = XKeysymToKeycode(dpy, at->keysym);
		XGrabKey(dpy, kcode, at->modmasks ? at->modmasks : AnyModifier, rootwin, False,
				 GrabModeAsync, GrabModeAsync);
#ifdef DEBUG
		fprintf(stderr, "Add-in command: %s ", at->cmd.cmd_path);
		for (i = 0; i < (at->cmd.argc - 1); i++) {
			fprintf(stderr, "%s ", at->cmd.argv[i]);			
		}
		fprintf(stderr, "\n");
#endif
	}
	if (addins->atable)
		XSelectInput(dpy, rootwin, KeyPressMask);
}

/**
 * unset_grab_keys:
 **/
static void
unset_grab_keys(void)
{
	Window	rootwin;
	KeyCode	kcode;
	AddinTable *at;
	
	rootwin = DefaultRootWindow(dpy);
	for (at = addins->atable; at; at = at->next) {
		kcode = XKeysymToKeycode(dpy, at->keysym);
		XUngrabKey(dpy, kcode, at->modmasks, rootwin);
	}
}


/**
 * str2modmasks:
 * Converter from string to Modmasks
 * E.g. "CS" => ControlMask | ShiftMask 
 **/
static uint
str2modmasks(const char *str)
{
	uint modmasks = 0;
	char c;

	while (*str) {
		c = tolower(*str);
		switch (c) {
		case 'c':
			modmasks |= ControlMask;
			break;
		case 'm':
			modmasks |= Mod1Mask;
			break;
		case 'l':
			modmasks |= LockMask;
			break;
		case 's':
			modmasks |= ShiftMask;
			break;
		case 'a':
			modmasks |= (ControlMask | ShiftMask | Mod1Mask | LockMask);
			break;
		case 'x':
			modmasks = 0;
			break;
		default:
			modmasks = ~0;	/* error */
			break;
		}
		str++;
	}
	
	return modmasks;
}


/**
 * get_cmd_path:
 * Search the path for the specified command.
 * Path name is allocated here.
 * Input:
 * const char *cmd; command name.
 * Output:
 * Return value; path name for the command. allocated here.
 **/
static char*
get_cmd_path(const char *cmd)
{
	char path[PATH_MAX+1];
	int i;
	
	/* Search the addin command in specified paths */
	for (i = 0; i < addins->num_path; i++) {
		sprintf(path, "%s/%s", addins->module_path[i], cmd);
		if (access(path, X_OK) == 0) {
			return strdup(path);
		}
	}

	fprintf(stderr, "%s: Not exists or no execute bit. Not registered.\n", cmd);
	return NULL;
}

/**
 * determine_id:
 * Return IdentSymbol by word.
 **/
static const struct {
	IdentSymbol id;
	const char *keyword;
	int keyword_lenb;
} ident_table[] = {
	{N_IDENT_VERSION, IDENT_VERSION, sizeof(IDENT_VERSION)-1},
	{N_IDENT_MODULE_PATH, IDENT_MODULE_PATH, sizeof(IDENT_MODULE_PATH)-1},
	{N_IDENT_KEY, IDENT_KEY, sizeof(IDENT_KEY)-1},
	{N_IDENT_AUTOKEY, IDENT_AUTOKEY, sizeof(IDENT_AUTOKEY)-1},
};
#define NUM_IDENT_TABLE		(sizeof(ident_table)/sizeof(ident_table[0]))

static IdentSymbol
determine_id(const char *word)
{
	int i;

	for (i = 0; i < NUM_IDENT_TABLE; i++) {
		if (strncasecmp(word, ident_table[i].keyword, ident_table[i].keyword_lenb) == 0) {
			return ident_table[i].id;
		}
	}
	return N_IDENT_UNKNOWN;
}

static int
determine_use_cache(const char *word)
{
	if (strcasecmp(word, "true") == 0 || strcasecmp(word, "1") == 0)
		return 1;
	else
		return 0;
}

/* for debug */
static void
show_parse_result(void)
{
	int i;
	AddinTable *at;
	
	printf("Config file path: %s\n\n", config_path);
	printf("Parse result:\n");
	printf("Version: %d\n\n", addins->version);
	for (i = 0; i < addins->num_path; i++) {
		printf("Module path: %s\n", addins->module_path[i]);
	}
	for (at = addins->atable; at; at = at->next) {
		int j;
		printf("\nAddin command:\n cmd_path=%s", at->cmd.cmd_path);
		for (j = 0; j < at->cmd.argc - 1; j++) {
			printf(" %s", at->cmd.argv[j]);
		}
		printf("\n max_nlines=%d", at->etc.max_nlines);
		printf("\n max_bufsiz=%d", at->etc.max_bufsiz);
		printf("\n max_width=%d", at->etc.max_width);
		printf("\n use_cache=%d", at->etc.f_use_cache);
		printf("\n automode=%d", at->etc.f_automode);
		printf("\n interval=%ld[millisecond]\n", at->etc.interval);
	}
}
