/* 
 *  Xfmedia Remote Plugin - Xfmedia Remote Control Plugin for Xfce4 Panel
 *  Copyright (C) 2005  Pasi Orovuo <pasi.ov@gmail.com>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#ifdef HAVE_STRING_H
#include <string.h>
#endif

#define DBUS_API_SUBJECT_TO_CHANGE
#include <dbus/dbus.h>
#include <dbus/dbus-glib.h>
#include <dbus/dbus-glib-lowlevel.h>

#include <glib.h>
#include <libxfce4util/libxfce4util.h>
#include <xfmedia-remote-common.h>
#include "xfmedia-remote.h"

#ifdef DBUS_USE_OLD_API
#define dbus_message_set_auto_start dbus_message_set_auto_activation
#endif

#define PLUGIN_NAME                     "xfmedia-remote-plugin"

struct _XfmediaRemote {
    gint                instance_no;
 
    /*DBusConnection      *connection;*/
    gchar               *name;
    gchar               *path;
};

static DBusConnection       *_dbus_connection = NULL;

static DBusHandlerResult
_dbus_message_filter_cb( DBusConnection *connection,
                         DBusMessage *message,
                         void *user_data )
{
    if ( dbus_message_is_signal( message,
#ifdef DBUS_USE_OLD_API
                                 DBUS_INTERFACE_ORG_FREEDESKTOP_LOCAL,
#else
                                 DBUS_INTERFACE_LOCAL,
#endif
                                 "Disconnected" ) ) {
        g_warning( PLUGIN_NAME ": DBus disconnected" );

        dbus_connection_unref( _dbus_connection );
        _dbus_connection = NULL;
    }
    return ( DBUS_HANDLER_RESULT_NOT_YET_HANDLED );
}

static DBusConnection *
_get_dbus_connection( void )
{
    if ( !_dbus_connection ) {
        DBusError       error;

        dbus_error_init( &error );
        
        _dbus_connection = dbus_bus_get( DBUS_BUS_SESSION, &error );
        if ( !_dbus_connection ) {
            g_warning( PLUGIN_NAME ": Failed to connect session bus: %s", error.message );
            dbus_error_free( &error );
        }
        else {
            dbus_connection_setup_with_g_main( _dbus_connection, NULL );
            dbus_connection_set_exit_on_disconnect( _dbus_connection, FALSE );

            if ( !dbus_connection_add_filter( _dbus_connection,
                                              _dbus_message_filter_cb,
                                              NULL,
                                              NULL ) ) {
                g_warning( PLUGIN_NAME ": Failed to set up message bus filter" );

                /*
                 * You may not close a shared connection. Connections created with 
                 * dbus_connection_open() or dbus_bus_get() are shared. These 
                 * connections are owned by libdbus, and applications should only 
                 * unref them, never close them. Applications can know it is safe 
                 * to unref these connections because libdbus will be holding a 
                 * reference as long as the connection is open. Thus, either the 
                 * connection is closed and it is OK to drop the last reference, 
                 * or the connection is open and the app knows it does not have 
                 * the last reference.
                dbus_connection_close( _dbus_connection );
                */
                dbus_connection_unref( _dbus_connection );

                _dbus_connection = NULL;
            }
        }
    }
    return ( _dbus_connection );
}

static gchar *
_get_xfmedia( DBusConnection *connection, gchar *bus_name )
{
    DBusMessage     *message, *reply;
    DBusError       error;
    gchar           *unique_name = NULL;

    message = dbus_message_new_method_call(
#ifdef DBUS_USE_OLD_API
                                            DBUS_SERVICE_ORG_FREEDESKTOP_DBUS,
                                            DBUS_PATH_ORG_FREEDESKTOP_DBUS,
                                            DBUS_INTERFACE_ORG_FREEDESKTOP_DBUS,
                                            "GetServiceOwner"
#else
                                            DBUS_SERVICE_DBUS,
                                            DBUS_PATH_DBUS,
                                            DBUS_INTERFACE_DBUS,
                                            "GetNameOwner"
#endif
                                            );
    if ( !message ) {
        return ( NULL );
    }
    
    dbus_message_set_auto_start( message, FALSE );
    
    if ( !dbus_message_append_args( message,
#ifdef DBUS_USE_OLD_API
                                    DBUS_TYPE_STRING, bus_name,
#else
                                    DBUS_TYPE_STRING, &bus_name,
#endif
                                    DBUS_TYPE_INVALID ) ) {
        dbus_message_unref( message );
        return ( NULL );
    }

    dbus_error_init( &error );

    reply = dbus_connection_send_with_reply_and_block( connection,
                                                       message,
                                                       -1,
                                                       &error );
    if ( !reply || dbus_message_get_type( reply ) == DBUS_MESSAGE_TYPE_ERROR ) {
        dbus_error_free( &error );
        
        if ( reply ) {
            dbus_message_unref( reply );
        }
        return ( NULL );
    }
    
    if ( dbus_message_get_args( reply,
                                &error,
                                DBUS_TYPE_STRING, &unique_name,
                                DBUS_TYPE_INVALID ) ) {
        unique_name = g_strdup( unique_name );
    }

    dbus_message_unref( message );
    dbus_message_unref( reply );

    return ( unique_name );
}

static int
_remote_xfmedia_connect( XfmediaRemote *remote, guint instance )
{
    DBusConnection      *connection;
    gchar               *name;
    gchar               bus_name[64];

    if ( !( connection = _get_dbus_connection() ) ) {
        return ( -1 );
    }

    g_snprintf( bus_name, sizeof( bus_name ) / sizeof( gchar ),
                XFMEDIA_DBUS_SERVICE_FMT, instance );

    if ( !( name = _get_xfmedia( connection, bus_name ) ) ) {
        /* Disconnect? */
        return ( -1 );
    }

    remote->instance_no     = instance;
    /*remote->connection      = connection;*/
    remote->name            = name;
    remote->path            = g_strdup_printf( XFMEDIA_DBUS_PATH_FMT, instance );

    return ( 0 );
}

static gboolean
_send_noparam_message( XfmediaRemote *remote, const gchar *method )
{
    DBusMessage     *message;
    gboolean        success = FALSE;

    g_return_val_if_fail( _dbus_connection != NULL, FALSE );

    message = dbus_message_new_method_call( remote->name,
                                            remote->path,
                                            XFMEDIA_DBUS_INTERFACE,
                                            method );
    if ( message ) {
        DBusMessage     *reply;
        DBusError       error;

        dbus_message_set_auto_start( message, FALSE );
        dbus_error_init( &error );

        reply = dbus_connection_send_with_reply_and_block( _dbus_connection,
                                                           message,
                                                           -1,
                                                           &error );
        if ( !reply ) {
            dbus_error_free( &error );
        }
        else {
            dbus_message_unref( reply );
            dbus_message_unref( message );
            success = TRUE;
        }
    }
    return ( success );
}

/* ====================== Public api ============================ */

void
xfmedia_remote_disconnect( void )
{
    if ( _dbus_connection ) {
        /*
        dbus_connection_close( _dbus_connection );
        */
        dbus_connection_unref( _dbus_connection );
        _dbus_connection = NULL;
    }
}

/*
 * XfmediaRemote *xfmedia_remote_new( guint );
 */
XfmediaRemote *
xfmedia_remote_new( guint instance_no )
{
    XfmediaRemote       remote, *p;

    if ( _remote_xfmedia_connect( &remote, instance_no ) < 0 ) {
        return ( NULL );
    }

    p = g_new0( XfmediaRemote, 1 );
    *p = remote;
    
    return ( p );
}

/*
 * void xfmedia_remote_free( XfmediaRemote * );
 */
void
xfmedia_remote_free( XfmediaRemote *remote )
{
    if ( remote->path ) {
        g_free( remote->path );
    }
    if ( remote->name ) {
        g_free( remote->name );
    }
    g_free( remote );
}

/*
 * gboolean xfmedia_remote_xfmedia_running( XfmediaRemote * );
 */
gboolean
xfmedia_remote_xfmedia_is_running( XfmediaRemote *remote )
{
    return ( _send_noparam_message( remote, XFMEDIA_REMOTE_IS_RUNNING ) );
}

/*
 * gboolean xfmedia_remote_song_previous( XfmediaRemote * );
 */
gboolean
xfmedia_remote_song_previous( XfmediaRemote *remote )
{
    return ( _send_noparam_message( remote, XFMEDIA_REMOTE_PREV ) );
}


/*
 * gboolean xfmedia_remote_song_play( XfmediaRemote *, gint );
 */
gboolean
xfmedia_remote_song_play( XfmediaRemote *remote, gint index )
{
    DBusMessage         *message;
    gboolean            success = FALSE;

    g_return_val_if_fail( _dbus_connection != NULL, FALSE );

    message = dbus_message_new_method_call( remote->name,
                                            remote->path,
                                            XFMEDIA_DBUS_INTERFACE,
                                            XFMEDIA_REMOTE_PLAY );
    if ( message ) {
        dbus_message_set_auto_start( message, FALSE );

        if ( dbus_message_append_args( message,
#ifdef DBUS_USE_OLD_API
                                       DBUS_TYPE_INT32, index,
#else
                                       DBUS_TYPE_INT32, &index,
#endif
                                       DBUS_TYPE_INVALID ) ) {
            DBusMessage     *reply;
            DBusError       error;

            dbus_error_init( &error );

            reply = dbus_connection_send_with_reply_and_block( _dbus_connection,
                                                               message,
                                                               -1,
                                                               &error );
            if ( reply ) {
                if ( dbus_message_get_type( reply ) != DBUS_MESSAGE_TYPE_ERROR ) {
                    success = TRUE;
                }
                dbus_message_unref( reply );
            }
            else {
                dbus_error_free( &error );
            }
        }
        dbus_message_unref( message );
    }
    return ( success );
}

/*
 * gboolean xfmedia_remote_song_pause( XfmediaRemote * );
 */
gboolean
xfmedia_remote_song_pause( XfmediaRemote *remote )
{
    return ( _send_noparam_message( remote, XFMEDIA_REMOTE_PAUSE ) );
}

/*
 * gboolean xfmedia_remote_song_stop( XfmediaRemote * );
 */
gboolean
xfmedia_remote_song_stop( XfmediaRemote *remote )
{
    return ( _send_noparam_message( remote, XFMEDIA_REMOTE_STOP ) );
}

/*
 * gboolean xfmedia_remote_song_next( XfmediaRemote * );
 */
gboolean
xfmedia_remote_song_next( XfmediaRemote *remote )
{
    return ( _send_noparam_message( remote, XFMEDIA_REMOTE_NEXT ) );
}

/*
 * gboolean xfmedia_remote_quit( XfmediaRemote * );
 */
gboolean
xfmedia_remote_quit( XfmediaRemote *remote )
{
    return ( _send_noparam_message( remote, XFMEDIA_REMOTE_QUIT ) );
}

/*
 * gboolean xfmedia_remote_playlist_add_file( XfmediaRemote *, const gchar *, gint );
 */
gboolean
xfmedia_remote_playlist_add_file( XfmediaRemote *remote,
                                  const gchar *fn,
                                  gint index )
{
    DBusMessage         *message;
    gboolean            success = FALSE;

    g_return_val_if_fail( _dbus_connection != NULL, FALSE );

    message = dbus_message_new_method_call( remote->name,
                                            remote->path,
                                            XFMEDIA_DBUS_INTERFACE,
                                            XFMEDIA_REMOTE_ADD_FILE );
    if ( message ) {
        gchar           *utf8_fn;
        
        dbus_message_set_auto_start( message, FALSE );

        utf8_fn = ( !g_utf8_validate( fn, -1, NULL )
                    ? g_filename_to_utf8( fn, -1, NULL, NULL, NULL )
                    : g_strdup( fn ) );

        if ( utf8_fn ) {
            if ( dbus_message_append_args( message,
#ifdef DBUS_USE_OLD_API
                                           DBUS_TYPE_STRING, utf8_fn,
                                           DBUS_TYPE_INT32, index,
#else
                                           DBUS_TYPE_STRING, &utf8_fn,
                                           DBUS_TYPE_INT32, &index,
#endif
                                           DBUS_TYPE_INVALID ) ) {
                DBusMessage         *reply;
                DBusError           error;

                dbus_error_init( &error );

                reply = dbus_connection_send_with_reply_and_block( _dbus_connection,
                                                                   message,
                                                                   -1,
                                                                   &error );
                if ( reply ) {
                    if ( dbus_message_get_args( reply,
                                                &error,
                                                DBUS_TYPE_INT32, &index,
                                                DBUS_TYPE_INVALID ) ) {
                        success = TRUE;
                    }
                    dbus_message_unref( reply );
                }

                if ( dbus_error_is_set( &error ) ) {
                    dbus_error_free( &error );
                }
            }

            g_free( utf8_fn );
        }

        dbus_message_unref( message );
    }
    return ( success );
}

/*
 * gboolean xfmedia_remote_playlist_remove_file( XfmediaRemote *, guint );
 */
gboolean
xfmedia_remote_playlist_remove_file( XfmediaRemote *remote, guint index )
{
    DBusMessage         *message;
    gboolean            success = FALSE;

    g_return_val_if_fail( _dbus_connection != NULL, FALSE );

    message = dbus_message_new_method_call( remote->name,
                                            remote->path,
                                            XFMEDIA_DBUS_INTERFACE,
                                            XFMEDIA_REMOTE_REMOVE_FILE );
    if ( message ) {
        dbus_message_set_auto_start( message, FALSE );

        if ( dbus_message_append_args( message,
#ifdef DBUS_USE_OLD_API
                                       DBUS_TYPE_UINT32, index,
#else
                                       DBUS_TYPE_UINT32, &index,
#endif
                                       DBUS_TYPE_INVALID ) ) {
            DBusMessage     *reply;
            DBusError       error;

            dbus_error_init( &error );

            reply = dbus_connection_send_with_reply_and_block( _dbus_connection,
                                                               message,
                                                               -1,
                                                               &error );
            if ( reply ) {
                if ( dbus_message_get_type( reply ) != DBUS_MESSAGE_TYPE_ERROR ) {
                    success = TRUE;
                }
                dbus_message_unref( reply );
            }
            else {
                dbus_error_free( &error );
            }
        }
        dbus_message_unref( message );
    }
    return ( success );
}

/*
 * gboolean xfmedia_remote_playlist_load( XfmediaRemote *, const gchar * );
 */
gboolean
xfmedia_remote_playlist_load( XfmediaRemote *remote, const gchar *fn )
{
    DBusMessage         *message;
    gboolean            success = FALSE;

    g_return_val_if_fail( _dbus_connection != NULL, FALSE );

    message = dbus_message_new_method_call( remote->name,
                                            remote->path,
                                            XFMEDIA_DBUS_INTERFACE,
                                            XFMEDIA_REMOTE_LOAD_PLAYLIST );
    if ( message ) {
        gchar           *utf8_fn;

        dbus_message_set_auto_start( message, FALSE );

        utf8_fn = ( !g_utf8_validate( fn, -1, NULL ) ?
                    g_filename_to_utf8( fn, -1, NULL, NULL, NULL ) :
                    g_strdup( fn ) );

        if ( utf8_fn ) {
            if ( dbus_message_append_args( message,
#ifdef DBUS_USE_OLD_API
                                           DBUS_TYPE_STRING, utf8_fn,
#else
                                           DBUS_TYPE_STRING, &utf8_fn,
#endif
                                           DBUS_TYPE_INVALID ) ) {
                DBusMessage         *reply;
                DBusError           error;

                dbus_error_init( &error );

                reply = dbus_connection_send_with_reply_and_block( _dbus_connection,
                                                                   message,
                                                                   -1,
                                                                   &error );
                if ( reply ) {
                    success = TRUE;

                    dbus_message_unref( reply );
                }
                else {
                    dbus_error_free( &error );
                }
            }
            g_free( utf8_fn );
        }
        dbus_message_unref( message );
    }
    return ( success );
}

/*
 * gboolean xfmedia_remote_playlist_save( XfmediaRemote *, const gchar * );
 */
gboolean
xfmedia_remote_playlist_save( XfmediaRemote *remote, const gchar *fn )
{
    DBusMessage         *message;
    gboolean            success = FALSE;

    g_return_val_if_fail( _dbus_connection != NULL, FALSE );

    message = dbus_message_new_method_call( remote->name,
                                            remote->path,
                                            XFMEDIA_DBUS_INTERFACE,
                                            XFMEDIA_REMOTE_SAVE_PLAYLIST );
    if ( message ) {
        gchar           *utf8_fn;
        
        dbus_message_set_auto_start( message, FALSE );

        utf8_fn = ( !g_utf8_validate( fn, -1, NULL ) ?
                    g_filename_to_utf8( fn, -1, NULL, NULL, NULL ) :
                    g_strdup( fn ) );

        if ( utf8_fn ) {
            if ( dbus_message_append_args( message,
#ifdef DBUS_USE_OLD_API
                                           DBUS_TYPE_STRING, utf8_fn,
#else
                                           DBUS_TYPE_STRING, &utf8_fn,
#endif
                                           DBUS_TYPE_INVALID ) ) {
                DBusMessage         *reply;
                DBusError           error;

                dbus_error_init( &error );

                reply = dbus_connection_send_with_reply_and_block( _dbus_connection,
                                                                   message,
                                                                   -1,
                                                                   &error );
                if ( reply ) {
                    success = TRUE;

                    dbus_message_unref( reply );
                }
                else {
                    dbus_error_free( &error );
                }
            }
            g_free( utf8_fn );
        }
        dbus_message_unref( message );
    }
    return ( success );
}

/*
 * gboolean xfmedia_remote_playlist_clear( XfmediaRemote * );
 */
gboolean
xfmedia_remote_playlist_clear( XfmediaRemote *remote )
{
    return ( _send_noparam_message( remote, XFMEDIA_REMOTE_CLEAR_PLAYLIST ) );
}

/*
 * gboolean xfmedia_remote_playlist_get( XfmediaRemote *,  );
 */
gboolean
xfmedia_remote_playlist_get( XfmediaRemote *remote,
                             gchar ***playlist,
                             guint *ntracks )
{
    DBusMessage         *message;
    gboolean            success = FALSE;
    
    g_return_val_if_fail( _dbus_connection != NULL, FALSE );

    message = dbus_message_new_method_call( remote->name,
                                            remote->path,
                                            XFMEDIA_DBUS_INTERFACE,
                                            XFMEDIA_REMOTE_GET_PLAYLIST );
    if ( message ) {
        DBusMessage         *reply;
        DBusError           error;

        dbus_message_set_auto_start( message, FALSE );
        dbus_error_init( &error );

        reply = dbus_connection_send_with_reply_and_block( _dbus_connection,
                                                           message,
                                                           -1,
                                                           &error );
        if ( reply ) {
            gchar           **tmp;
            guint           n;

            if ( dbus_message_get_args( reply,
                                        &error,
                                        DBUS_TYPE_ARRAY, DBUS_TYPE_STRING,
                                        &tmp, &n,
                                        DBUS_TYPE_INVALID ) ) {
                *playlist       = g_strdupv( tmp );
                *ntracks        = n;

                dbus_free_string_array( tmp );

                success = TRUE;
            }

            dbus_message_unref( reply );
        }

        if ( dbus_error_is_set( &error ) ) {
            dbus_error_free( &error );
        }

        dbus_message_unref( message );
    }
    return ( success );
}

/*
 * gboolean xfmedia_remote_playlist_entry_get( XfmediaRemote *, guint, gchar ** );
 */
/* Not currently needed */

/*
 * XfmediaTrackInfo *xfmedia_remote_now_playing( XfmediaRemote *);
 */
gboolean
xfmedia_remote_now_playing( XfmediaRemote *remote, gint *index, gchar **title, guint *length )
{
    DBusMessage         *message;
    gboolean            success = FALSE;

    g_return_val_if_fail( _dbus_connection != NULL, FALSE );

    message = dbus_message_new_method_call( remote->name,
                                            remote->path,
                                            XFMEDIA_DBUS_INTERFACE,
                                            XFMEDIA_REMOTE_NOW_PLAYING );
    if ( message ) {
        DBusMessage         *reply;
        DBusError           error;
        gint                i, len;
        gchar               *t, *fn;
        
        dbus_message_set_auto_start( message, FALSE );
        dbus_error_init( &error );

        reply = dbus_connection_send_with_reply_and_block( _dbus_connection,
                                                           message,
                                                           -1,
                                                           &error );
        if ( reply ) {
            if ( dbus_message_get_args( reply,
                                        &error,
                                        DBUS_TYPE_INT32, &i,
                                        DBUS_TYPE_STRING, &t,
                                        DBUS_TYPE_INT32, &len,
                                        DBUS_TYPE_STRING, &fn,
                                        DBUS_TYPE_INVALID ) ) {
                success = TRUE;

                *index      = i;
                *title      = g_strdup( t != NULL ? t : fn );
                *length     = len;

#ifdef DBUS_USE_OLD_API
                dbus_free( t );
                dbus_free( fn );
#endif
            }

            dbus_message_unref( reply );
        }
        else {
            if ( dbus_error_has_name( &error, XFMEDIA_DBUS_ERROR ) ) {
                *index = -1;
                success = TRUE;
            }
        }

        if ( dbus_error_is_set( &error ) ) {
            dbus_error_free( &error );
        }
        
        dbus_message_unref( message );
    }
    return ( success );
}

