/*
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <stdio.h>
#include <stdlib.h>

#include "node.h"
#include "entry.h"
#include "preamble.h"
#include "strings.h"
#include "tag.h"
#include "comment.h"
#include "debug.h"
#include "strings.h"

struct entry *add_entry(struct node **node, char *type, char *key)
{
	_DEBUG(("add_entry"));
	if(*node == NULL) {
		struct node *new = malloc(sizeof(struct node));
		new->node_type = ENTRY_NODE;
		new->next = NULL;
		new->node = new_entry(type, key);
		*node = new;
		return new->node;
	} else {
		return add_entry(&((*node)->next), type, key);
	}
}

void add_preamble(struct node **node, char *value)
{
	_DEBUG(("add_preamble"));

	if(*node == NULL) {
		struct node *new = malloc(sizeof(struct node));
		new->node_type = PREAMBLE_NODE;
		new->next = NULL;
		new->node = new_preamble(value);
		*node = new;
	} else {
		add_preamble(&((*node)->next), value);
	}
}

void add_comment(struct node **node, char *value)
{
	_DEBUG(("add_comment"));
	if(*node == NULL) {
		struct node *new = malloc(sizeof(struct node));
		new->node_type = COMMENT_NODE;
		new->next = NULL;
		new->node = new_comment(value);
		*node = new;
	} else {
		add_comment(&((*node)->next), value);
	}
}

void add_string(struct node **node, char *variable, char *value)
{
	_DEBUG(("add_string"));
	if(*node == NULL) {
		struct node *new = malloc(sizeof(struct node));
		new->node_type = STRING_NODE;
		new->next = NULL;
		new->node = new_string(variable, value); 
		*node = new;
	} else {
		add_string(&((*node)->next), variable, value);
	}
}

void add_tag(struct entry *entry, char *name, char *value, int sense)
{
	_DEBUG(("add_tag"));
	new_tag(&(entry->tags), name, value, sense);
}

void remove_tags(struct entry *entry)
{
	_DEBUG(("remove_tags"));
	free_tags(entry->tags);
	entry->tags = NULL;
}

void print_list(struct node *node)
{
	_DEBUG(("print_list"));
	write_list(node, stdout);
}

void write_list(struct node *node, FILE *file)
{
	_DEBUG(("write_list"));
	struct node *n;
	for (n = node; n != NULL; n = n->next) {
		switch(n->node_type) {
		case ENTRY_NODE:
			write_entry(n->node, file);
			break;
		case PREAMBLE_NODE:
			write_preamble(n->node, file);
			break;
		case STRING_NODE:
			write_string(n->node, file);
			break;
		case COMMENT_NODE:
			write_comment(n->node, file);
			break;
		default:
			fprintf(stderr, "Error: unrecogniced node type %d\n", n->node_type);
			break;
		}
	}
}

void free_list(struct node *n)
{
	_DEBUG(("free_list"));
	if (n != NULL) {
		free_list(n->next);
		switch(n->node_type) {
		case ENTRY_NODE:
			free_entry(n->node);
			break;
		case PREAMBLE_NODE:
			free_preamble(n->node);
			break;
		case STRING_NODE:
			free_string(n->node);
			break;
		case COMMENT_NODE:
			free_comment(n->node);
			break;
		default:
			fprintf(stderr, "Error: unrecogniced node type %d\n", n->node_type);
			break;
		}
		free(n);
	}
}

void remove_node(struct node **nodes, struct node *n)
{
	_DEBUG(("remove_node"));
	if (*nodes != NULL && *nodes == n) {
		_DEBUG(("Node match"));
		*nodes = n->next;
		switch(n->node_type) { 
			case ENTRY_NODE: 				
				free_entry(n->node);
				break; 
			case PREAMBLE_NODE: 
				free_preamble(n->node); 
				break; 
			case STRING_NODE: 
				free_string(n->node); 
				break;
			case COMMENT_NODE:
				free_comment(n->node);
				break;
			default:
				fprintf(stderr, "Error: unrecogniced node type %d\n", n->node_type);
				break;
			}
		free(n);	
	} else {
		if (*nodes != NULL)
			remove_node(&(*nodes)->next, n);
		else
			_DEBUG(("Warning: Node match could not be found!"));
	}
}

char *get_tag_value(struct entry *e, char *name)
{
	_DEBUG(("get_tag_value"));
	return get_value(e->tags, name);
}
