/*
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <stdlib.h>
#include <string.h>
#include <config.h>
#include <libxfcegui4/libxfcegui4.h>

#include "entry_edit_dialog.h"
#include "entry.h"
#include "tag.h"
#include "debug.h"
#include "treeview.h"
#include "node.h"

#define RED 0
#define GREEN 1
#define WHITE 2

const char *label_types[] = {"Article", "Book", "Booklet", "Conference", "Inbook",
	"Incollection", "Inproceedings", "Manual", "Mastersthesis", "Misc",
	"PhdThesis", "Proceedings", "Techreport", "Unpublished"};

const char *label_entry[] = {"Address", "Annote", "Author", "Booktitle", "Chapter",
	"Crossref", "Edition", "Editor", "E-print", "HowPublished",
	"Institution", "Journal", "Key", "Month", "Note",
	"Number", "Organization", "Pages", "Publisher", "School",
	"Series", "Title", "Type", "URL", "Volume", "Year"};
	
gint combobox_type;

static char* get_last_name(char *name)
{
	int i, size = strlen(name);
	for(i = size; i != size; i--) {
		if (name[i] == ' ') {
			return &name[i + 1];
		}
	}
	return name;
}		

static void generate_keys(GtkListStore *completion_model, GtkTreeModel *author_model)
{
	_DEBUG(("generate_keys"));
	GtkTreeIter iter, comp_iter;
	gchar *tmp = NULL;
	/* Clear the old entries */
	gtk_list_store_clear(completion_model);

	if (gtk_tree_model_get_iter_first(author_model, &iter)) {
		do {
			gtk_tree_model_get (author_model, &iter, 0, &tmp, -1);
			gtk_list_store_append (completion_model, &comp_iter);
			gtk_list_store_set (completion_model, &comp_iter, 0, get_last_name(tmp), -1);
			g_free(tmp);
		} while (gtk_tree_model_iter_next(author_model, &iter));
	}
}

static gboolean cb_focus_in_event (GtkWidget *widget, GdkEventFocus *event, struct edit_entry *edit_entry)
{
	_DEBUG(("cb_in_focus_event"));

	generate_keys(GTK_LIST_STORE(gtk_entry_completion_get_model(gtk_entry_get_completion(GTK_ENTRY(widget)))), 
			gtk_tree_view_get_model(GTK_TREE_VIEW(edit_entry[AUTHOR].entry)));
	
	return FALSE;
}

static void set_color_base(GtkWidget *entry, int selected_color)
{
	_DEBUG(("set_color"));
	
	GdkColor color;
	
	if (selected_color == RED) {
		color.red = 0xe7ff;
		color.green = 0xb3ff;
		color.blue = 0xb3ff;	
	} else if (selected_color == GREEN) {
		color.red = 0xcdff;
		color.green = 0xe7ff;
		color.blue = 0xb3ff;
	} else if (selected_color == WHITE) {
		color.red = 0xffff;
		color.green = 0xffff;
		color.blue = 0xffff;
	}
	if (entry != NULL)
		gtk_widget_modify_base(entry, GTK_STATE_NORMAL, &color);
}

static int change_color_entry(GtkEntry *entry)
{
	_DEBUG(("change_color_entry"));
	if (strcmp(gtk_entry_get_text(entry), "") == 0) {
		set_color_base(GTK_WIDGET(entry), RED);
		return 0;
	} else {
		set_color_base(GTK_WIDGET(entry), GREEN);
		return 1;
	}
}

static int change_color_view(GtkTreeView *entry)
{
	_DEBUG(("change_color_view"));
	GtkTreeIter iter;
	gboolean is_empty = gtk_tree_model_get_iter_first(gtk_tree_view_get_model(entry), &iter);
	if (is_empty == FALSE) {
		set_color_base(GTK_WIDGET(entry), RED);
		return 0;
	} else {
		set_color_base(GTK_WIDGET(entry), GREEN);
		return 1;
	}
}

static int get_entry_pos(char *str)
{
	_DEBUG(("get_entry_pos"));
	int i;
	if(str != NULL) {
		for (i = 0; i < N_ENTRIES; i++) {
			if (strcasecmp(str, label_entry[i]) == 0)
				return i;
		}
	}
	return -1;
}

static void entry_colors(struct edit_entry *entries)
{
	_DEBUG(("entry_colors"));
	
	extern gint combobox_type;
	
	set_color_base(entries[get_entry_pos("author")].entry, WHITE);
	set_color_base(entries[get_entry_pos("editor")].entry, WHITE);
	set_color_base(entries[get_entry_pos("chapter")].entry, WHITE);
	set_color_base(entries[get_entry_pos("title")].entry, WHITE);
	set_color_base(entries[get_entry_pos("journal")].entry, WHITE);
	set_color_base(entries[get_entry_pos("year")].entry, WHITE);
	set_color_base(entries[get_entry_pos("publisher")].entry, WHITE);
	set_color_base(entries[get_entry_pos("booktitle")].entry, WHITE);
	set_color_base(entries[get_entry_pos("pages")].entry, WHITE);
	set_color_base(entries[get_entry_pos("school")].entry, WHITE);
	set_color_base(entries[get_entry_pos("institution")].entry, WHITE);
	set_color_base(entries[get_entry_pos("note")].entry, WHITE);
	
	switch(combobox_type) {
		case ARTICLE:
			_DEBUG(("Article"));
			change_color_view(GTK_TREE_VIEW(entries[get_entry_pos("author")].entry));
			change_color_entry(GTK_ENTRY(entries[get_entry_pos("title")].entry));
			change_color_entry(GTK_ENTRY(entries[get_entry_pos("journal")].entry));
			change_color_entry(GTK_ENTRY(entries[get_entry_pos("year")].entry));
			break;
		case BOOK:
			_DEBUG(("book"));
			if (!change_color_view(GTK_TREE_VIEW(entries[get_entry_pos("author")].entry)))
				if (change_color_view(GTK_TREE_VIEW(entries[get_entry_pos("editor")].entry)))
					set_color_base(entries[get_entry_pos("author")].entry, WHITE);
			change_color_entry(GTK_ENTRY(entries[get_entry_pos("title")].entry));
			change_color_entry(GTK_ENTRY(entries[get_entry_pos("publisher")].entry));
			change_color_entry(GTK_ENTRY(entries[get_entry_pos("year")].entry));
			break;
		case BOOKLET:
			_DEBUG(("booklet"));
			change_color_entry(GTK_ENTRY(entries[get_entry_pos("title")].entry));
			break;
		case CONFERENCE:
		case INCOLLECTION:
		case INPROCEEDINGS:
			_DEBUG(("conference/incollection/inproceedings"));
			change_color_view(GTK_TREE_VIEW(entries[get_entry_pos("author")].entry));
			change_color_entry(GTK_ENTRY(entries[get_entry_pos("title")].entry));
			change_color_entry(GTK_ENTRY(entries[get_entry_pos("booktitle")].entry));
			change_color_entry(GTK_ENTRY(entries[get_entry_pos("year")].entry));
			break;
		case INBOOK:
			_DEBUG(("inbook"));
			if (!change_color_view(GTK_TREE_VIEW(entries[get_entry_pos("author")].entry)))
				if (change_color_view(GTK_TREE_VIEW(entries[get_entry_pos("editor")].entry)))
					set_color_base(entries[get_entry_pos("author")].entry, WHITE);
			change_color_entry(GTK_ENTRY(entries[get_entry_pos("title")].entry));
			if (!change_color_entry(GTK_ENTRY(entries[get_entry_pos("chapter")].entry)))
				if (change_color_entry(GTK_ENTRY(entries[get_entry_pos("pages")].entry)))
					set_color_base(entries[get_entry_pos("chapter")].entry, WHITE);
			change_color_entry(GTK_ENTRY(entries[get_entry_pos("publisher")].entry));
			change_color_entry(GTK_ENTRY(entries[get_entry_pos("year")].entry));
			break;
		case MANUAL:
			_DEBUG(("manual"));
			change_color_entry(GTK_ENTRY(entries[get_entry_pos("title")].entry));
			break;
		case MASTERSTHESIS:
			_DEBUG(("mastersthesis"));
			change_color_view(GTK_TREE_VIEW(entries[get_entry_pos("author")].entry));
			change_color_entry(GTK_ENTRY(entries[get_entry_pos("title")].entry));
			change_color_entry(GTK_ENTRY(entries[get_entry_pos("school")].entry));
			change_color_entry(GTK_ENTRY(entries[get_entry_pos("year")].entry));
			break;
		case MISC:
			_DEBUG(("misc"));
			break;
		case PHDTHESIS:
			_DEBUG(("phdthesis"));
			change_color_entry(GTK_ENTRY(entries[get_entry_pos("title")].entry));
			change_color_entry(GTK_ENTRY(entries[get_entry_pos("year")].entry));
			break;
		case TECHREPORT:
			_DEBUG(("techreport"));
			change_color_view(GTK_TREE_VIEW(entries[get_entry_pos("author")].entry));
			change_color_entry(GTK_ENTRY(entries[get_entry_pos("title")].entry));
			change_color_entry(GTK_ENTRY(entries[get_entry_pos("institution")].entry));
			change_color_entry(GTK_ENTRY(entries[get_entry_pos("year")].entry));
			break;
		case UNPUBLISHED:
			_DEBUG(("unpublished"));
			change_color_view(GTK_TREE_VIEW(entries[get_entry_pos("author")].entry));
			change_color_entry(GTK_ENTRY(entries[get_entry_pos("title")].entry));
			change_color_entry(GTK_ENTRY(entries[get_entry_pos("note")].entry));
	}
}

static void entry_sensitivity(struct edit_entry *entries)
{
	_DEBUG(("entry_sensitivity"));
	
	extern gint combobox_type;
	
	/* These entries are not used by any of the standard types in BibTeX */
	//gtk_widget_set_sensitive(GTK_WIDGET(entries[get_entry_pos("url")].entry), FALSE);
	//gtk_widget_set_sensitive(GTK_WIDGET(entries[get_entry_pos("e-print")].entry), FALSE);
	//gtk_widget_set_sensitive(GTK_WIDGET(entries[get_entry_pos("crossref")].entry), FALSE);
	//gtk_widget_set_sensitive(GTK_WIDGET(entries[get_entry_pos("annote")].entry), FALSE);
	
	/* These entries are use by all types */
	//gtk_widget_set_sensitive(GTK_WIDGET(entries[get_entry_pos("title")].entry), TRUE);
	//gtk_widget_set_sensitive(GTK_WIDGET(entries[get_entry_pos("month")].entry), TRUE);
	//gtk_widget_set_sensitive(GTK_WIDGET(entries[get_entry_pos("year")].entry), TRUE);
	//gtk_widget_set_sensitive(GTK_WIDGET(entries[get_entry_pos("key")].entry), TRUE);
	//gtk_widget_set_sensitive(GTK_WIDGET(entries[get_entry_pos("note")].entry), TRUE);
	
	gtk_widget_set_sensitive(GTK_WIDGET(entries[get_entry_pos("author")].entry), combobox_type == PROCEEDINGS ? FALSE : TRUE);
	gtk_widget_set_sensitive(GTK_WIDGET(entries[get_entry_pos("editor")].entry), combobox_type == BOOK || combobox_type == CONFERENCE || combobox_type == INBOOK || combobox_type == INCOLLECTION || combobox_type == INPROCEEDINGS || combobox_type == PROCEEDINGS ? TRUE : FALSE);
	gtk_widget_set_sensitive(GTK_WIDGET(entries[get_entry_pos("institution")].entry), combobox_type == TECHREPORT || combobox_type == MASTERSTHESIS ? TRUE : FALSE);
	gtk_widget_set_sensitive(GTK_WIDGET(entries[get_entry_pos("organization")].entry), combobox_type == CONFERENCE || combobox_type == INCOLLECTION || combobox_type == INPROCEEDINGS || combobox_type == MANUAL || combobox_type == PROCEEDINGS ? TRUE : FALSE);
	gtk_widget_set_sensitive(GTK_WIDGET(entries[get_entry_pos("school")].entry), combobox_type == MASTERSTHESIS || combobox_type == PHDTHESIS ? TRUE : FALSE);
	gtk_widget_set_sensitive(GTK_WIDGET(entries[get_entry_pos("type")].entry), combobox_type == TECHREPORT ? TRUE : FALSE);
	gtk_widget_set_sensitive(GTK_WIDGET(entries[get_entry_pos("booktitle")].entry), combobox_type == CONFERENCE || combobox_type == INCOLLECTION || combobox_type == INPROCEEDINGS ? TRUE : FALSE);
	gtk_widget_set_sensitive(GTK_WIDGET(entries[get_entry_pos("chapter")].entry), combobox_type == INBOOK ? TRUE : FALSE);
	gtk_widget_set_sensitive(GTK_WIDGET(entries[get_entry_pos("edition")].entry), combobox_type == BOOK || combobox_type == INBOOK || combobox_type == MANUAL ? TRUE : FALSE);
	gtk_widget_set_sensitive(GTK_WIDGET(entries[get_entry_pos("number")].entry), combobox_type == ARTICLE || combobox_type == TECHREPORT ? TRUE : FALSE);
	gtk_widget_set_sensitive(GTK_WIDGET(entries[get_entry_pos("pages")].entry), combobox_type == ARTICLE || combobox_type == CONFERENCE || combobox_type == INBOOK || combobox_type == INCOLLECTION || combobox_type == INPROCEEDINGS ? TRUE : FALSE);
	gtk_widget_set_sensitive(GTK_WIDGET(entries[get_entry_pos("volume")].entry), combobox_type == ARTICLE || combobox_type == BOOK || combobox_type == INBOOK ? TRUE : FALSE);
	gtk_widget_set_sensitive(GTK_WIDGET(entries[get_entry_pos("address")].entry), combobox_type == ARTICLE || combobox_type == MISC || combobox_type == UNPUBLISHED ? FALSE : TRUE);
	gtk_widget_set_sensitive(GTK_WIDGET(entries[get_entry_pos("howpublished")].entry), combobox_type == BOOKLET || combobox_type == MISC ? TRUE : FALSE);
	gtk_widget_set_sensitive(GTK_WIDGET(entries[get_entry_pos("journal")].entry), combobox_type == ARTICLE ? TRUE : FALSE);
	gtk_widget_set_sensitive(GTK_WIDGET(entries[get_entry_pos("publisher")].entry), combobox_type == BOOK || combobox_type == CONFERENCE || combobox_type == INBOOK || combobox_type == INCOLLECTION || combobox_type == INPROCEEDINGS || combobox_type == PROCEEDINGS ? TRUE : FALSE);
	gtk_widget_set_sensitive(GTK_WIDGET(entries[get_entry_pos("series")].entry), combobox_type == BOOK || combobox_type == INBOOK ? TRUE : FALSE);
}

static void combobox_changed_event(GtkComboBox *widget, struct edit_entry *edit_entries)
{
	_DEBUG(("combobox_changed_event"));
	extern gint combobox_type;
	combobox_type = gtk_combo_box_get_active(widget);
	entry_colors(edit_entries);
	entry_sensitivity(edit_entries);
}

static void entry_changed_event(GtkEntry *widget, struct edit_entry *edit_entries)
{
	_DEBUG(("entry_changed_event"));
	entry_colors(edit_entries);
}

static GtkWidget *create_hbox(GtkWidget **combobox, GtkWidget **key_entry, gpointer edit_entry)
{
	_DEBUG(("create_hbox"));
	GtkWidget *hbox;
	GtkEntryCompletion *completion;
	GtkListStore *store;
	int i;

	*key_entry = gtk_entry_new();
	gtk_entry_set_width_chars(GTK_ENTRY (*key_entry), 25);
	gtk_widget_set_tooltip_text(*key_entry, _("Key used when citing the publication"));

	store = gtk_list_store_new (1, G_TYPE_STRING);

	completion = gtk_entry_completion_new ();
	gtk_entry_set_completion (GTK_ENTRY (*key_entry), completion);
	g_object_unref(completion);
	gtk_entry_completion_set_model (completion, GTK_TREE_MODEL (store));
	g_object_unref(store);
	gtk_entry_completion_set_text_column (completion, 0);
	gtk_entry_completion_set_minimum_key_length(completion, 0);

    g_signal_connect(GTK_WIDGET(*key_entry), "focus-in-event", G_CALLBACK(cb_focus_in_event), edit_entry);


	*combobox = gtk_combo_box_new_text();
	for (i = 0; i < N_TYPES; i++)
		gtk_combo_box_append_text (GTK_COMBO_BOX (*combobox),label_types[i] );
	
	g_signal_connect (G_OBJECT (GTK_COMBO_BOX(*combobox)), "changed",
                      G_CALLBACK (combobox_changed_event), edit_entry);

	hbox = gtk_hbox_new(FALSE, 0);
	gtk_box_pack_start(GTK_BOX(hbox), gtk_label_new(_("Type: ")), TRUE, FALSE, 10);
	gtk_box_pack_start(GTK_BOX(hbox), *combobox, TRUE, FALSE, 10);
	gtk_box_pack_end(GTK_BOX(hbox), *key_entry, TRUE, FALSE, 10);
	gtk_box_pack_end(GTK_BOX(hbox), gtk_label_new(_("Key: ")), TRUE, FALSE,10);
	
	return hbox;
}

static int get_type_pos(char *str)
{
	_DEBUG(("get_type_pos"));
	int i;
	if(str != NULL) {
		for (i = 0; i < N_TYPES; i++) {
			if (strcasecmp(str, label_types[i]) == 0)
				return i;
		}
	}
	return -1;
}

static void set_entry_text(GtkEntry *entry, char *text)
{
	_DEBUG(("set_entry_text"));
	if (text != NULL && entry != NULL)
		gtk_entry_set_text(entry, text);
}

static void add_author_event_handler (struct edit_entry *edit_entries)
{
	_DEBUG(("add_event_handler"));
	struct edit_entry *entry = (struct edit_entry *) &edit_entries[get_entry_pos("author")];
	GtkListStore *list = GTK_LIST_STORE(gtk_tree_view_get_model(GTK_TREE_VIEW(entry->entry)));
	GtkTreeIter iter;
	gtk_list_store_append(list, &iter);
	gtk_list_store_set(list, &iter, 0, "NewAuthor", -1);
	entry_colors(edit_entries);
}

static void remove_author_event_handler (struct edit_entry *edit_entries)
{
	_DEBUG(("remove_event_handler"));
	GtkTreeModel *model;
	GtkTreeIter iter;
	GtkTreeSelection *selection;
	
	struct edit_entry *entry = (struct edit_entry *) &edit_entries[get_entry_pos("author")];
	model = gtk_tree_view_get_model(GTK_TREE_VIEW(entry->entry));
	selection = gtk_tree_view_get_selection(GTK_TREE_VIEW(entry->entry));
	if (gtk_tree_selection_get_selected(selection, &model, &iter)) {
		gtk_list_store_remove(GTK_LIST_STORE(model), &iter);
		entry_colors(edit_entries);
	} else {
		_DEBUG(("No row selected"));
	}
}

static void add_editor_event_handler (struct edit_entry *edit_entries)
{
	_DEBUG(("add_event_handler"));
	struct edit_entry *entry = (struct edit_entry *) &edit_entries[get_entry_pos("editor")];
	GtkListStore *list = GTK_LIST_STORE(gtk_tree_view_get_model(GTK_TREE_VIEW(entry->entry)));
	GtkTreeIter iter;
	gtk_list_store_append(list, &iter);
	gtk_list_store_set(list, &iter, 0, "NewEditor", -1);
	entry_colors(edit_entries);
}

static void remove_editor_event_handler (struct edit_entry *edit_entries)
{
	_DEBUG(("remove_event_handler"));
	GtkTreeModel *model;
	GtkTreeIter iter;
	GtkTreeSelection *selection;
	
	struct edit_entry *entry = (struct edit_entry *) &edit_entries[get_entry_pos("editor")];
	model = gtk_tree_view_get_model(GTK_TREE_VIEW(entry->entry));
	selection = gtk_tree_view_get_selection(GTK_TREE_VIEW(entry->entry));
	if (gtk_tree_selection_get_selected(selection, &model, &iter)) {
		gtk_list_store_remove(GTK_LIST_STORE(model), &iter);
		entry_colors(edit_entries);
	} else {
		_DEBUG(("No row selected"));
	}
}

static void set_list_text (GtkListStore *list, char *str)
{
	_DEBUG(("set_list_text"));
	GtkTreeIter iter;
	int n, max, len;
	char *prev, *tmp;
	
	max = strlen(str);
	for(n = 0, prev = str; n < max; n++) {
		if (str[n] == ' ' && str[n+1] == '\\' && str[n+2] == 'a' && str[n+3] == 'n' && str[n+4] == 'd' && str[n+5] == ' ') {
			len = &str[n] - prev;
			tmp = malloc(sizeof(char) * (len+1));
			strncpy(tmp, prev, len);
			tmp[len] = '\0';
			gtk_list_store_append(list, &iter);
			gtk_list_store_set(list, &iter, 0, tmp, -1);
			prev = &str[n+6];
		}
	}
	len = &str[n] - prev;
	tmp = malloc(sizeof(char) * (len+1));
	strncpy(tmp, prev, len);
	tmp[len] = '\0';
	gtk_list_store_append(list, &iter);
	gtk_list_store_set(list, &iter, 0, tmp, -1);
}

void show_entry_edit_dialog(struct xfbib *xfbib, struct entry *selected)
{
	_DEBUG(("show_entry_edit_dialog"));
	
	GtkWidget *dialog, *notebook, *hbox, *key_entry, *combobox, *vbox[5], *vboxes[2], *hboxes, *adds[2], *removes[2], *buttonboxes[2];
	struct edit_entry edit_entry[N_ENTRIES];
	int i, pos, n;
	struct tag *tags;
	char *type, *value, *key;
	const char *tooltips[] = {_("Address of publisher"),
	_("Annotation for annotated bibliography styles"), 
	_("Name(s) of the author(s), separated by 'and' if more than one"), 
	_("Title of the book, if only part of it is being cited"), 
	_("Chapter number"), 
	_("Citation key of the cross-referenced entry"), 
	_("Edition of the book (such as \"first\" or \"second\")"), 
	_("Name(s) of the editor(s), separated by 'and' if more than one"), 
	_("Specification of electronic publication"), 
	_("Publishing method if the method is nonstandard"), 
	_("Institution that was involved in the publishing"), 
	_("Journal or magazine in which the work was published"), 
	_("Hidden field used for specifying or overriding the alphabetical order of entries"), 
	_("Month of publication or creation if unpublished"), 
	_("Miscellaneous extra information"), 
	_("Number of journal, magazine, or tech-report"), 
	_("Sponsor of the conference"), 
	_("Page numbers separated by commas or double-hyphens"), 
	_("Name of publisher"), 
	_("School where thesis was written"), 
	_("Series of books in which the book was published"), 
	_("Title of the work"), 
	_("Type of technical report"), 
	_("Internet address"), 
	_("Number of the volume"), 
	_("Year of publication or creation if unpublished")};
	
	xfce_textdomain (PACKAGE, LOCALEDIR, "ISO-8859-1");

	dialog = xfce_titled_dialog_new_with_buttons ((selected == NULL) ? _("Add entry") : _("Edit entry"),
						NULL,
						GTK_DIALOG_MODAL,
						GTK_STOCK_OK, GTK_RESPONSE_OK,
						GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
						NULL);
	gtk_dialog_set_default_response (GTK_DIALOG (dialog), GTK_RESPONSE_OK);
	
	gtk_window_set_icon_name(GTK_WINDOW(dialog), "xfbib");

	hbox = create_hbox(&combobox, &key_entry, edit_entry);

	for (i = 0; i < 5; i++)
		vbox[i] = gtk_vbox_new(FALSE, 10);
		
	hboxes = gtk_hbox_new(FALSE, 10);

	for (i = 0; i < N_ENTRIES; i++)  {
		edit_entry[i].label = gtk_label_new(i == HOWPUBLISHED ? N_("How Published") : label_entry[i]);
		edit_entry[i].toggle = gtk_toggle_button_new_with_label ("aA");
		gtk_widget_set_tooltip_text (edit_entry[i].toggle, _("Toggle case sensitivity."));
		if (i != AUTHOR && i != EDITOR) {
			edit_entry[i].entry = gtk_entry_new();
			gtk_entry_set_width_chars(GTK_ENTRY (edit_entry[i].entry), 30);
			gtk_widget_set_tooltip_text (edit_entry[i].entry, tooltips[i]);
			edit_entry[i].hbox = gtk_hbox_new(FALSE, 10);
			gtk_box_pack_start(GTK_BOX(edit_entry[i].hbox), edit_entry[i].label, FALSE, FALSE, 10);
			gtk_box_pack_end(GTK_BOX(edit_entry[i].hbox), edit_entry[i].toggle, FALSE, TRUE, 10);
			gtk_box_pack_end(GTK_BOX(edit_entry[i].hbox), edit_entry[i].entry, FALSE, TRUE, 10);
			g_signal_connect (G_OBJECT (edit_entry[i].entry), "changed", G_CALLBACK (entry_changed_event), edit_entry);
		
			if (i == INSTITUTION || i == ORGANIZATION || i == SCHOOL || i == TITLE || i == TYPE)
				gtk_box_pack_start (GTK_BOX (vbox[0]), edit_entry[i].hbox, FALSE, FALSE, 2);
			else if (i == BOOKTITLE || i == CHAPTER || i == EDITION || i == MONTH || i == NUMBER || i == PAGES || i == VOLUME || i == YEAR)
				gtk_box_pack_start (GTK_BOX (vbox[2]), edit_entry[i].hbox, FALSE, FALSE, 2);
			else if (i == ADDRESS || i == HOWPUBLISHED || i == JOURNAL || i == PUBLISHER || i == SERIES)
				gtk_box_pack_start (GTK_BOX (vbox[3]), edit_entry[i].hbox, FALSE, FALSE, 2);
			else
				gtk_box_pack_start (GTK_BOX (vbox[4]), edit_entry[i].hbox, FALSE, FALSE, 2);
		} else {
			_DEBUG(("Author/Editor"));
			n = (i == EDITOR);
			edit_entry[i].entry = create_author_editor_treeview(xfbib, (char*) label_entry[i]);
			adds[n] = gtk_button_new_from_stock(GTK_STOCK_ADD);
			removes[n] = gtk_button_new_from_stock(GTK_STOCK_REMOVE);
			vboxes[n] = gtk_vbox_new(FALSE, 10);
			buttonboxes[n] = gtk_hbox_new(FALSE, 10);
			gtk_box_pack_start(GTK_BOX(vboxes[n]), edit_entry[i].entry, TRUE, TRUE, 10);
			gtk_box_pack_start(GTK_BOX(buttonboxes[n]), adds[n], TRUE, FALSE, 1);
			gtk_box_pack_start(GTK_BOX(buttonboxes[n]), removes[n], TRUE, FALSE, 1);
			if (i == AUTHOR) {
				g_signal_connect_swapped(G_OBJECT(adds[n]), "clicked", G_CALLBACK(add_author_event_handler), edit_entry);
				g_signal_connect_swapped(G_OBJECT(removes[n]), "clicked", G_CALLBACK(remove_author_event_handler), edit_entry);
			} else {
				g_signal_connect_swapped(G_OBJECT(adds[n]), "clicked", G_CALLBACK(add_editor_event_handler), edit_entry);
				g_signal_connect_swapped(G_OBJECT(removes[n]), "clicked", G_CALLBACK(remove_editor_event_handler), edit_entry);
			}
			gtk_box_pack_start(GTK_BOX(buttonboxes[n]), edit_entry[i].toggle, TRUE, FALSE, 1);
			gtk_box_pack_start(GTK_BOX(vboxes[n]), buttonboxes[n], FALSE, FALSE, 10);
			gtk_box_pack_start(GTK_BOX(hboxes), vboxes[n], FALSE, FALSE, 10);
		}
	}	

	notebook = gtk_notebook_new ();
	gtk_notebook_set_tab_pos (GTK_NOTEBOOK (notebook), GTK_POS_TOP);
	
	gtk_notebook_append_page (GTK_NOTEBOOK (notebook), vbox[0], gtk_label_new(_("Title")));
	gtk_notebook_append_page (GTK_NOTEBOOK (notebook), hboxes, gtk_label_new(_("Author/Editor")));
	gtk_notebook_append_page (GTK_NOTEBOOK (notebook), vbox[2], gtk_label_new(_("Publication")));	
	gtk_notebook_append_page (GTK_NOTEBOOK (notebook), vbox[3], gtk_label_new(_("Publisher")));	
	gtk_notebook_append_page (GTK_NOTEBOOK (notebook), vbox[4], gtk_label_new(_("Misc")));	

	gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dialog)->vbox), hbox, FALSE, FALSE, 10);
	gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dialog)->vbox), notebook, TRUE, TRUE, 10);

	gtk_widget_show_all(dialog);

	GtkTreeModel *model;

	if (selected != NULL) {
		_DEBUG(("Entry selected"));
		pos = get_type_pos(selected->type);
		if (pos == -1)
			_DEBUG(("Type equals -1"));
		else {
			gtk_combo_box_set_active(GTK_COMBO_BOX(combobox), pos);
			set_entry_text(GTK_ENTRY(key_entry), selected->key);
			for (tags = selected->tags; tags != NULL; tags = tags->next) {
				pos = get_entry_pos(tags->name);
				if (pos == -1) {
					_DEBUG(("Tag equals -1: Name: %s Value: %s", tags->name, tags->value));
				} else {
					if (pos == AUTHOR || pos == EDITOR)
						set_list_text(GTK_LIST_STORE(gtk_tree_view_get_model(GTK_TREE_VIEW(edit_entry[pos].entry))), strdup(tags->value));					
					else 
						set_entry_text(GTK_ENTRY(edit_entry[pos].entry), tags->value);
					if (tags->sense)
						gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (edit_entry[pos].toggle), TRUE);
					else
						gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (edit_entry[pos].toggle), FALSE);
				}
			}
		}
	}
	
	GtkTreeIter iter;
	char *name;
	
	while ( TRUE ) {
		if (gtk_dialog_run(GTK_DIALOG(dialog)) == GTK_RESPONSE_OK) {
			_DEBUG(("Response OK"));
			key = (char *) gtk_entry_get_text(GTK_ENTRY(key_entry));
			if ((type = gtk_combo_box_get_active_text(GTK_COMBO_BOX(combobox))) == NULL) {
				_DEBUG(("No type selected"));
				xfce_warn(_("No type was selected"));
				continue;
			} else if (strcmp(key, "") == 0) {
				_DEBUG(("No key selected"));
				xfce_warn(_("No key was selected"));
				continue;
			} else {
				if (selected == NULL) {
					_DEBUG(("Add new entry"));
					selected = add_entry(&xfbib->list, type,(char*) key);
				} else {
					_DEBUG(("Edit old entry"));
					selected->key = strdup(key);
					selected->type = strdup(type);
					remove_tags(selected);
				}
				for (i = 0; i < N_ENTRIES; i++) {
					if(GTK_WIDGET_IS_SENSITIVE(GTK_WIDGET(edit_entry[i].entry))) {
						_DEBUG(("Widget is sensitive: %u", i));
						if (i == AUTHOR || i == EDITOR) {
							model = gtk_tree_view_get_model(GTK_TREE_VIEW(edit_entry[i].entry));
							value = NULL;
							if (gtk_tree_model_get_iter_first(model, &iter)) {
								_DEBUG(("Pasting together Author/Editor"));
								do {
									gtk_tree_model_get (model, &iter, 0, &name, -1);
									if (value == NULL)
										value = strdup(name);
									else  {
										value = (char *) realloc(value, sizeof(char)*(strlen(value) + strlen(" \\and ") + strlen(name) + 1));
										strcat(strcat((char *) value, " \\and "), name);
									}
								} while (gtk_tree_model_iter_next(model, &iter));
							}
						} else
							value = (char *) gtk_entry_get_text(GTK_ENTRY(edit_entry[i].entry));
						_DEBUG(("Checking if empty: %s", value));
						if(value != NULL && strcmp(value, "") != 0) {
							_DEBUG(("Is empty"));
							fprintf(stderr, "label = %s\tvalue = %s\n", gtk_label_get_text(GTK_LABEL(edit_entry[i].label)), value);
							add_tag(selected,(char*) gtk_label_get_text(GTK_LABEL(edit_entry[i].label)), value,
									gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (edit_entry[i].toggle)));
						}
					}
				}
				update_treeview(xfbib);
				break;
			}
		} else {
			_DEBUG(("Response Cancel"));
			break;
		}
	}
	gtk_widget_destroy (dialog);
}
