#include <xen/xen.h>
#include "xen/x86_32.h"
#include "xen/x86_64.h"

#include "shared.h"
#include "list.h"
#include "xenner-emudev.h"

/* convert megabytes to pages and back */
#define MB_TO_PG(x) (x << 8)
#define PG_TO_MB(x) (x >> 8)

enum xenmode {
    XENMODE_32   = 32,
    XENMODE_PAE  = 36,
    XENMODE_64   = 64,
};

union xenguestcontext {
    struct vcpu_guest_context_x86_32  ct32;
    struct vcpu_guest_context_x86_64  ct64;
};

struct xennic {
    int                               configured;
    char                              mac[32];
    char                              ifname[32];
    char                              bridge[32];
    int                               fd;
};

struct xendisk {
    int                               configured;
    char                              diskimage[256];
    int                               cdrom;
};

struct xenvcpu {
    int                               id;
    struct xenvm                      *vm;
    pthread_t                         thread;
    int                               thread_id;

    /* kvm vcpu state */
    struct kvm_regs                   regs;
    struct kvm_sregs                  sregs;
    int                               regs_up2date;
    int                               sregs_up2date;
    int                               regs_dirty;
    int                               sregs_dirty;

    /* timekeeping */
    struct vcpu_time_info             *xentime;
    uint64_t                          systime;
};

struct xenclock {
    uint32_t wc_version;
    uint32_t wc_sec;
    uint32_t wc_nsec;
};

struct qemu_config {
    char   *path;
    char   *vnc;
    char   *kbd;
    char   *serial;
    char   *disk[16];
    char   *nic[16];
    int    upstream;
};

struct xenvm {
    /* kvm state */
    kvm_context_t   kvm;
    enum xenmode    mode;
    pthread_mutex_t biglock;
    pthread_t       mainthread;
    uint32_t        features;

    /* vm core file mapping */
    char       corefile[256];
    int        corefd;
    size_t     coresize;
    void       *coremap;

    /* memory */
    void       *memory;
    uint64_t   pg_total;
    uint64_t   pg_emu;
    uint64_t   pg_m2p;
    uint64_t   pg_guest;
    uint64_t   mfn_emu;
    uint64_t   mfn_m2p;
    uint64_t   mfn_guest;

    /* parameters */
    char       *kernel;
    char       *ramdisk;
    char       *cmdline;
    int        unlink_kernel;
    int        unlink_ramdisk;

    /* logging */
    int        debug;
    FILE       *logfile;
    int        nostderr;

    /* timekeeping */
    int                   tsc_shift;
    uint32_t              tsc_mul_frac;
    uint64_t              boot;

    /* xen emu state */
    struct elf_binary     *emu;
    uint64_t              emu_vs;
    uint64_t              emu_ve;
    uint64_t              emu_pgd_mfn;
    uint32_t              *m2p_32;
    uint64_t              *m2p_64;
    struct emudev_state   e;
    char                  eline[256];
    int                   elen;

    /* xen guest state */
    uint64_t              console_pfn;
    uint64_t              xenstore_pfn;
    union xenguestcontext boot_ctxt;

    /* virtual cpus */
    struct xenvcpu        vcpu[VCPUS_MAX];
    int                   vcpus;
    int                   vcpus_stopped;
    int                   ready_to_run;

    /* xen devices */
    struct xendisk        disks[4];
    struct xennic         nics[2];
    char                  bootdisk[256];
    char                  mon_buffer[256];
    int                   mon_connected;
    struct qemu_config    qemu;
    pid_t                 qemu_pid;

    /* event channels */
    int                   evtchnd;
    uint32_t              next_event;
    uint32_t              console_event;
    uint32_t              xenstore_event;

    /* ipc */
    uint32_t              domid;
    char                  name[64];
    char                  vm[48];
    struct xs_handle      *xenstore;
    struct list_head      callbacks;
};

typedef int (*xenner_callback)(struct xenvm *xen, int handle, void *cb_data);

struct xenner_callback {
    int                 fd;
    char                *name;
    void                *cb_data;
    xenner_callback     callback;
    struct list_head    list;
};

/* xenner.c */
extern char *search_path[];
extern int search_path_length;
int qemu_cleanup(struct xenvm *xen);

/* kvmbits.c */
void alive(void);
uint64_t get_systime(void);
void setup_timer(struct xenvcpu *vcpu);
void update_time(struct xenvcpu *vcpu);
int raise_event(struct xenvcpu *vcpu, int port);
void wait_event(struct xenvcpu *vcpu);
int kvmbits_signal_vcpus(struct xenvm *xen, int signal, int self);

int xenner_register_cb(struct xenvm *xen, char *name, int fd, void *data,
		       int (*callback)(struct xenvm *xen, int fd, void *data));

void print_guest_stack(struct xenvcpu *vcpu, uint64_t rsp);
void print_emu_stack(struct xenvm *xen, uint64_t rsp);
void vm_kill(struct xenvcpu *vcpu, const char *reason, uint64_t guest_esp);
void xenner_cleanup(struct xenvm *xen, int print_stats);

void need_regs(struct xenvcpu *vcpu);
void need_sregs(struct xenvcpu *vcpu);
void flush_regs(struct xenvcpu *vcpu);
void flush_sregs(struct xenvcpu *vcpu);

int kvmbits_features_enable(char *name, int state);
int kvmbits_features_check(struct xenvm *xen);
int kvmbits_init1(struct xenvm *xen);
int kvmbits_init2(struct xenvm *xen, int populate);
int kvmbits_mainloop(struct xenvm *xen);
int kvmbits_vcpus_stop(struct xenvm *xen);
int kvmbits_vcpus_cont(struct xenvm *xen);
int kvmbits_vcpus_ready(struct xenvm *xen);

/* build.c */
int xen_emu_map(struct xenvm *xen);
int domain_builder(struct xenvm *vm);
void setup_regs(struct xenvcpu *vcpu);

/* hypercalls.c */
int do_xen_hypercall(struct xenvcpu *vcpu, int nr);
void hypercall_stats(struct xenvm *xen);

/* cpuid.c */
void setup_cpuid(struct xenvm *xen);
void set_msr(struct xenvcpu *vcpu, uint32_t index, uint64_t data);
uint64_t get_msr(struct xenvcpu *vcpu, uint32_t index);
void print_msrs(struct xenvcpu *vcpu);

/* x86_emu.c */
int do_instr_emu(struct xenvm *xen);
int do_general_protection(struct xenvm *xen);
int do_page_fault(struct xenvm *xen);

/* debug.c */
int parse_asm(char *filename);
int do_debug_trap(struct xenvcpu *vcpu);

/* log.c */
/*
 * Message lewels:
 *  0 - FIXME.  Nuff sayed ;)
 *  1 - informational init stuff (which is rarely called
 *      and doesn't flood the screen when enabled).
 *  2 - default for debug messages.
 *  3 - debug messages causing *excessive* flood.
 */
#define d0printf(fmt, ...) dbgprintf(xen, 0, fmt, __VA_ARGS__)
#define d1printf(fmt, ...) dbgprintf(xen, 1, fmt, __VA_ARGS__)
#define d2printf(fmt, ...) dbgprintf(xen, 2, fmt, __VA_ARGS__)
#define d3printf(fmt, ...) dbgprintf(xen, 3, fmt, __VA_ARGS__)
void loginit(char *name);
void dbgprintf(struct xenvm *xen, int msglevel, const char *fmt, ...)
    __attribute__ ((format(printf, 3, 4)));
void logprintf(struct xenvm *xen, const char *fmt, ...)
    __attribute__ ((format(printf, 2, 3)));
void conemu_print(struct xenvm *xen, const char *buf);
void banner_print(struct xenvm *xen, const char *msg);
void section_print(struct xenvm *xen, const char *func, char *msg);

/* vmcore.c */
int vmcore_init1(struct xenvm *xen);
int vmcore_init2(struct xenvm *xen, int populate);
void vmcore_fini(struct xenvm *xen);
int vmcore_write_headers(struct xenvm *xen);

/* tsctest.c */
void tsc_calibrate(struct xenvm *xen, int *shift, uint32_t *mul_frac);

/* console.c */
int console_init(struct xenvm *xen);

/* xenstore.c */
int xenstore_init_early(struct xenvm *xen);
int xenstore_init_late(struct xenvm *xen);
int xenstore_shutdown(struct xenvm *xen, char *action);
void xenstore_fini(struct xenvm *xen);

/* evtchn.c */
int evtchn_init(struct xenvm *xen);
int evtchn_port(struct xenvm *xen, char *name);
int evtchn_notify(struct xenvm *xen, int port);
int evtchn_close(struct xenvm *xen, int port);
int evtchn_unmask(struct xenvm *xen, int port);
void evtchn_fini(struct xenvm *xen);

/* statistics.c */
void print_fault_stats(struct xenvm *xen, uint64_t *faults, uint64_t *last);
void print_hcall_stats(struct xenvm *xen, uint64_t *hcalls, uint64_t *last);
void print_event_stats(struct xenvm *xen, uint64_t *events, uint64_t *last,
		       char *enames);
